/*!
 * \file galileo_e1_tcp_connector_tracking_cc.h
 * \brief Interface of a TCP connector block based on code DLL + carrier PLL VEML (Very Early
 *  Minus Late) tracking block for Galileo E1 signals
 * \author David Pubill, 2012. dpubill(at)cttc.es
 *         Luis Esteve, 2012. luis(at)epsilon-formacion.com
 *         Javier Arribas, 2011. jarribas(at)cttc.es
 *
 * Code DLL + carrier PLL according to the algorithms described in:
 * K.Borre, D.M.Akos, N.Bertelsen, P.Rinder, and S.H.Jensen,
 * A Software-Defined GPS and Galileo Receiver. A Single-Frequency Approach,
 * Birkhauser, 2007
 *
 * -----------------------------------------------------------------------------
 *
 * Copyright (C) 2010-2020  (see AUTHORS file for a list of contributors)
 *
 * GNSS-SDR is a software defined Global Navigation
 *          Satellite Systems receiver
 *
 * This file is part of GNSS-SDR.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 *
 * -----------------------------------------------------------------------------
 */

#ifndef GNSS_SDR_GALILEO_E1_TCP_CONNECTOR_TRACKING_CC_H
#define GNSS_SDR_GALILEO_E1_TCP_CONNECTOR_TRACKING_CC_H

#include "cpu_multicorrelator.h"
#include "gnss_synchro.h"
#include "tcp_communication.h"
#include <gnuradio/block.h>
#include <volk_gnsssdr/volk_gnsssdr_alloc.h>  // for volk_gnsssdr::vector
#include <fstream>
#include <map>
#include <string>
#if GNURADIO_USES_STD_POINTERS
#include <memory>
#else
#include <boost/shared_ptr.hpp>
#endif


class Galileo_E1_Tcp_Connector_Tracking_cc;

#if GNURADIO_USES_STD_POINTERS
using galileo_e1_tcp_connector_tracking_cc_sptr = std::shared_ptr<Galileo_E1_Tcp_Connector_Tracking_cc>;
#else
using galileo_e1_tcp_connector_tracking_cc_sptr = boost::shared_ptr<Galileo_E1_Tcp_Connector_Tracking_cc>;
#endif

galileo_e1_tcp_connector_tracking_cc_sptr
galileo_e1_tcp_connector_make_tracking_cc(
    int64_t fs_in, uint32_t vector_length,
    bool dump,
    const std::string &dump_filename,
    float pll_bw_hz,
    float dll_bw_hz,
    float early_late_space_chips,
    float very_early_late_space_chips,
    size_t port_ch0);

/*!
 * \brief This class implements a code DLL + carrier PLL VEML (Very Early
 *  Minus Late) tracking block for Galileo E1 signals
 */
class Galileo_E1_Tcp_Connector_Tracking_cc : public gr::block
{
public:
    ~Galileo_E1_Tcp_Connector_Tracking_cc();

    void set_channel(uint32_t channel);
    void set_gnss_synchro(Gnss_Synchro *p_gnss_synchro);
    void start_tracking();

    int general_work(int noutput_items, gr_vector_int &ninput_items,
        gr_vector_const_void_star &input_items, gr_vector_void_star &output_items);

    void forecast(int noutput_items, gr_vector_int &ninput_items_required);

private:
    friend galileo_e1_tcp_connector_tracking_cc_sptr
    galileo_e1_tcp_connector_make_tracking_cc(
        int64_t fs_in, uint32_t vector_length,
        bool dump,
        const std::string &dump_filename,
        float pll_bw_hz,
        float dll_bw_hz,
        float early_late_space_chips,
        float very_early_late_space_chips,
        size_t port_ch0);

    Galileo_E1_Tcp_Connector_Tracking_cc(
        int64_t fs_in, uint32_t vector_length,
        bool dump,
        const std::string &dump_filename,
        float pll_bw_hz,
        float dll_bw_hz,
        float early_late_space_chips,
        float very_early_late_space_chips,
        size_t port_ch0);

    void update_local_code();

    void update_local_carrier();

    // tracking configuration vars
    uint32_t d_vector_length;
    bool d_dump;

    Gnss_Synchro *d_acquisition_gnss_synchro;
    uint32_t d_channel;

    int64_t d_fs_in;

    int32_t d_correlation_length_samples;
    int32_t d_n_correlator_taps;
    float d_early_late_spc_chips;
    float d_very_early_late_spc_chips;

    volk_gnsssdr::vector<gr_complex> d_ca_code;

    gr_complex *d_Very_Early;
    gr_complex *d_Early;
    gr_complex *d_Prompt;
    gr_complex *d_Late;
    gr_complex *d_Very_Late;

    // remaining code phase and carrier phase between tracking loops
    double d_rem_code_phase_samples;
    float d_next_rem_code_phase_samples;
    float d_rem_carr_phase_rad;

    // acquisition
    float d_acq_code_phase_samples;
    float d_acq_carrier_doppler_hz;

    // correlator
    volk_gnsssdr::vector<float> d_local_code_shift_chips;
    volk_gnsssdr::vector<gr_complex> d_correlator_outs;
    Cpu_Multicorrelator multicorrelator_cpu;

    // tracking vars
    double d_code_freq_chips;
    float d_carrier_doppler_hz;
    float d_acc_carrier_phase_rad;
    float d_acc_code_phase_secs;
    float d_code_phase_samples;
    size_t d_port_ch0;
    size_t d_port;
    int32_t d_listen_connection;
    float d_control_id;
    Tcp_Communication d_tcp_com;

    // PRN period in samples
    int32_t d_current_prn_length_samples;
    int32_t d_next_prn_length_samples;

    // processing samples counters
    uint64_t d_sample_counter;
    uint64_t d_acq_sample_stamp;

    // CN0 estimation and lock detector
    int32_t d_cn0_estimation_counter;
    volk_gnsssdr::vector<gr_complex> d_Prompt_buffer;
    float d_carrier_lock_test;
    float d_CN0_SNV_dB_Hz;
    float d_carrier_lock_threshold;
    int32_t d_carrier_lock_fail_counter;

    // control vars
    bool d_enable_tracking;
    bool d_pull_in;

    // file dump
    std::string d_dump_filename;
    std::ofstream d_dump_file;

    std::map<std::string, std::string> systemName;
    std::string sys;
};

#endif  // GNSS_SDR_GALILEO_E1_TCP_CONNECTOR_TRACKING_CC_H
