/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "cell.hh"
#include "format_spec.hh"
#include <iomanip>
#include <sstream>
#include <cstdlib>

using namespace table;
using namespace std;

//======================================  Default conversion format
static format_spec def_cell_format("p16");

//======================================  Cell constructor
Cell::Cell(void) 
    :  mType(kNull), mNumeric(0)
{
}

Cell::Cell(const string& str) 
  :  mType(kUnknown), mContent(str)
{
    setType();
}

Cell::Cell(const char* str) 
  :  mType(kUnknown), mContent(str)
{
    setType();
}

Cell::Cell(double num) {
    setNumeric(num);
}

//======================================  Append a string
void 
Cell::append(const string& x) {
    mContent += x;
    mType = kString;
}

Cell&
Cell::operator|=(const Cell& c) {
    mContent += c.mContent;
    mType = kString;
    return *this;
}

//======================================  Sum cells
Cell&
Cell::operator+=(const Cell& c) {
    if (mType == kNumeric && c.mType == kNumeric) {
        setNumeric(mNumeric + c.mNumeric);
    } else {
        mContent += c.mContent;
	mType = kString;
    }
    return *this;
}

//======================================  Cell data accessors
void
Cell::reformat(const format_spec& fmt) {
    if (mType == kNumeric) mContent = fmt.put_string(mNumeric);
}

//======================================  Set the type
void
Cell::setType(void) {
    if (mContent.empty()) {
	mType = kNull;
    } else {
        char* p = const_cast<char*>(mContent.c_str());
	mNumeric = strtod(p, &p);
	if (*p) mType = kString;
	else    mType = kNumeric;
    }
}

//======================================  Cell data accessors
double 
Cell::getNumeric(void) const {
    if (mType == kNull) return 0.0;
    if (mType != kNumeric) throw runtime_error("Data not numeric");
    return mNumeric;
}

//======================================  Cell data accessors
void
Cell::setNumeric(double val) {
    mContent = def_cell_format.put_string(val);
    mNumeric = val;
    mType    = kNumeric;
}
