/* -*- mode: c++; c-basic-offset: 4; -*- */
#include <string>
#include <vector>
class ParseLine;

/**  The VArg class is used to parse a command string containing a series of
  *  keywords followed by values. A typical use would be as follows:
  *  \verbatim
     VArg va << "--start" = "0" << "--end" = "100" << "--loop" = "yes";
     va.scan(pl);
     double start, stop;
     if (va.getBool("--loop")) {
          start = va.getDouble("--start");
          stop  = va.getDouble("--end");
     }
     \endverbatim
  *  \author John Zweizig (john.zweizig@ligo.org)
  *  \version 1.0; $Id"
  */
class VArg {
public:
    /**  Internal data type used to hold string lists.
     */
    typedef std::vector<std::string> string_vector;

    /**  Iterator for string list data type.
     */
    typedef unsigned int const_arg_iter;

public:
    /**  Construct a Varg and pre allocate spece in the keyword lists.
      *  \brief Constructor
      *  \param N number of arguments to preallocate.
      */
    VArg(int N=0);

    /**  Add a keyword to the scan list.
      *  \brief add a keyword.
      *  \param a keyword string.
      *  \return Reference to this VArg.
      */
    VArg& operator<<(const std::string& a);

    /**  Set the default value for the most recently defined keyword.
      *  \brief Set keyword default value.
      *  \param a Default value string.
      *  \return Reference to this VArg.
     */
    VArg& operator=(const std::string& a);

    /**  Get the string value for the specified keyword.
      *  \brief Get keyword value.
      *  \param a Keyword name.
      *  \return Constant reference to keyword value string.
      */
    const std::string& operator[](const std::string& a) const;

    /**  Start iterator of keyword list.
      *  \brief Start of keyword list.
      *  \return Constant iterator to start of keyword list.
      */
    const_arg_iter begin(void) const;

    /**  End iterator of keyword list.
      *  \brief End of keyword list.
      *  \return Constant iterator to end of keyword list.
      */
    const_arg_iter end(void) const;

    /**  Remove all keywords and values.
      *  \brief Erase keywords.
      */
    void erase(void);

    /**  Find a specified keyword in the keyword list. An exception is thrown
      *  if the keyword is not found.
      *  \brief Find a keyword.
      *  \exception std::runtime_error is thrown if the keyword is not found.
      *  \param a Keyword name.
      *  \return Constant iterator to the keyword  in the list.
      */
    const_arg_iter find(const std::string& a) const;

    /**  Find a specified keyword in the keyword list. findx() returns end()
      *  if the specified key is not found.
      *  \brief Find a keyword.
      *  \param a Keyword name.
      *  \return Constant iterator to the keyword  in the list.
      */
    const_arg_iter findx(const std::string& a) const;

    /**  Get a boolean keyword value.
      *  \brief Get a boolean keyword value.
      *  \param a Keyword name.
      *  \return false if the keyword was not specified or if the value 
      *          is "false".
      */
    bool getBool(const std::string& a) const;

    /**  Get a numeric keyword value.
      *  \brief Get a numeric keyword value.
      *  \param a Keyword name.
      *  \return Numeric keyword value or 0.0.
      */
    double getDouble(const std::string& a) const;

    /**  Scan a parsed command line for the specified keywords and values.
      *  \brief Scan a command line.
      *  \exception std::runtime_error is thrown if a command argument
      *             was not found in the keyword list.
      *  \param pl ParseLine containing a parsed command line.
      *  \param first First argument position to scan.
      *  \param last End of argument list.
      */
    void scan(const ParseLine& pl, int first=0, int last=0);


    /**  Scan a parsed command line for the specified keywords and values.
      *  \brief Scan a command line.
      *  \param pl ParseLine containing a parsed command line.
      *  \param first First argument position to scan.
      *  \param last End of argument list.
      *  \return Number of parsed arguments
      */
    int scanx(const ParseLine& pl, int first, int last);
private:
    string_vector mKeyList;
    string_vector mArgList;
    std::string   mScanError;
};

//======================================  Inline methods.
inline VArg::const_arg_iter 
VArg::begin(void) const {
    return 0;
}

inline VArg::const_arg_iter 
VArg::end(void) const {
    return mKeyList.size();
}

inline bool 
VArg::getBool(const std::string& a) const {
    return (*this)[a] == "true";
}
