/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    DMT Data processing base class implementation.
//
//======================================  Header files.
#include "MatchTrig.hh"

#include <iostream>
#include <fstream>
#include "Interval.hh"
#include "ParseLine.hh"
#include "SBTrigger.hh"
#include "Time.hh"
#include "TError.h"
#include "TFile.h"
#include "TROOT.h"
#include "TApplication.h"
#include "TCanvas.h"
#include "TStyle.h"
#include <stdexcept>
#include <string>
#include <stdio.h>

using FrameCPP::FrSimEvent;
using FrameCPP::FrameH;

using namespace std;

//=====================================  Main program
TROOT root("MatchTrig", "Graphical background monitor");

int 
main(int argc, const char* argv[]) {
    int bc(2);
    const char* bv[2] = {"MatchTrig", "-b"};
    TApplication theApp("App", &bc, const_cast<char**>(bv));
    gROOT->SetBatch(true);
    gErrorIgnoreLevel=1500;

    try {
      MatchTrig testPgm(argc,argv);
      if (testPgm.isActive()) testPgm.MainLoop();
    } catch (std::exception& e) {
      cerr << "Caught exception: " << e.what() << endl;
    }
    return 0;
}


double 
getParVal(const FrameCPP::FrSimEvent& e, const string& name) {
    const FrSimEvent::ParamList_type& l(e.GetParam() );
    for (FrSimEvent::ParamList_type::const_iterator i=l.begin(); i!=l.end(); i++) {
      if (i->first == name) return i->second;
    }
    throw std::runtime_error("Nonexistant parameter requested");
}


//======================================  MatchTrig constructor.
MatchTrig::MatchTrig(int argc, const char *argv[]) 
    : mActive(false), mEpoch(1.0), mOffset(0.0), mStride(1.0), 
      matchMode(best_time), mDebug(0), mTrigRead(0), mTrigSelect(0), 
      mTrigProc(0), mTrigMatch(0), mPlot("MatchTrig Plots"), maxDiff(0.2), 
      tLimit(1.0), dtmax(50.0), tLast(0)
{
    for (int i=0; i<hNumHist; i++) mHisto[i] = 0;
    double aMin=0.0;
    double aMax=100.0;
    double fMin=0.0;
    double fMax=4000.0;
    string tableFile, trigName, trigSubID;

    //----------------------------------  Parse the arguments
    bool syntax=false;
    for (int i=1 ; i<argc && argv ; i++) {
        string argi = argv[i];
        if (argi == "-infile") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -infile." << endl;
		syntax= true;
	        break;
	    }
	    mIn.addFile(argv[i]);
	} else if (argi == "-inlist") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -inlist." << endl;
		syntax= true;
	        break;
	    }
	    ParseLine pl(argv[i]);
	    if (!pl.isOpen()) {
	        cerr << "Unable to open file " << argv[i] << endl;
		continue;
	    }
	    while (pl.getLine() >= 0) {
	        if (pl.getCount() > 0) mIn.addFile(pl[0]);
	    }
	}
	else if (argi == "-mode") {
	    if (++i != argc) {
		string modename = argv[i];
		if (modename == "time") {
		    matchMode = best_time;
		} else if (modename == "snr") {
		    matchMode = best_snr;
		} else {
		    cerr << "Unrecognized matching mode: " << modename << endl;
		    syntax = true;
		}
	    }
	}
	else if (argi == "-debug") {
	    if (++i == argc) mDebug = 1;
	    else             mDebug = strtol(argv[i], 0, 0);
	    mIn.setDebug(mDebug);
	} else if (argi == "-ttable") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -ttable." << endl;
		syntax= true;
	        break;
	    }
	    tableFile = argv[i];
	} else if (argi == "-maxdiff") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -maxdiff." << endl;
		syntax= true;
	        break;
	    }
	    maxDiff = strtod(argv[i], 0);
	} else if (argi == "-dtmax") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -dtmax." << endl;
		syntax= true;
	        break;
	    }
	    dtmax = strtod(argv[i], 0);
	} else if (argi == "-tlimit") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -tlimit." << endl;
		syntax= true;
	        break;
	    }
	    tLimit = strtod(argv[i], 0);
	} else if (argi == "-amin") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -amin." << endl;
		syntax= true;
	        break;
	    }
	    aMin = strtod(argv[i], 0);
	} else if (argi == "-amax") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -amax." << endl;
		syntax= true;
	        break;
	    }
	    aMax = strtod(argv[i], 0);
	} else if (argi == "-fmin") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -fmin." << endl;
		syntax= true;
	        break;
	    }
	    fMin = strtod(argv[i], 0);
	} else if (argi == "-fmax") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -fmax." << endl;
		syntax= true;
	        break;
	    }
	    fMax = strtod(argv[i], 0);
	} else if (argi == "-histo") {
	    mHistOut = argv[++i];
	} else if (argi == "-par") {
	    string pname = argv[++i];
	    double xmin(0), xmax(1.0);
	    string::size_type l = pname.find(",");
	    if (l < pname.npos) {
	        pname.erase(l++);
		const char* p = argv[i] + l;
		xmin = strtod(p, (char**)&p);
		if (*p) syntax |= (*p++ != ',');
		xmax = strtod(p, (char**)&p);
		syntax |= *p;
	    }
	    if (syntax) {
	        cerr << "Syntax error in parameter specification: " << argv[i]
		     << endl;
		break;
	    }
	    if (Debug()) cout << "Histogram Parameter " << pname 
			      << " from " << xmin << " to " << xmax 
			      << endl;
	    mParList.push_back(ParHist(pname, xmin, xmax));
	} else if (argi == "-select") {
	    trigName = argv[++i];
	    string::size_type p = trigName.find("/");
	    if (p != trigName.npos) {
	        trigSubID = trigName.substr(p+1);
		trigName.erase(p);
	    }
	} else {
	    cerr << "Unrecognized argument: " << argi << endl;
	    syntax = true;
	    break;
	}
    }

    //----------------------------------  Get the table file
    if (mIn.refList().empty()) {
	const char* input=getenv("DMTINPUT");
	if (input && *input) {
	    mIn.addFile(input);
	} else {
	    cerr << "MatchTrig: No input frames specfied!" << endl;
	    syntax = true;
	}
    }

    //----------------------------------  Get the table file
    if (tableFile.empty()) {
        cout << "Syntax error in MatchTrig: No table specified." << endl;
	syntax = true;
    }

    //----------------------------------  Get the table file
    if (syntax) {
        cerr << "Error in MatchTrig command line. Syntax is:" << endl;
	cerr << "MatchTrig {-infile <files> | -inlist <list-file>} " 
	     << "-ttable <trigs> [-debug <debug-level>]" << endl;
	cerr << "          [-maxdiff <max-dt>] [-tlimit <histo-time-limit>] "
	     << "[-amin <ampl-min>] [-amax <ampl-max>]" << endl;
	cerr << "[-histo <hist-file>] [-par <par-name>,<min>,<max>]"
	     << "[-select <trigger-id>/<sub-id>]" << endl;
	finish();
	return;
    }

    //----------------------------------  Open the metaio parser.
    if (metaio.open(tableFile.c_str())) {
        cout << "Unable to open trigger table file: " << tableFile << endl;
	finish();
	return;
    } else {
        cout << "Opened trigger table file: " << tableFile << endl;
    }

    string tableName(metaio.getTableName());
    mTableType = kNone;
    if (tableName == "gds_triggergroup:gds_trigger:table") {
        mTableType = kGDSTrig;
    } else if (tableName == "sngl_burstgroup:sngl_burst:table") {
        mTableType = kSngl_burst;
    } else {
        cout << "Don't know how to unpack table ID: " << tableName << endl;
        finish();
        return;
    }

    //----------------------------------  Get the list of Triggers
    if (trigName  == "*") trigName.clear();
    if (trigSubID == "*") trigSubID.clear();
    while (!readTrigger()) {
	if (Debug() > 3) cout << "Read trigger: " << mTrigger.getID()  
			      << " subID: " << mTrigger.getSubID() 
			      << " Time: " << mTrigger.getPeakTime();
	mTrigRead++;
	if ((trigName.empty()  || trigName == mTrigger.getID()) &&
	    (trigSubID.empty() || trigSubID == mTrigger.getSubID())) {
	    trig_iter iter=mTrigList.end();
	    while (iter != mTrigList.begin()) {
	        --iter;
		if (mTrigger.getTime() > iter->getTrigTime()) {
		    iter++;
		    break;
		}
	    }
	    mTrigList.insert(iter, trigEvt(mTrigger));
	    if (Debug() > 3) cout << " Selected!" << endl;
	    mTrigSelect++;
	} else if (Debug() > 3) {
	    cout << " Rejected!" << endl;
	}
    }

    //----------------------------------  Check the trigger ordering
    if (Debug()) {
        cout <<  "Trigger input complete. " << mTrigList.size() 
	     << " triggers selected" << endl;
        cout << "Checking trigger ordering... ";
	for (unsigned int i=1; i<mTrigList.size(); ++i) {
	    if (mTrigList[i].getTrigTime() < mTrigList[i-1].getTrigTime()) {
	        cout << endl << "Trigger at " << mTrigList[i].getTrigTime() 
		     << "out of order";
	    }
	}
	cout << "Done!" << endl;
    }

    //----------------------------------  Close the trigger table I/O object
    metaio.close();
    if (mTrigList.empty()) {
        finish();
	return;
    }

    //----------------------------------  Read entire data.
    mIn.setTOCMode(false);

    //----------------------------------  Handle signals
    mTerm.add(SIGTERM);
    mTerm.add(SIGHUP);
    mTerm.add(SIGINT);

    //----------------------------------  Create a peck of histos
    mHisto[hEventAmpl]   = new TH1D("EventAmpl", "Event Amplitude",
				    100, aMin, aMax);
    mHisto[hEffvsAmpl]   = new TH1D("EffvsAmpl", "Ntrig vs Event Amplitude",
				    100, aMin, aMax);
    mHisto[hTrigAmpl]    = new TH1D("TrigAmpl", "Trigger Amplitude",
				    100, aMin, aMax);
    mHisto[hDeltaT]      = new TH1D("DeltaT", "Time difference",
				    100, -tLimit, tLimit);
    mHisto[hDeltaTvsA]   = new TH1D("DeltaTvaA", "Delta-t vs Amplitude",
				    100, aMin, aMax);
    mHisto[hDeltaTSqvsA] = new TH1D("DeltaTSqvsA", "Delta-t^2 vs Amplitude",
				    100, aMin, aMax);
    mHisto[hAmplvsA]     = new TH1D("AmplvsA", "Trigger Ampl vs Amplitude",
				    100, aMin, aMax);
    mHisto[hDeltaFvsA]   = new TH1D("DeltaFvsA", "Delta-F vs Amplitude",
				    100, aMin, aMax);
    mHisto[hDeltaFSqvsA] = new TH1D("DeltaFSqvsA", "Delta-F^2 vs Amplitude",
				    100, aMin, aMax);
    mHisto[hEventDt]     = new TH1D("EventDt", "Time to previous event",
				    100,  0.0, dtmax);
    mHisto[hEffvsDt]     = new TH1D("EffvsDt", "Efficiency vs Dt",
				    100,  0.0, dtmax);
    mHisto[hEffvsF]      = new TH1D("EffvsF", "Efficiency vs Frequency",
				    100, fMin, fMax);
    mHisto[hDeltaFvsF]   = new TH1D("DeltaFvsF", "Delta-F vs Frequency",
				    100, fMin, fMax);
    mHisto[hDeltaFSqvsF] = new TH1D("DeltaFSqvsF", "Delta-F^2 vs Frequency",
				    100, fMin, fMax);

    //----------------------------------  Open a frame stream (FrameCPP)
    mActive = true;
}

//======================================  inline utility functions
static void
wHist(TH1D* Hist) {
    if (Hist) Hist->Write(Hist->GetName());
}

//======================================  MatchTrig object destructor.
MatchTrig::~MatchTrig(void) {

    //----------------------------------  Loop over trigger list entris
    for (const_trig_iter i=mTrigList.begin(); i!=mTrigList.end(); i++) {
        if (!(i->isValid())) continue;
        double trigAmpl = i->getTrigAmpl();
        double Ampl     = i->getEvtAmpl();
	Time   evTime   = GDSTime(i->refEvent().GetGTime());
	Time   trigT    = i->refTrig().getPeakTime();
	if (Debug()) cout << "Event at: " << evTime << " (Ampl " 
			  << Ampl << ") matched with trigger at: "
			  << trigT << " (Ampl " << trigAmpl << ")" 
			  << endl;
	double tDiff = i->getTDiff();
	mHisto[hTrigAmpl]->Fill(trigAmpl);
	mHisto[hEffvsAmpl]->Fill(Ampl);
	mHisto[hDeltaTvsA]->Fill(Ampl, tDiff);
	mHisto[hDeltaTSqvsA]->Fill(Ampl, tDiff*tDiff);
	double dAmpl = (trigAmpl - Ampl)/Ampl;
	mHisto[hAmplvsA]->Fill(Ampl, dAmpl);
	double f0(0.0);
	double dF(0.0);
	try {
	    f0 = i->getEvtFreq();
	    mHisto[hEffvsF]->Fill(f0);
	    dF = i->getFDiff();
	    double dF2 = dF*dF;
	    mHisto[hDeltaFvsA]->Fill(Ampl, dF);
	    mHisto[hDeltaFSqvsA]->Fill(Ampl, dF2);	    
	    mHisto[hDeltaFvsF]->Fill(f0, dF);
	    mHisto[hDeltaFSqvsF]->Fill(f0, dF2);	    
	} catch (std::runtime_error& e) {
	    dF = 0.0;
	}
	mHisto[hEffvsDt]->Fill(i->getDtLast());
	for (par_iter j=mParList.begin(); j!=mParList.end(); j++) {
	    j->histTrig(i->refEvent(), tDiff, dAmpl, dF);
	}
    }

    //----------------------------------  DMT has terminater
    cout << "Trigger environment has terminated with Term/finish =" 
	 << bool(mTerm) << "/" << !mActive << endl;
    cout << "Event Statistics:" << endl;
    cout << "Triggers read:      " << mTrigRead   << endl;
    cout << "Triggers selected:  " << mTrigSelect << endl;
    cout << "Events processed:   " << mTrigProc   << endl;
    cout << "Events matched:     " << mTrigMatch  << endl;
 
    //----------------------------------  Open a root file.
    int nPar = mParList.size();
    if (!mHistOut.empty()) {
        TFile f(mHistOut.c_str(), "RECREATE", "Trigger efficiency Plots");
	for (int i=0; i<hNumHist; i++) wHist(mHisto[i]);
	for (int i=0; i<nPar; i++) mParList[i].writeHistos(f);
	f.Close();
    }

    //----------------------------------  Plot Plots
    mPlot.setFile("MatchTrig.ps");
    if (mHisto[hDeltaT]) {
	mPlot.hPlot(*mHisto[hDeltaT]);
    }
    if (mHisto[hEffvsAmpl] && mHisto[hEventAmpl]) {
	mPlot.effPlot(*mHisto[hEffvsAmpl], *mHisto[hEventAmpl], 
		      "Efficiency vs. Ampl");
    }
    if (mHisto[hEffvsAmpl] && mHisto[hDeltaTvsA]) {
	mPlot.avgPlot(*mHisto[hEffvsAmpl], *mHisto[hDeltaTvsA], 
		      *mHisto[hDeltaTSqvsA], "Time offset vs Amplitude");
	mPlot.resPlot(*mHisto[hEffvsAmpl], *mHisto[hDeltaTvsA], 
		      *mHisto[hDeltaTSqvsA], "Time resolution vs Amplitude");
    }
    if (mHisto[hEffvsAmpl] && mHisto[hDeltaFvsA] && 
	mHisto[hDeltaFvsA]->GetEntries() != 0) {
        mPlot.avgPlot(*mHisto[hEffvsAmpl], *mHisto[hDeltaFvsA], 
		      *mHisto[hDeltaFSqvsA], "Freq offset vs Amplitude");
	mPlot.resPlot(*mHisto[hEffvsAmpl], *mHisto[hDeltaFvsA], 
		      *mHisto[hDeltaFSqvsA], "Freq resolution vs Amplitude");
    }
    if (mHisto[hEffvsDt] && mHisto[hEventDt]) {
	mPlot.effPlot(*mHisto[hEffvsDt], *mHisto[hEventDt], 
		      "Efficiency vs. Dt");
    }
    if (mHisto[hEffvsF] && mHisto[hDeltaFvsF] && 
	mHisto[hDeltaFvsF]->GetEntries() != 0) {
        mPlot.avgPlot(*mHisto[hEffvsF], *mHisto[hDeltaFvsF], 
		      *mHisto[hDeltaFSqvsF], "Freq offset vs Frequency");
	mPlot.resPlot(*mHisto[hEffvsF], *mHisto[hDeltaFvsF], 
		      *mHisto[hDeltaFSqvsF], "Freq resolution vs Frequency");
    }
    for (int i=0; i<nPar; i++) {
        mParList[i].genPlots(mPlot);
    }

    //----------------------------------  Erase the histograms
    //for (int i=0; i<hNumHist; ++i) {
    //    delete mHisto[i];
    //    mHisto[i] = 0;
    //}

    //----------------------------------  Close the trigger table I/O object
    metaio.close();

    //----------------------------------  Close the Input file/partition.
    mIn.close();
}

//======================================  Main processing loop function.
void
MatchTrig::MainLoop(void) {
    trig_iter startIt = mTrigList.begin();
    
    //----------------------------------  Loop until terminated.
    while(mActive && !mTerm) {

        //------------------------------  Check for an attention interrupt.
        int rc = mIn.nextFrame();
	if (rc) {
	    mActive = false;
	    continue;
	}
	frameh_pointer frame(mIn.getFrame());

	//------------------------------  Loop over events
	for (FrameH::const_simEvent_iterator i=frame->RefSimEvent().begin();
	     i != frame->RefSimEvent().end() && !mTerm ; i++) {

	    //---------------------------  Get the event parameters.
	    double Ampl   = (*i)->GetAmplitude();
	    Time   tEvent = GDSTime((*i)->GetGTime());

	    //---------------------------  Histogram the event
	    mHisto[hEventAmpl]->Fill(Ampl);
	    mHisto[hEventDt]->Fill(double(tEvent-tLast));
	    for (par_iter j=mParList.begin(); j!=mParList.end(); j++) {
	        j->histEvent(**i);
	    }

	    //---------------------------  Find a good starting point
	    if (startIt == mTrigList.end() || 
		startIt->getTDiff(**i) > -tLimit) startIt=mTrigList.begin();
	    if (startIt == mTrigList.end()) continue;
	    while (startIt < mTrigList.end()-1 && 
		   (startIt+1)->getTDiff(**i) <= -tLimit) {
	        startIt++;
	    }

	    //---------------------------  Match a trigger
	    double tDiff  = tLimit;
	    double snrMax = 0;
	    trig_iter jSv = mTrigList.end();
	    switch (matchMode) {
	    case best_time:
		for (trig_iter j=startIt; j != mTrigList.end(); j++) {
		    double delta = j->getTDiff(**i);
		    if (delta >= tLimit) break;
		    if (fabs(delta) < fabs(tDiff)) {
			tDiff = delta;
			jSv   = j;
		    }
		}
		break;
	    case best_snr:
		for (trig_iter j=startIt; j != mTrigList.end(); j++) {
		    double delta = j->getTDiff(**i);
		    double snr = j->getSNR();
		    if (delta >= tLimit) break;
		    if (snr > snrMax) {
			snrMax = snr;
			tDiff = delta;
			jSv   = j;
		    }
		}
		break;
	    }
	    if (jSv != mTrigList.end()) {
		mHisto[hDeltaT]->Fill(tDiff);
	        if (fabs(tDiff) < maxDiff) {
		    if (!jSv->isValid()) {
		        jSv->setEvent(**i, tLast);
			mTrigMatch++;
		    } else if (fabs(tDiff) < fabs(jSv->getTDiff())) {
		        jSv->setEvent(**i, tLast);
		    }
		}
	    } else if (Debug() > 1) {
	        cout << "Unmatched event at: " << GDSTime((*i)->GetGTime())
		     << " comment: " << (*i)->GetComment() << endl; 
	    }
	    tLast = GDSTime((*i)->GetGTime());
	    mTrigProc++;
	}
    }
}

//======================================  Read/Process trigger.
int
MatchTrig::readTrigger(void) {

    int rc(0);
    try {
        //------------------------------  Get the next row; return if done
        rc = metaio.getRow();
	if (rc != 1) {
	    if (Debug()) cout << "readTrigger failed with " << rc 
			      << " from  MetaioGetRow." << endl;
	    return rc-1;
	}

	//------------------------------  Dispatch to the appropriate
	switch (mTableType) {
	case kGDSTrig:
	    rc = readGDSTrigger();
	    break;
	case kSngl_burst:
	    rc = readSingleBurst();
	    break;
	default:
	    rc = -1;
	}
    } catch (std::exception& e) {
        cerr << "Exception reading trigger: " << e.what() << endl;
	rc = -1;
    } catch (...) {
        cerr << "Unidentified exception caught in readTrigger" << endl;
	rc = -1;
    }
    return rc;
}

//======================================  Unpack trigger from gds_trigger table
int
MatchTrig::readGDSTrigger(void) {

    //----------------------------------  Get all specified fields.
    string name    = metaio.getString("name");
    string subtype = metaio.getString("subtype");
    //cout << "Reading trigger: " << name << ":" << subtype << endl;

    mTrigger = trig::TrigBase(name.c_str(), subtype.c_str(),
			      Time(metaio.getInt("start_time"),
				   metaio.getInt("start_time_ns")),
			      metaio.getFloat("duration"),
			      metaio.getFloat("size"));

    string ifo = metaio.getString("ifo");
    mTrigger.setIfos (ifo.c_str());
    mTrigger.setPriority(trig::TrigPrio(metaio.getInt("priority")));
    mTrigger.setDisposition(metaio.getInt("disposition"));
    mTrigger.setSignificance(metaio.getFloat("significance"));
    mTrigger.setFrequency(metaio.getFloat("frequency"));
    mTrigger.setData (metaio.getString("binarydata").c_str(),
		      metaio.getInt("binarydata_length"));
    mTrigger.setPeakOffset(metaio.getFloat("time_peak"));
    mTrigger.setAvgOffset(metaio.getFloat("time_average"));
    mTrigger.setTimeSigma(metaio.getFloat("time_sigma"));
    mTrigger.setFreqPeak(metaio.getFloat("freq_peak"));
    mTrigger.setFreqAvg(metaio.getFloat("freq_average"));
    mTrigger.setFreqSigma(metaio.getFloat("freq_sigma"));
    mTrigger.setNoisePower(metaio.getFloat("noise_power"));
    mTrigger.setSignalPower(metaio.getFloat("signal_power"));
    mTrigger.setPixelCount(metaio.getInt("pixel_count"));
    mTrigger.setConfidence(metaio.getFloat("confidence"));
    return 0;
}

//======================================  Unpack event from sngl_burst table
int
MatchTrig::readSingleBurst(void) {

    //----------------------------------  Get all specified fields.
    string search  = metaio.getString("search");
    string channel = metaio.getString("channel");
    Time   start(metaio.getInt("start_time"), metaio.getInt("start_time_ns"));
    Time   peak(metaio.getInt("peak_time"), metaio.getInt("peak_time_ns"));

    trig::SBTrigger sb(search, channel, start,
		       metaio.getFloat("duration"),
		       metaio.getFloat("central_freq"),
		       metaio.getFloat("bandwidth"),
		       metaio.getFloat("snr")
		       );

    string ifo = metaio.getString("ifo");
    sb.ifo(ifo);
    sb.peak_time(peak);
    mTrigger = static_cast<const trig::SBTrigger&>(sb);
    return 0;
}

//======================================  Construct a triggered event object
MatchTrig::trigEvt::trigEvt(const trig::TrigBase& t)
  : mTrig(t), mValidEvt(false)
{}

//======================================  Set the event descriptor
void 
MatchTrig::trigEvt::setEvent(const FrameCPP::FrSimEvent& e, const Time& tLast) {
    mEvent    = e;
    mValidEvt = true;
    tDiff     = getTDiff(e);
    dtLast    =  double(GDSTime(e.GetGTime()) - tLast);
}

//======================================  Get the event frequency
double 
MatchTrig::trigEvt::getEvtFreq(void) const {
    return getParVal(mEvent, "F");
}

//======================================  Get the frequency difference
double
MatchTrig::trigEvt::getFDiff(void) const {
    if (mValidEvt && (mTrig.getFrequency() != 0.0 || 
		      mTrig.getBandwidth()  != 0.0   )) {
	return  mTrig.getFrequency() - getEvtFreq();
    }
    return 0.0;
}

//======================================  Get the Dignal to Noise ratio
double
MatchTrig::trigEvt::getSNR(void) const {
    double signal = mTrig.getSignalPower();
    if (signal < 0) return 0;
    double noise  = mTrig.getNoisePower();
    if (noise <= 0) noise = 1.0;
    return sqrt(signal/noise);
}

//======================================  Define a parameter for histogramming
MatchTrig::ParHist::ParHist(const std::string& name, double xMin, double xMax)
  : _name(name), _xMin(xMin), _xMax(xMax)
{
    string hName  = string("Event") + _name;
    string hTitle = string("Distribution of ") + _name + " in Events";
    _EventHist = new TH1D(hName.c_str(), hTitle.c_str(), 100, _xMin, _xMax);

    hName     = string("Trig") + _name;
    hTitle    = string("Distribution of ") + _name + " in Triggered events";
    _TrigHist = new TH1D(hName.c_str(), hTitle.c_str(), 100, _xMin, _xMax);

    hName     = string("dA") + _name;
    hTitle    = string("Amplitude fractional offset versus") + _name;
    _dAHist   = new TH1D(hName.c_str(), hTitle.c_str(), 100, _xMin, _xMax);

    hName     = string("dt") + _name;
    hTitle    = string("Time offset versus ") + _name;
    _dtHist   = new TH1D(hName.c_str(), hTitle.c_str(), 100, _xMin, _xMax);


    hName     = string("dF") + _name;
    hTitle    = string("Freq offset versus ") + _name;
    _dFHist   = new TH1D(hName.c_str(), hTitle.c_str(), 100, _xMin, _xMax);

    hName     = string("sA") + _name;
    hTitle    = string("Amplitude resolution versus") + _name;
    _sAHist   = new TH1D(hName.c_str(), hTitle.c_str(), 100, _xMin, _xMax);

    hName     = string("st") + _name;
    hTitle    = string("Time resolution versus ") + _name;
    _stHist   = new TH1D(hName.c_str(), hTitle.c_str(), 100, _xMin, _xMax);

    hName     = string("sF") + _name;
    hTitle    = string("Freq resolution versus ") + _name;
    _sFHist   = new TH1D(hName.c_str(), hTitle.c_str(), 100, _xMin, _xMax);

}

//======================================  Destroy a histogrammed parameter
MatchTrig::ParHist::~ParHist(void) {
    delete _EventHist;
    _EventHist = 0;
    delete _TrigHist;
    _TrigHist  = 0;
    delete _dAHist;
    _dAHist    = 0;
    delete _dtHist;
    _dtHist    = 0;
    delete _dFHist;
    _dFHist    = 0;
    delete _sAHist;
    _sAHist    = 0;
    delete _stHist;
    _stHist    = 0;
    delete _sFHist;
    _sFHist    = 0;
}

//======================================  Copy a parameter histogram descriptor
MatchTrig::ParHist::ParHist(const ParHist& p) {
    *this = p;
}

MatchTrig::ParHist& 
MatchTrig::ParHist::operator=(const ParHist& p) {
    _name = p._name;
    _xMin = p._xMin;
    _xMax = p._xMax;
    _EventHist = new TH1D(*p._EventHist);
    _TrigHist  = new TH1D(*p._TrigHist);
    _dAHist    = new TH1D(*p._dAHist);
    _dtHist    = new TH1D(*p._dtHist);
    _dFHist    = new TH1D(*p._dFHist);
    _sAHist    = new TH1D(*p._sAHist);
    _stHist    = new TH1D(*p._stHist);
    _sFHist    = new TH1D(*p._sFHist);
    return *this;
}

//======================================  Get parameter value
double 
MatchTrig::ParHist::getParVal(const FrSimEvent& e) const {
    const FrSimEvent::ParamList_type& pars(e.GetParam());
    int N = pars.size();
    for (int i=0 ; i<N; i++) if (pars[i].first == _name) return pars[i].second;
    return 0.0;
}

//======================================  Histogram one trigger
void 
MatchTrig::ParHist::histEvent(const FrSimEvent& e) {
    double val = getParVal(e);
    _EventHist->Fill(val);
}

//======================================  Destroy a histogrammed parameter
void 
MatchTrig::ParHist::histTrig(const FrSimEvent& e, double dt, 
			    double dA, double dF) {
    double val = getParVal(e);
    _TrigHist->Fill(val);
    _dAHist->Fill(val, dA);
    _dtHist->Fill(val, dt);
    _sAHist->Fill(val, dA*dA);
    _stHist->Fill(val, dt*dt);
    if (dF != 0.0) {
        _dFHist->Fill(val, dF);
	_sFHist->Fill(val, dF*dF);
    }
}

//======================================  Write parameter histograms to TFile
void 
MatchTrig::ParHist::writeHistos(TFile& f) {
    wHist(_EventHist);
    wHist(_TrigHist);
    wHist(_dAHist);
    wHist(_dtHist);
    wHist(_dFHist);
    wHist(_sAHist);
    wHist(_stHist);
    wHist(_sFHist);
}

//======================================  Plot efficiency, dT and dA 
//                                        dependence on parameter
void 
MatchTrig::ParHist::genPlots(Plotter& c) {
    string hTitle("Efficiency vs. ");
    hTitle += _name;
    c.effPlot(*_TrigHist, *_EventHist, hTitle.c_str());
    c.avgPlot(*_TrigHist, *_dtHist, *_stHist, _dtHist->GetTitle());
    c.resPlot(*_TrigHist, *_dtHist, *_stHist, _stHist->GetTitle());
    if (_dFHist->GetEntries() != 0) {
        c.avgPlot(*_TrigHist, *_dFHist, *_sFHist, _dFHist->GetTitle());
	c.resPlot(*_TrigHist, *_dFHist, *_sFHist, _sFHist->GetTitle());
    }
    c.avgPlot(*_TrigHist, *_dAHist, *_sAHist, _dAHist->GetTitle());
    c.resPlot(*_TrigHist, *_dAHist, *_sAHist, _sAHist->GetTitle());
}

Plotter::Plotter(const char* title) 
  : mCanvas("cPlotter", title)
{
    gStyle->SetOptStat("nemro");
    gStyle->SetPadTickX(1);
    gStyle->SetPadTickY(1);
}

Plotter::~Plotter(void) {
    if (!mFile.empty()) mCanvas.Print((mFile+"]").c_str());
}

void 
Plotter::hPlot(const TH1D& h, const char* opt) {
    if (!opt) opt = "";
    gPad->SetTicks(1,1);
    h.DrawCopy(opt);
    update();
}

void 
Plotter::avgPlot(const TH1D& n, const TH1D& x, const TH1D& xx, const char* t) {
    TH1D avg(x);
    avg.Divide(&n);
    TH1D sig(getSig(n, x, xx));
    int N=avg.GetNbinsX();
    for (int i=1 ; i<=N ; i++) {
        double ni = n.GetBinContent(i);
	if (ni > 0.0) avg.SetBinError(i, sig.GetBinContent(i)/sqrt(ni));
	else          avg.SetBinError(i, 0.0);
    }
    avg.SetStats(0);
    if (t) avg.SetTitle(t);
    hPlot(avg, "E");
}

void 
Plotter::effPlot(const TH1D& n, const TH1D& d, const char* t) {
    TH1D eff(n);
    eff.Divide(&d);
    int N=eff.GetNbinsX();
    for (int i=1 ; i<=N ; i++) {
        double di = d.GetBinContent(i);
	if (di <= 0.0) di = 1.0;
	double ei = eff.GetBinContent(i);
	double err = sqrt(ei*(1-ei)/di);
	eff.SetBinError(i, err);
    }
    eff.SetStats(0);
    if (t) eff.SetTitle(t);
    hPlot(eff, "E");
}

TH1D
Plotter::getSig(const TH1D& n, const TH1D& x, const TH1D& xx) {
    TH1D sig(xx);
    sig.Divide(&n);
    int N=sig.GetNbinsX();
    for (int i=1 ; i<=N ; i++) {
        double ni = n.GetBinContent(i);
	if (ni <= 0.0) ni = 1.0;
	double xi  =   x.GetBinContent(i);
	double var = sig.GetBinContent(i) - xi*xi/(ni*ni);
	if (var > 0.0) sig.SetBinContent(i, sqrt(var));
	else           sig.SetBinContent(i, 0.0);
    }
    return sig;
}

void 
Plotter::resPlot(const TH1D& n, const TH1D& x, const TH1D& xx, const char* t) {
    TH1D sig(getSig(n, x, xx));
    int N=sig.GetNbinsX();
    for (int i=1 ; i<=N ; i++) {
        double ni = n.GetBinContent(i);
	if (ni > 0.0) sig.SetBinError(i, sig.GetBinContent(i)/sqrt(ni));
    }
    sig.SetStats(0);
    if (t) sig.SetTitle(t);
    hPlot(sig, "E");
}

void 
Plotter::update(void) {
    mCanvas.Update();
    if (!mFile.empty()) mCanvas.Print((mFile+"(").c_str());
}

void 
Plotter::setFile(const char* file) {
    mFile = file;
    mCanvas.Print((mFile+"[").c_str());
}
