#ifndef FRDUMP_HH
#define FRDUMP_HH

#include <string>
#include <list>
#include <vector>
#include <map>
#include <iosfwd>
#ifndef NOTGDS
#include "SigFlag.hh"
#endif

#define MaxID 256

class FrameF;

/**  \page FrameDump
  *  %FrameDump is a frame dump program that discovers the frames contents 
  *  from the structure header information inside the frame. The current 
  *  FrameDump version handles all frame formats up to version 8. Frame data 
  *  may be read either from a frame file or from a DMT shared memory 
  *  partition.
  *
  *  <H1> Running FrameDump</H1>
  *
  *  The command line syntax for FrDump is as follows:
  *  \begin{verbatim}
     FrameDump [-i[nfile] <file> | -p[artition] <part>] [+l <struct>] \
               [-chanlist] [-maxprt <nwds>] [-debug <level>]
     \end{verbatim}
  *  If the "+l" option is specified one or more times, only those structures
  *  named by "<struct>" will be listed. The structure specifiers are the 
  *  case-sensitive structure names as specified in the Frame specification,
  *  \e e.g. \c FrAdcData. \c \<nwds\> is the maximum number of word of data to 
  *  dump from an internal array.
  *
  *  @memo Frame dump utility
  *  @author J. Zweizig
  *  @version 1.2
  */


//======================================  Element structure
class Elem {
public:
  Elem(const std::string& Name, const std::string& Type);
  ~Elem(void);
  std::string getName(void) const;
  std::string getType(void) const;
private:
  std::string mName;
  std::string mType;
};

inline
Elem::Elem(const std::string& Name, const std::string& Type)
    : mName(Name), mType(Type) {}

inline
Elem::~Elem(void) {}

inline std::string 
Elem::getName(void) const {
    return mName;
}

inline std::string
Elem::getType(void) const {
    return mType;
}

//======================================  Structure class
class Struct {
public:
  typedef std::list<Elem> ElemList;
  typedef ElemList::const_iterator Elem_iter;
public:
  Struct(const std::string& Name, const std::string& Comment);
  Struct(void);
  ~Struct(void);
  void addElem(const Elem& E);
  std::string getName(void) const;
  Elem_iter begin(void) const;
  Elem_iter end(void) const;
private:
  std::string   mName;
  std::string   mComment;
  ElemList mList;
};

inline 
Struct::Struct(const std::string& Name, const std::string& Comment) 
  : mName(Name), mComment(Comment) {}

inline
Struct::Struct(void) {}

inline
Struct::~Struct(void) {}

inline std::string 
Struct::getName(void) const {
    return mName;
}

inline void
Struct::addElem(const Elem& E) {
    mList.push_back(E);
}

inline Struct::Elem_iter 
Struct::begin(void) const {
    return mList.begin();
}

inline Struct::Elem_iter 
Struct::end(void) const {
    return mList.end();
}

//======================================  Frame dump utility
class FrDump {
  public:
    FrDump(int argc, const char *argv[]);
    virtual ~FrDump();
    void ProcessFrame(void);
    void MainLoop(void);
    void finish(void);
    int  Debug(void) const;
    int  getNFrames(void) const;
    bool isOnline(void) const;
    bool inList(const std::string& x) const;
    void Dump(int len, int id, int inst, std::istream& In);
    void DumpH(void);
    void DumpS(void);
    void procSH(void);
    void procSE(void);
    void procAdc(void);
  private:
    bool    mActive;
#ifndef NOTGDS
    SigFlag mTerm;
#else
    bool    mTerm;
#endif
    std::streambuf* mBuffer;
    std::istream* mStream;
    int      mDebug;
    int      mNFrames;
    FrameF*  mFile;
    int      mLastID;
    int      mEOFID;
    int      mVectID;
    int      mAdcID;
    int      mProcID;
    typedef std::map<short,Struct> StructMap;
    StructMap mStructMap;
    bool    mOnline;
    std::vector<std::string> mLBlock;
    int     maxprt;
    int     mFlags;
};

inline void 
FrDump::finish(void) {
    mActive = false;
}

inline int  
FrDump::Debug(void) const {
    return mDebug;
}

inline int  
FrDump::getNFrames(void) const {
    return mNFrames;
}

inline bool 
FrDump::isOnline(void) const {
    return mOnline;
}

#endif // !defined(FRDUMP_HH)
