/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    FrDump Utility
//

//======================================  Header files.
#include "FrDump.hh"
#include "FrameLenTab.hh"
#include <time.h>
#include <iostream>
#include <iomanip>
#include <stdlib.h>
#include <fstream>
//#include <streambuf>
#include <stdio.h>
#include "FrameF.hh"
#ifndef NOTGDS
#include "iSMbuf.hh"
#endif
#include <ctype.h>

using namespace std;

//======================================  FrDump main function.
int main(int argc, const char *argv[]) {
    FrDump MyObject(argc,argv); 
    MyObject.MainLoop(); 
    return 0;}

//======================================  FrDump constructor.
FrDump::FrDump(int argc, const char *argv[]) 
  : mActive(false), mBuffer(0), mStream(0), mDebug(0), mNFrames(0),
    mFile(0), mLastID(-1), mEOFID(-1), mVectID(-1), mAdcID(-1), mProcID(-1),
    mOnline(false), maxprt(8), mFlags(0)
{
    const char*     partition = 0;
    const char*     file      = (const char*) 0;

    //---------------------------------  Parse the arguments
    bool syntax = false;
    for (int i=1 ; i<argc && argv ; i++) {
        string argi = argv[i];
        if (argi == "-infile" || argi == "-i") {
	    if (++i >= argc) continue;
	    file = argv[i];
#ifndef NOTGDS
	} else if (argi == "-partition" || argi == "-p") {
	    if (++i >= argc) continue;
	    partition = argv[i];
#endif
	} else if (argi == "-chanlist") {
	    mFlags |= 1;
	} else if (argi == "-debug") {
	    if (++i == argc) mDebug = 1;
	    else             mDebug = strtol(argv[i], 0, 0);
	} else if (argi == "-maxprt") {
	    maxprt = strtol(argv[++i], 0, 0);
	} else if (argi == "+l") {
	    mLBlock.push_back(string(argv[++i]));
	} else {
	    cerr << "Invalid argument: " << argi << endl;
	    syntax = true;
	    break;
	}
    }

    //----------------------------------  Default to LIGOSMPART partition.
    if (!file && !partition) {
#ifndef NOTGDS
	partition = getenv("LIGOSMPART");
	if (!partition) {
	    cerr << "Input file/partition not specified" << endl;
	    syntax = true;
	}
#else
	cerr << "Input file not specified" << endl;
	syntax = true;
#endif
    }

    if (syntax) {
	    cerr << "Syntax:" << endl;
#ifndef NOTGDS
	    cerr << "FrameDump -i[nfile] <file> " 
#else
	    cerr << "FrameDump {-i[nfile] <file> | -p[artition] <part>} " 
#endif
		 << "[+l <struct-name>] [-chanlist] \\" << endl;
	    cerr << "       [-maxprt <elements to print>] [-debug <debug-lvl>]"
		 << endl;
	    return;
    }

    if ((mFlags & 1) != 0) mLBlock.push_back(string("FrAdcData"));

#ifndef NOTGDS
    //---------------------------------  Handle signals
    mTerm.add(SIGTERM);
    mTerm.add(SIGHUP);
    mTerm.add(SIGINT);
#else
    mTerm = false;
#endif

    //---------------------------------  Open a frame stream (FrameCPP)
    if (file) {
        filebuf* fbuf = new filebuf;
	if (fbuf->open((char*)file, ios::in)) {
	    cout << "FrameDump: Opened file " << file << " for frame input." 
		 << endl;
	} else {
	    cerr << "FrameDump: Unable to open file " << file 
		 << " for frame input." << endl;
	    return;
	}
	mBuffer = fbuf;
    } else {
#ifdef NOTGDS
        cerr << "FrameDump: No file specified with -i or -infile " << endl;
#else
        iSMbuf* sbuf = new iSMbuf;
	if (sbuf->open((char*)partition, ios::in)) {
	    cout << "FrameDump: Opened partition " << partition
		 << " for frame input." << endl;
	    mOnline = true;
	} else {
	    cerr << "FrameDump: Unable to access partiton " << partition
		 << " for frame input." << endl;
	    return;
	}
	mBuffer = sbuf;
#endif
    }
    mStream = new istream(mBuffer);
    mFile = new FrameF(*mStream);
    mStructMap.insert(StructMap::value_type(1, Struct("FrSH", "")));
    mStructMap.insert(StructMap::value_type(2, Struct("FrSE", "")));
    mActive = true;
}

//======================================  FrDump object destructor.
FrDump::~FrDump(void) 
{
    //---------------------------------  Close the Input file/partition.
    if (mStream) delete mStream;
    if (mBuffer) delete mBuffer;

    //---------------------------------  Print statistics
    if (mDebug != 999) {
        cout << "Frame reading terminated after " << mNFrames 
	     << " frames." << endl;
    }
}

//======================================  Main processing loop function.
void
FrDump::MainLoop(void) 
{
    if (!mActive) return;

    //---------------------------------  make sure the header is red
    mFile->ReadHeader();

    //---------------------------------  Loop until terminated.
    while(mActive && !mTerm) {
        //-----------------------------  Process a frame.
	try {
	    ProcessFrame();
	    mNFrames++;
	} catch (exception& e) {
	    cout << "Caught exception: " << e.what() << endl;
	    mActive = false;
	}
	if (isOnline()) mFile->resetHeader();
    }
}

//======================================  Process asingle frame
void
FrDump::ProcessFrame(void) {

    //---------------------------------  Print version before first frame.
    if (!mNFrames) {
        cout << "Frame version is: " << mFile->getVersion() 
	     << "." << mFile->getPatch() << endl;
    }

    //---------------------------------  Loop over structures
    if (!mLBlock.empty() && inList("SH")) {
        mStructMap[1] = Struct("FrSH", "Frame Structure Header");
	mStructMap[1].addElem(Elem("name",    "STRING"));
	mStructMap[1].addElem(Elem("class",   "INT_2U"));
	mStructMap[1].addElem(Elem("comment", "STRING"));
    }
    if (!mLBlock.empty() && inList("SE")) {
        mStructMap[2] = Struct("FrSE", "Frame Structure Element");
	mStructMap[2].addElem(Elem("name",    "STRING"));
	mStructMap[2].addElem(Elem("class",   "STRING"));
	mStructMap[2].addElem(Elem("comment", "STRING"));
    }
    mLastID = 2;
    while (!mTerm) {
        mFile->NxStruct();
        short ID = mFile->getID();
	if (ID == 1) 	        procSH();
	else if (ID == 2) 	procSE();
	else if (ID == mAdcID)  procAdc();
	else if (ID == mProcID) procAdc();
	else                    DumpS();

	if (mFile->getID() == mEOFID) break;
    }
    if (mDebug) cout << "End of Frame " << getNFrames() << endl;
}

//======================================  Test whether the strcture is selected
bool
FrDump::inList(const string& name) const {
    if (mLBlock.empty()) return true;
    for (unsigned int i=0 ; i< mLBlock.size() ; i++) {
        if (mLBlock[i] == name) return true;
    }
    return false;
}

//======================================  Process a structure header
void
FrDump::procSH(void) {
    string Name    = mFile->getString();
    short  ID      = mFile->getShort();
    string Comment = mFile->getString();
    int    ChkSum  = 0;
    if (mFile->getVersion() >= 8) ChkSum = mFile->getInt();
    if (Name == "FrVect")           mVectID = ID;
    else if (Name == "FrAdcData")   mAdcID  = ID;
    else if (Name == "FrProcData")  mProcID = ID;
    else if (Name == "FrEndOfFile") mEOFID  = ID;
    mStructMap[ID] = Struct(Name, Comment);
    mLastID = ID;
    if (!mLBlock.empty() && inList("FrSH")) {
        DumpH();
	cout << "   name: "    << Name    << endl;
	cout << "   class: "   << ID      << endl;
	cout << "   comment: " << Comment << endl;
	cout << "   chkSum: "  << ChkSum  << endl;
    }
    if (mDebug) cout << "Structure type: " << ID << " is: " << Name << endl;
    return;
}

//======================================  Process an SE structure
void
FrDump::procSE(void) {
    string Name    = mFile->getString();
    string Class   = mFile->getString();
    string Comment = mFile->getString();
    int    ChkSum  = 0;
    if (mFile->getVersion() >= 8) ChkSum = mFile->getInt();
    mStructMap[mLastID].addElem(Elem(Name, Class));
    if (!mLBlock.empty() && inList("FrSE")) {
        DumpH();
	cout << "   name: "    << Name    << endl;
	cout << "   class: "   << Class   << endl;
	cout << "   comment: " << Comment << endl;
	cout << "   chkSum: "  << ChkSum  << endl;
    }
    return;
}

//======================================  Dump the structure header contents
void
FrDump::DumpH(void) {
    short ID = mFile->getID();
    cout << mStructMap[ID].getName() << " (Instance: " 
	 << mFile->getInstance() << ", Length: " << mFile->getLength() 
	 << ")" << endl;
}

//======================================  Dump a structure
void
FrDump::DumpS(void) {
    FrameLenTab lenMap;

    short ID = mFile->getID();
    const Struct& s = mStructMap[ID];
    if (!inList(s.getName())) return;
    DumpH();
    for (Struct::Elem_iter i=s.begin(); i != s.end(); i++) {
        string name = i->getName();
	string type = i->getType();
	long length = 1;
	string eltype;
	lenMap.parseType(type, eltype, length);
	if (mDebug > 9) cout << "Element name/type/length=" << name
			     << "/" << type << "/" << length << endl;
        cout << "   " << name << ":  ";
	if (length != 0) {
	    int nskip = length;
	    if (length > maxprt) length = maxprt;
	    nskip -= length;
	    if (eltype == "INT_4U") {
	        int val = mFile->getInt();
		cout << val;
		lenMap.addSym(name, val);
	        for (int j=1 ; j<length ; j++) cout << " " << mFile->getInt();
		for (int j=0 ; j<nskip  ; j++) mFile->getInt();
	    } else if (eltype == "INT_4S") {
	        int val = mFile->getInt();
		cout << val;
		lenMap.addSym(name, val);
	        for (int j=1 ; j<length ; j++) cout << " " << mFile->getInt();
		for (int j=0 ; j<nskip  ; j++) mFile->getInt();
	    } else if (eltype == "INT_2U") {
	        short val = mFile->getShort();
		cout << val;
		lenMap.addSym(name, val);
	        for (int j=1 ; j<length ; j++) cout << " " << mFile->getShort();
		for (int j=0 ; j<nskip  ; j++) mFile->getShort();
	    } else if (eltype == "INT_2S") {
	        short val = mFile->getShort();
		cout << val;
		lenMap.addSym(name, val);
	        for (int j=1 ; j<length ; j++) cout << " " << mFile->getShort();
		for (int j=0 ; j<nskip  ; j++) mFile->getShort();
	    } else if (eltype == "INT_8S") {
	        long val = mFile->getLong();
		cout << val;
		lenMap.addSym(name, val);
	        for (int j=1 ; j<length ; j++) cout << " " << mFile->getLong();
		for (int j=0 ; j<nskip  ; j++) mFile->getLong();
	    } else if (eltype == "INT_8U") {
	        long val = mFile->getLong();
		cout << val;
		lenMap.addSym(name, val);
	        for (int j=1 ; j<length ; j++) cout << " " << mFile->getLong();
		for (int j=0 ; j<nskip  ; j++) mFile->getLong();
	    } else if (eltype == "REAL_8") {
	        for (int j=0 ; j<length ; j++) cout <<mFile->getDouble()<< " ";
		for (int j=0 ; j<nskip  ; j++) mFile->getDouble();
	    } else if (eltype == "REAL_4") {
	        for (int j=0 ; j<length ; j++) cout << mFile->getFloat()<< " ";
		for (int j=0 ; j<nskip  ; j++) mFile->getFloat();
	    } else if (eltype == "STRING") {
	        for (int j=0 ; j<length ; j++) {
		    string temp = mFile->getString();
		    for (unsigned int k=0 ; k<temp.size() ; k++) {
		        if (!isprint(temp[k])) {
			    char x[4] = {'\\', '0', '0', '0'};
			    x[1] += (int(temp[k]) & 255) >> 6;
			    x[2] += (temp[k] >> 3) & 7;
			    x[3] += temp[k] & 7;
			    temp.replace(k, 1, x, 4);
			}
		    }
		    cout << "\"" << temp << "\" ";
		}
	        for (int j=0 ; j<nskip ; j++) mFile->getString();
	    } else if (eltype == "CHAR") {
	        char* temp = new char[length];
	        bool nprint(false);
	        for (int j=0 ; j<length ; j++) {
		    temp[j] = mFile->getChar();
		    nprint  |= !isprint(temp[j]) || ispunct(temp[j]);
		}
		if (nprint) {
		    for (int j=0 ; j<length ; j++) cout << int(temp[j]) << " ";
		} else {
		    cout << "'" << string(temp, length) << "'";
		}
		delete[] temp;
		mFile->Skip(nskip);
	    } else if (eltype.substr(0,10) == "PTR_STRUCT") {
                FrameF::StrLink link;
		mFile->getLink(link);
		if (link.ID || link.Instance) {
		    cout << mStructMap[link.ID].getName() << "["
			 << link.Instance << "]";
		} else {
		    cout << "NULL";
		}
	    } else {
	        cout << "Unidentified type: " << i->getType() << endl;
		return;
	    }
	    cout << dec << setw(0);
	}
	cout << endl;
	if (mFile->testOffset()) {
	    throw BadFile("Offset is outside of structure");
	}
    }
}

//======================================  Dump a structure
void
FrDump::procAdc(void) {
    if ((mFlags & 1) == 0) {
        DumpS();
    } else {
	if (mDebug) cout << "Found Data Structure" << endl;
        string chName;
	double sampleRate(0);
	FrameLenTab lenMap;

	short ID = mFile->getID();
	const Struct& s = mStructMap[ID];
	for (Struct::Elem_iter i=s.begin() ; i != s.end() ; i++) {
	    string name = i->getName();
	    string type = i->getType();
	    long length = 0;
	    string eltype;
	    lenMap.parseType(type, eltype, length);

	    if (name == "name") {
	        chName = mFile->getString();
	    } else if (name == "sampleRate") {
	        sampleRate = mFile->getDouble();
	    } else if (type == eltype) {
	        if (type == "INT_4U") {
		    int val = mFile->getInt();
		    lenMap.addSym(name, val);
		} else if (type == "INT_4S") {
		    int val = mFile->getInt();
		    lenMap.addSym(name, val);
		} else if (type == "INT_2U") {
		    short val = mFile->getShort();
		    lenMap.addSym(name, val);
		} else if (type == "INT_2S") {
		    short val = mFile->getShort();
		    lenMap.addSym(name, val);
		} else if (type == "INT_8S") {
		    long val = mFile->getLong();
		    lenMap.addSym(name, val);
		} else if (type == "INT_8U") {
		    long val = mFile->getLong();
		    lenMap.addSym(name, val);
		} else if (type == "REAL_8") {
		    mFile->getDouble();
		} else if (type == "REAL_4") {
		    mFile->getFloat();
		} else if (type == "STRING") {
		    mFile->getString();
		} else if (type.substr(0,10) == "PTR_STRUCT") {
		    FrameF::StrLink link;
		    mFile->getLink(link);
		} else {
		    cout << "Unidentified type: " << i->getType() << endl;
		    break;
		}
	    } else {
	        if (eltype == "INT_4U") {
		    for (int j=0 ; j<length ; j++) mFile->getInt();
		} else if (eltype == "INT_4S") {
		    for (int j=0 ; j<length ; j++) mFile->getInt();
		} else if (eltype == "INT_2U") {
		    for (int j=0 ; j<length ; j++) mFile->getShort();
		} else if (eltype == "INT_2S") {
		    for (int j=0 ; j<length ; j++) mFile->getShort();
		} else if (eltype == "INT_8S") {
		    for (int j=0 ; j<length ; j++) mFile->getLong();
		} else if (eltype == "INT_8U") {
		    for (int j=0 ; j<length ; j++) mFile->getLong();
		} else if (eltype == "REAL_8") {
		    for (int j=0 ; j<length ; j++) mFile->getDouble();
		} else if (eltype == "REAL_4") {
		    for (int j=0 ; j<length ; j++) mFile->getFloat();
		} else if (eltype == "STRING") {
		    for (int j=0 ; j<length ; j++) mFile->getString();
		} else if (eltype == "CHAR") {
		    mFile->Skip(length);
		} else if (type.substr(0,10) == "PTR_STRUCT") {
		    FrameF::StrLink link;
		    mFile->getLink(link);
		} else {
		    cout << "Unidentified type: " << i->getType() << endl;
		    break;
		}
	    }
	}
	if (!chName.empty()) cout << chName << "    " << sampleRate << endl;

    }
}

#define TEXTLEN 16

//======================================  Do a Hex Dump of a structure
void 
FrDump::Dump(int len, int id, int inst, istream& In) {
    char strng[TEXTLEN+1];

    //-----------------------------  Read in a frame
    int nbyt = len - 8;
    int nwd  = (nbyt + sizeof(int) - 1)/sizeof(int);
    int* buffer =  new int[nwd];
    if (!buffer) return;
    buffer[nwd-1] = 0;
    In.read((char*)buffer, nbyt);

    //-----------------------------  Dump the frame to stdout.
    printf ("Structure type %i, instance %i, Length %i\n", 
	   id, inst, len);
    for (int i=0 ; i<nwd ; i+=4) {
        for (int j=0 ; j<TEXTLEN ; j++) {
	    char tchar = *((char*) (buffer + i) +j);
	    if      (tchar >= 'a' && tchar <= 'z') strng[j] = tchar;
	    else if (tchar >= 'A' && tchar <= 'Z') strng[j] = tchar;
	    else if (tchar >= '0' && tchar <= '9') strng[j] = tchar;
	    else if (tchar == ' ' || tchar == '_') strng[j] = tchar;
	    else if (tchar == ':' || tchar == '-') strng[j] = tchar;
	    else                                   strng[j] = '.';
	}
	strng [TEXTLEN] = 0;
        printf ("%8lx  %08x %08x %08x %08x |%s|\n", i*sizeof(int),
	       buffer[i], buffer[i+1], buffer[i+2], buffer[i+3], strng);
    }
    delete[] buffer;
}
