#ifndef _LIGO_EVENTWINDOWITERATOR_H
#define _LIGO_EVENTWINDOWITERATOR_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: WindowIterator						*/
/*                                                         		*/
/* Module Description: Iterator for moving a window thru an event set	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: WindowIterator.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <iterator>
#include "Time.hh"
#include "Interval.hh"
#include "events/TimeWindow.hh"
#include "events/Iterator.hh"
#include "events/Window.hh"


namespace events {


/** Basic window iterator class. Helper class for WindowIterator
    and ConstWindowIterator.

    @memo Basic window iterator
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class BasicWindowIterator {
   public:
      /// State of iterator in input
      struct InputState {
         /// Begin
         ConstIterator	mBegin;
         /// End
         ConstIterator	mEnd;
         /// First event (for window list)
         ConstIterator	mFirst;
         /// Next event (for window list)
         ConstIterator	mNext;
         /// Current event
         ConstIterator	mCur;
         /// Constructor
         InputState() {
         }
         /// Constructor 
         InputState (const ConstIterator& beg,
                    const ConstIterator& end)
         : mBegin (beg), mEnd (end), mFirst (beg), mNext (beg), 
         mCur (beg) {
         }
      };
      /// List of iterator states
      typedef std::vector<InputState> InputStateList;
   
      /** Creates an NULL window iterator. Equivalent to an end iterator
          @memo Default constructor
       ******************************************************************/
      BasicWindowIterator () : mPermMax (0), mPermI (0) {
         InitWindow (); }
      /** Creates an window iterator from a an iterator range.
          Works from a single input stream.
          @memo Constructor
          @param beg Beginning of event range
          @param end End of event range
          @param order Coincidence order
          @param window Anlysis window (in sec)
          @param offset Offset of analysis window (in sec)
       ******************************************************************/
      BasicWindowIterator (const ConstIterator& beg, 
                        const ConstIterator& end, 
                        int order = 1,
                        const TimeWindow& window = 1.0);
      /** Creates an window iterator from two iterator ranges.
          This is a simple coincidence between two input lists,
          hence the coincidence order is 2.
          @memo Constructor
          @param beg1 Beginning of first event range
          @param end1 End of first event range
          @param beg2 Beginning of second event range
          @param end2 End of second event range
          @param window Anlysis window (in sec)
          @param offset Offset of analysis window (in sec)
       ******************************************************************/
      BasicWindowIterator (const ConstIterator& beg1, 
                        const ConstIterator& end1, 
                        const ConstIterator& beg2, 
                        const ConstIterator& end2, 
                        const TimeWindow& window = 1.0);
      /** Creates an window iterator from three iterator ranges.
          This is a simple coincidence between three input lists,
          hence the coincidence order is 3.
          @memo Constructor
          @param beg1 Beginning of first event range
          @param end1 End of first event range
          @param beg2 Beginning of second event range
          @param end2 End of second event range
          @param beg3 Beginning of third event range
          @param end3 End of third event range
          @param window Anlysis window (in sec)
          @param offset Offset of analysis window (in sec)
       ******************************************************************/
      BasicWindowIterator (const ConstIterator& beg1, 
                        const ConstIterator& end1, 
                        const ConstIterator& beg2, 
                        const ConstIterator& end2, 
                        const ConstIterator& beg3, 
                        const ConstIterator& end3, 
                        const TimeWindow& window = 1.0);
      /** Creates an window iterator from multiple iterator ranges.
          This is a simple coincidence between multiple input lists.
          The coincidence order is the equal the length of the
          input list.
          @memo Constructor
          @param state List of input event ranges
          @param window Anlysis window (in sec)
          @param offset Offset of analysis window (in sec)
       ******************************************************************/
      BasicWindowIterator (const InputStateList& state, 
                        const TimeWindow& window = 1.0);
   
      /** Get the event window order.
          @memo Get event window order
       ******************************************************************/
      int GetOrder() const {
         return mWindow.GetOrder(); }
   
   protected:
      /** Initializes the window
          @memo Init window
       ******************************************************************/
      void InitWindow();
      /** Increments the iterator
          @memo Increment operator
       ******************************************************************/
      void Increment();
      /** Gets the number of coincidence permutations for the 
          present event list.
          @memo Get number of permutations
       ******************************************************************/
      int GetPermutations() const;
      /** Set the current events in window according to the
          permutations index. 
          @memo Set permutation to window
       ******************************************************************/
      void SetPermutation();
      /** Get the event window.
          @memo Get event window
       ******************************************************************/
      Window& GetWindow() {
         return mWindow; }
      /** Get the event window.
          @memo Get event window
       ******************************************************************/
      const Window& GetWindow() const {
         return mWindow; }
   
   private:
      /// True if working on a single event stream
      bool		mInline;
      /// Current Window
      Window		mWindow;
      /// State of input event lists
      InputStateList	mState;
      /// First time initialization?
      bool		mFirstTime;
      /// Number of permutation
      int		mPermMax;
      /// Permutation index
      int		mPermI;
   };


/** An event window iterator is used to move am event window through an 
    event set. This is a forward iterator. The postfix increment 
    operator is rather inefficient and should be avoided!
    An event window iterator is initialized with a begin and end
    iterator as well as the coincidence order and the window with and 
    offset.

    A window iterator returns a window if dereferenced. The window
    contains the current event(s) and the list of all events
    falling within the time window around the Current(0) event.
    The number of current events is equal to the coincidence order.

    The window iterator returns a Window if dereferenced. A typical
    usage would be as follows:
    \begin{verbatim}
    Set set ("input.xml");
    ConditionPtr cond = IfoSet ("H1[0]") && IfoSet ("L1[1]");
    WindowIterator i (set.Begin(), set.End(), 2);
    WindowIterator end (set.End(), set.End(), 2);
    for (; i != end; ++i) {
       bool ret;
       if (cond->Evaluate (*i, ret) && ret) {
          // found a coincidence event between H1 and L1
       }
    }
    \end{verbatim}

    If the event order is larger than 1, the time order of the 
    coincidence events is not guaranteed. Meaning, the averaged
    time of a coincidence returned by the window iterator later
    may actually be earlier than a previously returned coincidence.
    However, it is always guaranteed that the Current(0) is 
    either the same event or a later event for the next window
    iterator. It is also guaranteed that the returned event is
    not younger or older than the time interval covered by the window.
    Algorithms which perform a coincidence analysis should temporarily
    store a resulting coincidence events (preferrablay in a 
    priortity queue) until the window iterator has moved beyond
    the event time. (See the Coincidence algorithm as an example.)

    The Current(0) event is part of the event list of the window,
    if the time window includes zero. For a higher order analysis
    current events Current(n) with n > 1 are always part of the 
    list.

    When using a window iterator the caller MUST NOT change the 
    list of events and the list of current events which are 
    managed by the Window object returned by the iterator. However, 
    the caller may modify the events which are referenced by these 
    lists.

    @memo Defines an event window iterator
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class WindowIterator : public std::iterator 
   <std::forward_iterator_tag, Window, int, Window*, Window&>,
   public BasicWindowIterator {
      friend class ConstWindowIterator;
   
   public:
      /// Iterator to be used
      typedef Iterator iterator;
   
      /** Creates an NULL window iterator. Equivalent to an end iterator
          @memo Default constructor
       ******************************************************************/
      WindowIterator () { 
      }
      /** Creates an window iterator from an iterator range.
          @memo Constructor
          @param beg Beginning of event range
          @param end End of event range
          @param order Coincidence order
          @param window Anlysis window
       ******************************************************************/
      WindowIterator (const iterator& beg, 
                     const iterator& end, 
                     int order = 1,
                     const TimeWindow& window = 1.0)
      : BasicWindowIterator (beg, end, order, window) {
      }
      /** Creates an window iterator from two iterator ranges.
          The coincidence order is 2.
          @memo Constructor
          @param beg1 Beginning of first event range
          @param end1 End of first event range
          @param beg2 Beginning of second event range
          @param end2 End of second event range
          @param window Anlysis window (in sec)
       ******************************************************************/
      WindowIterator (const iterator& beg1, 
                     const iterator& end1, 
                     const iterator& beg2, 
                     const iterator& end2, 
                     const TimeWindow& window = 1.0)
      : BasicWindowIterator (beg1, end1, beg2, end2, window) {
      }
      /** Creates an window iterator from three iterator ranges.
          The coincidence order is 3.
          @memo Constructor
          @param beg1 Beginning of first event range
          @param end1 End of first event range
          @param beg2 Beginning of second event range
          @param end2 End of second event range
          @param beg3 Beginning of third event range
          @param end3 End of third event range
          @param window Anlysis window (in sec)
       ******************************************************************/
      WindowIterator (const iterator& beg1, 
                     const iterator& end1, 
                     const iterator& beg2, 
                     const iterator& end2, 
                     const iterator& beg3, 
                     const iterator& end3, 
                     const TimeWindow& window = 1.0)
      : BasicWindowIterator (beg1, end1, beg2, end2, beg3, end3,
                        window) {
      }
      /** Creates an window iterator from multiple iterator ranges.
          The coincidence order is the equal the length of the
          input list.
          @memo Constructor
          @param state List of input event ranges
          @param window Anlysis window (in sec)
       ******************************************************************/
      WindowIterator (const InputStateList& state, 
                     const TimeWindow& window = 1.0)
      : BasicWindowIterator (state, window) {
      }
   
      /** Equality operator
          @memo Equality operator
       ******************************************************************/
      bool operator== (const WindowIterator& i) const {
         return GetWindow() == i.GetWindow(); }
      /** Inequality operator
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const WindowIterator& i) const {
         return !(*this == i); }
   
      /** Increment operator (prefix)
          @memo Increment operator (prefix)
       ******************************************************************/
      WindowIterator& operator++ () {
         Increment(); 
         return *this; }
      /** Increment operator (postfix)
          @memo Increment operator (postfix)
       ******************************************************************/
      WindowIterator operator++ (int) {
         WindowIterator i = *this; ++*this; 
         return i; }
      /** Dereference operator
          @memo Dereference operator
       ******************************************************************/
      reference operator*() {
         return GetWindow(); }
      /** Pointer operator
          @memo Pointer operator
       ******************************************************************/
      pointer operator->() {
         return &GetWindow(); }
   };


/** A constant event window iterator.

    @memo Defines a const event window iterator
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class ConstWindowIterator : public std::iterator 
   <std::bidirectional_iterator_tag, Window, int, 
   const Window*, const Window&>, public BasicWindowIterator {
   public:
      /// Iterator to be used
      typedef ConstIterator iterator;
   
      /** Creates an NULL window iterator. Equivalent to an end iterator
          @memo Default constructor
       ******************************************************************/
      ConstWindowIterator () { 
      }
      /** Creates an window iterator form a an iterator range.
          @memo Constructor
          @param beg Beginning of event range
          @param end End of event range
          @param order Coincidence order
          @param window Anlysis window (in sec)
          @param offset Offset of analysis window (in sec)
       ******************************************************************/
      ConstWindowIterator (const iterator& beg, 
                        const iterator& end, 
                        int order = 1,
                        const TimeWindow& window = 1.0)
      : BasicWindowIterator (beg, end, order, window) {
      }
      /** Creates an window iterator from two iterator ranges.
          The coincidence order is 2.
          @memo Constructor
          @param beg1 Beginning of first event range
          @param end1 End of first event range
          @param beg2 Beginning of second event range
          @param end2 End of second event range
          @param window Anlysis window (in sec)
       ******************************************************************/
      ConstWindowIterator (const iterator& beg1, 
                        const iterator& end1, 
                        const iterator& beg2, 
                        const iterator& end2, 
                        const TimeWindow& window = 1.0)
      : BasicWindowIterator (beg1, end1, beg2, end2, window) {
      }
      /** Creates an window iterator from three iterator ranges.
          The coincidence order is 3.
          @memo Constructor
          @param beg1 Beginning of first event range
          @param end1 End of first event range
          @param beg2 Beginning of second event range
          @param end2 End of second event range
          @param beg3 Beginning of third event range
          @param end3 End of third event range
          @param window Anlysis window (in sec)
       ******************************************************************/
      ConstWindowIterator (const iterator& beg1, 
                        const iterator& end1, 
                        const iterator& beg2, 
                        const iterator& end2, 
                        const iterator& beg3, 
                        const iterator& end3, 
                        const TimeWindow& window = 1.0)
      : BasicWindowIterator (beg1, end1, beg2, end2, beg3, end3,
                        window) {
      }
      /** Creates an window iterator from multiple iterator ranges.
          The coincidence order is the equal the length of the
          input list.
          @memo Constructor
          @param state List of input event ranges
          @param window Anlysis window (in sec)
       ******************************************************************/
      ConstWindowIterator (const InputStateList& state, 
                        const TimeWindow& window = 1.0)
      : BasicWindowIterator (state, window) {
      }
      /** Convert from non-const
          @memo Convert from non-const
       ******************************************************************/
      ConstWindowIterator (const WindowIterator& i) {
         *this = i; }
      /** Assignment from non-const
          @memo Assignment opetator
       ******************************************************************/
      ConstWindowIterator& operator= (const WindowIterator& i) {
         this->BasicWindowIterator::operator= (i); 
         return *this; }
   
      /** Equality operator
          @memo Equality operator
       ******************************************************************/
      bool operator== (const ConstWindowIterator& i) const {
         return GetWindow() == i.GetWindow(); }
      /** Inequality operator
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const ConstWindowIterator& i) const {
         return !(*this == i); }
   
      /** Increment operator (prefix)
          @memo Increment operator (prefix)
       ******************************************************************/
      ConstWindowIterator& operator++ () {
         Increment(); 
         return *this; }
      /** Increment operator (postfix)
          @memo Increment operator (postfix)
       ******************************************************************/
      ConstWindowIterator operator++ (int) {
         ConstWindowIterator i = *this; ++*this; 
         return i; }
      /** Dereference operator
          @memo Dereference operator
       ******************************************************************/
      reference operator*() const {
         return GetWindow(); }
      /** Pointer operator
          @memo Pointer operator
       ******************************************************************/
      pointer operator->() const {
         return &GetWindow(); }
   };

}

#endif // _LIGO_EVENTWINDOWITERATOR_H
