#ifndef _LIGO_EVENTLAYOUTINFO_H
#define _LIGO_EVENTLAYOUTINFO_H

/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: LayoutInfo						*/
/*                                                         		*/
/* Module Description: Defines an event					*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: LayoutInfo.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/
#ifndef __CINT__
#include <vector>
#include <iosfwd>
#include "Time.hh"
#include "events/Type.hh"
#include "events/Value.hh"
#include "events/ColumnType.hh"
#include "events/ColumnInfo.hh"
#include "events/IndexList.hh"


namespace events {


/** An event layout is used to describe an event. The layout infromation
    contains the (sub)type name and id, and the column information. An 
    event layout is uniquely identified by its type and subtype ID. An 
    event will store a pointer to its corresponding event layout.
   
    @memo Defines an event layout information record
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class LayoutInfo : public ColumnType {
      friend class Factory;
   
   public:
      /// Column list
      typedef IndexList<ColumnInfo> ColumnList;
   
      /** Creates an empty event layout.
          @memo Default constructor
       ******************************************************************/
      LayoutInfo();
      /** Creates an event layout.
          @memo Default constructor
       ******************************************************************/
      LayoutInfo (const Type& type);
      /** Returns a copy of the layout information. This method must 
          be overriden by all descendents.
          @memo Copy the layout information
          @return Layout information copy
       ******************************************************************/
      LayoutInfo* Copy() const {
         return new LayoutInfo (*this); }
   
      /** Equality operator. Two layouts are the same if the 
          consist of identical columns. Columns have to be identical
          both in name and type.
          @memo Equality operator
       ******************************************************************/
      bool operator== (const LayoutInfo& info) const;
      /** Inequality operator.
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const LayoutInfo& info) const {
         return !(*this == info); }
   
      /** Is this a valid registered layout?
          @memo Registered layout?
       ******************************************************************/
      bool IsRegistered() const {
         return mRegistered; }
      /**  Reference counting
           @memo Reference counting
       ******************************************************************/
      void RefCount (bool up) {
         mRefCount += (up ? 1 : -1); }
      /**  Get reference count
           @memo Get reference count
       ******************************************************************/
      int GetRefCount () const {
         return mRefCount; }
   
      /**  Returns the data value of the specified column. 
           If the specified column was added to the layout after the 
           event was created, the default value for this column type 
           will be returned. The event data block will not be changed.
           @memo Gets the column value of an event
           @param name Name of column
           @param data Pointer to event data block
           @param val Return value (return)
           @return true if column exists and can be read
       ******************************************************************/
      bool GetValue (const char* name, const_data_ptr data, 
                    Value& val) const;
      /**  Sets the data value of the specified column. If the specified
           column was added to the layout after the event was created,
           the event data block will be automatically extended.
           @memo Sets the column value of an event
           @param name Name of column
           @param data Pointer to event data block
           @param val Set value
           @return true if column exists and can be set
       ******************************************************************/
      bool SetValue (const char* name, data_ptr& data, 
                    const Value& val);
   
      /**  Returns the event type of the layout
           @memo Gets the event type
       ******************************************************************/
      const Type& GetType() const {
         return mType; }
      /**  Checks if the layout corresponds to the specified type.
           @memo Compatible type?
       ******************************************************************/
      bool IsCompatible (const Type& type) const {
         return mType == type; }
   
      /** Get the column information
          @memo Column information
       ******************************************************************/
      const ColumnInfoList& GetColumnList() const;
      /** Add a column to the layout.
          @memo Add a column
       ******************************************************************/
      bool AddColumn (const ColumnInfo& col);
      /** Add a column to the layout.
          @memo Add a column
       ******************************************************************/
      bool AddColumn (const char* name, Enum type) {
         return AddColumn (ColumnInfo (name, type)); }
      /** Remove a column from the layout.
          @memo Remove a column
       ******************************************************************/
      bool RemoveColumn (const char* name);
      /** Get a column from the layout.
          @memo Get a column
       ******************************************************************/
      const ColumnInfo* GetColumn (const char* name) const;
   
      /**  Size of data block described by layout.
           @memo Data block size
       ******************************************************************/
      int DataSize () const {
         return mDataSize; }
      /**  Construct the event data (optional copy construct)
           @memo Construct event data
       ******************************************************************/
      bool Construct (data_ptr data, const_data_ptr init = 0);
      /**  Destruct the event data
           @memo Destruct event data
       ******************************************************************/
      bool Destruct (data_ptr data);
      /** Updates the event data block, if columns have been added. 
          Fills added columns with default values.
          @memo Update the event
       ******************************************************************/
      bool Update (data_ptr& data);
      /**  Compares event data
           @memo Compares event data
           @return true if equal
       ******************************************************************/
      bool Compare (const_data_ptr d1, const_data_ptr d2) const;
   
      /**  Register the current layout with the event factory.
           @memo Register the layout globaly
           @param usesubtype True if layout should be subtype specific
           @return Pointer to global layout information (0 if failed)
       ******************************************************************/
      const LayoutInfo* Register();
   
      /**  Dump column names to specified output stream.
           @memo Dump column names to specified output stream.
           @param os output stream
       ******************************************************************/
      void Dump (std::ostream& os) const;
   
      /**  Parse a column name. For a name of the form "(n)" or "(n,...)". The 
           function returns the first section of the name and its remainder.
           @memo Get column event
           @param nameOrg column name
           @param nameFirst First part of the name (return)
           @param nameRem Remainder of the name (return)
           @return true if suucess.
       ******************************************************************/
      static bool Parse (const char* nameOrg,
                        std::string& nameFirst, std::string& nameRem);
   
      /**  Parse an array index. The str argument must point to a 
           valid array index of the form "(n)" or "(n,...)". The 
           function returns the first index and the remainder if
           the second form was used (i.e., multiple indices).
           @memo Get column event
           @param str Array index string
           @param first First index (return)
           @param left Remainder of indices (return)
           @return true if valid array index
       ******************************************************************/
      static bool ParseArrayIndex (const char* str, 
                        int& first, std::string& left);
   
      /**  Lookup a registered layout.
           @memo Lookup a layout
       ******************************************************************/
      static const LayoutInfo* Lookup (const Type& type);
      /**  Get the minimal layout.
           @memo Get "simple" layout
           @return Simple layout
       ******************************************************************/
      static const LayoutInfo* GetSimple();
      /**  Get the standard layout.
           @memo Get "standard" layout
           @return Standard layout
       ******************************************************************/
      static const LayoutInfo* GetStandard();
   
      /// Name of simple event layout
      static const char* Simple() {
         return "Simple"; }
      /// Name of standard event layout
      static const char* Standard() {
         return "Standard"; }
      /// Name of coincidence event layout
      static std::string Coincidence (int order);
      /// Name of cluster event layout
      static std::string Cluster (int num);
      /// Name of gds trigger event layout
      static const char* GdsTrigger() {
         return "GDS_Trigger"; }
      /// Name of single inspiral event layout
      static const char* SnglInspiral() {
         return "Sngl_Inspiral"; }
      /// Name of single burst event layout
      static const char* SnglBurst() {
         return "Sngl_Burst"; }
      /// Name of single ringdown event layout
      static const char* SnglRingdown() {
         return "Sngl_Ringdown"; }
      /// Name of single unmodeled event layout
      static const char* SnglUnmodeled() {
         return "Sngl_Unmodeled"; }
      /// Name of single directed periodic event layout
      static const char* SnglDPeriodic() {
         return "Sngl_DPeriodic"; }
       /// Name of single directed periodic event layout
      static const char* MultiInspiral() {
         return "Multi_Inspiral"; }
      /// Name of single directed periodic event layout
      static const char* MultiBurst() {
         return "Multi_Burst"; }
      /// Name of single directed periodic event layout
      static const char* Segment() {
         return "Segment"; }
   
   protected:
      /// Recalculate data offset, column index and total size
      void Recalculate();
   
   private:
      /** Set the registered flag?
          @memo Set registered
       ******************************************************************/
      void SetRegistered (bool set = true) {
         mRegistered = set; }
   
      /// Is layout registered?
      bool		mRegistered;
      /// Event type
      Type		mType;
      /// Data size
      int		mDataSize;
      /// List of column information
      ColumnList	mColumns;
      /// Reference count
      int		mRefCount;
   };

/** @name LayoutList
    A list of layout information records.
    @memo  A list of layout information records
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
   /// Layout smart pointer
   typedef gdsbase::ref_ptr<LayoutInfo> LayoutPtr;
   /// Layout list
   typedef std::vector <LayoutPtr> LayoutList;
//@}


}
#endif // __CINT__
#endif // _LIGO_EVENTLAYOUTINFO_H
