#ifndef _LIGO_EVENTITERATORIMP_H
#define _LIGO_EVENTITERATORIMP_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: IteratorImp						*/
/*                                                         		*/
/* Module Description: Abstract event iterator implementation		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: IteratorImp.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

namespace events {

   class Event;


/** Abstract class for implementing an event iterator.

    @memo Defines an event iterator implementation
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class IteratorImp {
   public:
      /** Create an iterator implementation
          @memo Default constructor
       ******************************************************************/
      IteratorImp() {
      }
      /** Destroy the iterator
          @memo Destructor
       ******************************************************************/
      virtual ~IteratorImp() {
      }
      /** Get a copy
          @memo Get a copy
       ******************************************************************/
      virtual IteratorImp* Copy() const = 0;
      /** Get a pointer to the event
          @memo Get event pointer
       ******************************************************************/
      virtual Event* Get() const = 0;
      /** Increment the iterator
          @memo Increment
       ******************************************************************/
      virtual void Inc() = 0;
      /** Decrement the iterator
          @memo Decrement
       ******************************************************************/
      virtual void Dec() = 0;
   
   private:
   };



/** Implementing an event iterator based on standard iterators

    @memo Defines an event iterator implementation
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
template <class iter>
   class StdIteratorImp : public IteratorImp {
   public:
      /** Create an event iterator from a standard iterator
          @memo Default constructor
       ******************************************************************/
      StdIteratorImp (iter i) : fI (i) {
      }
      /** Get a copy
          @memo Get a copy
       ******************************************************************/
      virtual StdIteratorImp* Copy() const {
         return new StdIteratorImp (*this); }
      /** Get a pointer to the event
          @memo Get event pointer
       ******************************************************************/
      virtual Event* Get() const {
         return const_cast<Event*> (&*fI); }
      /** Increment the iterator
          @memo Increment
       ******************************************************************/
      virtual void Inc() {
         ++fI; }
      /** Decrement the iterator
          @memo Decrement
       ******************************************************************/
      virtual void Dec() {
         --fI; }
      /** Get the iterator
          @memo Get iterator
       ******************************************************************/
      virtual const iter& GetIterator() const {
         return fI; }
   protected:
      iter		fI;
   };


}

#endif // _LIGO_EVENTITERATORIMP_H
