#ifndef _LIGO_EVENTINFO_H
#define _LIGO_EVENTINFO_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Info							*/
/*                                                         		*/
/* Module Description: Defines the event info function			*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Info.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <string>
#include "events/Function.hh"
#include "events/FunctionPtr.hh"
#include "events/Column.hh"


namespace events {

   class Event;

/** Event information access. Like column values, the information
    describing the event layout can be accessed through an event 
    function. Rather than reading values from the event data block,
    this class accessed the event layout information.

    The event information is accessed by name. Recognized names are:
    \begin{verbatim}
    -------------------------------------------------------------------
    Name                Return  Description
    -------------------------------------------------------------------
    Type                String  Event type name
    Name                String  Event name
    TypeId              Int     Event type identification number
    NameId              Int     Event name identification number
    Size                Int     Size of event data block in bytes
    RefCount            Int     Number of identical events
    ColumnNumber        Int     Number of columns
    Column(i)::Name     String  Column name
    Column(i)::Type     String  Column type name
    Column(i)::TypeId   Int     Column type identification number
    Column(i)::TypeSize Int     Type size
    Column(i)::Fixed    Int     One if fixed column
    Column(i)::Index    Int     Column index
    Column(i)::Offset   Int     Column byte offset
    -------------------------------------------------------------------
    \end{verbatim}

    The column index i can be specified as a number or as a column
    name. The notation "m:n" can be used to specify that the index should
    be enumerated over indices m to n. If the second argumen is 
    missing, i.e., "m:", the enumeration goes till the last column.
    When enumerating columns, the result is always returned as a 
    string of comma separated values.

    The special literal "dollar" can be used to represent a column index
    which is specified at run-time. For example, 
    \begin{verbatim}
    Info ("Column($)::Name", Info ("ColumnNumber") - 1)
    \end{verbatim}
    can be used to get the name of the last column.

    If a information item is specified without an event index, the
    default is zero. An event index can be explicitly specified as 
    part of the string or as a separate argument.
    Examples:
    \begin{verbatim}
    Info ("Type[1]") + Value("::") + Info ("Name", 1)
    \end{verbatim}
    constructs a type identifier string on the second event.

    As with the column class the information function can also 
    access events stored as event columns. The same notation 
    applies. For example:
    \begin{verbatim}
    Info ("Event(1).Type")
    Info ("Type(1)")
    \end{verbatim}
    to access the type inormation from the event stored in a 
    column named "Event(1)".

    @memo Defines an event information function
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Info : public Function {
   public:
      /// Info token
      enum infotoken {
      /// Invalid
      kInvalid = 0,
      /// Type
      kType,
      /// Name
      kName,
      /// Type ID
      kTypeId,
      /// Name ID
      kNameId,
      /// Event data size
      kSize,
      /// Reference count
      kRefCount,
      /// Column number
      kColNum,
      /// Column name
      kColName,
      /// Column type
      kColType,
      /// Column type ID
      kColTypeId,
      /// Column type size
      kColTypeSize,
      /// Column fixed?
      kColFixed,
      /// Column index
      kColIndex,
      /// Column offset
      kColOffset
      };
   
      /** Create an event information item with the specified name.
          @memo Constructor
          @param name Name of event information item
       ******************************************************************/
      explicit Info (const char* name) 
      : mIndex (0) {
         SetName (name); }
      /** Create an event information item with the specified name.
          @memo Constructor
          @param name Name of event information item
       ******************************************************************/
      explicit Info (const char* name, int index) 
      : mIndex (0) {
         SetName (name); SetIndex (index); }
      /** Create an event information item with the specified name.
          @memo Constructor
          @param name Name of event information item
       ******************************************************************/
      explicit Info (const std::string& name, int index) 
      : mIndex (0) {
         SetName (name.c_str());  SetIndex (index); }
      /** Create an event information item with the specified name 
          and argument
          @memo Constructor
          @param name Name of event information item
          @param name full name
       ******************************************************************/
      Info (const char* name, const Function& arg1) 
      : mIndex (0), mArg1 (arg1) {
         SetName (name); }
      Info (const char* name, const FunctionPtr& arg1) 
      : mIndex (0), mArg1 (arg1) {
         SetName (name); }
      /** Returns a copy of the event information item. 
          This method must be overriden by all descendents.
          @memo Copy the event info
          @return event copy
       ******************************************************************/
      virtual Info* Copy() const {
         return new Info (*this); }
      /** Checks if this is a valid info item.
          @memo Is valid?
          @return true if valid
       ******************************************************************/
      bool IsValid() const {
         return mToken != kInvalid; }
      /** Equality operator (case insensitive)
          @memo Equality operator
       ******************************************************************/
      bool operator== (const Info& info) const;
      /** Inequality operator.
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const Info& info) const {
         return !(*this == info); }
   
      /** Returns the value of the event information item.
          @memo Value of information item
          @param event Events to pick info value from
          @param n Number of events passed to method
          @param val Info value (return)
          @return true if info exists and event is of correct type
       ******************************************************************/
      virtual bool InfoValue (const Event& event, Value& val) const;
   
      /** Returns the value of the event information item.
          @memo Value of information item
          @param arg Event argument list
          @param val Column value (return)
          @return true if info exists and event is of correct type
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, Value& val) const;
   
      /** Sets the name of information item.
          @memo Sets information item
          @param name Name of information item
       ******************************************************************/
      virtual void SetName (const char* name);
      /** Returns the name of information item.
          @memo Name of information item
          @return Name
       ******************************************************************/
      const char* GetName() const {
         return mName.c_str(); }
      /** Set the event index.
          @memo Set event index
          @param index Event index
       ******************************************************************/
      void SetIndex (int index) {
         mIndex = index; }
      /** Returns the event index.
          @memo Get event index
          @return Event index
       ******************************************************************/
      int GetIndex() const {
         return mIndex; }
   
   private:
      /// Event index
      int		mIndex;
      /// Name of information item
      std::string 	mName;
      /// First argument (if any)
      FunctionPtr	mArg1;
      /// Information token
      infotoken		mToken;
      /// Column index for info token (begin)
      int		mColIndex1;
      /// Column index for info token (end if iteration)
      int		mColIndex2;
      /// Column name for info token
      std::string	mColName;
      /// Column access for array indices
      Column		mColumn;
   };

}

#endif // _LIGO_EVENTINFO_H
