/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: EventPtr						*/
/*                                                         		*/
/* Module Description: Defines a event smart pointer			*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: EventPtr.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _LIGO_EVENTPTR_H
#define _LIGO_EVENTPTR_H


#include "events/Event.hh"


namespace events {

/** A smart pointer for events. When copying an event smart pointer,
    a new event is created with the Copy method.
   
    @memo Smart pointer for events.
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class EventPtr {
   public:
      /** Creates an pointer to an event. The event gets adopted.
          @memo Default constructor
       ******************************************************************/
      explicit EventPtr (Event* ptr = 0) : mPtr (0) {
         Reset (ptr); }
      /** Creates a new copy of the event. The event is not adopted.
          @memo Constructor
       ******************************************************************/
      EventPtr (const Event& event) : mPtr (0) {
         Reset (event); }
      /** Copies an event list.
          @memo Copy constructor
       ******************************************************************/
      EventPtr (const EventPtr& ptr) : mPtr (0) {
         *this = ptr; }
      /** Destructs the event.
          @memo Destructor
       ******************************************************************/
      ~EventPtr () {
         Reset (0); }
      /** Assignment. Makes a new pointer with a true copy.
          @memo Assignment operator
       ******************************************************************/
      EventPtr& operator= (const EventPtr& ptr) {
         if (this != &ptr) Reset (ptr.Get() ? ptr->Copy() : 0); 
         return *this; }
      /** Get a pointer to the event.
          @memo Get
       ******************************************************************/
      Event* Get () {
         return mPtr; }
      /** Get a pointer to the event.
          @memo Get
       ******************************************************************/
      const Event* Get () const {
         return mPtr; }
      /** Dereference operator.
          @memo Dereference oprator
       ******************************************************************/
      Event& operator*() {
         return *mPtr; }
      /** Dereference operator.
          @memo Dereference oprator
       ******************************************************************/
      const Event& operator*() const {
         return *mPtr; }
      /** Member access oprator.
          @memo Member access oprator
       ******************************************************************/
      Event* operator->() {
         return mPtr; }
      /** Member access oprator.
          @memo Member access oprator
       ******************************************************************/
      const Event* operator->() const {
         return mPtr; }
      /** Release the event.
          @memo Release
       ******************************************************************/
      Event* Release () {
         Event* e = mPtr; mPtr = 0; 
         return e; }
      /** Resets the event. The new event gets adopted.
          @memo Reset
       ******************************************************************/
      void Reset (Event* ptr = 0) {
         if (mPtr) delete mPtr; 
         mPtr = ptr; }
      /** Resets the event. The new event is copied.
          @memo Reset
       ******************************************************************/
      void Reset (const Event& ptr) {
         if (mPtr) delete mPtr; 
         mPtr = ptr.Copy(); }
   
   private:
      /** Event pointer.
          @memo Pointer to event
       ******************************************************************/
      Event*	mPtr;
   };


}

#endif // _LIGO_EVENTPTR_H
