/* -*- mode: c++; c-basic-offset: 3; -*- */
//////////////////////////////////////////////////////////////////////////
//  									//
//  Event								//
//  									//
//////////////////////////////////////////////////////////////////////////

#include <time.h>
#include "events/Event.hh"
#include "events/ColumnType.hh"
#include "events/ColumnInfo.hh"
#include "events/Value.hh"
#include <iostream>
#include <iomanip>
#include <string>
#include <cstdio>
#include <cstring>

   using namespace std;

namespace events {

//______________________________________________________________________________

   static std::string Escape (const char* p, int len, bool* ishex = 0)
   {
      if (ishex) *ishex = false;
      std::string s;
      if (!p) {
         return s;
      }
      if (len < 0) len = strlen (p);
   
      // non printable characters?
      bool nonprint = false;
      for (int i = 0; i < len; ++i) {
         if ((p[i] < 32) || (p[i] >= 127)) {
            nonprint = true;
            break;
         }
      }
   
      // hex representation
      if (nonprint) {
         s = "0x";
         char buf[10];
         for (int i = 0; i < len; ++i) {
            sprintf (buf, "%02X", (unsigned char)p[i]);
            s += buf;
         }
         if (ishex) *ishex = true;
      }
      
      // ASCII representation
      else {
         s = p;
      }
      return s;
   }

//______________________________________________________________________________
   Event* Event::gDefault = 0;

//______________________________________________________________________________
   bool Event::operator== (const Event& event) const
   {
      return (mLayout == event.mLayout) &&
         ((mLayout == Layout()) || mLayout.Compare (mData, event.mData));
   }

//______________________________________________________________________________
   const Event& Event::Default()
   {
      if (!gDefault) {
         gDefault = new Event (Layout::GetSimple());
      }
      return *gDefault;
   }

//______________________________________________________________________________
   bool Event::Init (const Layout& layout, const_data_ptr init)
   {
      if (!layout.IsRegistered()) {
         mData = 0;
         mLayout = Layout();
         return false;
      }
      mLayout = layout;
      mData = new (std::nothrow) char [mLayout.DataSize()];
      mLayout.Construct (mData, init);
      return true;
   }

//______________________________________________________________________________
   void Event::Destroy ()
   {
      mLayout.Destruct (mData);
      delete [] (char*) mData; 
      mData = 0;
   }
//______________________________________________________________________________
   void Event::Swap (Event& e)
   {
      std::swap (mData, e.mData); 
      std::swap (mLayout, e.mLayout);
   }

//______________________________________________________________________________
   void Event::Dump (std::ostream& os, int indent) const
   {
      std::string tab;
      for (int i = 0; i < indent; ++i) {
         tab += "\t";
      }
   
      const Layout l = GetLayout();
   
      Type type;
      l.GetType(type);
   
      os << tab << "Type = " << type.GetName() << endl;
   
      ColumnInfoList cl = l.GetColumnList();
      for (ColumnInfoList::iterator j = cl.begin(); j != cl.end(); ++j) {
      
         string colname = j->GetName();
         int    offset  = j->GetOffset();
      
         std::string ctype;
         if (j->IsFixed()) {
            ctype = "Fixed";
         }
         else {
            ctype = "Variable";
         }
      
         Value val;
         GetValue(colname.c_str(), val);
      
         ColumnType::Enum type = val.Type();
         std::string vtype = j->GetTypeName();
      
         os << tab << colname << " [" 
            << ctype << "," 
            << vtype << "," 
            << offset << "] = ";
      
         if (type ==  ColumnType::kComplex) {
            ColumnType::Complex c;
            val.Write(c);
            os << "(" << c.real() << "," << c.imag() << ")";
         }
         else if (type == ColumnType::kTime) {
            Time t;
            val.Write(t);
            os << setprecision(9)
               << t.getS() 
               << ".";
            os.width(9);
            os.fill('0');
            os	<< t.getN();
            os.fill(' ');
         }
         else if (type == ColumnType::kReal) {
            double data;
            val.Write(data);
            os << data;
         }
         else if (type == ColumnType::kInt) {
            int data;
            val.Write(data);
            os << data;
         }
         else if (type == ColumnType::kString) {
            std::string s;
            val.Write(s);
            bool ishex;
            s = Escape (s.c_str(), s.size(), &ishex);
            if (ishex) os << s; 
            else os << "\"" << s << "\"";
         }
         else if (type == ColumnType::kEvent) {
            Event e;
            val.Write(e);
            os << endl;
            e.Dump(os,indent+1);
         }
      
         if (j->IsFixed()) {
            if ((colname == "Name") && GetName().GetName()) {
               os << " \"" << GetName().GetName() << "\"";
            }
            else if (colname == "Ifo") {
               os << " \"" << GetIfoStr() << "\"";
            }
         }
      
         os << endl;
      }
   
   }

//______________________________________________________________________________
   void Event::Dump (int indent) const
   {
      Dump (cout, indent);
   }

//______________________________________________________________________________
   void Event::DumpColumn (const char* name, std::ostream& os) const
   {
      Value val;
      GetValue(name,val);
      ColumnType::Enum type = val.Type();
      if (type ==  ColumnType::kComplex) {
         ColumnType::Complex c;
         val.Write(c);
         os << "(" << c.real() << "," << c.imag() << ")";
      }
      else if (type == ColumnType::kTime) {
         Time t;
         val.Write(t);
         os << setprecision(18) 
            << t.totalS() 
            << setprecision(6);
      }
      else if (type == ColumnType::kReal) {
         double data;
         val.Write(data);
         os << data;
      }
      else if (type == ColumnType::kInt) {
         int data;
         val.Write(data);
         os << data;
      }
      else if (type == ColumnType::kString) {
         std::string s;
         val.Write(s);
         bool ishex;
         s = Escape (s.c_str(), s.size(), &ishex);
         if (ishex) os << s; 
         else os << "\"" << s << "\"";
      }
      else if (type == ColumnType::kEvent) {
         Event e;
         val.Write(e);
         os << endl;
         e.Dump(os);
      }
      os << endl;
   }

//______________________________________________________________________________
   void Event::DumpColumn (const char* name) const
   {
      DumpColumn (name, cout);
   }


}
