//////////////////////////////////////////////////////////////////////////
//  									//
//  ColumnType								//
//  									//
//////////////////////////////////////////////////////////////////////////


#include <stddef.h>
#include <string.h>
#include <strings.h>
#include "events/ColumnType.hh"
#include "events/Event.hh"


namespace events {

//______________________________________________________________________________
   const int ColumnType::kNum = ColumnTypekNum;

//______________________________________________________________________________
   const char* const ColumnType::kColumnNumberName = "ColumnNumber";
   const char* const ColumnType::kColumnNameName = "Name";
   const char* const ColumnType::kColumnTimeName = "Time";
   const char* const ColumnType::kColumnIfoName = "Ifo";

//______________________________________________________________________________
   std::string ColumnType::TypeName (Enum type)
   {
      switch (type) {
         case kInt:
            return "Int";
         case kReal:
            return "Real";
         case kComplex:
            return "Complex";
         case kTime:
            return "Time";
         case kString:
            return "String";
         case kEvent:
            return "Event";
         default:
            return "";
      }
   }

//______________________________________________________________________________
   int ColumnType::TypeSize (Enum type)
   {
      switch (type) {
         case kInt:
            return sizeof (Int);
         case kReal:
            return sizeof (Real);
         case kComplex:
            return sizeof (Complex);
         case kTime:
            return sizeof (Time);
         case kString:
            return sizeof (String);
         case kEvent:
            return sizeof (Event);
         default:
            return 0;
      }
   }

//______________________________________________________________________________
   int ColumnType::AlignmentSize (Enum type)
   {
      struct ComplexAlign {
         char dummy;
         Complex c;
      };
      struct TimeAlign {
         char dummy;
         Time t;
      };
      struct StringAlign {
         char dummy;
         String s;
      };
      struct EventAlign {
         char dummy;
         Event e;
      };
   
      switch (type) {
         case kInt:
            return sizeof (Int);
         case kReal:
            return sizeof (Real);
         case kComplex:
            return offsetof (ComplexAlign, c);
         case kTime:
            return offsetof (TimeAlign, t);
         case kString:
            return offsetof (StringAlign, s);
         case kEvent:
            return offsetof (EventAlign, e);
         default:
            return 0;
      }
   }

//______________________________________________________________________________
   bool ColumnType::ConstructValue (Enum type, data_ptr data,
                     const_data_ptr init)
   {
      // construct if necessary
      switch (type) {
         case kString:
            if (init) {
               new (data) String (*(const String*)init);
            } 
            else {
               new (data) String ();
            }
            break;
         case kEvent:
            if (init) {
               new (data) Event (*(const Event*)init);
            } 
            else {
               new (data) Event ();
            }
            break;
         default:
            if (init) {
               memcpy (data, init, TypeSize (type));
            } 
            else {
               memset (data, 0, TypeSize (type));
            }
            break;
      }
      return true;
   }

//______________________________________________________________________________
   bool ColumnType::DestructValue (Enum type, data_ptr data)
   {
      // desctruct if necessary
      switch (type) {
         case kString:
            ((String*)data)->~String();
            break;
         case kEvent:
            ((Event*)data)->~Event();
            break;
         default:
            // nothing
            break;
      }
      return true;
   }

//______________________________________________________________________________
   bool ColumnType::CopyValue (Enum type, data_ptr d1, 
                     const_data_ptr d2)
   {
      if (!d1 || !d2) {
         return false;
      }
      switch (type) {
         case kString:
            *(String*)d1 = *(String*)d2;
            break;
         case kEvent:
            *(Event*)d1 = *(Event*)d2;
            break;
         default:
            memcpy (d1, d2, TypeSize (type));
            break;
      }
      return true;
   }

//______________________________________________________________________________
   bool ColumnType::CompareValue (Enum type, const_data_ptr d1, 
                     const_data_ptr d2)
   {
      // compare
      switch (type) {
         case kInt:
            return *(Int*)d1 == *(Int*)d2;
         case kReal:
            return *(Real*)d1 == *(Real*)d2;
         case kComplex:
            return *(Complex*)d1 == *(Complex*)d2;
         case kTime:
            return *(Time*)d1 == *(Time*)d2;
         case kString:
            return strcasecmp (((String*)d1)->c_str(), 
                              ((String*)d2)->c_str()) == 0;
         case kEvent:
            return *(Event*)d1 == *(Event*)d2;
         default:
            return true;
      }
      return true;
   }

//______________________________________________________________________________
   bool ColumnType::IsZeroValue (Enum type, const_data_ptr data)
   {
      // test for zero/empty
      switch (type) {
         case kInt:
            return *(Int*)data == 0;
         case kReal:
            return *(Real*)data == 0;
         case kComplex:
            return *(Complex*)data == Complex(0,0);
         case kTime:
            return *(Time*)data == Time(0);
         case kString:
            return ((String*)data)->empty();
         case kEvent:
            return ((Event*)data)->IsValid();
         default:
            return true;
      }
      return true;
   }

}
