#ifndef _LIGO_EVENTCOLUMN_H
#define _LIGO_EVENTCOLUMN_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Column							*/
/*                                                         		*/
/* Module Description: Defines an event	column				*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Column.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <string>
#include "events/Function.hh"
#include "events/Value.hh"


namespace events {

   class Event;
   class Argument;
   class Value;
   class ColumnCache;


/** An event column represents a column value of an event. The
    basic tag of a column is its name. The column class will
    look up the name in the column information list assciated 
    with the event and extract the corresponding column value
    from the event data block.

    A column name is typically just an ASCII string. Example:
    \begin{verbatim}
    Column ("Time")
    \end{verbatim}
    will pick up the even time. A column can also specifiy an
    event index--denoted by index brackets--which is used in 
    a coincidence analysis to determine which event to use. 
    Example:
    \begin{verbatim}
    Column ("Time[1]") - Column ("Time[0]")
    \end{verbatim}
    will calculate the time difference bewteen the two coincidence
    events.

    A special case are columns which are themselves events. If
    one want to access the amplitude of an event column with name 
    "Event(1)", one can simply specify:
    \begin{verbatim}
    Column ("Event(1).Amplitutde")
    \end{verbatim}
    For the above case a short for exists: 
    \begin{verbatim}
    Column ("Amplitutde(1)")
    \end{verbatim}
    will be expanded to the same column amplitude value. The short
    form can be used if the name of the event column has a name
    of the format "Event(n)", with n an non-zero positive integer.
    The index 0 is reserved for the original event, i.e., 
    "Amplitude(0)" and "Amplitue" correspond to identical columns.

    Even so not a likely case, it is possible to have an event
    with an event column which contains yet another event column
    with its own event. These columns can be access through:
    \begin{verbatim}
    Column ("Event(m).Event(n).Amplitutde")   or
    Column ("Event(m).Amplitutde(n)")         or
    Column ("Amplitutde(m,n)")                or
    Column ("Event(m,n).Amplitutde")
    \end{verbatim}
    with m and n the event column indices of the original and the 
    sub event, respectively. The first form is the canonical 
    representation which is used for internal storage.

    If an event column of an event with non-zero index has 
    to be accessed, they can both be specified simultanously,
    but the index must be specifed last. For example:
    \begin{verbatim}
    Column ("Event(1).Time[1]")
    Column ("Time(1)[1]")
    \end{verbatim}
    represent the time of the event stored at column "Event(1)".

    Column names can not contain the characters "[", "]", ".", "*" 
    or "?". They should not contain "(", ")" unless they are of the
    event type and contain the string "Event". Using something like
    "Amplitude(1)" as a column name is confusing. In case such
    a name has been used, the column name specification has to be 
    "Amplitude(1)." with a dot at the end. Using a double index in
    any column name is forbidden, i.e., "Amplitude(1,2)" is not 
    allowed, nor is "Event(1,2)".

    A column can also be used to access the column value of an event.
    If a series of events with identical column layout is processed,
    this is more efficient than calling the SetValue method of the 
    event. The event SetValue method requires a string search in the
    column table for every access, whereas the column class uses a
    caching algorithm to increase performance. Example:
    \begin{verbatim}
    // Create a new coincidence event from two events represented by
    // i->Current(0) and i->Current(1)
    Column col0 ("Event(0)");
    Column col1 ("Event(1)");
    Column ifo ("Ifo");
    // loop over events here
    for (; i != end; ++i) {
       Event ne ("Coincidence", "2");
       if (ne.IsValid()) {
          col0.Set (ne, i->Current(0));
          col1.Set (ne, i->Current(1));
          ne.SetTime (i->GetTime());
          Value ifo0, ifo1;
          ifo.Get (i->Current(0), ifo0); 
          ifo.Get (i->Current(1), ifo1);
          ifo.Set (ne, ifo0 | ifo1);
       }
    }
    \end{verbatim}
   
    @memo Defines an event column
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Column : public Function {
   public:
      /** Creates an event column withou a name.
          @memo Default constructor
       ******************************************************************/
      Column() : mIndex (0), mCache (0) {
         SetName (0); }
      /** Create an event column with the specified name. Allow
          automatic type conversion from const char*.
          @memo Constructor
          @param name full name
       ******************************************************************/
      Column (const char* name) : mIndex (0), mCache (0) {
         SetName (name); } 
      /** Create an event column with the specified name. Allow
          automatic type conversion from string.
          @memo Constructor
          @param name full name
       ******************************************************************/
      Column (const std::string& name) : mIndex (0), mCache (0) {
         SetName (name); }
      /** Create an event column with the specified name and event 
          index.
          @memo Constructor
          @param name column name only
          @param eindex Event index (for coincidence)
       ******************************************************************/
      Column (const char* name, int eindex) : mIndex (0), mCache (0) {
         SetName (name); SetIndex (eindex); }
      /** Copy constructor.
          @memo Copy constructor
       ******************************************************************/
      Column (const Column& col) : mIndex (0), mCache (0) {
         *this = col; }
      /** Destructs an event column.
          @memo Denstructor
       ******************************************************************/
      virtual ~Column();
      /** Assignment operator.
          @memo Assignment operator
       ******************************************************************/
      Column& operator= (const Column& col);
   
      /** Returns a copy of the event column. This method must be 
          overriden by all descendents.
          @memo Copy the event column
          @return event copy
       ******************************************************************/
      virtual Column* Copy() const {
         return new Column (*this); }
      /** Checks if this is a valid column.
          @memo Is valid?
          @return true if valid
       ******************************************************************/
      bool IsValid() const;
   
      /** Returns the column value of the event.
          @memo Get column value (Value)
          @param events Event to pick column value from
          @param val Column value (return)
          @return true if column exists
       ******************************************************************/
      virtual bool Get (const Event& event, Value& val) const;
      /** Returns the column value of the event.
          @memo Get column value (Real)
          @param events Event to pick column value from
          @param x Column value (return)
          @return true if column exists
       ******************************************************************/
      virtual bool Get (const Event& event, ColumnType::Real& x) const {
         Value val; 
         return Get (event, val) && val.Write (x); }
      /** Returns the column value of the event.
          @memo Get column value (Int)
          @param events Event to pick column value from
          @param i Column value (return)
          @return true if column exists
       ******************************************************************/
      virtual bool Get (const Event& event, ColumnType::Int& i) const {
         Value val; 
         return Get (event, val) && val.Write (i); }
      /** Returns the column value of the event.
          @memo Get column value (Complex)
          @param events Event to pick column value from
          @param c Column value (return)
          @return true if column exists
       ******************************************************************/
      virtual bool Get (const Event& event, ColumnType::Complex& c) const {
         Value val; 
         return Get (event, val) && val.Write (c); }
      /** Returns the column value of the event.
          @memo Get column value (Time)
          @param events Event to pick column value from
          @param t Column value (return)
          @return true if column exists
       ******************************************************************/
      virtual bool Get (const Event& event, ColumnType::time_type& t) const {
         Value val; 
         return Get (event, val) && val.Write (t); }
      /** Returns the column value of the event.
          @memo Get column value (string)
          @param events Event to pick column value from
          @param s Column value (return)
          @return true if column exists
       ******************************************************************/
      virtual bool Get (const Event& event, std::string& s) const {
         Value val; 
         return Get (event, val) && val.Write (s); }
      /** Returns the column value of the event.
          @memo Get column value (string)
          @param events Event to pick column value from
          @param p Column value (return)
          @param len Maximum length of buffer
          @return true if column exists
       ******************************************************************/
      virtual bool Get (const Event& event, char* p, int& len) const {
         Value val; 
         return Get (event, val) && val.Write (p, len); }
   
      /** Sets the column value of the event.
          @memo Set column value (Value)
          @param events Event for which to set column value
          @param val New column value
          @return true if column exists and value could be set
       ******************************************************************/
      virtual bool Set (Event& event, const Value& val);
      /** Sets the column value of the event.
          @memo Set column value (Real)
          @param events Event for which to set column value
          @param x New column value
          @return true if column exists and value could be set
       ******************************************************************/
      virtual bool Set (Event& event, const ColumnType::Real& x) {
         return Set (event, Value (x)); }
      /** Sets the column value of the event.
          @memo Set column value (Int)
          @param events Event for which to set column value
          @param i New column value
          @return true if column exists and value could be set
       ******************************************************************/
      virtual bool Set (Event& event, const ColumnType::Int& i) {
         return Set (event, Value (i)); }
      /** Sets the column value of the event.
          @memo Set column value (Complex)
          @param events Event for which to set column value
          @param c New column value
          @return true if column exists and value could be set
       ******************************************************************/
      virtual bool Set (Event& event, const ColumnType::Complex& c) {
         return Set (event, Value (c)); }
      /** Sets the column value of the event.
          @memo Set column value (Time)
          @param events Event for which to set column value
          @param t New column value
          @return true if column exists and value could be set
       ******************************************************************/
      virtual bool Set (Event& event, const ColumnType::time_type& t) {
         return Set (event, Value (t)); }
      /** Sets the column value of the event.
          @memo Set column value (string)
          @param events Event for which to set column value
          @param s New column value
          @return true if column exists and value could be set
       ******************************************************************/
      virtual bool Set (Event& event, const std::string& s) {
         return Set (event, Value (s)); }
      /** Sets the column value of the event.
          @memo Set column value (string)
          @param events Event for which to set column value
          @param p New column value
          @param len length of buffer, or -1 for NULL terminated
          @return true if column exists and value could be set
       ******************************************************************/
      virtual bool Set (Event& event, const char* p, int len = -1) {
         return Set (event, Value (p, len)); }
   
      /** Returns a pointer to the last event. This method returns 
          the event which is used to pick the column. This is useful for
          Event which contains events (array index). For example:
          \begin{verbatim}
          Column name            Returns
          Amplitude              current event
          Amplitude(1)           Event(1)
          Event(2)               Event(2)
          Event(2).Event(1).E    Event(2).Event(1) // even if E is an event
          \end{verbatim}
          All array indices are resolved and the last event is returned.
          If the event column does not exists, NULL is returned.
          @memo Get column event
          @return Event if exist
       ******************************************************************/
      virtual const Event* GetEvent (const Argument& arg) const {
         return GetEvent (arg, mIndex); }
      /** Returns a pointer to the "last" event. This method explicitly
          specifies the event index.
          @memo Get column event
          @param arg Event argument list
          @param index Event index
          @return Event if exist
       ******************************************************************/
      virtual const Event* GetEvent (const Argument& arg, 
                        int index) const;
      /** Returns a pointer to the last event. This method specifies
          the first event.
          @memo Get column event
          @param event Event
          @return Event if exist
       ******************************************************************/
      virtual Event* GetEvent (Event& event);
      /** Returns a pointer to the last event. This method specifies
          the first event.
          @memo Get column event
          @param event Event
          @return Event if exist
       ******************************************************************/
      virtual const Event* GetEvent (const Event& event) const;
   
      /** Returns the column value of the specified event.
          @memo Column value
          @param arg Event argument list
          @param val Column value (return)
          @return true if column exists and event is of correct type
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, Value& val) const;
   
      /** Set the name of the column.
          @memo Set column name
       ******************************************************************/
      void SetName (const char* name);
      /** Set the name of the column.
          @memo Set column name
       ******************************************************************/
      void SetName (const std::string& name) {
         SetName (name.c_str()); }
      /** Get the name of the column in its canonical representation.
          @memo Get column name
          @return Name
       ******************************************************************/
      const char* GetName() const {
         return mName.c_str(); }
      /** Set the event index.
          @memo Set event index
          @param index Event index
       ******************************************************************/
      void SetIndex (int index) {
         mIndex = index; }
      /** Gets the event index.
          @memo Get event index
          @return Event index
       ******************************************************************/
      int GetIndex() const {
         return mIndex; }
      /** Clear the column cache.
          @memo Clear cache
       ******************************************************************/
      void ResetCache() const;
   
   private:
      /// Column name.
      std::string 	mName;
      /// Event index.
      int		mIndex;
      /// Column cache
      ColumnCache*	mCache;
   };

}

#endif // _LIGO_EVENTCOLUMN_H
