#ifndef _LIGO_EVENTCHAIN_H
#define _LIGO_EVENTCHAIN_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Chain							*/
/*                                                         		*/
/* Module Description: Defines a chain of event lists			*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Chain.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/List.hh"
#include "events/ListPtr.hh"
#include "events/Function.hh"
#include "events/FunctionPtr.hh"

namespace events {

/** An event chain consists of a series of event lists. The idea is
    that for really large sets of events, multiple files are
    referenced through a chain. When they are needed, they are read 
    in, worked on and discared as the analysis algorithm moves 
    through the events of the chain.
   
    An event chain contains events ordered by time with the oldest 
    event being stored first. For efficiency reasons Chain provides
    algorithms which let you add events without checking that 
    they are in correct orderer. In this case the user has to 
    guarantee that the sort order is maintained, or call the Sort
    method afterwards.

    @memo Chain of event lists
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Chain {
   public:
      /// Basic event chain type
      typedef std::vector<ListPtr> eventchain;
      /// Basic list type
      typedef List::eventlist eventlist;
      /// Value type
      typedef List::value_type value_type;
      /// Size type
      typedef List::size_type size_type;
      /// Difference type 
      typedef List::difference_type difference_type;
      /// Reference type
      typedef List::reference reference;
      /// Const reference type
      typedef List::const_reference const_reference;
      /// Pointer type
      typedef List::pointer pointer;
      /// Const pointer type
      typedef List::const_pointer const_pointer;      
      /// Iterator
      typedef Iterator iterator;
      /// Const iterator
      typedef ConstIterator const_iterator;
      /// Iterator
      typedef ReverseIterator reverse_iterator;
      /// Const iterator
      typedef ConstReverseIterator const_reverse_iterator;
   
      /** Creates an empty event chain.
          @memo Default constructor
       ******************************************************************/
      Chain() {
      }
      /** Creats a event chain with one event list.
          @memo Constructor
       ******************************************************************/
      explicit Chain (const char* filename) {
         AddList (filename); }
   
      /** Returns a copy of the event chain. This method must be 
          overriden by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      Chain* Copy() const {
         return new Chain (*this); }
   
      /** Add an event list to the chain.
          @memo Add
       ******************************************************************/
      bool AddList (const char* filename);
      /** Add an event list to the chain.
          @memo Add
       ******************************************************************/
      bool AddList (const List& eventlist);
      /** Get an event list of the chain.
          @memo Get
       ******************************************************************/
      List& GetList (int lnum) {
         return *mChain[lnum]; }
      /** Get an event list of the chain.
          @memo Get
       ******************************************************************/
      const List& GetList (int lnum) const {
         return *mChain[lnum]; }
      /** Remove an event list form the chain.
          @memo Remove
       ******************************************************************/
      bool RemoveList (int lnum);
      /** Number of event lists.
          @memo N
       ******************************************************************/
      int N() const {
         return mChain.size(); }
      /** Merge all event lists form the chain into one.
          @memo Merge
       ******************************************************************/
      void Merge ();
      /** Add the event lists specified in the configuration file.
          The configuration file must contain a list of files
          (one per line) each representing an event list. Lines starting
          with a UNIX comment letter are ignored.
          @memo Configure
       ******************************************************************/
      bool Configure (const char* filename);
      /** Save the chain to the specified file(s). If a non-negative 
          perfile is specified and if the number of events in the chain 
          is larger than perfile, multiple files are created with
          a maximum of perfile events each. The file number is increased
          automatically until maxevents is reached or all events are
          processed.
          @memo Save
          @param filename Name of file
          @param perfile Number of events per written file
          @param maxevents Maximum number of events written
       ******************************************************************/
      bool Save (const char* filename, int perfile = 0,
                int maxevents = -1) const;
   	/** Restore a event chain with one event list from the spcified file.
          @memo Restore
          @param filename Name of file
       ******************************************************************/
      bool Restore (const char* filename);
      /** Size of list (number of events in chain).
          @memo Size
       ******************************************************************/
      int Size() const;
      /** Empty list?
          @memo Empty
       ******************************************************************/
      bool Empty() const;
      /** Equality operator.
          @memo Equality
       ******************************************************************/
      bool operator== (const Chain& l) const;
      /** Inequality operator.
          @memo Equality
       ******************************************************************/
      bool operator!= (const Chain& l) const {
         return !(*this == l); }
      /** Check if events are stored in proper time order.
          @memo Check order
       ******************************************************************/
      bool CheckOrder() const;
      /** Swap the list.
          @memo Swap
       ******************************************************************/
      void Swap (Chain& l);
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      iterator LowerBound (const Time& t) {
         return LowerBound (Event (t)); }
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      const_iterator LowerBound (const Time& t) const {
         return LowerBound (Event (t)); }
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      iterator UpperBound (const Time& t) {
         return UpperBound (Event (t)); }
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      const_iterator UpperBound (const Time& t) const {
         return UpperBound (Event (t)); }
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      iterator LowerBound (const Event& t);
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      const_iterator LowerBound (const Event& t) const;
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      iterator UpperBound (const Event& t);
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      const_iterator UpperBound (const Event& t) const;
      /** Sort the list by event time (the list will be merged first).
          @memo Sort
       ******************************************************************/
      void Sort();
      /** Sort the chain by the specified function in ascending or 
          descending order (the chain will be merged first).
          @memo Sort
       ******************************************************************/
      void Sort (const Function& func, bool ascending = true);
      void Sort (const FunctionPtr& func, bool ascending = true) {
         Sort (*func, ascending); }
   
      /** At.
          @memo At
       ******************************************************************/
      reference At (size_type idx);
      /** At.
          @memo At
       ******************************************************************/
      const_reference At (size_type idx) const;
      /** Operator [].
          @memo Operator[]
       ******************************************************************/
      reference operator[] (size_type idx) {
         return At (idx); }
      /** Operator [].
          @memo Operator[]
       ******************************************************************/
      const_reference operator[] (size_type idx) const {
         return At (idx); }
      /** Front.
          @memo Front
       ******************************************************************/
      reference Front();
      /** Front.
          @memo Front
       ******************************************************************/
      const_reference Front() const;
      /** Back.
          @memo Back
       ******************************************************************/
      reference Back();
      /** Back.
          @memo Back
       ******************************************************************/
      const_reference Back() const;
      /** Begin.
          @memo Begin
       ******************************************************************/
      iterator Begin ();
      /** Begin.
          @memo Begin
       ******************************************************************/
      const_iterator Begin() const;
      /** End.
          @memo End
       ******************************************************************/
      iterator End ();
      /** End.
          @memo End
       ******************************************************************/
      const_iterator End() const;
      /** RBegin.
          @memo RBegin
       ******************************************************************/
      reverse_iterator RBegin () {
         return reverse_iterator (End()); }
      /** RBegin.
          @memo RBegin
       ******************************************************************/
      const_reverse_iterator RBegin () const {
         return const_reverse_iterator (End()); }
      /** REnd.
          @memo REnd
       ******************************************************************/
      reverse_iterator REnd () {
         return reverse_iterator (Begin()); }
      /** REnd.
          @memo REnd
       ******************************************************************/
      const_reverse_iterator REnd () const {
         return const_reverse_iterator (Begin()); }
   
      /** Insert an event to the list at its proper location.
          @memo Insert
       ******************************************************************/
      iterator Insert (const Event& event);
      /** Insert an event to the list at the specified location.
          @memo Insert
       ******************************************************************/
      iterator Insert (iterator pos, const Event& event);
      /** Inserts a range of events.
          @memo Add
       ******************************************************************/
      void Insert (iterator beg, iterator end);
      /** Inserts an event at the back.
          @memo PushBack
       ******************************************************************/
      void PushBack (const Event& event);
      /** Erase an event.
          @memo Erase
       ******************************************************************/
      iterator Erase (iterator pos);
      /** Erase a rane of events.
          @memo Erase
       ******************************************************************/
      iterator Erase (iterator beg, iterator end);
      /** Removes an event from the back.
          @memo PopBack
       ******************************************************************/
      void PopBack();
      /** Clear the chain.
          @memo Clear
       ******************************************************************/
      void Clear () {
         mChain.clear(); }
   
   protected:
   
   private:
      /// Series of event lists
      eventchain		mChain;
   };


}

#endif // _LIGO_EVENTCHAIN_H
