// Wavelet Analysis Tool
//--------------------------------------------------------------------
// Implementation of 
// Symlet wavelets using Fast Wavelet Transform 
// References:
//   I.Daubechies, Ten lectures on wavelets
//   ISBN 0-89871-274-2, 1992
//--------------------------------------------------------------------
//$Id: Symlet.cc 2921 2004-10-14 17:35:03Z sergei.klimenko $

#define SYMLET_CC

#include "Symlet.hh"
#include <time.h>
#include <iostream>

//namespace datacondAPI {
//namespace wat {

extern const double sym2[4]=
{0.48296291314469,0.83651630373747,0.22414386804186,-0.12940952255092};

extern const double sym3[6]=
{0.33267055295096,0.80689150931334,0.45987750211933,-0.13501102001039,
-0.08544127388224,0.03522629188210};

extern const double sym4[8]=
{0.03222310060404,-0.01260396726204,-0.09921954357685,0.29785779560528,
0.80373875180592,0.49761866763202,-0.02963552764600,-0.07576571478927};

extern const double sym5[10]=
{0.01953888273529,-0.02110183402476,-0.17532808990845,0.01660210576452,
0.63397896345821,0.72340769040242,0.19939753397739,-0.03913424930238,
0.02951949092577,0.02733306834508};

extern const double sym6[12]=
{-0.00780070832503,0.00176771186424,0.04472490177067,-0.02106029251230,
-0.07263752278646,0.33792942172762,0.78764114103019,0.49105594192675,
-0.04831174258563,-0.11799011114819,0.00349071208422,0.01540410932703};

extern const double sym7[14]=
{0.01026817670851,0.00401024487153,-0.10780823770382,-0.14004724044296,
0.28862963175151,0.76776431700316,0.53610191709176,0.01744125508686,
-0.04955283493713,0.06789269350137,0.03051551316596,-0.01263630340325,
-0.00104738488868,0.00268181456826};

extern const double sym8[16]=
{0.00188995033276,-0.00030292051472,-0.01495225833705,0.00380875201389,
0.04913717967361,-0.02721902991706,-0.05194583810771,0.36444189483533,
0.77718575170052,0.48135965125837,-0.06127335906766,-0.14329423835081,
0.00760748732492,0.03169508781149,-0.00054213233179,-0.00338241595101};

extern const double sym9[18]=
{0.00106949003291,-0.00047315449868,-0.01026406402763,0.00885926749340,
0.06207778930289,-0.01823377077940,-0.19155083129729,0.03527248803528,
0.61733844914095,0.71789708276441,0.23876091460730,-0.05456895843084,
0.00058346274613,0.03022487885828,-0.01152821020768,-0.01327196778182,
0.00061978088899,0.00140091552591};

extern const double sym10[20]=
{-0.00045932942100,0.00005703608362,0.00459317358531,-0.00080435893202,
-0.02035493981231,0.00576491203358,0.04999497207738,-0.03199005688243,
-0.03553674047382,0.38382676106708,0.76951003702110,0.47169066693844,
-0.07088053578324,-0.15949427888492,0.01160989390371,0.04592723923109,
-0.00146538258130,-0.00864129927702,0.00009563267072,0.00077015980911};

extern const double sym11[22]=
{0.00048926361026,0.00011053509764,-0.00638960366645,-0.00200347190011,
0.04300019068155,0.03526675956446,-0.14460234370532,-0.20465479449578,
0.23768990904928,0.73034354908840,0.57202297801007,0.09719839445889,
-0.02283265102257,0.06997679961073,0.03703741597886,-0.02408084159587,
-0.00985793482879,0.00651249567477,0.00058835273540,-0.00173436626730,
-0.00003879565574,0.00017172195070};

extern const double sym12[24]=
{-0.00017906658698,-0.00001815807886,0.00235029761418,0.00030764779631,
-0.01458983644923,-0.00260439103133,0.05780417944551,0.01530174062248,
-0.17037069723887,-0.07833262231632,0.46274103121933,0.76347909778368,
0.39888597239018,-0.02216230617039,-0.03584883073697,0.04917931829966,
0.00755378061168,-0.02422072267502,-0.00140890924433,0.00741496551766,
0.00018021409009,-0.00134975575557,-0.00001135392804,0.00011196719425};

extern const double sym13[26]=
{0.00007042986691,0.00003690537342,-0.00072136438514,0.00041326119884,
0.00567485376012,-0.00149244727426,-0.02074968632552,0.01761829688065,
0.09292603089915,0.00881975767043,-0.14049009311369,0.11023022302128,
0.64456438390117,0.69573915056158,0.19770481877127,-0.12436246075151,
-0.05975062771796,0.01386249743584,-0.01721164272631,-0.02021676813340,
0.00529635973872,0.00752622538997,-0.00017094285853,-0.00113606343893,
-0.00003573862365,0.00006820325263};

extern const double sym14[28]=
{0.00004461897799,0.00001932901697,-0.00060576018247,-0.00007321421357,
0.00453267747195,0.00101314198718,-0.01943931426363,-0.00236504883674,
0.06982761636183,0.02589858753105,-0.15999741114654,-0.05811182331763,
0.47533576263445,0.75997624196128,0.39320152196202,-0.03531811211520,
-0.05763449835147,0.03743308836280,0.00428052049898,-0.02919621776406,
-0.00275377479123,0.01003769371768,0.00036647657366,-0.00257944172593,
-0.00006286542481,0.00039843567298,0.00001121086581,-0.00002587909027};

extern const double sym15[30]=
{0.00002866070853,0.00002171789015,-0.00040216853760,-0.00010815440169,
0.00348102873707,0.00152613827819,-0.01717125278165,-0.00874478888650,
0.06796982904489,0.06839331006054,-0.13405629845629,-0.19662635876643,
0.24396270543204,0.72184302963632,0.57864041521527,0.11153369514269,
-0.04108266663544,0.04073547969677,0.02193764271975,-0.03887671687685,
-0.01940501143095,0.01007997708790,0.00342345073635,-0.00359016544737,
-0.00026731644647,0.00107056721946,0.00005512254786,-0.00016066186637,
-0.00000735966680,0.00000971241974};

extern const double sym16[32]=
{-0.00001079798210,-0.00000539648318,0.00016545679579,0.00003656592483,
-0.00133872060669,-0.00022211647621,0.00693776113081,0.00135984474247,
-0.02495275804631,-0.00351027506830,0.07803785290360,0.03072113906330,
-0.15959219218533,-0.05404060138707,0.47534280601285,0.75652498787649,
0.39712293361998,-0.03457422841817,-0.06698304907080,0.03233309161055,
0.00486927440478,-0.03105120284367,-0.00312651717226,0.01266673165989,
0.00071821197882,-0.00388091225262,-0.00010844562231,0.00085235471081,
0.00002807858213,-0.00010943147930,-0.00000311355641,0.00000623000670};

extern const double sym17[34]=
{0.00000379125319,-0.00000245271634,-0.00007607124406,0.00002520793314,
0.00071982706421,0.00005840042870,-0.00393232527979,-0.00190540768987,
0.01239698836660,0.00995298252353,-0.01803889724182,-0.00726163475091,
0.01615880872602,-0.08607087472032,-0.15507600534989,0.18053958457921,
0.68148899534246,0.65071662920429,0.14239835041670,-0.11856693260976,
0.01727117821070,0.10475461484209,0.01790395221460,-0.03329138349213,
-0.00481921280320,0.01048236693298,0.00085677007020,-0.00274167597567,
-0.00013864230268,0.00047599638026,-0.00001350638340,-0.00006293702598,
0.00000278012669,0.00000429734333};

extern const double sym18[36]=
{-0.00000151315307,0.00000078472981,0.00002955743762,-0.00000985881603,
-0.00026583011024,0.00004741614518,0.00142808632708,-0.00018877623940,
-0.00523978968302,0.00108778478960,0.01501235634429,-0.00326074419997,
-0.03171268473169,0.00627794455441,0.02852959703912,-0.07379920729075,
-0.03248057329092,0.40148386056943,0.75362914010133,0.47396905989475,
-0.05202915898285,-0.15993814866901,0.03399566710369,0.08421992997028,
-0.00507708516062,-0.03032509108928,0.00164298639727,0.00950216439096,
-0.00041152110922,-0.00231387181450,0.00007021273459,0.00039616840638,
-0.00001402099258,-0.00004524675787,0.00000135491576,0.00000261261256};

extern const double sym19[38]=
{0.00000175093680,0.00000206231706,-0.00002815113866,-0.00001682138703,
0.00027621877686,0.00012930767651,-0.00170496026116,-0.00061792232780,
0.00826223695551,0.00431935187495,-0.02770989693110,-0.01690823486119,
0.08407267627915,0.09363084341623,-0.11624173010592,-0.17659686625066,
0.25826616923642,0.71955552571378,0.57814494533691,0.10902582508132,
-0.06752505803991,0.00895459117312,0.00701557385747,-0.04663598353443,
-0.02265199337798,0.01579743929571,0.00796843832065,-0.00512220500254,
-0.00116070325720,0.00212142502817,0.00015915804768,-0.00063576451500,
-0.00004612039600,0.00011553923334,0.00000887331217,-0.00001188051827,
-0.00000064636513,0.00000054877328};

extern const double sym20[40]=
{-0.00000063291290,-0.00000032567026,0.00001228725278,0.00000452542221,
-0.00011739133516,-0.00002661555034,0.00074761085974,0.00012544091727,
-0.00347164780267,-0.00061112638609,0.01215704094790,0.00193859706772,
-0.03537333675507,-0.00684370196894,0.08891966801896,0.03625095165273,
-0.16057829841234,-0.05108834294808,0.47199147504138,0.75116272838989,
0.40583144438308,-0.02981936882107,-0.07899434489912,0.02557934951284,
0.00812322835892,-0.03162943714139,-0.00331385738520,0.01700404902119,
0.00142308736006,-0.00660658579824,-0.00030526283204,0.00208899470791,
0.00007215991195,-0.00049473109149,-0.00001928412302,0.00007992967834,
0.00000302566606,-0.00000791936141,-0.00000019015676,0.00000036955375};

extern const double sym21[42]=
{0.00000023116904,-0.00000007401089,-0.00000516158836,0.00000183899845,
0.00005820786308,-0.00001055398996,-0.00041349561806,-0.00002060745324,
0.00203118441578,0.00069591568201,-0.00674898277369,-0.00409543219752,
0.01450786941750,0.01118865626885,-0.01508147033057,0.00625464069426,
0.02791728151204,-0.08709884432141,-0.15131078012891,0.19202706524796,
0.68158332059527,0.64618599743615,0.14412386119274,-0.12356548353363,
0.00736738941236,0.10148310770052,0.01067013591536,-0.04716326398300,
-0.00899202924623,0.01540083464058,0.00176596179226,-0.00535395747976,
-0.00043315590952,0.00142132044885,0.00006208657527,-0.00028647517579,
0.00000607913038,0.00004694778085,-0.00000183116834,-0.00000509819139,
0.00000007895915,0.00000024662467};

extern const double sym22[44]=
{0.00000015701037,0.00000009966118,-0.00000328417896,-0.00000146277694,
0.00003442102521,0.00001076285381,-0.00023615708448,-0.00004790058918,
0.00120819750454,0.00021700733081,-0.00470892831539,-0.00077852943445,
0.01485841354863,0.00255497233268,-0.03922962637142,-0.00752552739132,
0.09413257032833,0.04132666895742,-0.15420831552203,-0.04123051452708,
0.47683319799820,0.74899293929327,0.40287666044285,-0.03659053895049,
-0.09061187088512,0.01915428943071,0.00842098867472,-0.03179026919182,
-0.00355937471201,0.01864608663048,0.00168314925131,-0.00800078674386,
-0.00046737262697,0.00282051712122,0.00011079462032,-0.00079412754687,
-0.00003361349829,0.00016465262503,0.00000770422674,-0.00002357636865,
-0.00000098649840,0.00000210708665,0.00000005624840,-0.00000008861606};

extern const double sym23[46]=
{0.00000010235430,0.00000011947301,-0.00000207996803,-0.00000168882731,
0.00002269111797,0.00001337545480,-0.00016515274988,-0.00006891398065,
0.00092065028998,0.00035751068028,-0.00393795248714,-0.00159865021268,
0.01387834485221,0.00751868829733,-0.03854919903172,-0.02759831960490,
0.09194189389449,0.10791667041303,-0.11151590218887,-0.17895622157283,
0.24859349726852,0.70931934236184,0.58652666181596,0.12546897118256,
-0.07028820351936,-0.00454376049568,0.00255211302562,-0.04643623117797,
-0.02186836373024,0.02077476122165,0.01143485977922,-0.00717111121596,
-0.00281604932182,0.00304595670399,0.00045268316044,-0.00123284622195,
-0.00009789651829,0.00036373350955,0.00002965227285,-0.00007386036564,
-0.00000627559386,0.00001006623708,0.00000074523049,-0.00000084388462,
-0.00000003876629,0.00000003321165};

extern const double sym24[48]=
{-0.00000003763894,-0.00000001379902,0.00000092122723,0.00000035313033,
-0.00001083968061,-0.00000471080885,0.00007954430272,0.00003554828409,
-0.00041089637963,-0.00015315078955,0.00165852411624,0.00036533725724,
-0.00578932461719,-0.00070874601122,0.01791452935815,0.00422746501837,
-0.04281324592714,-0.01507710368464,0.08102456049784,0.02392107501492,
-0.16088779689114,-0.02988259975726,0.48880974855408,0.74688385743468,
0.39288115166264,-0.04379342297629,-0.08777599016841,0.03444149388902,
0.02574605129641,-0.02426279125326,-0.00322429774344,0.01824651181612,
0.00027829218166,-0.01009299895819,-0.00061288026541,0.00390135636880,
0.00032248385033,-0.00117205334361,-0.00010277190304,0.00027429214178,
0.00002161714042,-0.00004875059458,-0.00000276094813,0.00000635141112,
0.00000020725965,-0.00000054069132,-0.00000000809775,0.00000002208787};

extern const double sym25[50]=
{0.00000001417393,-0.00000000128584,-0.00000035883307,0.00000008450483,
0.00000460200134,-0.00000106655594,-0.00003862052166,0.00000525387369,
0.00023387577301,0.00000581099069,-0.00106357085882,-0.00024665256419,
0.00363827754098,0.00168360713385,-0.00899375078539,-0.00591958921067,
0.01498023419939,0.00993789458552,-0.01340322915072,0.01798969208769,
0.03862447178667,-0.08574203560471,-0.14658183869341,0.20129994479568,
0.68175692878985,0.64245780610168,0.14440656141754,-0.12888353966089,
-0.00288020261395,0.09620356820102,0.00454479703964,-0.05697322565402,
-0.01086875280482,0.02115133223651,0.00336812528826,-0.00801783983806,
-0.00080537486804,0.00276454631473,0.00021102111165,-0.00074974673235,
-0.00002424720226,0.00016847624768,-0.00000317175456,-0.00003152810174,
0.00000107502980,0.00000434483509,-0.00000008618136,-0.00000036987182,
0.00000000130255,0.00000001435812};

extern const double sym26[52]=
{-0.00000000543571,0.00000000199024,0.00000014430002,-0.00000005275976,
-0.00000183707240,0.00000073688296,0.00001516517917,-0.00000654194941,
-0.00009206773058,0.00003757555029,0.00043268934169,-0.00014414125172,
-0.00160482533375,0.00038470529620,0.00477959384708,-0.00058695506598,
-0.01115011462549,0.00028029270803,0.01851220018216,-0.00454082260968,
-0.02417683712649,0.02788593725265,0.03357868590107,-0.08973261009127,
-0.04066610256625,0.39654996642049,0.74534691300998,0.48659675911554,
-0.03054816636870,-0.16263026270583,0.02381623987822,0.08335972926050,
-0.01571336409339,-0.04542223587036,0.00529771387456,0.02060700942608,
-0.00097581041791,-0.00728998611856,0.00040607789503,0.00226218548420,
-0.00019898334754,-0.00062562025657,0.00005915042437,0.00014332547319,
-0.00001083563455,-0.00002507873243,0.00000124228948,0.00000309673285,
-0.00000008854202,-0.00000024216780,0.00000000335850,0.00000000917271};

extern const double sym27[54]=
{0.00000000200486,-0.00000000225655,-0.00000005780336,0.00000005816294,
0.00000081043393,-0.00000067142671,-0.00000724188402,0.00000454696437,
0.00004551014829,-0.00001987214817,-0.00020932941145,0.00006267580812,
0.00071471422728,-0.00020855313323,-0.00183995581592,0.00103837743414,
0.00396141302209,-0.00484585447809,-0.00946085230775,0.01431280731090,
0.02593596176648,-0.01789276436445,-0.04191593809368,0.00091947692675,
-0.01754596460237,-0.07886544762866,0.13063831337891,0.58946164295955,
0.70449404469039,0.24869269195981,-0.17335255579376,-0.10550196328364,
0.11576172687488,0.09374636823657,-0.03849177869134,-0.04783974419199,
0.01091220407678,0.01925636749894,-0.00314457482910,-0.00679065270580,
0.00076187111254,0.00197699552151,-0.00018852424968,-0.00048322234864,
0.00004468020830,0.00009726406937,-0.00000899977529,-0.00001546107195,
0.00000144610528,0.00000186359765,-0.00000015046913,-0.00000015232407,
0.00000000686339,0.00000000609786};

extern const double sym28[56]=
{-0.00000000223841,-0.00000000080814,0.00000006348994,0.00000002240654,
-0.00000087514214,-0.00000032744949,0.00000772104424,0.00000313803239,
-0.00004848031767,-0.00002021976593,0.00023114897537,0.00008673264856,
-0.00088921450226,-0.00024089432929,0.00295035213651,0.00045574888521,
-0.00885635272453,-0.00132060610611,0.02310526980043,0.00629724042357,
-0.04771096755582,-0.01621457161075,0.08536229609472,0.02357730093902,
-0.16437040480472,-0.03154320812556,0.48424408416030,0.74397532266483,
0.40014593426660,-0.03744630472127,-0.09120194632275,0.03285695485769,
0.02986054897631,-0.02404713054345,-0.00585973849229,0.01865510958775,
0.00038509113319,-0.01202490038160,-0.00054004768399,0.00563725023110,
0.00043338521493,-0.00208037206914,-0.00018786535509,0.00062860489607,
0.00005764318298,-0.00015428555363,-0.00001257590187,0.00003035437524,
0.00000189746857,-0.00000466642977,-0.00000019675201,0.00000052609490,
0.00000001351981,-0.00000003830230,-0.00000000048383,0.00000000134012};

extern const double sym29[58]=
{0.00000000086914,0.00000000007683,-0.00000002476128,0.00000000304434,
0.00000035944572,-0.00000006997480,-0.00000343511732,0.00000067174032,
0.00002415810963,-0.00000310632315,-0.00013140817964,0.00000160162253,
0.00056818760343,0.00008532968392,-0.00195185738910,-0.00065489520898,
0.00527183982333,0.00284145407341,-0.01048311108995,-0.00699646454543,
0.01478801128217,0.00757358653480,-0.01212866901975,0.02896370174289,
0.04899578905306,-0.08385363336444,-0.14426113167429,0.20571029834063,
0.68048314526649,0.64077273046963,0.14693981020727,-0.13193541112313,
-0.01228374764213,0.08980937772282,-0.00093576808478,-0.06412063880974,
-0.01161867252201,0.02656533167184,0.00487002331131,-0.01083258821344,
-0.00136198949647,0.00428940824406,0.00040722859294,-0.00143251730877,
-0.00009078820672,0.00040339823174,0.00000745768225,-0.00009793537610,
0.00000196559708,0.00002001819326,-0.00000066704682,-0.00000321133820,
0.00000007745993,0.00000036644309,-0.00000000281205,-0.00000002591084,
-0.00000000007491,0.00000000084744};

extern const double sym30[60]=
{0.00000000055439,0.00000000025363,-0.00000001650488,-0.00000000612778,
0.00000024412962,0.00000008995017,-0.00000232354978,-0.00000089828469,
0.00001589700483,0.00000647382806,-0.00008272690614,-0.00003293176436,
0.00034385221446,0.00011735869759,-0.00119692134911,-0.00027628415438,
0.00371380699054,0.00054379852399,-0.01038876691564,-0.00162932203336,
0.02556854601492,0.00760939671239,-0.04889669601845,-0.01523693636380,
0.08910160894494,0.02619852893355,-0.16204680547674,-0.02843781124261,
0.48469802756342,0.74269377583417,0.40072254149852,-0.03849152799019,
-0.09630227409198,0.02927099877092,0.02965858799887,-0.02521794463036,
-0.00795200551976,0.01829724962657,0.00037212410631,-0.01284118848030,
-0.00051919347319,0.00645416697646,0.00046642596970,-0.00258964861453,
-0.00023156691354,0.00086145323605,0.00008209441595,-0.00023724380919,
-0.00002120204181,0.00005384704875,0.00000402586750,-0.00000989667625,
-0.00000055432800,0.00000142300528,0.00000005465707,-0.00000014986092,
-0.00000000380480,0.00000001014810,0.00000000014934,-0.00000000032641};


// constructors

template<class DataType_t> Symlet<DataType_t>::
Symlet(const Wavelet &w) : 
WaveDWT<DataType_t>(w) 
{ 
   setFilter();
}

template<class DataType_t> Symlet<DataType_t>::
Symlet(const Symlet<DataType_t> &w) : 
WaveDWT<DataType_t>(w) 
{ 
   setFilter();
}

template<class DataType_t> Symlet<DataType_t>::
Symlet(int m, int tree, enum BORDER border) :
WaveDWT<DataType_t>(m,m,tree,border) 
{
   setFilter();
}

// destructor
template<class DataType_t>
Symlet<DataType_t>::~Symlet()
{ 
   if(pLForward) delete [] pLForward;
   if(pLInverse) delete [] pLInverse;
   if(pHForward) delete [] pHForward;
   if(pHInverse) delete [] pHInverse;
}

// clone
template<class DataType_t>
Symlet<DataType_t>* Symlet<DataType_t>::Clone() const
{
  return new Symlet<DataType_t>(*this);
}

template<class DataType_t>
void Symlet<DataType_t>::setFilter()
{
   const double* pF;
   this->m_H = (this->m_H>>1)<<1;
   int n = this->m_H/2;
   switch(n)
   {
      case  2: pF =  sym2; this->m_H =  4; break;
      case  3: pF =  sym3; this->m_H =  6; break;
      case  4: pF =  sym4; this->m_H =  8; break;
      case  5: pF =  sym5; this->m_H = 10; break;
      case  6: pF =  sym6; this->m_H = 12; break;
      case  7: pF =  sym7; this->m_H = 14; break;
      case  8: pF =  sym8; this->m_H = 16; break;
      case  9: pF =  sym9; this->m_H = 18; break;
      case 10: pF = sym10; this->m_H = 20; break;
      case 11: pF = sym11; this->m_H = 22; break;
      case 12: pF = sym12; this->m_H = 24; break;
      case 13: pF = sym13; this->m_H = 26; break;
      case 14: pF = sym14; this->m_H = 28; break;
      case 15: pF = sym15; this->m_H = 30; break;
      case 16: pF = sym16; this->m_H = 32; break;
      case 17: pF = sym17; this->m_H = 34; break;
      case 18: pF = sym18; this->m_H = 36; break;
      case 19: pF = sym19; this->m_H = 38; break;
      case 20: pF = sym20; this->m_H = 40; break;
      case 21: pF = sym21; this->m_H = 42; break;
      case 22: pF = sym22; this->m_H = 44; break;
      case 23: pF = sym23; this->m_H = 46; break;
      case 24: pF = sym24; this->m_H = 48; break;
      case 25: pF = sym25; this->m_H = 50; break;
      case 26: pF = sym26; this->m_H = 52; break;
      case 27: pF = sym27; this->m_H = 54; break;
      case 28: pF = sym28; this->m_H = 56; break;
      case 29: pF = sym29; this->m_H = 58; break;
      case 30: pF = sym30; this->m_H = 60; break;
      default: pF =  sym4; this->m_H =  8; break;
   }
   
   pLInverse = new double[this->m_H];
   pLForward = new double[this->m_H];
   pHInverse = new double[this->m_H];
   pHForward = new double[this->m_H];

   for(int i=0; i<this->m_H; i+=2){
      pLForward[i]   = pF[i];
      pLForward[i+1] = pF[i+1];
      pHForward[i]   = pF[this->m_H-1-i];
      pHForward[i+1] = -pF[this->m_H-2-i];
      if(n&1){
	 pLInverse[i]   = pF[this->m_H-2-i];
	 pLInverse[i+1] = pF[i+1];
	 pHInverse[i]   = pF[this->m_H-1-i];
	 pHInverse[i+1] = -pF[i];	 
      }
      else{
	 pLInverse[i]   = pF[this->m_H-1-i];
	 pLInverse[i+1] = -pF[i];
	 pHInverse[i]   = pF[this->m_H-2-i];
	 pHInverse[i+1] = pF[i+1];
      }
   }

   this->m_WaveType = SYMLET;
}

// forward function does one step of forward transformation.
// <level> input parameter is the level to be reconstructed
// <layer> input parameter is the layer to be reconstructed.
template<class DataType_t>
void Symlet<DataType_t>::forward(int level,int layer)
{ 
   this->forwardFWT(level, layer, pLForward, pHForward); 
}

// inverse function does one step of inverse transformation.
// <level> input parameter is the level to be reconstructed
// <layer> input parameter is the layer to be reconstructed.
template<class DataType_t>
void Symlet<DataType_t>::inverse(int level,int layer)
{
   this->inverseFWT(level, layer, pLInverse, pHInverse); 
}

// instantiations

#define CLASS_INSTANTIATION(class_) template class Symlet< class_ >;

CLASS_INSTANTIATION(float)
CLASS_INSTANTIATION(double)
//CLASS_INSTANTIATION(std::complex<float>)
//CLASS_INSTANTIATION(std::complex<double>)

#undef CLASS_INSTANTIATION

//}  // end namespace wat
//}  // end namespace datacondAPI
