/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIGMSGWRITER_HH
#define TRIGMSGWRITER_HH

#include "TrigWriter.hh"
#include "TrigProc.hh"
#include "AppClient.hh"
#include "lmsg/MsgAddr.hh"
#include <string>

namespace trig {
    class TrigBase;
    class Segment;

    /**  The MsgWriter class provides the client API for the Trigger manager.
      *  The trigger manager maintains a registry of monitors that may produce 
      *  triggers. The client API requests that it be added to the registry 
      *  with the enroll method. The client may then sends one or more triggers
      *  or segments to the trigger manager with the #addTrigger# or 
      *  #addSegment# method, respectively.
      *  @memo Trigger Client API
      *  @author John G. Zweizig
      *  @version 1.5; Modified June 18, 2009
      */
    class MsgWriter : public TrigWriter {
    public:
	/**  Construct a trigger client.
 	 *  @memo Construct a trigger client.
	 */
	MsgWriter(void);

	/**  Destroy a trigger client.
	  *  @memo Destroy a trigger client.
	  */
	virtual ~MsgWriter(void);

	/**  Send a segment to be logged by the trigger manager.
	  *  @memo Log a Segment with the trigger manager.
	  *  @return Error code enumerated in lmsg::Errorlist.hh.
	  *  @param segm Segment to be logged in the metadatabase.
	  */
	lmsg::error_type addSegment(const Segment& segm, const TrigProc& p);

	/**  Send a trigger to the trigger manager.
	  *  @memo Log a trigger with the trigger manager.
	  *  @return Error code enumerated in lmsg::Errorlist.hh.
	  *  @param trigger Trigger to be logged in the metadatabase.
	  */
	lmsg::error_type addTrigger(const TrigBase& trigger, const TrigProc& p);

	void clear(const Time& start, const Time& end) {}

	/**  The process is registered with the trigger manager. On successful
	  *  enrolment, the trigger manager returns a unique process identifier
	  *  which is used to identify the process in subsequent requests.
	  *  @memo Enroll process with the Trigger Manager.
	  *  @return Error code enumerated in lmsg::Errorlist.hh.
	  */
	lmsg::error_type enroll(void);

	/**  Get athe debug level (from AppClient)
	  */
	lmsg::index_type getDebug(void) const;

	trig_mode getMode(void) const;

	/**  This function removes all processes from a specified node.
	  *  This is dangerous and a hideous kludge, but it should come in 
	  *  handy if a node fails and the processes running on that node 
	  *  don't get a chance to unregister.
	  *  @memo Tell TrigMgr to remove all processes on a specified node.
	  *  @return Error code enumerated in lmsg::Errorlist.hh.
	  *  @param node Name of node to be dumped.
	  */
	lmsg::error_type removeByNode(const std::string& node);

	/**  Set the debug level.
	  *  @memo Set the debug level.
	  *  @param level New debugging level.
	  */
	void setDebug(lmsg::index_type level);

	/**  Register a process with the trigger manager.
	  *  @memo Register a process with the trigger manager.
	  *  @return Error code enumerated in lmsg::Errorlist.hh.
	  *  @param p Process ID toe be registered.
	  */
	lmsg::error_type setProcess(const TrigProc& p);

	lmsg::error_type write(const std::string& f, const Time& start, 
			       const Time& end) const;

    private:
	/**  This function sends a data message (i.e. a message containing 
	  *  trigger or segment object) to the trigger manager. SendMessage 
	  *  will enroll the trigger process with the trigger manager if it
	  *  is not currently registered. It will also invalidate the trigger
	  *  manager address (i.e. mark the process as not registered) if the
	  *  message fails with a BadAddress code.
	  *  @memo Send a data message to the trigger manager.
	  *  @return Error code enumerated in lmsg::Errorlist.hh.
	  *  @param msg Message to be logged in the metadatabase.
	  */
	lmsg::error_type sendMessage(const lmsg::Message& msg);

    private:
	lmsg::AppClient mClient;
	lmsg::MsgAddr   mTrigManager;
    };

    //====================================  Inline methods.
    inline lmsg::index_type 
    MsgWriter::getDebug(void) const {
	return mClient.getDebug();
    }

    inline TrigWriter::trig_mode 
    MsgWriter::getMode(void) const {
	return kMgr;
    }

} // namespace trig

#endif  //   TRIGMGRCLIENT_HH
