/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "LdasDBWriter.hh"
#include "Segment.hh"
#include "TrigBase.hh"
#include "TrigProc.hh"
#include "TrigTable.hh"
#include "xsil/Xwriter.hh"
#include "xsil/ligolw.hh"
#include "lmsg/ErrorList.hh"
#include <fstream>
#include <sstream>
#include <algorithm>

using namespace std;
using namespace trig;

//====================================  Ldas writer constructor
LdasDBWriter::LdasDBWriter(void) 
    : mNPrcWrt(0)
{
}

//====================================  Ldas writer destructor
LdasDBWriter::~LdasDBWriter(void) {
}

//====================================  Add a trigger to the list
lmsg::error_type 
LdasDBWriter::addTrigger(const TrigBase& t, const TrigProc& p) {
    proc_iter pit = insert_proc(p);
    if (!mTrgList.empty() && mTrgList.back() == t) {
	cerr << "Trigger repeated. ID: " << t.getID() 
	     << " SubID: " << t.getSubID()  << endl;
	return lmsg::Invalid;
    }
    mTrgList.push_back(t);
    mTrgList.back().setProcess(pit->getProcessID());
    if (getDebug() > 1) cout << "Trigger " << t.getID() << ":" << t.getSubID()
			     << " inserted at " << t.getTime() << endl;
    return lmsg::OK;
}

//======================================  Add a segment to the list
lmsg::error_type 
LdasDBWriter::addSegment(const Segment& s,  const TrigProc& p) {
    if (s.getActivity() <= 0) return lmsg::OK;
    proc_iter pit = insert_proc(p);
    if (!mSegList.empty() && mSegList.back() == s) {
	cerr << "Segment repeated. Group: " << s.getGroup() << endl;
	return lmsg::Invalid;
    }
    mSegList.push_back(s);
    mSegList.back().setProcess(pit->getProcessID());
    if (getDebug() > 2) cout << "Segment " << s.getGroup() << " inserted at " 
			     << s.getStartTime() << "-" << s.getEndTime()
			     << endl;
    return lmsg::OK;
}

//======================================  Clear temporary tables
void 
LdasDBWriter::clear(const Time& start, const Time& end) {
    // mPrcList.clear();
    mNPrcWrt = mPrcList.size();

    //----------------------------------  Clear the used segments
    for (seg_iter i=mSegList.begin(); i!=mSegList.end(); ) {
	seg_iter j = i++;
	if (!end) {
	    mSegList.erase(j);
	} else if (j->getStartTime() < end) {
	    if (j->getEndTime() > end) j->setStartTime(end);
	    else                       mSegList.erase(j);
	}
    }
 
    //----------------------------------  Clear the used triggers
    for (trig_iter i=mTrgList.begin(); i!=mTrgList.end(); ) {
	trig_iter j = i++;
	if (!end || j->getTime() < end) {
	    mTrgList.erase(j);
	}
    }
}

//======================================  Get earliest time
Time
LdasDBWriter::getEarly(void) const {
    Time t(0);
    for (const_seg_iter i=mSegList.begin(); i!=mSegList.end(); ++i) {
	if (!t || i->getStartTime() < t) t =  i->getStartTime();
    }
    for (const_trig_iter i=mTrgList.begin(); i!=mTrgList.end(); ++i) {
	if (!t || i->getTime() < t) t =  i->getTime();
    }
    return t;
}

//======================================  Data size methods
int
LdasDBWriter::getNSegs(const Time& t) const {
    if (!t) return mSegList.size();
    int N = 0;
    for (const_seg_iter i=mSegList.begin(); i!=mSegList.end(); ++i) {
	if (i->getStartTime() < t) N++;
    }
    return N;
}

int
LdasDBWriter::getNTrigs(const Time& t) const {
    if (!t) return mTrgList.size();
    int N = 0;
    for (const_trig_iter i=mTrgList.begin(); i!=mTrgList.end(); ++i) {
	if (i->getTime() < t) N++;
    }
    return N;
}

//======================================  Add a segment to the list
lmsg::error_type 
LdasDBWriter::setProcess(const TrigProc& p) {
    refProcess()  = p;
    Time idTime(Now());
    idTime.setN(mPrcList.size()+1); // replace ns field with unique ID
    char line[128];
    TimeStr(idTime, line, "%Y%02m%02d%02H%02N%02S%06n010000");
    refProcess().setProcessID(line);
    //proc_iter pit = 
    insert_proc(refProcess());
    return lmsg::OK;
}

//======================================  Write out tables as xml.
lmsg::error_type 
LdasDBWriter::write(const string& file, const Time& start, 
		    const Time& end) const {
    if (mTrgList.empty() && mSegList.empty()) return lmsg::OK;
    ofstream out(file.c_str());
    if (!out.is_open()) {
	cerr << "LdasDBWriter is unable to open file: " << file << endl;
	return lmsg::Failure;
    }
    xsil::Xwriter xw(out);
    xw.setDocType("SYSTEM \"http://gateway/doc/ligolwAPI/html/ligolw_dtd.txt\"");
    xsil::ligolw lwfile("ligo:ldas:file");
 
    //----------------------------------  Fill in the (new) processes
    if (int(mPrcList.size()) > mNPrcWrt) {
	const_proc_iter pStart = mPrcList.begin();
	for (int i=0; i<mNPrcWrt; ++i) ++pStart;

	ProcTable* pTab = new ProcTable(false);
	for (const_proc_iter i=pStart; i!=mPrcList.end(); ++i) {
	    pTab->addRow(*i);
	    //mNPrcWrt++;
	}
	lwfile.addObject(pTab);
    }
 
    //----------------------------------  Fill in the segments
    int nSegs = 0;
    SegTable* sTab = new SegTable(false);
    for (const_seg_iter i=mSegList.begin(); i!=mSegList.end(); ++i) {
	if (!end || i->getEndTime() <= end) {
	    sTab->addRow(*i);
	    nSegs++;
	} else if (i->getStartTime() < end) {
	    trig::Segment seg(*i);
	    seg.setEndTime(end);
	    sTab->addRow(seg);
	    nSegs++;
	}
    }
    if (nSegs) lwfile.addObject(sTab);
    else       delete sTab;
 
    //----------------------------------  Fill in the triggers
    int nTrgs = 0;
    TrigTable* tTab = new TrigTable(false);
    for (const_trig_iter i=mTrgList.begin(); i!=mTrgList.end(); ++i) {
	if (!end || i->getTime() < end) {
	    tTab->addRow(*i);
	    nTrgs++;
	}
    }
    if (nTrgs) lwfile.addObject(tTab);
    else       delete tTab;

    //----------------------------------  Spew the file.
    lwfile.Spew(xw);
    if (out.fail()) {
	cerr << "LdasDBWriter: Writing triggers/segments to file: " 
	     << file << " failed." << endl;
	return lmsg::Failure;
    } else if (getDebug()) {
	cout << "Wrote " << nTrgs << " triggers and " << nSegs 
	     << " segments to file " << file << endl;
    }
    return lmsg::OK;
}

//======================================  Add/find process in list
LdasDBWriter::proc_iter 
LdasDBWriter::insert_proc(const TrigProc& p) {
    proc_iter pit = find(mPrcList.begin(), mPrcList.end(), p);
    if (pit == mPrcList.end()) {
	int pid = mPrcList.size();
	mPrcList.push_back(p);
	pit = mPrcList.end(); --pit;
	if ( ! *(pit->getProcessID())) {
	    ostringstream id;
	    id << "process:process_id:" << pid;
	    pit->setProcessID(id.str());
	}
	if (getDebug()) {
	    cerr << "LdasDBWriter: Added process: " << pit->getName()
		 << " pID: " << pit->getProcessID() << endl;
	}
    } else if (getDebug() > 1) {
	cerr << "LdasDBWriter: Found process: " << pit->getName()
	     << " pID: " << pit->getProcessID() << endl;
    }
    return pit;
}

