/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "MonServer.hh"
#include "lmsg/MsgTypes.hh"
#include "lmsg/ErrorList.hh"
#ifndef DMTOFFLINE
#include "MonDataMsg.hh"
#include "lmsg/MsgHandler.hh"
#include "NameMsg.hh"
#endif
#include <iostream>

using namespace lmsg;
using namespace std;

//==========================================  Request handler class
#ifndef DMTOFFLINE
class ReqHandler : public MsgHandler {
public:
    ReqHandler(const MonServer::Dictionary& dict) : mDict(dict) {}
    ~ReqHandler(void) {}
    error_type handleMsg(AppServer& app, const MsgHeader& hdr);
    void* getMsgbufAddr(void);
    lmsg::size_type getMsgbufLength(void);
    Message& getMessage(void);
private:
    const MonServer::Dictionary& mDict;
    MDM_Request mReq;
};

//======================================  Request handler inline methods
Message&
ReqHandler::getMessage(void) {
    return mReq;
}

void* 
ReqHandler::getMsgbufAddr(void) {
    return reinterpret_cast<void*>(&mReq);
}

size_type  
ReqHandler::getMsgbufLength(void) {
    return sizeof(mReq);
}

template<class T>
inline const T& deref(const void* p) {
    return *reinterpret_cast<const T*>(p);
}

template<class T>
inline T& deref(void* p) {
    return *reinterpret_cast<const T*>(p);
}

error_type 
ReqHandler::handleMsg(AppServer& app, const MsgHeader& hdr) {
    error_type rc;
    Message* reply = 0;

    MonServer::const_dict_iter dp = mDict.find(string(mReq.getName()));
    if (dp != mDict.end() && dp->second.getAddr() != 0) {
        const void* addr = dp->second.getAddr();
	switch (dp->second.getType()) {
	case DataDesc::t_void:
	    break;

	case DataDesc::t_long:
	    reply = new MDM_Int(mReq.getName(), deref<long>(addr));
	    break;

	case DataDesc::t_double:
	    reply = new MDM_Double(mReq.getName(), deref<double>(addr));
	    break;

	case DataDesc::t_string:
	    reply = new MDM_String(mReq.getName(), deref<string>(addr));
	    break;

	case DataDesc::t_TSeries:
	    reply = new MDM_TSeries(mReq.getName(), deref<TSeries>(addr));
	    break;

	case DataDesc::t_FSeries:
	    reply = new MDM_FSeries(mReq.getName(), deref<FSeries>(addr));
	    break;

	case DataDesc::t_FSpectrum:
	    reply = new MDM_FSpectrum(mReq.getName(), deref<FSpectrum>(addr));
	    break;

	case DataDesc::t_Histogram1:
	    reply = new MDM_Histogram1(mReq.getName(), deref<Histogram1>(addr));
	    break;

	case DataDesc::t_asd:
	    reply = new MDM_ASD(mReq.getName(), deref<containers::ASD>(addr));
	    break;

	case DataDesc::t_dft:
	    reply = new MDM_DFT(mReq.getName(), deref<containers::DFT>(addr));
	    break;

	case DataDesc::t_psd:
	    reply = new MDM_PSD(mReq.getName(), deref<containers::PSD>(addr));
	    break;
	}
    }
    if (!reply) reply = new MDM_NAck;
    rc = app.reply(hdr, *reply);
    delete reply;
    return rc;
}

//======================================  Index handler implementation
class IndexHandler : public MsgHandler {
public:
    IndexHandler(const MonServer::Dictionary& dict) : mDict(dict) {}
    ~IndexHandler(void) {}
    error_type handleMsg(AppServer& app, const MsgHeader& hdr);
    void* getMsgbufAddr(void);
    lmsg::size_type getMsgbufLength(void);
    Message& getMessage(void);
private:
    const MonServer::Dictionary& mDict;
    MDM_ReqIndex mReq;
};

Message&
IndexHandler::getMessage(void) {
    return mReq;
}

error_type
IndexHandler::handleMsg(AppServer& app, const MsgHeader& hdr) {
    const string::size_type trunc(65000);
    string toc;
    for (MonServer::const_dict_iter i=mDict.begin(); i != mDict.end(); ++i) {
        toc += i->first;
	toc += ";";
	switch (i->second.getType()) {
	case DataDesc::t_void:
	    toc += "void";
	    break;
	case DataDesc::t_long:
	    toc += "long";
	    break;
	case DataDesc::t_double:
	    toc += "double";
	    break;
	case DataDesc::t_string:
	    toc += "string";
	    break;
	case DataDesc::t_TSeries:
	    toc += "TSeries";
	    break;
	case DataDesc::t_FSeries:
	case DataDesc::t_dft:
	    toc += "FSeries";
	    break;
	case DataDesc::t_FSpectrum:
	case DataDesc::t_psd:
	case DataDesc::t_asd:
	    toc += "FSpectrum";
	    break;
	case DataDesc::t_Histogram1:
	    toc += "Histogram1";
	    break;
	}
	toc += ";";
	toc += i->second.getComment();
	toc += "\n";
    }

    //-----------------------------------  Truncate the string if necessary
    if (toc.size() > trunc) {
        string::size_type i = toc.rfind('\n', trunc);
	if (i != string::npos) toc.erase(i+1);
    }
	
    //-----------------------------------  Send the reply
    return app.reply(hdr, MDM_Index(toc));
}

void* 
IndexHandler::getMsgbufAddr(void) {
    return (void*)0;
}

lmsg::size_type  
IndexHandler::getMsgbufLength(void) {
    return 0;
}
#endif // !def(DMTOFFLINE)

//======================================  MonServer Implementation
MonServer::MonServer(const char* Process) 
{
#ifndef DMTOFFLINE
    AppServer::open(o_async);
    setServerName(Process);
    setTimeOut(0.0);
    addHandler(mdm_Request,  new ReqHandler(mDict));
    addHandler(mdm_ReqIndex, new IndexHandler(mDict));
#endif
}

MonServer::~MonServer(void) {
}

void
MonServer::processMsg(void) {
#ifndef DMTOFFLINE
    while (waitMsg(0.0)) {    // waitMsg returns true if data available
        error_type rc = handleMessage();
	if ((rc && getDebug()) || getDebug() > 2) {
	    cout << "AppServer::handleMsg returned with rc=" << rc << endl;
	}
    }
#endif
}

void
MonServer::Attention(void) {
    // cout << "Got a Message!" << endl;
    processMsg();
}

void 
MonServer::setServerName(const char* Name) {
#ifndef DMTOFFLINE
    if (Name) AppServer::register_name(Name, p_Monitor);
#endif
}

//======================================  Construct a data descriptor
DataDesc::DataDesc(void) {
    mType = t_void;
    mAddr = 0;
}

DataDesc::DataDesc(const long* addr, const char* Comm) {
    mType = t_long;
    mAddr = reinterpret_cast<const void*>(addr);
    if (Comm) mComment = Comm;
}

DataDesc::DataDesc(const double* addr, const char* Comm) {
    mType = t_double;
    mAddr = reinterpret_cast<const void*>(addr);
    if (Comm) mComment = Comm;
}

DataDesc::DataDesc(const string* addr, const char* Comm) {
    mType = t_string;
    mAddr = reinterpret_cast<const void*>(addr);
    if (Comm) mComment = Comm;
}

DataDesc::DataDesc(const TSeries* addr, const char* Comm) {
    mType = t_TSeries;
    mAddr = reinterpret_cast<const void*>(addr);
    if (Comm) mComment = Comm;
}

DataDesc::DataDesc(const FSeries* addr, const char* Comm) {
    mType = t_FSeries;
    mAddr = reinterpret_cast<const void*>(addr);
    if (Comm) mComment = Comm;
}

DataDesc::DataDesc(const FSpectrum* addr, const char* Comm) {
    mType = t_FSpectrum;
    mAddr = reinterpret_cast<const void*>(addr);
    if (Comm) mComment = Comm;
}

DataDesc::DataDesc(const Histogram1* addr, const char* Comm) {
    mType = t_Histogram1;
    mAddr = reinterpret_cast<const void*>(addr);
    if (Comm) mComment = Comm;
}

DataDesc::DataDesc(const containers::ASD* addr, const char* Comm) {
    mType = t_asd;
    mAddr = reinterpret_cast<const void*>(addr);
    if (Comm) mComment = Comm;
}

DataDesc::DataDesc(const containers::DFT* addr, const char* Comm) {
    mType = t_dft;
    mAddr = reinterpret_cast<const void*>(addr);
    if (Comm) mComment = Comm;
}

DataDesc::DataDesc(const containers::PSD* addr, const char* Comm) {
    mType = t_psd;
    mAddr = reinterpret_cast<const void*>(addr);
    if (Comm) mComment = Comm;
}

DataDesc::~DataDesc(void) {
}
