#ifndef MONCLIENT_HH
#define MONCLIENT_HH

#include "lmsg/MsgHeader.hh"
#include "AppClient.hh"
#include<string>

class TSeries;
class FSeries;
class FSpectrum;
class Histogram1;

/**  The monitor data client API allows a user to get data from any 
  *  monitor process running a data server. The Name server must be 
  *  running to allow the clent to find the monitor, and it's location 
  *  must be specified using the DMTNAMESERVER environment variable.
  *  The client process must know the name and type of data to access 
  *  served data. A table of contents containing the data name, the data 
  *  type, and a comment field is available from the servers. 
  *  @memo Monitor Data Client API.
  *  @author John Zweizig
  *  @version 1.1; Modified March 28, 2000
  */
class MonClient : private lmsg::AppClient {
public:

  /** Construct a client.
   */
  MonClient(void);

  /**  Construct a client and connect it logically to the specified server 
    *  process.
    *  @memo Construct a client and connect it to a server.
    *  @param Process Server process name the client will connect to.
    */
  MonClient(const char* Process);

  /**  Destroy the client and release any ports currently in use by the 
    *  client.
    *  @memo Disconnect and delete the Client.
    */
  ~MonClient(void);

  /**  Look up the address of the specified server. All subsequent 
    *  data requests will be sent to the named server until another
    *  setSever() request is made.
    *  @memo Connect to a server.
    *  @return An error code from lmsg::ErrorList.
    *  @param Server Name of server to which the client will 
    */
  lmsg::error_type setServer(const char* Server);

  /**  Get an integer data word with the specified name.
    *  @memo Get integer data.
    *  @return An error code from lmsg::ErrorList.
    *  @param  Name Name of data value to be fetched from the server.
    *  @param  iVal pointer to the long word to receive the data.
    */
  lmsg::error_type getData(const char* Name, long* ival);

  /**  Get a double precision floating point data item with the specified 
    *  name.
    *  @memo Get float data.
    *  @return An error code from lmsg::ErrorList.
    *  @param  Name Name of data value to be fetched from the server.
    *  @param  dVal pointer to the double float word to received the data.
    */
  lmsg::error_type getData(const char* Name, double* dVal);

  /**  Get a character string with the specified name.
    *  @memo Get character string data.
    *  @return An error code from lmsg::ErrorList.
    *  @param  Name Name of data value to be fetched from the server.
    *  @param  sVal pointer to a string into which the data will be stored.
    */
  lmsg::error_type getData(const char* Name, std::string* sVal);

  /**  Get a Time Series with the specified name.
    *  @memo Get TSeries data.
    *  @return An error code from lmsg::ErrorList.
    *  @param  Name Name of data value to be fetched from the server.
    *  @param  ts   pointer to a TSeries into which the data will be stored.
    */
  lmsg::error_type getData(const char* Name, TSeries* ts);

  /**  Get a Frequency series with the specified name.
    *  @memo Get FSeries data.
    *  @return An error code from lmsg::ErrorList.
    *  @param  Name Name of data value to be fetched from the server.
    *  @param  fs   Pointer to an FSeries into which the data will be stored.
   */
  lmsg::error_type getData(const char* Name, FSeries* fs);

  /**  Get a power spectrum with the specified name.
    *  @memo Get FSpectrum data.
    *  @return An error code from lmsg::ErrorList.
    *  @param  Name Name of data value to be fetched from the server.
    *  @param  fs   Pointer to an FSeries into which the data will be stored.
   */
  lmsg::error_type getData(const char* Name, FSpectrum* fs);

  /**  Get a 1-d histogram with the specified name.
    *  @memo Get Histogram1 data.
    *  @return An error code from lmsg::ErrorList.
    *  @param  Name Name of data value to be fetched from the server.
    *  @param  fs   Pointer to an Histogram1 into which the data will be stored.
   */
  lmsg::error_type getData(const char* Name, Histogram1* hg);

  /**  Get the index of named data elements available from the current 
    *  server. The index is returned as a single character string. The 
    *  string contains one entry for each data element known to the server.
    *  Entries contain the data name (key), the data type and a comment 
    *  string  separated by semicolons (";"). The entries are in turn 
    *  separated by new-line characters.
    *  @memo List the data available from the server.
    *  @return An error code from lmsg::ErrorList.
    *  @param  sVal pointer to the string to receive the index.
    */
  lmsg::error_type getIndex(std::string& sVal);

  /**  Make AppClient methods public.
   */
  void setDebug(lmsg::index_type level);

private:
  lmsg::MsgAddr mServer;

};

#endif  //  MONCLIENT_HH

