/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "AppServer.hh"
#include "lmsg/Buffer.hh"
#include "lmsg/BufferPool.hh"
#include "lmsg/ErrorList.hh"
#include "lmsg/MsgAddr.hh"
#include "lmsg/Message.hh"
#include "lmsg/MsgHandler.hh"
#include "NameClient.hh"
#include "lmsg/SocketPool.hh"
#ifdef TCP_TRANSPORT
#include "lmsg/TransportTCP.hh"
#else
#include "lmsg/TransportMsg.hh"
#endif
#include <stdio.h>
#include <iostream>
#include <cstdlib>

using namespace lmsg;
using namespace std;

//======================================  AppServer destructor
AppServer::~AppServer(void) {
    if (isOpen()) {
        close();
        delete mServer;
	mServer = 0;
    }
    for (Handler_iterator i=mHandler.begin(); i != mHandler.end(); ++i) {
	delete i->second;
    }
    mHandler.clear();
}

//======================================  AppServer constructor
AppServer::AppServer(const char* Name, long ofl, NameProcs pType) 
  :  mServerType(pType), mRegistered(false), 
     mBuffPool(&defaultLMsgBufferPool), mTimeout(3600.0), mDebug(0)
{
    const char* deblvl=getenv("APPSERVER_DEBUG");
    if (deblvl) {
        if (*deblvl) mDebug = strtol(deblvl, 0, 0);
        else         mDebug = 1;
    }
#ifdef TCP_TRANSPORT
    mServer = new TransportTCP();
#else
    mServer = new TransportMsg();
#endif
    mServer->setDebug(mDebug);
    if (Name) {
        mServerName = Name;
	open(ofl | o_register);
    } else if (ofl != o_none) {
        open(ofl);
    }
}

//======================================  Close an application port
error_type 
AppServer::close(void) {
    if (isOpen()) {
	unregister();
        mServer->close();
    }
    // delete mServer;
    // mServer = 0;
    return OK;
}

//======================================  Test whether socket is open
bool
AppServer::isOpen(void) const {
    return (mServer && mServer->isOpen());
}

//======================================  Open server socket
error_type 
AppServer::open(long flags) {
    if (mServer->isOpen()) close();
#ifdef TCP_TRANSPORT
    long fl = TransportTCP::o_none;
    if ((flags & o_async))  fl |= TransportTCP::o_async;
#else
    long fl = TransportMsg::o_none;
    if ((flags & o_async))  fl |= TransportMsg::o_async;
#endif
    error_type rc = mServer->open(fl, &mPort);
    if (rc) return rc;
    mServer->setDebug(mDebug);
    if (!mServerName.empty()) {
        NameClient nc;
        register_name(nc);
    }
    return rc;
}

//======================================  Receive message with default timeout
error_type 
AppServer::receive(MsgHeader& hdr, void* data, size_type length) {
    return receive(hdr, data, length, mTimeout);
}

//======================================  Receive message as header, data
error_type 
AppServer::receive(MsgHeader& hdr, void* data, size_type length, 
		   double timeout) {
    if (!mServer->isOpen()) return NotOpen;
    return mServer->receive(hdr, (char*)data, length, timeout);
}

//======================================  Reply to message
error_type 
AppServer::reply(const MsgHeader& to, const MsgHeader& rhdr, 
		       const void* data) {
    if (!mServer->isOpen()) return NotOpen;
    MsgHeader h(rhdr);
    h.setDest(to.getSource());
    h.setTransID(to.getTransID());
    return send(h, (const char*)data);
}

//======================================  Reply to message
error_type 
AppServer::reply(const MsgHeader& to, const Message& msg) {
    MsgHeader h(msg.getType(), msg.size(), to.getSource());
    h.setTransID(to.getTransID());
    return send(h, msg);
}

//======================================  Send message from server port
error_type 
AppServer::send(const MsgHeader& to, const Message& msg) {
    if (!mServer->isOpen()) return NotOpen;
    return mServer->send(to, msg);
}

//======================================  Send message from server port
error_type 
AppServer::send(const MsgHeader& hdr, const void* data) {
    if (!mServer->isOpen()) return NotOpen;
    return mServer->send(hdr, (const char*)data);
}

//======================================  Set default buffer pool
void 
AppServer::setBufferPool(BufferPool* pool) {
    if (pool) mBuffPool = pool;
    if (mServer) mServer->setBufferPool(mBuffPool);
}

//======================================  Set debug message level
void 
AppServer::setDebug(index_type debug) {
    mDebug = debug;
    if (mServer) mServer->setDebug(debug);
}

//======================================  Set the name server address
void 
AppServer::setDomainName(const char* domain) {
    mDomainName = domain;
}

//======================================  Set the server port
void 
AppServer::setServerPort(const MsgAddr& addr) {
    mPort = addr;
}

//======================================  Set the default timeout
void 
AppServer::setTimeOut(wtime_type time) {
    mTimeout = time;
}

//======================================  Add a message handler to the lsit
void 
AppServer::addHandler(MsgHeader::MsgType type, MsgHandler* handler) {
    mHandler[type] = handler;
}

//======================================  Get message. Handle it appropriately
error_type
AppServer::handleMessage(void) {
    error_type rc = mServer->waitMsg(mTimeout);
    if (rc) {
        if (mDebug) {
	    if (rc != SystemError) cerr << "Error in waitMsg: " << rc << endl;
	    else                   perror("Error in waitMsg");
	}
	return rc;
    }

    //----------------------------------  Get a message buffer.
    Buffer* buf;
    rc = mServer->receive(&buf);
    if (rc) {
        if (mDebug) {
	    if (rc != SystemError) cerr << "Error in receive: " << rc << endl;
	    else                   perror("Error in receive");
	}
	return rc;
    }

    //----------------------------------  Find a handler for this message
    MsgHeader hdr = buf->getHeader();
    if (mDebug) {
        std::cout << "Message of length " << hdr.getMsgLength() 
		  << " received from " << hdr.getSource() << std::endl;
    }
    Handler_iterator maptr = mHandler.find(hdr.getMsgType());
    if (maptr == mHandler.end()) {
        if (getDebug()) {
	    std::cerr << "AppServer: No Handler for message type "
		 << hdr.getMsgType() << std::endl;
	}
	buf->Return();
	return NoHandler;
    }
    MsgHandler* handler = maptr->second;

    //----------------------------------  Run it by the message handler.
    try {
        rc = handler->handleBuffer(*this, *buf);
    } catch (std::exception& e) {
        if (getDebug()) {
	    std::cerr << "Exception: "<<e.what()<<" in handler for message "
		      << buf->getHeader().getMsgType() << std::endl;
	}
        rc = Invalid;
    }
    buf->Return();

    //----------------------------------  Disconnect the server port.
    error_type r1 = mServer->disconnect();
    if (r1 && getDebug()) perror("Error in disconnect");
    return rc;
}

bool
AppServer::waitMsg(wtime_type time) {
    if (!mServer->isOpen()) return false;
    error_type rc = mServer->waitMsg(time);
    while (time == 0.0 && rc == Continue) rc = mServer->waitMsg(time);
    if (rc == SystemError) perror("Error in AppServer::waitMsg");
    return (rc == OK);
}

error_type
AppServer::register_name(const char* Server, NameProcs type) {
    error_type rc = OK;
    if (isOpen()) {
        NameClient nc;
	if (mRegistered && !mServerName.empty()) {
	    nc.remName(mServerName.c_str());
	    mRegistered = false;
	}
	mServerName = Server;
	mServerType = type;
	rc = register_name(nc);
    } else {
	mServerName = Server;
	mServerType = type;
    }
    return rc;
}

error_type
AppServer::register_name(NameClient& nc) {
    error_type rc;
    if (!mServer->isOpen()) {
        rc = OK;
    } else if (mServerName.empty()) {
        rc = Invalid;
    } else {
        rc = nc.addName(mServerName.c_str(), mServer->getAddr(), mServerType);
	mRegistered = (rc == OK);
	if (!mRegistered) {
	    cerr << "%%error%% Unable to register service: " << mServerName
		 << endl;
	    cerr << "%%error%% try \"NameCtrl add -a " << mServer->getAddr()
		 << " -t " << mServerType << " " << mServerName << "\"" << endl;
	}
    }
    return rc;
}

error_type
AppServer::unregister(void) {
    if (!isOpen() || mServerName.empty() || !mRegistered) return OK;
    NameClient nc;
    mRegistered = false;
    return nc.remName(mServerName.c_str());
}
