/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "AlarmClient.hh"
#include <iostream>
#include <string>
#include <vector>
#include <cstdlib>

using namespace std;

enum cmdList {
    kNone = 0,
    kDefine,
    kSet,
    kRequest,
    kGetActive,
    kGetDefined,
    kProlong,
    kCancel,
    kAllActive,
    kEnable,
    kDisable,
    kGetTemplate,
    kRemove,
    kTemplateCmd
};

static void dumpAlarm(ostream& out, const AlarmData& x);

int 
main(int argc, const char* argv[]) {
    bool syntax(false);

    //----------------------------------  Get the command
    cmdList iCmd(kNone);
    if (argc < 2) {
        cerr << "No command specified" << endl;
	syntax = true;
    } else {
        string argi=argv[1];
	if (argi == "define" ) {
	    iCmd = kDefine;
	} else if (argi == "set" ) {
	    iCmd = kSet;
	} else if (argi == "request" ) {
	    iCmd = kRequest;
	} else if (argi == "getactive" ) {
	    iCmd = kGetActive;
	} else if (argi == "getdefined" ) {
	    iCmd = kGetDefined;
	} else if (argi == "prolong" ) {
	    iCmd = kProlong;
	} else if (argi == "cancel" ) {
	    iCmd = kCancel;
	} else if (argi == "allactive" ) {
	    iCmd = kAllActive;
	} else if (argi == "enable" ) {
	    iCmd = kEnable;
	} else if (argi == "disable" ) {
	    iCmd = kDisable;
	} else if (argi == "gettemplate" ) {
	    iCmd = kGetTemplate;
	} else if (argi == "dumptemplate" ) {
	    iCmd = kTemplateCmd;
	} else if (argi == "remove" ) {
	    iCmd = kRemove;
	} else {
	    iCmd = kNone;
	    cerr << "Unrecognized command: " << argi << endl;
	    syntax = true;
	}
    }

    //----------------------------------  Print help if requested
    if (argc >= 3 && (string(argv[2])=="-help" || string(argv[2])=="-h")) {
        switch (iCmd) {
	case kDefine:
	    cout << "define: defines an alarm template" << endl;
	    break;
	case kSet:
	    cout << "set: set an alarm" << endl;
	    break;
	case kGetActive:
	    cout << "getactive: List active instances of the specified alarm."
		 << endl;
	    break;
	case kGetDefined:
	    cout << "getdefined: list defined alarm types." << endl;
	    break;
	case kProlong:
	    cout << "prolong: prolong the specified alarm instance" << endl;
	    break;
	case kRequest:
	    cout << "request: Get full information on specified alarm instance"
		 << endl;
	    break;
	case kCancel:
	    cout << "cancel: delete an alarm instance" << endl;
	    break;
	case kAllActive:
	    cout << "allactive: get all information about active alarms" 
		 << endl;
	    break;
	case kEnable:
	    cout << "diable: Disable the specified alarm type" << endl;
	    break;
	case kDisable:
            cout << "enable: Reenable the specified alarm type." << endl;
	    break;
	case kGetTemplate:
	    cout << "getTemplate: Get template for specified alarm type" 
		 << endl;
	    break;
	case kRemove:
	    cout << "remove: Undefine the specified alarm type" << endl;
	    break;
	default:
	    cout << "Help not available" << endl;
	    break;
	}
        return 1;
    }

    //----------------------------------  Get positional arguments
    int endp(argc), Handle(0);
    string Monitor, ID;
    switch (iCmd) {
    case kDefine:
    case kSet:
    case kEnable:
    case kDisable:
    case kGetTemplate:
    case kRemove:
    case kTemplateCmd:
        if (argc < 4) {
	    cerr << "Missing positional arguments" << endl;
	    syntax = true;
	} else {
	    Monitor = argv[2];
	    ID      = argv[3];
	}
	endp = 4;
	break;
    case kGetDefined:
    case kGetActive:
    case kAllActive:
        if (argc < 3 || *argv[2] == '-') {
	    Monitor = "*";
	    ID      = "*";
	    endp    = 2;
	} else if (argc < 4 || *argv[3] == '-') {
	    Monitor = argv[2];
	    ID      = "*";
	    endp    = 3;
	} else {
	    Monitor = argv[2];
	    ID      = argv[3];
	    endp    = 4;
	}
	break;
    case kRequest:
    case kProlong:
    case kCancel:
        Monitor = "Alarm";
        if (argc < 3) {
	    cerr << "Missing positional arguments" << endl;
	    syntax = true;
	} else {
	    Handle = strtol(argv[2], 0, 0);
	}
	endp = 3;
	break;        
    default:
        syntax = true;
        break;
    }

    //----------------------------------  Get keyword parameters
    Interval dT(0.0);
    string desc, param, web, multi;
    int severity(0), debug(0), flags(0);
    bool help(false);
    for (int i=endp ; i<argc ; i++) {
        string argi(argv[i]);
	if (argi == "-time") {
	    dT = Interval(strtod(argv[++i], 0));
	} else if (argi == "-sever") {
	    severity = strtol(argv[++i], 0, 0);
	} else if (argi == "-desc") {
	    desc = argv[++i];
	} else if (argi == "-params") {
	    param = argv[++i];
	} else if (argi == "-webfile") {
	    web = argv[++i];
	} else if (argi == "-multimedia") {
	    multi = argv[++i];
	} else if (argi == "-debug") {
	    debug = strtol(argv[++i], 0, 0);
	} else if (argi == "-flags") {
	    flags = strtol(argv[++i], 0, 0);
	} else if (argi == "-help" || argi == "-h") {
	    help  = true;
	} else {
	    cerr << "Unrecognized argument: " << argi << endl;
	    syntax = true;
	}
    }

    //----------------------------------  Bail if there was a comand error
    if (syntax) {
        cerr << "Command syntax: " << endl;
	cerr << "Alarm <cmd> {<monitor> <alarm-id> | <handle>} [-time <start>]"
	     << " [-sever <sevr>]" << endl;
	cerr << "            [-desc <descr>] [-params <params>]"
	     << " [-webfile <web>]" << endl;
	cerr << "            [-multimedia <multi>] [-flags <modifiers>]"
	     << " [-help]" << endl;
	cerr << endl;
	cerr << "<cmd> = { cancel | define | disable | enable | getactive"
	     << " | getdefined | " << endl
	     << "          gettemplate | prolong | remove | request | set |"
	     << " allactive }" << endl;
	return 1;
    }

    //----------------------------------  Open up the client
    AlarmClient ac(Monitor.c_str());
    ac.setDebug(debug);
    AlarmData alarm(Monitor, ID, dT, severity, desc, param);
    alarm.setWebFile(web);
    alarm.setMultiMedia(multi);
    alarm.jamFlags(flags);
    AlarmHandle handle(Handle);
    std::vector<AlarmHandle> hVector;
    string reply;
    lmsg::error_type rc=0;
    int disflag=1;

    //----------------------------------  Issue a command
    switch (iCmd) {
    case kDefine:
	rc = ac.defineAlarm(alarm);
	cout << "Alarm defined, rc = " << rc << endl;
	break;
    case kSet:
	rc = ac.setAlarm(alarm, handle);
	cout << "Alarm set, rc = " << rc << " Handle: " << handle.getInt()
	     << endl;
	break;
    case kGetActive:
        rc = ac.getActive(static_cast<const AlarmID&>(alarm), hVector);
	if (!rc) {
	    cout << "Active alarms: ";
	    for (int i=0 ; i<int(hVector.size()) ; i++) {
	        cout << hVector[i].getInt() << " ";
	    }
	    cout << endl;
	} else {
	    cout << "Get Active request failed, rc = " << rc << endl;
	}
	break;
    case kGetDefined:
        rc = ac.getDefined(static_cast<const AlarmID&>(alarm), reply);
	if (!rc) {
	    cout << "Defined alarms: " << reply << endl;
	} else {
	    cout << "Get Active request failed, rc = " << rc << endl;
	}
	break;
    case kProlong:
        rc = ac.prolongAlarm(handle, alarm.getTimeout());
	cout << "Alarm prolonged, rc = " << rc << endl;
	break;
    case kRequest:
        rc = ac.testAlarm(handle, alarm);
	if (!rc) {
	    dumpAlarm(cout, alarm);
	} else {
	    cout << "Alarm status not available, rc = " << rc << endl;
	}
	break;
    case kCancel:
        rc = ac.cancelAlarm(handle);
	cout << "Alarm cancelled, rc = " << rc << endl;
	break;
    case kAllActive:
        rc = ac.getActive(static_cast<const AlarmID&>(alarm), hVector);
	if (!rc) {
	    for (int i=0 ; i<int(hVector.size()) ; i++) {
		rc = ac.testAlarm(hVector[i], alarm);
		if (!rc) dumpAlarm(cout, alarm);
		else cout << "Alarm status not available, rc=" << rc << endl;
	    }
	} else {
	    cout << "Get Active request failed, rc = " << rc << endl;
	}
	break;
    case kEnable:
        disflag = 0;
    case kDisable:
        rc = ac.setDisable(static_cast<const AlarmID&>(alarm), disflag);
	break;
    case kGetTemplate:
        rc = ac.getTemplate(static_cast<const AlarmID&>(alarm), alarm);
	if (!rc) {
	    dumpAlarm(cout, alarm);
	} else {
	    cout << "Alarm template not available, rc = " << rc << endl;
	}
	break;
    case kTemplateCmd:
        rc = ac.getTemplate(static_cast<const AlarmID&>(alarm), alarm);
	if (!rc) {
	    cout << "Alarm define " 
		 << alarm.getMonitor() << " " << alarm.getAlarm()
		 << " -sever "       << alarm.getSeverity()
		 << " -time "        << alarm.getTimeout()
		 << " -flags "       << alarm.getFlags()
		 << " -desc '"       << alarm.getDescription()  << "'"
		 << " -webfile '"    << alarm.getWebFile() << "'"
	         << " -multimedia '" << alarm.getMultiMedia() << "'"
		 << endl;
	    
	} else {
	    cout << "Alarm template not available, rc = " << rc << endl;
	}
	break;
    case kRemove:
        rc = ac.remove(static_cast<const AlarmID&>(alarm));
	cout << "Alarm removed, rc = " << rc << endl;
	break;
    default:
        cout << "Command: " << argv[1] << " not implemented." << endl;
	break;
    }
    return rc;
}

//======================================  dump an alarm data block
static void
dumpAlarm(ostream& out, const AlarmData& x) {
    cout << "Monitor:     " << x.getMonitor()     << endl;
    cout << "Alarm:       " << x.getAlarm()       << endl;
    cout << "Severity:    " << x.getSeverity()    << endl;
    cout << "Duration:    " << x.getTimeout()     << endl;
    cout << "Flags:       " << x.getFlags()       << endl;
    cout << "Start time:  " << x.getStartTime()   << endl;
    cout << "Description: " << x.getDescription() << endl;
    cout << "Parameters:  " << x.getParameters()  << endl;
    cout << "Web file:    " << x.getWebFile()     << endl;
    cout << "Multimedia:  " << x.getMultiMedia()  << endl;
}
