#ifndef SUSCONSTRUCTS_HH
#define SUSCONSTRUCTS_HH

/************************
 *** Including standard C++ headers (I/O, STL, math, etc.)
 *** necessary for defining (but not implementing) classes
 ************************/
#include <string>
#include <deque>

/************************
 *** Including GDS headers (containers, servers, clients, etc.)
 *** necessary for defining (but not implementing) classes
 ************************/
#include "TSeries.hh"
#include "Histogram1.hh"
#include "TrigClient.hh"
#include "OperStateCondList.hh"
#include "TrigRslt.hh"
#include "Trend.hh"



/************************
 *** Including chInterface base classes
 ************************/
#include "chConstructs.hh"

/************************
 *** Note that we do NOT "use" namespaces in the header file; this way we don't
 *** accidentally overwrite classes with the same name in different namespaces.
 *** Rather, we define everything necessary in the context of the channel
 *** namespace.
 ************************/

namespace channel {
/* susData structure  */
struct susData {
  /// a std::deque for long-term RMS
  std::deque<TSeries> lt;
  /// a std::deque for short-term RMS
  std::deque<TSeries> st;
  /// the sample rate
  int dequeRate;
};

/** @name chInterface Templates
  * Some Example channel interface templates
  * @memo example channel interface templates
  * @author Rauha Rahkola
  * @version 0.1 June 6th, 2002
  ***************************************************************************/
//@{

/**
  * @memo a version of ResultConstruct with data/log file handling
  */

class susLog : public ResultConstruct {
private:
  std::string dataPrefix;
  std::string logPrefix;
  Trend* mTrend;

public:
  susLog() {}

  susLog( const susLog& log, unsigned int debug=0 ):
    ResultConstruct(log, debug) {
    if (debug > 2) std::cout <<"Copying susLog -- "<<log.getName()<<std::endl;
    *this = log;
    if (ResultConstruct::mDebug > 2) std::cout << "susLog copy constructor used" << std::endl;
    //ResultConstruct::mDebug=debug;
  }

  susLog(const std::string& resultID="", const updateType type=uUnknown,
	 unsigned int outmask=0, std::ostream& out=std::cout,
	 const Interval& update=0.0, unsigned int debug=0 );


  /// checks whether the susLog object has been setup correctly
  bool isInitialized(void) const;

  /// set trend
  void setTrend(Trend* trend) { mTrend = trend; }

  /// susLog "clone" method, to allocate new susLogs
  susLog* clone( void ) const;

  ///outputs the result
  virtual bool output( const statResult& result, unsigned int outmask=0,
		       const bool override=false, std::ostream& out=std::cout );

  virtual ~susLog() {
    if (ResultConstruct::mDebug > 2)
      std::cout <<"Destroying susLog -- " <<getName()<<std::endl;
    if (ResultConstruct::mDebug > 3)
      std::cout <<"   (susLog "<<getName()<<") "<<this<<std::endl;
  }

};



/**
  * @memo a version of ResultConstruct with event handling
  */

//----------------------------begin susEvent class----------------------------

class susEvent : public ResultConstruct {
private:
  std::string dataPrefix;
  std::string logPrefix;
  OperStateCondList* pOSC;
  TrigClient*        pTrig;
  std::string nameOSC;
  double stThresh;
  double ltThresh;

public:
  susEvent() {}

  susEvent( const susEvent& log, unsigned int debug=0 ):
    ResultConstruct(log, debug) {
    if (debug > 2) std::cout <<"Copying susEvent -- "<<log.getName()<<std::endl;
    *this = log;
    if (ResultConstruct::mDebug > 2) std::cout << "susEvent copy constructor used" << std::endl;
    //ResultConstruct::mDebug=debug;
  }

  void setOSC(OperStateCondList& oscpointer, const std::string oscname);
  void setTrig(TrigClient& trigref){pTrig = &trigref;}

  /* susEvent(const std::string& resultID="", const updateType type=uUnknown,
	      const unsigned int outmask=0, std::ostream& out=std::cout,
	      const Interval& update=0.0, const unsigned int debug=0 );*/

  susEvent(const std::string& resultID="", updateType type=uUnknown,
	   unsigned int outmask=0, std::ostream& out=std::cout,
	   const Interval& update=0.0, unsigned int debug=0, 
	   double shortThresh=75, double longThresh=75);


  /// susEvent object has been setup correctly
  bool isInitialized(void) const;

  /// susEvent "clone" method, to allocate new susEvents
  susEvent* clone( void ) const;

  ///outputs the result
  virtual bool output( const statResult& result, unsigned int outmask=0,
		       bool override=false, std::ostream& out=std::cout );

  virtual ~susEvent() {
    if (ResultConstruct::mDebug > 2)
      std::cout <<"Destroying susEvent -- " <<getName()<<std::endl;
    if (ResultConstruct::mDebug > 3)
      std::cout <<"   (susEvent "<<getName()<<") "<<this<<std::endl;
  }
  virtual susEvent& operator =( const susEvent& rhs );
};


//-----------------------------end of susEvent class------------------------

/**
  * @memo a version of susStatistic which calculates the RMS
  */

class susStatistic : public channel::StatisticsConstruct { 

protected: 

  std::map<std::string, susData> mData;

public:
  unsigned int ltlength;
  unsigned int stlength;

  explicit susStatistic(const std::string& statID="",
			const channel::updateType type=channel::uUnknown,
			const Interval& rate=0.0,
			const unsigned int lt=10, unsigned int st=1,
			const double init=0.0,
			const unsigned int debug=0 );



  /// susStatistic copy constructor
  susStatistic( const susStatistic& stat, const unsigned int debug=0 );

  /// susStatistic destructor
  virtual ~susStatistic();

  /// susStatistic "clone" method, to allocate new susStatistics
  susStatistic *clone( void ) const;

  /* ********************
   * * susStatistic modifiers
   * ********************/
  void setLTLength( const unsigned int lt ) { ltlength = lt; }

  void setSTLength( const unsigned int st ) { stlength = st; }

  /* ********************
   * * susStatistic accessors
   * ********************/

  int getLTLength( void ) const { return ltlength; }

  int getSTLength( void ) const { return stlength; }

  /// returns the list of constructed statistics results
  //  std::list<channel::statResult> const* getResults( void ) const { return &results; }

  /* susStatistic virtual functions
   */
  /** resets the statistics -- continuous stats must be overridden to be reset
    * overridden.
    * @param override if true, resets continuous susStatistics as well
    */
  void Reset( const bool override=false );

  /** updates the statistics for (possibly filtered) data-- a unique statID
    * must be given for the result to be distinguished among other data for
    * that channel
    * @param  ts     the TSeries to update statistics for
    * @param  statID try matching an existing statResult
    * @return list of statistics results obtained so far
    */
 
  std::list<channel::statResult>* updateStats( const TSeries& ts,
						const std::string statID );

  /// checks whether the susStatistic object has been setup correctly
  bool isInitialized() const;

  /* susStatistic misc functions
   */
  // /// returns false if any statistics still need more data to update
  // bool isDone();
}; // end of susStatistic class declaration;




/**
  * @memo a version of susStatistic which calculates the Kurtosis
  */
/* class susKurt : public channel::susStatistic {
private:
public:
};
//@}*/
}

#endif // SUSCONSTRUCTS_HH








