#include "DQ_Bit.hh"

using namespace std;

/**  The DQ_OvflCount tool looks for incrementing of a counter by flagging
  *  any time span where the maximum value of one or more channels is 
  *  greater than the maximum value of the previous time span of that channel.
  *  Note that each time span is compared to the previous time span, so if 
  *  one or more of the counters is reset, the to a smaller value, the next 
  *  segment with a value greter than the reset value is flagged.
  *  \brief Adc Overflo counter flag tool.
  *  \author John Zweizig (john.zweizig _at_ ligo.org)
  *  \version $Id$
  */
class DQ_OvflCount :public DQ_bit {

public:
  /**  Tool initialization method.
    */
  void
  init(void) {
     _first = true;
     _last_count.clear();
     _startGPS = 0;
     _print = numeric_param("print") != 0;
  }

  /**  Evaluate the flag for the input data channels. The argument data contain
    *  one or more channels to be checked for incrementa values.
    *  \brief Flag evaluation method. 
    *  \param data The data vector contains one or more time series (TSeries) 
    *              instances to be checked for incrented values.
    *  \return True if the defined condition is satisfied.
    */
  bool 
  bit_value(const tser_vect& data) {
     size_t N=data.size();
     if (_first) _last_count.resize(N, 0);
     bool retcd = false;
     for (size_t i=0; i<N; i++) {
         uint32_t max_count = data[i].getMaximum();
         if (!_first && max_count > _last_count[i]) retcd = true;
         _last_count[i] = max_count;
     }
     _first = false;
     if (_print) {
         uint32_t now = data[0].getStartTime().getS();
         if (!retcd) {
	    if (_startGPS && _lastGPS < now) {
	        cout << dq_name() << " segment: " << _startGPS << ":" 
		     << now << endl;
		_startGPS = 0;
	    }
	 } else {
	    if (!_startGPS) _startGPS = now;
	    _lastGPS  = now;
	 }
     }
     return retcd;
  }
//-------------------------------------  Local data for the too instance.
private:
  bool     _first;
  bool     _print;
  std::vector<uint32_t> _last_count;
  uint32_t _startGPS;
  uint32_t _lastGPS;
};

DQ_PLUGIN(DQ_OvflCount)
