/* file WaveMon.hh 
   author: Sergey Klimenko, University of Florida
*/


#ifndef WAVEMON_HH
#define WAVEMON_HH

#include "DatEnv.hh"
#include "MonServer.hh"
#include "OperStateCondList.hh"
#include "Trend.hh"
#include "TrigPar.hh"
#include "wseries.hh"
#include "cluster.hh"
#include "LineFilter.hh"

typedef char * pchar;

class ostream;
//class LineTable;

namespace xsil {
  class Xwriter;
}


class WaveMon : public DatEnv, MonServer {

   public:
  
      WaveMon(int argc, const char *argv[]);     // Constructor
      ~WaveMon();                          // Destructor

      void ProcessData(void);
      void DumpHelp();
      void Attention(void);                // for Data Viewer
      void addTrend(wavearray<float>, int);

      wavearray<float> getRate(wavecluster&);
      // wavearray<float> getAcceptance(WSeries<double>&);
      
      /**  read input parameters from configuration file
       *  @memo read configuration file
       *  @param input file name.
       *  @return true if .
       */
      bool readConfig(char *);
      void readLockLoss(bool);

      /** fill triggerList 
       *  @memo read cluster information and fill triggers
       *  @param wseries, channel name index
       *  @return return number of clusters with size >= minClusterSize
       */
      size_t addTriggers(wavecluster&, size_t);

      /** set selection cuts in WSeries object 
       *  @memo set cuts on cluster size, asymmetry and quality
       *  @param wseries
       *  @return return number of clusters pussed the cuts
       */
      size_t setCuts(wavecluster&);

      void DumpTrigger(char*, int=0);
      void DumpSummary(char*);

      inline int getDeep(double, double);

      char *getToken(pchar &, bool = false);
      inline bool isParameter(const char *);
      inline bool getParameter(char *, const char *, int&);
      inline bool getParameter(char *, const char *, double&);

      /** set trend sample data for specified channel  
       *  @memo set trend sample data for specified channel  
       *  @param wavearray where sample is taken from, channel name
       *  @return true if sample is set, false if no samples to set
       */
      inline bool setTrend(const wavearray<float>&, char*);

   private:
 
      pchar pString;

      char masterChannel[256];
      std::vector<char *>    slaveChannel;
      std::vector<char *>    channelName;
      std::vector<TSeries *> dmtviewerTS;

      char configFile[256];
      char trendFile[256];
      char indexFile[256];
      char triggerFile[256];
      char summaryFile[256];
      char backupFile[256];
      char lockCondition[256];
      char suffix[256];
      char gpsTime[256];
      
      double   timeGate;
      double   pfraction;
      double   timeStride;
      double   timeInterval;
      double   resolution;
      double   samplingRate;
      double   liveTime;
      double   totalTime;
      double   asymmetryCut;
      double   likelihoodCut;
      double   offset;
      int      minClusterSize;
      int      nDump;
      int      mDump;      
      int      maxDump;      
      int      nBuffer;   // lengts of buffer to serve trend data to data viewer
      int      verbose;   
      bool     firstcall;
      int      line60Hz;
      int      waveorder;

      Time  currentTime;

      Trend mTrend;
      OperStateCondList mOSC;

      wavearray<double>* master;
      TSeries *masterTS;
      WSeries<double> wAlarm;
      WSeries<double> wBurst;

      std::vector<TSeries*> slaveTS;
      std::vector<wavearray<double>*> slave;
      std::vector<wavearray<float>*>  wavebuffer;
      std::vector<wavearray<double>*> clusterRate;
      std::vector<wavearray<double>*> currentRate;
      std::vector<wavearray<double>*> clusterEffy;
      std::vector<wavearray<double>*> currentEffy;

      std::list<char*>   wTrigger;
      LineFilter* Filter;
};

inline int WaveMon::getDeep(double f1, double f2){
  if(f1<=0. || f2<=0. || f1==f2) return 0;

  int n=0;
  while(int(f1/f2+0.01)>1){ f1*=0.5; n+=1; }
  while(int(f2/f1+0.01)>1){ f2*=0.5; n-=1; }
  return n;
}

inline bool WaveMon::isParameter(const char *s){
   int n = strlen(s);
   bool isP = (n>0) ? true : false;
   for(int i=0; i<n; i++)
      if(!isdigit(s[i]) && s[i] != '-' && s[i] != '.')
	 isP = false;
   return isP;
}

inline bool WaveMon::getParameter(char *w, const char *c, int &x){
  if ( strcmp(w,c) == 0 ){
    if( isParameter(w = getToken(pString)) )
      sscanf(w,"%d",&x);
    return true;
  }
  return false;
}

inline bool WaveMon::getParameter(char *w, const char *c, double &x){
  if ( strcmp(w,c) == 0 ){
    if( isParameter(w = getToken(pString)) )
      sscanf(w,"%lf",&x);
    return true;
  }
  return false;
}

// survival probability for Gamma distribution
/*
double gammaCL(double x, int n){   
   double z,y,s;
   z = 0;

   if(x<=n){
     y = x; s = 1.;
     for(int k=n-1; k>0; k--){ 
       s += y; 
       y*=double(x)/(n-k+1); 
       if(y>1.e290) break;
     }
     return double(x-log(s));
   }
   else{
     y = n/x; s = n/x; z = log(x/n);
     for(int k=n-1; k>0; k--){ 
       y *= double(k/x); 
       s += y; 
       z += log(x/k); 
       if(y<1.e-290) break;
     }
     return float(x-z-log(s));
   }
}
*/

#endif     //  WaveMon_HH












