#include "TrigDsply.hh"
#include "TrigBase.hh"
#include "TSeries.hh"
#include "ParseLine.hh"

//----------   Supported filters
#include "BaseLine.hh"
#include "Blackman.hh"
#include "Difference.hh"
#include "FilterDesign.hh"
#include "Hamming.hh"
#include "Hanning.hh"
#include "MultiPipe.hh"
#include "IIRFilter.hh"
//----------

#include "TROOT.h"
#include "TApplication.h"
#include "TH1.h"
#include "TPad.h"
#include "TLine.h"
#include "TText.h"
#include "TStyle.h"

#include <unistd.h>

using namespace std;

//======================================  Generate the main function
EXEROOT(TrigDsply)

//======================================  Construct the monitor
TrigDsply::TrigDsply(int argc, const char* argv[]) 
  : TrigEnv(argc, argv), mCanvas(0), minPad(0.25),
    mFracPad(0.20), mSleep(10)
{
    const char* cfile = 0;
    for (int i=1 ; i<argc ; i++) {
        string argi(argv[i]);
        if (isTrigEnvArg(argv[i])) {
	    i++;
	} else if (argi == "-conf") {
	    cfile = argv[++i];
	} else if (argi == "-toc") {
	    getDacc().setTOCMode(false);
	} else if (argi == "+toc") {
	    getDacc().setTOCMode(true);
	} else if (argi == "-mpad") {
	    minPad = strtod(argv[++i], 0);
	} else if (argi == "-fpad") {
	    mFracPad = strtod(argv[++i], 0);
	} else if (argi == "-outfile") {
	    mOutfile = argv[++i];
	} else if (argi == "-sleep") {
	    mSleep = strtol(argv[++i], 0, 0);
	} else {
	    cerr << "Unrecognized argument: " << argv[i] << endl;
	    cerr << "Syntax: TrigDsply -ttable <trig-file> [-mpad <min-pad>]"
		 << " [-fpad <fract-pad>] [-toc | +toc] [-outfile <ps-file>]"
		 << endl;
	    finish();
	    return;
	}
    }
    //getDacc().setTOCMode(false);
    if (!cfile || ReadConfig(cfile)) {
        finish();
	return;
    }
    if (mSelect.empty()) mSelect.push_back(TrigID("*", "*"));
    if (!mOutfile.empty()) {
        gROOT->SetBatch(true);
	mCanvas = new TCanvas("tCanvas", "Trigger Display");
	mCanvas->Print((mOutfile+"[").c_str());
    } else {
        gROOT->SetBatch(false);
        mCanvas = new TCanvas("tCanvas", "Trigger Display");
    }
}

//======================================  Destroy the monitor
TrigDsply::~TrigDsply(void) {
    if (mCanvas) {
        if (!mOutfile.empty()) mCanvas->Print((mOutfile+"]").c_str());
	delete mCanvas;
    }
}

//======================================  Generate the trigger epoch
bool 
TrigDsply::ProcessTrigger(const trig::TrigBase& t) {

    //----------------------------------  Clear the channel selection
    if (!mChannel.empty()) {
        getDacc().rmChannel(mChannel.c_str());
	mChannel.erase();
    }

    //----------------------------------  See if this trigger is selected
    for (mTrig=mSelect.begin() ; mTrig != mSelect.end() ; mTrig++) {
        if (mTrig->isTrig(t)) break;
    }
    if (mTrig == mSelect.end()) {
        if (Debug()) cout << "Trigger " << t.getID() << "/" << t.getSubID()
			  << " is not selected" << endl;
        return false;
    }

    //----------------------------------  See if this trigger is deselected
    for (trig_iter i=mDeSelect.begin() ; i != mDeSelect.end() ; i++) {
        if (i->isTrig(t)) {
	    if (Debug()) cout << "Trigger " << t.getID() << "/" 
			      << t.getSubID() << " is deselected" << endl;
	    return false;
	}
    }

    //----------------------------------  Get the display channel
    mChannel = mTrig->getDisplay();
    if (mChannel.empty()) mChannel = t.getSubID();
    getDacc().addChannel(mChannel.c_str());

    //----------------------------------  Set the time epoch
    Interval dT = t.getDt();
    Interval pad = mFracPad * dT;
    if (pad < Interval(minPad)) pad = minPad;
    setEpochOffset(-(pad + mTrig->getSettle()));
    Interval length = dT + 2.0*pad + mTrig->getSettle();
    setEpochLength(length);
    setStride(length);
    cout << "Epoch start, length= " << -pad << ", " << length << endl;
    return true;
}

//======================================  Display and wait
void
TrigDsply::ProcessData(void) {
    TSeries* ts = getDacc().refData(mChannel.c_str());
    if (!ts) {
        cout << "Unable to fetch channel " << mChannel << endl;
	return;
    }
    TSeries fts = mTrig->Filter(*ts);
    if (mTrig->getSettle()) fts.eraseStart(mTrig->getSettle());
    Plot(fts);
    if (mOutfile.empty()) {
        sleep(mSleep);
    } else {
        mCanvas->Print(mOutfile.c_str());
    }
}

//======================================  Plot the time series
void 
TrigDsply::Plot(const TSeries& t) {
    //----------------------------------  Formatting parameters
    float ymargin = 0.1;  // Fractional margin in y.

    //----------------------------------  Get the TSeries parameters.
    int nbin = t.getNSample();
    float stime = t.getStartTime().totalS();
    if (stime > 1000000.0) stime = 0;
    float step  = t.getTStep().GetSecs();

    //----------------------------------  Get the data and real length.
    float* DVec = new float[nbin];
    int nwd = t.getData(nbin,DVec);
    float dtime = stime + step * nwd;

    //----------------------------------  Create and fill the histogram.
    string title(t.getName());
    title += string(" - Trigger: ") + getTrigger().getID() 
           + " " + getTrigger().getSubID();
    TH1F hist("TS001", title.c_str(), nwd, stime, dtime);
    float ymin = DVec[0];
    float ymax = DVec[0];
    for (int i=0 ; i<nwd ; i++) {
        //-----------------------------  Histogram content in bins 1-N. Bin 0
        //                               has UnderFlow, bin N+1 has OverFlow
        hist.SetBinContent(i+1, DVec[i]);
	if (DVec[i] < ymin) ymin = DVec[i];
	if (DVec[i] > ymax) ymax = DVec[i];
    }
    delete[] DVec;
    float yavg = 0.5 * (ymin + ymax);
    float ydif = yavg - ymin;
    ymin = yavg - (1.0 + ymargin)*ydif;
    ymax = yavg + (1.0 + ymargin)*ydif;
    if (ymin < 0.0 && yavg-ydif >= 0.0) ymin = 0.0;
    hist.SetMinimum(ymin);
    hist.SetMaximum(ymax);

    //----------------------------------  Set the plotting options
    gStyle->SetOptStat(0);
    mCanvas->cd();
    mCanvas->Clear();
    TPad* padT = new TPad("padT", "Time Series", 0.0, 0.0, 1.0, 1.0);
    padT->Draw();
    padT->cd();
    padT->SetBottomMargin(0.12);
    padT->SetLeftMargin(0.15);
    padT->SetTicks(1,1);
    hist.DrawCopy("");

    double evtTime = getTrigger().getTime() - t.getStartTime();
    TLine line;
    line.SetLineColor(3);
    line.DrawLine(evtTime, ymin, evtTime, ymax);

    evtTime = getTrigger().getTime() + getTrigger().getDt() 
            - t.getStartTime();
    line.DrawLine(evtTime, ymin, evtTime, ymax);

    TText lTxt;
    lTxt.SetTextAlign(22);
    lTxt.SetTextSize(0.05);
    lTxt.SetTextAngle(90.);
    lTxt.DrawTextNDC(0.03, 0.50, "ADC Signal (adc units)");

    char ttxt[128];
    if (stime) strcpy(ttxt, "Time (seconds)");
    else      TimeStr(t.getStartTime(), ttxt, "Time (seconds from GPS %s:%n)");

    lTxt.SetTextAngle(0.);
    lTxt.DrawTextNDC(0.50, 0.03, ttxt);

    mCanvas->Update();
}

//======================================  Read in the configuration file.
bool
TrigDsply::ReadConfig(const string& file) {

    //----------------------------------  Set up the line parser
    ParseLine lp(file.c_str());
    if (!lp.isOpen()) {
        cerr << "Unable to open configuration file " << file << endl;
	return true;
    }
    lp.setLog(cout);

    bool syntax = false;
    for (int nArg=lp.getLine() ; nArg>=0 ; nArg=lp.getLine()) {

        //------------------------------  Empty command line
        if (!lp[0]) {
	    continue;

        //------------------------------  Parse parameter line
	    //} else if (!strcmp(lp[0], "Parameter")) {
	    //for (int i=1 ; i<nArg ; i+=2) {
	    //    if (!mDict.exists(lp[i])) {
	    //    cout << "No such parameter: " << lp[i] << endl;
	    //    break;
	    //} else {
	    //    if (Debug()) cout << "Set parameter: " << lp[i] << endl;
	    //    mDict.setPar(lp[i], lp[i+1]);
	    //}
	    //}

        //------------------------------  Parse Select definition
        } else if (!strcmp(lp[0], "Select")) {
	    Pipe* pipe(0);
	    TrigID temp(lp[1], lp[2], pipe);
	    for (int i=3 ; i<nArg ; i++) {
	        string lpi=lp[i];
		if (lpi == "-display") {
		    temp.setDisplay(lp[++i]);
		} else if (lpi == "-filter") {
		    FiltrIterator f = mFiltr.find(lp[++i]);
		    if (f != mFiltr.end()) {
		        temp.setPipe(f->second);
		    } else {
		        cerr << "Filter not defined: " << lp[i] << endl;
			syntax = true;
		    }
		} else if (lpi == "-settle") {
		    temp.setSettle(lp.getDouble(++i));
		} else {
		    syntax = true;
		    cout << "Invalid Select parameter" << endl;
		}
	    }
	    if (nArg < 3 || syntax) {
	        cout << "Command syntax is:" << endl;
		cout << "Select <trigger-id> <sub-id> [-display <chan>] "
		     << "[-filter <filter>] [-settle <settle>]" << endl;
		syntax = true;
		break;
	    }
	    mSelect.push_back(temp);

        //------------------------------  Parse Select definition
        } else if (!strcmp(lp[0], "Deselect")) {
	    string ID, subID;
	    if (nArg == 3) {
	        mDeSelect.push_back(TrigID(lp[1], lp[2]));
	    } else {
	        cout << "Command syntax is:" << endl;
		cout << "Deselect <trigger-id> <sub-id>" << endl;
		syntax = true;
		break;
	    }

        //------------------------------  Parse Filter definition
        } else if (!strcmp(lp[0], "Filter")) {
	    if (nArg < 3) {
	        cout << "*** Invalid Filter specifier. Syntax: " << endl;
		cout << "    Filter <name> <type> <parameters>" << endl;
		syntax = true;
		continue;
	    }
	    string fName(lp[1]);
	    string fType(lp[2]);
	    Pipe* fPipe=0;
	    if (fType == "BaseLine") {
	        double tConst(1.0);
		if (nArg > 3) tConst = lp.getDouble(3);
	        fPipe = new BaseLine(3);
	    } else if (fType == "Blackman") {
	        fPipe = new Blackman;
	    } else if (fType == "Difference") {
	        if (nArg < 4) {
		    cout << "*** Difference filter requires a frequency!" 
			 << endl;
		    syntax = true;
		    continue;
		}
	        fPipe = new Difference(lp.getDouble(3));
	    } else if (fType == "Design") {
	        if (nArg < 5) {
                    cout << "*** Filter design parameters are: "
                         << "<sample-rate> <string>" << endl;
		    syntax = true;
                    continue;
                }
		double sRate = lp.getDouble(3);
		if (sRate <= 0.0) {
		    cerr << "Error - Invalid sample rate: " << lp[3] << endl;
		    syntax = true;
		    continue;
		}
                FilterDesign fd(lp[4], sRate);
                fPipe = fd.release();
	    } else if (fType == "FIRFilter") {
	        cout << "FIR Filters not implemented" << endl;
	    } else if (fType == "Hamming") {
	        fPipe = new Hamming;
	    } else if (fType == "Hanning") {
	        fPipe = new Hanning;
	    } else if (fType == "IIRFilter") {
	        if (nArg < 6) {
		    cout << "Specify rate, poles and zeros" << endl;
		    syntax = true;
		    continue;
		}
		double rate = strtod(lp[3], 0);
		if (Debug()) cout << "Filter: " << fName << " rate " 
				  << rate << endl;
	        vector<dComplex> poles, zeros;
		const char* ptr = lp[4];
		double twopi=2*3.1415926535;
		while (*ptr) {
		    double re = strtod(ptr, const_cast<char**>(&ptr));
		    double im = strtod(ptr, const_cast<char**>(&ptr));
		    poles.push_back(dComplex(re,im)/twopi);
		    if (Debug()) cout << "pole " << poles.size() << ": "
				      << re << " + " << im << "i" << endl;
		}
		ptr = lp[5];
		while (*ptr) {
		    double re = strtod(ptr, const_cast<char**>(&ptr));
		    double im = strtod(ptr, const_cast<char**>(&ptr));
		    zeros.push_back(dComplex(re,im)/twopi);
                    if (Debug()) cout << "zero " << zeros.size() << ": "
                                      << re << " + " << im << "i" << endl;
		}
		fPipe = new IIRFilter(poles.size(), &poles[0], 
				      zeros.size(), &zeros[0], rate);
	    } else if (fType == "MultiPipe") {
	        fPipe = new MultiPipe;
	        for (int i=3 ; i<nArg ; i++) {
		    if (mFiltr.find(lp[i]) != mFiltr.end()) {
		        cout << "Filter " << lp[i] 
			     << " could not be added to " << fName << endl;
			syntax = true;
		        continue;
		    }
		    dynamic_cast<MultiPipe*>(fPipe)->addPipe(*(mFiltr[lp[i]]));
		}
	    } else if (fType == "LineFilter") {
	        cout << "Line Filters not implemented" << endl;
		syntax = true;
	    } else {
	        cout << "*** Unrecognized filter type: " << fType << endl;
		syntax = true;
	    }
	    if (fPipe) mFiltr[fName] = fPipe;

	//------------------------------  Invalid statement.
	} else {
	    cout << "Invalid command in " << file << ": " << lp[0] << endl;
	    syntax = true;
	}
    }
    return syntax;
}

//======================================  TrigID data constructor
TrigID::TrigID(const string& ID, const string& subID, const Pipe* f) 
  : mTrigID(ID), mSubID(subID), mFilter(0), mSettle(0)
{
    if (f) mFilter = f->clone();
}

//======================================  TrigID copy constructor
TrigID::TrigID(const TrigID& t) 
  : mTrigID(t.mTrigID), mSubID(t.mSubID), mDisplay(t.mDisplay), mFilter(0),
    mSettle(t.mSettle)
{
    if (t.mFilter) mFilter = t.mFilter->clone();
}

//======================================  TrigID destructor
TrigID::~TrigID(void) {
    if (mFilter) delete mFilter;
    mFilter = 0;
}

//======================================  Is trigger covered by this ID
bool 
TrigID::isTrig(const trig::TrigBase& t) {
    return ((mTrigID == t.getID()    || mTrigID == "*") && 
	    (mSubID  == t.getSubID() || mSubID  == "*"));
}

//======================================  Filter a TSeries
TSeries 
TrigID::Filter(const TSeries& t) {
    if (!mFilter) return t;
    return mFilter->apply(t);
}

//======================================  Set display channel
void 
TrigID::setDisplay(const std::string& d) {
    mDisplay = d;
}

void 
TrigID::setSettle(double dt) {
    mSettle = Interval(dt);
}

void 
TrigID::setPipe(const Pipe* p) {
    if (mFilter) delete mFilter;
    if (p) mFilter = p->clone();
}
