//  GPS Timing monitor for LIGO DAQ
//
//  DMT/Base Code by John Zweizig
//  Written  by Szabolcs Marka
//

#include "TimeMon.hh"
#include "DVecType.hh"
#include "DaccAPI.hh"
#include <fstream>
#include <time.h>
#include <string.h>
#include <cstdlib>
#include "FixedLenTS.hh"
#include <iostream>
#include <sstream>

static const int      skipStart(1);         // Number of bins to dispose at the beginning of the RAMP
static const int      skipEnd(3);           // Number of bins to dispose at the end of the RAMP
static const Interval rLength(0.07);        // Length of sample (seconds) used to compute zero level offset
static const Interval rNext(0.1);           // Skip this many seconds after the previous trigger...also use for offset calculation
static const double   TrigThreshold(0.5);   // Trigger threshold (relative to range!)
             int      cnt=0;
            char      Alive(0);

char CaPutDir[256]  = "/cvs/cds/epics/extensions/bin/solaris";
char CaPutCom[256]  = "caput";
char TTAG_Ch[40]    = "xxx";
char ALIVE_Ch[40]   = "xxx";

bool Epics         = false;

using namespace std;

//======================================   GPSTimer Constructor
GPSTimer::GPSTimer(const string& ramp, const string& trend, 
                   const string& epics, const string& alarm, 
		   const float MarginalTh, const float ErrorTh, DaccAPI& In)
  : mRampChan(ramp), 
    mTrend(trend),
    mEpics(epics),
    mAlarm(alarm),
    mMarginalTh(MarginalTh),
    mErrorTh(ErrorTh),
    mRampSeries(0), 
    mTrigTime(0),
    fSlope(-9999999.9),
    fVoltage(-9999999.9),
    fOffset(-9999999.9),
    fRsquare(-1),
    mRampN(0) 
{
  In.addChannel(mRampChan.c_str(), 0, &mRampSeries);            // Link Channel to local variable
}

const char* 
GPSTimer::getTrendName(void) const {
    return mTrend.c_str();
}

const char* 
GPSTimer::getEpicsName(void) const {
    return mEpics.c_str();
}

const float 
GPSTimer::getMarginalTh(void) const {
    return mMarginalTh;
}

const float 
GPSTimer::getErrorTh(void) const {
    return mErrorTh;
}

const char* 
GPSTimer::getAlarmName(void) const {
    return mAlarm.c_str();
}

GPSTimer::~GPSTimer(void) {
}

const char* 
GPSTimer::getRampName(void) const {
    return mRampChan.c_str();
}

Time     
GPSTimer::getTrigTime(void) const {
    return mTrigTime;
}

double 
GPSTimer::getRampOff(void) const {
    return fOffset;
}

double 
GPSTimer::getRampSlope(void) const {
    return fSlope;
}

double 
GPSTimer::getRampMag0(void) const {
    return fVoltage;
}

double 
GPSTimer::getRampR2(void) const {
    return fRsquare;
}

int
GPSTimer::getRampFitQ(void) const {
  if ( fRsquare < 0.99   ) return  0;                    // Useless...
  int Q = (int)((fRsquare - 0.99) * 1030.9);             // Set the scale from 0 to 10...
  if ( Q >= 10 || fRsquare > 0.9997 ) return 10;         // Very good (Best)!
  return Q;
}

bool
GPSTimer::isComplete(void) const {
  return (mTrigTime != Time(0)); // && (mRampN == rLen);
}

void
GPSTimer::process(void) {
    if (!mRampSeries) return;
	 
#ifdef T_DEBUG  
    cout << "\n\n---+++ ------------------------------------------------------------" << mRampChan.c_str() << "\n" << endl;
#endif

	 // Set delay such to avoid partial RAMP
	 findTrig(mRampSeries->getStartTime() + 2.0 * rNext);
    getRamp();
}

void
GPSTimer::findTrig(const Time& t0) {                                    // Detects peaks exceeding TrigThreshold
#ifdef T_DEBUG  
  cout << " RAMP  Channel: " << mRampChan.c_str() << "\n" 
       << " Trend Channel: " << mTrend.c_str() << "\n" 
       << " Epics Channel: " << mEpics.c_str() << "\n" 
       << " Alarm Channel: " << mAlarm.c_str() << "\n"
       << "Look for trigger starting at: " << mTrigTime << " ?=? " << t0 << endl;
  cout << "Time series at " << mRampSeries->getStartTime() << " with length " 
       << mRampSeries->getNSample() << endl;
#endif 

    mRampSeries->Convert(DVecType<VecType>::getDataType());                 
    VecType* pTrig = reinterpret_cast<VecType*>(mRampSeries->refData());
    int nTrigSample = mRampSeries->getNSample();
    double mRampMax = (mRampSeries->getMaximum());
    double mRampMin = (mRampSeries->getMinimum());
    double mRampAve = (mRampSeries->getAverage());
    double mRange   = 0;

    if ( abs(mRampMax) > abs(mRampMin)) {
      mRange = mRampMax-mRampAve;
    } else {
        mRange = mRampMin-mRampAve;
    }

    for (int i=mRampSeries->getBin(t0) ; i<nTrigSample ; i++) {
	 
	 //cout << i << " : " << t0 << " : " << mRampSeries->getBin(t0) << "\n";
	 
        if ( abs ( pTrig[i]-mRampAve )  > abs ( TrigThreshold * mRange ) ) {
	     mTrigTime = mRampSeries->getBinT(i);
#ifdef T_DEBUG  
    cout << "Data properties\n"
         << "   Maximum = " << mRampMax << "\n"
         << "   Minimum = " << mRampMin << "\n"
         << "   Average = " << mRampAve << "\n"
         << "   Range   = " << mRange   << "\n"
         << "   TrigOn  = " << (pTrig[i]-mRampAve)/mRange << endl;
         
    cout << "Trigger found at : " << mTrigTime << endl;
    for(int yy=i; yy<i+30;yy++) cout << pTrig[yy] << ",";
    cout << endl;
#endif 
	    break;
	}
    }
}

double GPSTimer::getRamp(void) {
  int i = 0;
  //  Interval dT = mRampSeries->getTStep();                                // Find sampling time

  mRampSeries->Convert(DVecType<VecType>::getDataType());
  VecType* pRamp = reinterpret_cast<VecType*>(mRampSeries->refData());

  // Determine the zero level offset
  Time OffsetTimeStart = mTrigTime - rNext;
  int  OffsetBinStart  = mRampSeries->getBin(OffsetTimeStart);         // Start of Offset level sample
  Time OffsetTimeEnd   = OffsetTimeStart + rLength;
  int  OffsetBinEnd    = mRampSeries->getBin(OffsetTimeEnd);           // End of Offset level sample

  if (!OffsetBinEnd) reset();                                          // Check success
  if (OffsetBinEnd <= OffsetBinStart) {
#ifdef T_DEBUG  
    cout << "Offset interval undetermined!\n     TriggerTime=" << mTrigTime 
	 << "\n     Ofset interval start (bin)="               << OffsetBinStart 
	 << "\n     Ofset interval end   (bin)="               << OffsetBinEnd    << endl;
#endif 
    return 0;
  }

  //     Compute offset level
  double mOffset = 0;
  for (i = OffsetBinStart ; i <= OffsetBinEnd; i++) mOffset += (double)pRamp[i];
  mOffset /= (double) (++i - OffsetBinStart);
#ifdef T_DEBUG  
  cout << "Offset level determined : " << (float) mOffset << endl;
#endif 

  // Determine the ramp start and stop time 

  double mRampMax   = (mRampSeries->getMaximum()) - mOffset;
  double mRampMin   = (mRampSeries->getMinimum()) - mOffset;

  double mRampHeight = abs(mRampMax);
  if (mRampHeight < abs(mRampMin)) mRampHeight = abs(mRampMin);

  double mRampStartHeight = 0.1 * mRampHeight;
  double mRampStopHeight  = 0.9 * mRampHeight;

  int mTrigBin   = mRampSeries->getBin(mTrigTime);                      // Bin position of Trigger Point
#ifdef T_DEBUG    
  Time mStrtTime = mRampSeries->getStartTime();                         // Find series start time
  int mSStrtBin  = mRampSeries->getBin(mStrtTime);
  cout <<   "Trigger time/bin      = " << mTrigTime << " / " << mTrigBin 
       << "\nSeries start time/bin = " << mStrtTime << " / " << mSStrtBin
       << "\nSeries Maximum/Minimum= " << mRampMax  << " / " << mRampMin
		 << "\nRamp Start/Stop height= " << mRampStartHeight << " / " << mRampStopHeight
       << endl;
#endif 

// Determine RAMP Start
  for( i = mTrigBin; i>=mTrigBin-50; i-- ) {
      if ( abs ( pRamp[i] - mOffset ) <= mRampStartHeight ) break;      // Break at the start of Fittable part of Ramp or at the beginning of series
  }
  int  mRampStartBin  = i + skipStart;
  Time mRampStartTime = mRampSeries->getBinT(mRampStartBin);
#ifdef T_DEBUG  
  cout <<   "Ramp Start time/bin   = " << mRampStartTime << " / " << mRampStartBin << endl;
#endif 

  if ( i == (mTrigBin-50) ) {
#ifdef T_DEBUG  
    cout << "Did not find the start of the RAMP..." << endl;
#endif 
    return 0;
  }

// Determine RAMP End

  int  mRampEndBin  = - skipEnd;
  for( i = mRampStartBin; i<= mRampStartBin + 100; i++ ) {
#ifdef T_DEBUG 
    cout << i << " : " << ( pRamp[i] - mOffset) << ",= " << mRampStopHeight << endl;
#endif
   if ( abs ( pRamp[i] - mOffset ) >= mRampStopHeight) {
      mRampEndBin = i - skipEnd;
      break;                         // Break at the end of Fittable part of Ramp or at the beginning of series (old style ramp)
    }

    if ( pRamp[i] >= 32700 )  {
      mRampEndBin = i - skipEnd;
      break;                         // Break at the end of Fittable part of Ramp or at the beginning of series (new style ramp)
    }
  }
  if ( i == mRampStartBin + 100 ) {
#ifdef T_DEBUG  
    cout << "Did not find the end of the RAMP..." << endl;
#endif 
    return 0;
  }

#ifdef T_DEBUG 
  Time mRampEndTime = mRampSeries->getBinT(mRampEndBin); 
  cout <<   "Ramp End   time/bin   = " << mRampEndTime << " / " << mRampEndBin << endl;
#endif 

  Time mTicSeconds(mRampStartTime.getS());
#ifdef T_DEBUG  
  cout <<   "Second below ramp     = " << mTicSeconds << endl;
#endif 

  if ( mRampEndBin <= mRampStartBin ) return 0;  // Sanity checks here...

  // Compute slope and baseline crossing based on the available info 
  // Compute necessary parameters
  double mRampT  = 0.0;
  double mRampTT = 0.0;
  double mRampTY = 0.0;
  double mRampY  = 0.0;
  double mRampYY = 0.0;
  mRampN  = 0;
#ifdef T_DEBUG  
  cout <<  "========= Ramp data ======================== \n" 
       <<  "Time[us]\tMagnitude[V]\tMagnitude [cnt]"
       <<  endl;
#endif
  for (i = mRampStartBin ; i <= mRampEndBin ; i++) {
    double t = double( mRampSeries->getBinT(i) - mTicSeconds);
    double y = ((double) pRamp[i] - mOffset )/16384.0;             // Subtract voltage offset and calibrate to Volts
#ifdef T_DEBUG  
    cout << (float) (1000000.0*t) << "    \t " 
	 << (float)((float)y) << " \t " << (int) (y*16384.0) 
	 << endl;
#endif    
    mRampT  += t;
    mRampTT += t*t;
    mRampTY += t*y;
    mRampY  += y;
    mRampYY += y*y;
    mRampN++;
  }

  //    Compute slope and offset
    fSlope          = double(mRampN)*mRampTY - mRampT*mRampY;
    fVoltage        =         mRampY*mRampTT - mRampT*mRampTY;
    double denomT   = double(mRampN)*mRampTT - mRampT*mRampT;
    double denomY   = double(mRampN)*mRampYY - mRampY*mRampY;
    if (denomT == 0) {
#ifdef T_DEBUG
      cerr << mTicSeconds << "Ramp is underconstrained or constant (division by zero)" << endl;
#endif
      fSlope   = -9999999.9;
      fVoltage = -9999999.9;
      fOffset  = -9999999.9;
      fRsquare = -1;
      return -9999999.9;
    }
    if (denomY == 0) {
#ifdef T_DEBUG
      cerr << mTicSeconds << "Rsquared cannot be computed! (division by zero)" << endl;
#endif
      fRsquare = -1;
    } else { 
        fRsquare  = fSlope / sqrt(denomT * denomY);
        fRsquare *= fRsquare;
      }
    fSlope   /= denomT;
    fVoltage /= denomT;
    fOffset   = ((mRampY)/fSlope - (mRampT))/double(mRampN);
    if (fOffset > 0.5) fOffset -= 1.0;
    if (fOffset < -0.5) fOffset += 1.0;
#ifdef T_DEBUG  
    cout << "-------------------------------------------\n"
	 << " Offset      = " << (double) (1000000.0*fOffset) << " us\n"
	 << " Slope       = " << (double) (fSlope/1000.0)  << " V/ms\n" 
	 << " Magn. at 0  = " << (double) (1000.0*fVoltage) << " mV\n"
	 << " Rsquare     = " << (double) (fRsquare)  << " \n" 
	 << " Fit Quality = " << (int)    (getRampFitQ()) << " (0-10, 10=Best!)\n"
	 << "===========================================\n\n" << endl;
#endif    

    if (Epics) {
      // EPICS communication every 2 seconds ...
      ostringstream Comi;

      Comi << CaPutDir << "/caput " << mEpics.c_str() << "_WF ";
      for (i = mRampStartBin-20 ; i <= mRampStartBin+31 ; i++) Comi << (int)floor((29.5/32)*(pRamp[i] - mOffset+500)) << " ";
      Comi << " >> /dev/null 2>&1";
    
      system(Comi.str().c_str()); 
      // ...end of epics communication...
    }

    return fOffset;
}

//======================================  Reset ramp calculation
void
GPSTimer::reset(void) {
    mTrigTime = Time(0);
    mRampN    = 0;
    fSlope   = -9999999.9;
    fVoltage = -9999999.9;
    fOffset  = -9999999.9;
    fRsquare = -9999999.9;
}

//======================================  DAQ Monitor Constructor
EXECDAT(TimeMon)

  bool RAMP          = false;
  bool IRIG          = false;
  bool SecondTrend   = false;
  bool MinuteTrend   = false;
  bool HourTrend     = false;
  bool SecondLog     = false;
  bool MinuteLog     = false;
  bool HourLog       = false;
  bool DMTViewer     = false;
  bool Tigger        = false;
  bool StateOfHealth = false;
  bool EMail         = false;
  bool Alarm         = false;
  bool Unix          = false;
  bool Histogram     = false;  bool Verbose       = false;
  bool Web           = false;

  char WEBDIR[256]    = "X";
  char HTMLDir[256]   = "X";
  char PlotDir[256]   = "X";
  char TRENDDIR[256]  = "trend";				  
  char LOGDIR[256]    = "log";
  char HISTDIR[256]   = "histogram";
  char EMAIL[256]     = "smarka@ligo.caltech.edu";
  char UNIXM[256]     = "date";
  char UNIXH[256]     = "date";
  char LogFile[256]   = "X";
  char WebViewLink[256] = "http://blue.ligo-wa.caltech.edu/gds/WebViewer/WebIndex.html";

//======================================  DAQ Monitor Constructor
TimeMon::TimeMon(int argc, const char *argv[])
  : DatEnv(argc, argv), MonServer("TimeMon"), 
    mTrendSecond("TimeMonSecondTrend", Trend::kSecond, 900),
    mTrendMinute("TimeMonMinuteTrend", Trend::kMinute, 720),
    mTrendHour  ("TimeMonHourTrend"  , Trend::kNonStandard, 168),
    mAlarm("TimeMon")
{
  // ----------------------------------  Set up default values

  char ConfFile[256] = "TimeMon.conf";

  if (getenv("GDSAPACHE") == 0) {
    cerr << "Unable to parse the $GDSAPACHE variable. Please define it to point to the base of the HTML output directory tree! " << endl;
    finish();
    return;
  }
  strcpy(WEBDIR, getenv("GDSAPACHE"));
  strcat(WEBDIR, "/monitor_reports/TimeMon");
  Web = true;
  strcpy(HTMLDir,WEBDIR);
  strcat(HTMLDir,"/index.html");
  strcpy(PlotDir,WEBDIR);
  strcat(PlotDir,"/TimeMon");

  //getDacc().setTOCMode(false);
  getDacc().setIgnoreMissingChannel(true);
  getDacc().setStride(2.0);

//########################################  Parse command line arguments

    if ( argc > 2 ) {
      cerr << "Too many command line arguments! " << endl << "Correct syntax:" << endl;
      cerr << argv[0] << " <Absolute path to configuration file (default: ./TimeMon.conf)> " << endl << endl;
      finish();
      return;
     } else if ( argc == 1 ) {
              ifstream in(ConfFile);
	      if (in.bad()) {
		cerr << "Unable to open configuration file: " << ConfFile  << endl;
		finish();
		return;
	      }
#ifdef T_DEBUG  
	      cout << "Configuration file " << ConfFile << " is found. Reading configuration info..." << endl;
#endif
	} else if ( argc == 2 ) {
                 ifstream in(argv[1]);
		 if (in.bad()) {
		   cerr << "Unable to open configuration file: " << argv[1] << endl;
		   finish();
		   return;
		 }
#ifdef T_DEBUG  
		 cout << "Configuration file " << argv[1] << " is found. Reading configuration info..." << endl;
#endif
		 strcpy(ConfFile, argv[1]);
	  }

//########################################  Interpreting configuration file 

    // Setting switches, channels, trends and directories...
    ifstream in(ConfFile);
    char line[256];
    char* argl[32];
 
#ifdef T_DEBUG  
      cout << "Parsing configuration file..." << endl;
#endif
    while (!in.getline(line, sizeof(line)).eof()) {
      if ( !( line[0] == '#' ) ) {
	int  len   = in.gcount();
        int  narg   = 0;
	bool inArg  = false;
	bool YesArg = false;
#ifdef T_DEBUG  
	cout << len << " - " << line << endl;
#endif
	for ( int i=0; i < len; i++) {
	  if (!line[i] || line[i] == '\n' ) {
	      line[i] = 0;
	      break;
	  } else if (line[i] == ' ' || line[i] == '\t' || line[i] == ',' ) {
	      inArg = false;
	      line[i] = 0;
	    } else if (!inArg) {
	        inArg  = true;
		YesArg = true;
	        argl[narg++] = line+i;
	      }
	}

       if ( YesArg ) {
#ifdef T_DEBUG  
	cout << len << " -" << argl[0] << "- -" << argl[1] << "-" << endl;
#endif

	if ( !strcmp(argl[0],"SWITCH") ) {

#ifdef T_DEBUG  
	  cout << argl[0] << " found!!" << endl;
#endif
	  //-----------------------------------------Switch Options
	  if ( !strcmp(argl[1], "RAMP") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  RAMP = true;
	  } else if ( !strcmp(argl[1], "IRIG") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  IRIG = true;
	  } else if ( !strcmp(argl[1], "SecondTrend") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  SecondTrend = true;
	  } else if ( !strcmp(argl[1], "MinuteTrend") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  MinuteTrend = true;
	  } else if ( !strcmp(argl[1], "HourTrend") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  HourTrend = true;
	  } else if ( !strcmp(argl[1], "SecondLog") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  SecondLog = true;
	  } else if ( !strcmp(argl[1], "MinuteLog") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  MinuteLog = true;
	  } else if ( !strcmp(argl[1], "HourLog") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  HourLog = true;
	  } else if ( !strcmp(argl[1], "DMTViewer") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  DMTViewer = true;
	  } else if ( !strcmp(argl[1], "Tigger") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  Tigger = true;
	  } else if ( !strcmp(argl[1], "StateOfHealth") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  StateOfHealth = true;
	  } else if ( !strcmp(argl[1], "EMail") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  EMail = true;
	  } else if ( !strcmp(argl[1], "Epics") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  Epics = true;
	  } else if ( !strcmp(argl[1], "Alarm") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  Alarm = true;
	  } else if ( !strcmp(argl[1], "Unix") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  Unix = true;
	  } else if ( !strcmp(argl[1], "Histogram") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  Histogram = true;
	  } else if ( !strcmp(argl[1], "Verbose") ) {
#ifdef T_DEBUG  
	  cout << argl[1] << " found!!" << endl;
#endif
	  Verbose = true;
	  }
	  //------------------------------------
	} else if ( !strcmp(argl[0], "WEBDIR") ) {
#ifdef T_DEBUG  
	  cout << argl[0] << " found!!" << endl;
#endif
	  strcpy(WEBDIR, argl[1]);
	  Web = true;
	  strcpy(HTMLDir,WEBDIR);
	  strcat(HTMLDir,"/index.html");
	  strcpy(PlotDir,WEBDIR);
	  strcat(PlotDir,"/TimeMon");
	} else if ( !strcmp(argl[0], "TRENDDIR") ) {
#ifdef T_DEBUG  
	  cout << argl[0] << " found!!" << endl;
#endif
	  strcpy(TRENDDIR, argl[1]);
	} else if (  !strcmp(argl[0], "LOGDIR") ) {
#ifdef T_DEBUG  
	  cout << argl[0] << " found!!" << endl;
#endif
	  strcpy(LOGDIR, argl[1]);
	} else if (  !strcmp(argl[0], "HISTDIR") ) {
#ifdef T_DEBUG  
	  cout << argl[0] << " found!!" << endl;
#endif
	  strcpy(HISTDIR, argl[1]);
	} else if (  !strcmp(argl[0], "EMAIL") ) {
#ifdef T_DEBUG  
	  cout << argl[0] << " found!!" << endl;
#endif
	  strcpy(EMAIL, argl[1]);
	} else if (  !strcmp(argl[0], "UNIXM") ) {
#ifdef T_DEBUG  
	  cout << argl[0] << " found!!" << endl;
#endif
	  strcpy(UNIXM, argl[1]);
	} else if (  !strcmp(argl[0], "UNIXH") ) {
#ifdef T_DEBUG  
	  cout << argl[0] << " found!!" << endl;
#endif
	  strcpy(UNIXH, argl[1]);
	} else if (  !strcmp(argl[0], "TTAG") ) {
#ifdef T_DEBUG
          cout << argl[0] << " found!!" << endl;
#endif
          strcpy(TTAG_Ch, argl[1]);
        } else if (  !strcmp(argl[0], "ALIVE") ) {
#ifdef T_DEBUG
          cout << argl[0] << " found!!" << endl;
#endif
          strcpy(ALIVE_Ch, argl[1]);
        } else if (  !strcmp(argl[0], "RAMP_CH") ) {
#ifdef T_DEBUG  
	  cout << argl[0] << " found!!" << endl;
#endif
	//----------------------------------  Set up RAMP channel 
	mGPSVect.push_back(new GPSTimer(argl[1], argl[2], argl[3], argl[4], atof(argl[5]), atof(argl[6]), getDacc()));
#ifdef T_DEBUG  
        cout << argl[1] << " " << argl[2] << " " << argl[3] << " " << argl[4] <<  " "  << argl[5] <<  " "  << argl[6] <<  " " << mGPSVect.size() << endl;
#endif
	//----------------------------------  Set up the trend channels
	const char* trend = argl[2];
	if (SecondTrend) {
	  mTrendSecond.addChannel(trend);
	  serveData(trend, &(mTrendSecond.find(trend).refAvgSeries()), "TimeMon Second Trend");
	}	
	if (MinuteTrend) {
	  mTrendMinute.addChannel(trend);
	  serveData(trend, &(mTrendMinute.find(trend).refAvgSeries()), "TimeMon Minute Trend");
          //mTrendMinute.writeIndex();
	}
	if (HourTrend) {
	  mTrendHour.addChannel(trend);
	  mTrendHour.setSample(Interval(1000.0));
	  serveData(trend, &(mTrendHour.find(trend).refAvgSeries()), "TimeMon Hour Trend");
	}	//----------------------------------  Statistics for this RAMP channel
	mNRamps.push_back(0);
	mAvgOff.push_back(0.0);
	mSigOff.push_back(0.0);
        mAvgTrg.push_back(0.0);
	mSigTrg.push_back(0.0);
        mLAvg.push_back(0.0);
        mLSig.push_back(0.0);
        mLN.push_back(0);

	} else if (  !strcmp(argl[0], "IRIG_CH") ) {
#ifdef T_DEBUG  
	  cout << argl[0] << " found!!" << endl;
#endif
	}
      }
     }
    }
     const char* hugeDelayDesc="Timing RAMP on channel $1 is shifted reltive to the GPS second tic $2 with: $3 us.";
     const char* webFile="../monitor_reports/TimeMon/";
     AlarmData HugeDelay("TimeMon", "Huge_Delay", Interval(120), 7, hugeDelayDesc);
     HugeDelay.jamFlags(AlarmData::kReTrigger);
     HugeDelay.setWebFile(webFile); 
     mAlarm.defineAlarm(HugeDelay); 

#ifdef T_DEBUG  
    cout   <<       "============== Values (1=ON, 0=OFF) ====\n"
	         << "Config. File  = " <<  ConfFile      << "\n"
	         << "RAMP          = " <<  RAMP          << "\n"
	         << "IRIG          = " <<  IRIG          << "\n"
	         << "SecondTrend   = " <<  SecondTrend   << "\n"
	         << "MinuteTrend   = " <<  MinuteTrend   << "\n"
	         << "HourTrend     = " <<  HourTrend     << "\n"
	         << "SecondLog     = " <<  SecondLog     << "\n"
	         << "MinuteLog     = " <<  MinuteLog     << "\n"
	         << "HourLog       = " <<  HourLog       << "\n"
	         << "DMTViewer     = " <<  DMTViewer     << "\n"
	         << "Tigger        = " <<  Tigger        << "\n"
	         << "StateOfHealth = " <<  StateOfHealth << "\n"
	         << "EMail         = " <<  EMail         << "\n"
	         << "Epics         = " <<  Epics         << "\n"
	         << "Alarm         = " <<  Alarm         << "\n"
	         << "Unix          = " <<  Unix          << "\n"
	         << "Histogram     = " <<  Histogram     << "\n"
	         << "Verbose       = " <<  Verbose       << "\n"
	         << "WEBDIR        = " <<  WEBDIR        << "\n"
	         << "TRENDDIR      = " <<  TRENDDIR      << "\n"			  
	         << "LOGDIR        = " <<  LOGDIR        << "\n"
	         << "HISTDIR       = " <<  HISTDIR       << "\n"
	         << "EMAIL         = " <<  EMAIL         << "\n"
	         << "UNIXM         = " <<  UNIXM         << "\n"
	         << "UNIXH         = " <<  UNIXH         << "\n"
	   <<       "========================================\n"
	   << endl;
#endif

    if (SecondLog || MinuteLog || HourLog) {
      strcpy(LogFile, LOGDIR);
      strcat(LogFile, "/");
      strcat(LogFile,  "TimeMon.log");
    }

    // Get environmental variable for CAPUT command...

    if (getenv("EPICSBINDIR") != 0) {
       strcpy(CaPutDir, getenv("EPICSBINDIR"));
    }
}

//======================================  DAQ Monitor Destructor
TimeMon::~TimeMon() {
    mTrendSecond.close();
    mTrendMinute.close();
    mTrendHour.close();
}

//======================================  Process a DAQ frame
void
TimeMon::ProcessData(void) {
    ofstream OutLog(LogFile, ios::app );
    time_t TS;
    double GPS_T(0.0);
    double Off;
    double TD, TA;

    if (Epics) {	 
      // EPICS communication every 2 seconds ...
      ostringstream Comm0;
      Comm0 << CaPutDir << "/caput " << ALIVE_Ch << " " << abs((int)Alive) << " >> /dev/null 2>&1 \n";
      system(Comm0.str().c_str()); 
      Alive+=128;

      time(&TS);
      ostringstream Com00;
      Com00 << CaPutDir << "/caput " << TTAG_Ch << " \" `date -u` \"   >> /dev/null 2>&1 \n";
      system(Com00.str().c_str()); 
      // ...end of epics communication...
    }

	 for (unsigned int i=0 ; i<mGPSVect.size() ; i++) {
         time(&TS);
	  
         mGPSVect[i]->process();
	 mRamp = *(mGPSVect[i]->getRampSeries());

	if ((0==i) && (30==cnt)) {
	  ofstream WebOut(HTMLDir , ios::out );
	  if (Web)    WebOut    << "<!doctype html public \"-//w3c//dtd html 4.0 transitional//en\">\n<html>\n"
				<< "<head> \n"
				<< "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\"> \n"
				<< "<meta name=\"Author\" content=\"Szabolcs Marka\"> \n"
				<< "<meta name=\"GENERATOR\" content=\"Mozilla/4.78 [en] (Win98; U) [Netscape]\"> \n"
				<< "<title>Ramp Based Timing Monitor at LXO</title> \n"
				<< "<meta http-equiv=\"Refresh\" content=\"60; URL=index.html\">\n"
				<< "<META HTTP-EQUIV=\"Pragma\" CONTENT=\"no-cache\">"
				<< "</head>\n"
				<< "<body>\n" 
				<< "\n<b><font color=\"#FF6600\"><font size=+4>Ramp based timing monitor</font></font></b> \n"
                                << "<br><font color=\"#66AAAA\"><font size=+2>[ " 
                                << asctime(localtime(&TS)) << " ( <b>" << asctime(gmtime(&TS)) << "</b>UTC) ]</font></font>"
				<< endl;
	  for (unsigned int t=0 ; t<mGPSVect.size() ; t++) {
         
            float M_Th = (mGPSVect[t]->getMarginalTh())*1E-6;
            float E_Th = (mGPSVect[t]->getErrorTh())*1E-6;
#ifdef T_DEBUG
	    cout << "AgSum:  " << floor(mAvgOff[t]*1.000E+9)/1000 <<  " , Number of samples: " <<  mNRamps[t] << endl;
#endif
	    mAvgOff[t] /= mNRamps[t];
	    mSigOff[t] /= mNRamps[t];
	    double sigma0 = mSigOff[t] - mAvgOff[t]*mAvgOff[t];
	    if (sigma0 > 0) sigma0 = sqrt(sigma0);
	    else            sigma0 = 0.0;

#ifdef T_DEBUG
	    cout << "Average Delay:  " << floor(mAvgOff[t]*1.000E+9)/1000 << " , Sigma: " << floor(sigma0*1.0E+9) << endl;
#endif 
	    if (MinuteLog) OutLog << "\nMT:: " << mGPSVect[t]->getRampName() << " "
				  << floor(mAvgOff[t]*1.000E+9)/1000 << " us " 
				  << floor(sigma0*1.0E+9) << " ns "
				  << (long int)GPS_T << " " << asctime( gmtime(&TS) );
	    if (mLN[t] >= 60*60) {
	      mLAvg[t] /= mLN[t];
	      mLSig[t] /= mLN[t];
	      if (HourLog) OutLog << "\nHT:: " << mGPSVect[t]->getRampName() << " "
				  << floor(mAvgOff[t]*1.000E+9)/1000 << " us " 
				  << floor(sigma0*1.0E+9) << " ns "
				  << (long int)GPS_T << " " << asctime( gmtime(&TS) );

              mLAvg[t] = 0.0;
	      mLSig[t] = 0.0;
	      mLN[t]   = 0;
	    }

	    if (Epics) {
	      // EPICS communication by the minute...
	      ostringstream Comm1;
	      ostringstream Comm2;
          
	      TD = floor(mAvgOff[t]*1.000E+9)/1000;
	      TA = floor(sigma0*1.0E+9);

	      if ( abs(TD) > 0.51E+6 ) {
		TA = -4321;
		TD = -1234;
	      }
    
	      Comm1 << CaPutDir << "/caput " << mGPSVect[t]->getEpicsName() << " " << TD << " >> /dev/null 2>&1 \n";
	      Comm2 << CaPutDir << "/caput " << mGPSVect[t]->getEpicsName() << "_ACC " << TA << " >> /dev/null 2>&1 \n";

	      system(Comm1.str().c_str());
	      system(Comm2.str().c_str());
	      // ...end of epics communication...
	    }

       char LXO[4]      = "L?O";
       strncpy(LXO, getenv("LIGOSITE"), 3); 
       if (!(strcmp(LXO,"LLO") || strcmp(LXO,"LHO"))) {
        cerr << LXO << " cannot be identified as LIGO site! Check the $LIGOSITE environmental variable" << endl;
        exit(0);
       }
#ifdef T_DEBUG
  cout << "LIGO Site identified : " << LXO << endl;
#endif
       if (!strcmp(LXO,"LHO")) {
         strcpy(WebViewLink, "http://stone.ligo-wa.caltech.edu:9995/TimeMon/objects.html");
       } else if ((!strcmp(LXO,"LLO"))) {
                strcpy(WebViewLink, "http://delaronde.ligo-la.caltech.edu:9995/TimeMon/objects.html");
              }
       
	    if (Web) {
		   WebOut    << "<hr SIZE=1 WIDTH=\"100%\"> \n"
				       << "<br><b><tt><font color=\"#000099\"><font size=+1>DAQ Channel (DAQ time stamp) : " << endl
                   << "<a href=\"" << WebViewLink << "\">"
  				       << mGPSVect[t]->getRampName() << "</a> ( " << (long int)GPS_T << " ) <br></font><font size=+2> Delay = ";
			
			if ( mAvgOff[t] > -999999.9 ) {
				WebOut << floor(mAvgOff[t]*1.000E+9)/1000 << " us +/- " 
				       << floor(sigma0*1.0E+9) << " ns "
				       << "</font></font></tt></b> \n";
				} else {
				    WebOut << "<font color=\"#FF2200\"> Timing signal is missing ! </font> </font></font></tt></b> \n";
				}

	      if ( fabs(mAvgOff[t]) < M_Th ) {
		     WebOut  << "<b><font color=\"#00CC00\"><font size=+2>&nbsp;&nbsp;OK...</font></font></b> <br>" << endl;
	      } else if ( fabs(mAvgOff[t]) < E_Th ) {
		       WebOut  << "<b><blink><font color=\"#FF6600\"><font size=+3>&nbsp;&nbsp;Marginal!</font></font></blink></b><br>" << endl;
	        } else {
		         WebOut  << "<b><blink><font color=\"#FF0000\"><font size=+4WebViewLink>&nbsp;&nbsp;ERROR!!</font></font></blink></b><br>" << endl;
	          }
	    }

      if ( fabs(mAvgOff[t]) > M_Th ) {
         lmsg::error_t rc = 0;
	      AlarmHandle han;
	      ostringstream param;
	      param << mGPSVect[t]->getRampName() << " " << mGPSVect[t]->getTrigTime() << " " << floor(mAvgOff[t]*1.000E+9)/1000 << ends;
	      AlarmData al("TimeMon", "Huge_Delay", 0, 5, "", param.str());
	      rc = mAlarm.setAlarm(al, han);
	      if (rc) cout << "[TimeMon] Error sending alarm: " << rc << endl;
#ifdef T_DEBUG  
	      cout << "ALARM: " << mGPSVect[t]->getRampName() << " " << (long int)GPS_T << " " <<floor(mAvgOff[t]*1.000E+9)/1000 << endl;
#endif	
      }

	    mAvgOff[t] = 0.0;
	    mSigOff[t] = 0.0;
	    mNRamps[t] = 0;
	  }
	  
	  cnt = 0;
	  if (Web)    WebOut    << "\n<br>\n<hr SIZE=1 WIDTH=\"100%\"> \n"
				<< "<br><font size=-1>This page was last modified : "
				<< asctime(localtime(&TS)) << " ( " << asctime(gmtime(&TS)) << "(UTC) )" 
				<< " by Ramp Based Timing Monitor of DMT</font> \n<br><font size=-1>" 
				<< "Contact: <a href=\"mailto:smarka@ligo.caltech.edu\">Szabi Marka</a>.</font> \n"
				<< "<br><b><tt><font color=\"#000099\"><font size=+1></font></font></tt></b>&nbsp; \n"
//				<< "<SCRIPT LANGUAGE=\"JavaScript\"> \n"
//				<< "setTimeout(\"location.reload()\",360000) \n"
//				<< "</SCRIPT>\n"
            << "</body>\n</html>\n"
				<< endl;

	  if (Web)    WebOut.close();
	}
	if (0==i) cnt++;

	//--------------------------  Trend the data
	Time   trig = mGPSVect[i]->getTrigTime();
	Off  = mGPSVect[i]->getRampOff();
        if ( trig > Time(0) ) {
	  if (SecondTrend) mTrendSecond.trendData(mGPSVect[i]->getTrendName(), trig, Off);
	  if (MinuteTrend) mTrendMinute.trendData(mGPSVect[i]->getTrendName(), trig, Off);
#ifdef T_DEBUG
          cout << "," << mGPSVect[i]->getTrendName() << "," << trig << "," << Off << endl;
#endif
	  if (HourTrend)   mTrendHour.trendData(mGPSVect[i]->getTrendName(), trig, Off);
	}
	GPS_T = trig.totalS();
#ifdef T_DEBUG
	//--------------------------  Error Messages
	if ( fabs(Off) > 1.1900E-4 ) {
	  cerr << "ERR: " << mGPSVect[i]->getRampName() << " "
	       << floor(Off*1.000E+9)/1000 << " us " 
	       << (long int)GPS_T << " Unacceptable timing delay!!!!" << endl;
	}
#endif
	//--------------------------  Frequent output
	if ((i == 0) && SecondLog ) OutLog << "\n" << (long int)GPS_T << " ";
	if (SecondLog) OutLog << mGPSVect[i]->getRampName() << " " << floor(Off*1.000E+9)/1000 << " "; 
	//-------------------------- 
	mNRamps[i]++;
	mAvgOff[i] += Off;
	mSigOff[i] += Off*Off;
	mLN[i]++;
	mLAvg[i] += Off;
	mLSig[i] += Off*Off;
   }
}

void 
TimeMon::Attention(void) {
    MonServer::Attention();
}

void 
TimeMon::Reset(void) {
}

