///////////////////////////////////////////////////////////////////////////////
//
//	Class StrainbandsMon: Base class for monitoring strain in frequency bands
//	version 1.2 (2006.05.26)
//  author: Ramon S. Armen (rarmen@umich.edu)
//
///////////////////////////////////////////////////////////////////////////////

/************************************USAGE*************************************

StrainbandsMon [-verbosity <level>] [-OSCFile <filename>] [-config <filename>]
	[-calFile <filename>] [-OSCcond <condition>] [-window <windowtype>]
	[-help] [-ifo <name>] [-channel <name>] [-stride <length>]
	[-intervals <num>] [-history <length>] [-debug]

-verbosity <level>
	Defines the amount of text that StrainbandsMon will print out to cout.
	<level> must be >= 1. Verbosity of 1 yields no output (except for error
	messages). Verbosity 2 prints out minimal status information. 3 prints
	out every time data is being processed. 4 prints out parameter information
	and the bands that are being monitored. 5 prints out if data is not
	valid (OSC not met or dropouts). 6 prints out the values for the strain
	at every time stride.
	Defaults to 1.

-OSCFile <filename>
	Defines the file used to define operating state conditions.
	Defaults to "LockLoss.conf"

-config <filename>
	Defines the config file for StrainbandsMon. The config file defines
	parameters and bands to monitor.
	
	Format of config file:
	Any of the above arguments can be put in a config file with the same
	syntax. In addition, there is another argument, -bands, after which
	frequency bands may be defined. Each line thereafter, until the end
	of the config file is reached or the line "-end" is reached, should
	contain the information for one band. Each band should be in the
	following format:
	
	<lowerFreq> <upperFreq> [[notch <lowerFreq> <upperFreq>] ...] [name <bandName>]
	
	lowerFreq and upperFreq specify the lower and upper frequency limits of
	the band. You may then optionally specify notches within that band
	by using the keyword "notch" followed by the lower and upper frequency limits
	of the notch. This frequency range will be ignored when calculating
	strain for this band. You may specify an arbitrary number of notches,
	but each notch must start with the word "notch". 
	
	If the keyword "name" is encountered, the following word will be used to
	create the channel name. If a name is provided, the resulting channel name
	for this band will be <ifo>:DMT-SBND_AS_Q_<bandName> where <ifo> is the IFO
	being monitored (one of H1, H2, or L1), and <bandName> is the specified name.
	If no name is provided, a default of <ifo>:DMT-SBND_AS_Q_<lowerFreq>_<upperFreq>
	will be used for the channel name, where <ifo> is the IFO being monitored, 
	and <lowerFreq> and <upperFreq> are the frequency limits of the band. If 
	notches were specified, then an addition suffix of _notched will be added, 
	resulting in a channel name <ifo>:DMT-SBND_AS_Q_<lowerFreq>_<upperFreq>_notched
	NOTE: Only alphanumerical characters and the underscore are allowed in 
	<bandName>. I.e. whitespace, hyphens, or other characters are prohibited due 
	to the channel naming conventions.
	
	It is acceptable to use multiple lines for a band if necessary.
	
	Arguments specified on the command line after the -config argument will
	take precedence over parameters specified in the config file. Parameters in
	the config file take precedence over arguments on the command line which
	occur before the -config argument.
	
	Example config file:
	-verbosity 4
	-ifo H1			#the IFO to listen to
	-window hanning	#use a hanning window
	
	-bands
	50 100 notch 57 63 name 50_100_Hz_notched
	100 200 notch 118 122 notch 178 182
	150 160
	
	This would monitor the H1 interferometer. There would be three bands, as
	specified above.
	
	The config file defaults to "StrainbandsMon.cfg"

-calFile <filename>
	The filename to use as a reference file for FDEasyCalibrate.
	Defaults to "ReferenceCalibration_XX.xml", where XX is the interferometer
	being monitored by this DMT instance.

-OSCcond <condition>
	Defines an OSC condition. If this condition is not met, the data will be
	ignored, and the band value set to -1.
	Defaults to "XX:Both_arms_locked_strict_cm", where XX is the interferometer
	being monitored.

-window <windowtype>
	Defines the type of window to use for the fourier transform. <windowtype>
	must be one of "hanning", "hamming", "blackman" or "flattop"
	Defaults to a Hanning window

-help
	Show this information

-ifo <name>
	The name of the IFO to monitor. Must be one of H1, H2, or L1. Either -ifo
	must be specified, or -channel must be specified which includes an IFO name
	If -channel is specified as "XX:channelName", where XX is an interferometer, 
	and -ifo is not set, it will automatically be set to XX.

-channel <name>
	The name of the channel to monitor. This may either be in the format of
	":channelName" or "XX:channelName". If it is the former, -ifo must be
	specified, and the channel will be set to "ifo:channelName". If it is the
	latter, then -ifo need not be specified, but if it is, the user-defined
	IFO will be used. An IFO must be defined in either -channel or -ifo.
	Defaults to ":LSC-AS_Q"

-stride <length>
	The number of seconds in each timestride.
	Defaults to 60.

-intervals <num>
	The number of intervals used for each timestride. Must divide evenly
	into the timestride length.
	Defaults to 15.

-history <length>
	The length, in seconds, of the DMT viewer history. This number of
	seconds of data will be remembered.
	Defaults to 12 hours (43200)

-debug
	Enter debug mode. This will append _debug to the server name.

Note: An interferometer must be specified, either using the -ifo argument,
or by defining a -channel which contains an IFO name at the beginning. All
other arguments have default values (though the necessary files must be
present in order to function properly).

******************************************************************************/


#ifndef STRAINBANDSMON_HH
#define STRAINBANDSMON_HH

//Units for strain. Since, given current sensitivities, strain for most
//interesting bands is on the order of 10^-22 or larger, we multiply
//the strain values by 10^22, because the DMT viewer cannot handle
//very small data.
#define SCALE_FACTOR 1.0e22

//Standard library includes
#include <string>
#include <fstream>
#include <stdlib.h>
#include <stdio.h>
#include <iostream>

//DMT classes used in this program definition	
#include "DatEnv.hh"
#include "MonServer.hh"
#include "window_api.hh"
#include "OperStateCondList.hh"
#include "TSeries.hh"
#include "FDEasyCalibrate.hh"
#include "PSD.hh"
#include "FSpectrum.hh"
#include "Trend.hh"

//load std functions into class's scope
using namespace std;


/**
 * A structure containing all the pertainant information for a given
 * frequency band that we are monitoring.
 */
struct Band {
	float lowerFreq; //the lower frequency limit of the band
	float upperFreq; //the upper frequency limit of the band
	string channelName; //the name of the channel for this band
	TSeries *history; //information about past values
	float *historyVals; //past values of strain in this band
						//this array is what is sent to the DMT viewer.
	
	//notches within the above frequency which we do not pay attention to.
	//Each notch has two entries: a lower and an upper frequency. For
	//example, if notches={118, 121, 177, 182}, two notches will be taken
	//into account - one from 118-121, and one from 177 to 182
	vector<float> notches;
};

/**
 * A data monitoring tool to monitor the strain on a LIGO interferometer
 * within a certain frequency band. It obtains ASQ data and calibrates
 * it using FDEasyCalibrate. It then calculates the strain from
 * the calibrated data.
 */
class StrainbandsMon : public DatEnv, MonServer {
	public:
		/**
		 * Construct a new monitor with command-line arguments specifiying
		 * parameters. See above for list of valid parameters. (!!!)
		 * @param argc the number of command line arguments passed
		 * @param argv the command line arguments passed to the program
		 */
		StrainbandsMon( int argc, const char *argv[] );
		
		/**
		 * Destructor, called upon termination or completion of the
		 * monitor. The monitor will normally only exit on a termination,
		 * but additional parameters may be specified allowing for it
		 * to terminate after a certain number of timestrides. Cleans
		 * up data and deletes allocated memory.
		 */
		~StrainbandsMon( void );
		
		/**
		 * Called from the DMT framework every timestride when new
		 * data is received. This method is where all the computations
		 * are made, and where data is sent to the DMT viewer and the
		 * trend files.
		 */
		void ProcessData( void );
		
		/**
		 * Called from the DMT framework if some action has occured
		 * which requires our attention. This alert is not handled by
		 * StrainbandsMon, and is sent on to MonServer for handling.
		 */
		void Attention( void );

		/**
		 * Loads arguments from a stream (either from the command line or
		 * config file) into parameter settings. If an error occures, this
		 * method will terminate, returning the word that caused the error.
		 * s will be updated so that the next word read will be the word
		 * following the one that caused the error, or will be at the end
		 * of the input of no error occured.
		 * @param stream the stream that reads arguments
		 * @return the argument name that an error occured at, or the string
		 *		   "" if no error occured.
		 */
		string loadParams( istream *s );
		
		/**
		 * Loads the bands to monitor from a stream.
		 * @param stream the stream that will read the band limits
		 * @return the text that an error occured for, or the string ""
		 *		   if no error occured.
		 */
		string readBands( istream *s );
		
		/**
		 * Print out usage information.
		 */
		void dumpUsage( void );

		/**
		 * Set parameters to their default values.
		 */
		void setDefaultParams( void );
		
		/**
		 * Read command line arguments into parameters.
		 * @param argc the number of arguments
		 * @param argv the arguments, preparsed into words
		 */
		void parseCommandLine( int argc, const char *argv[] );
		
		/**
		 * Read the config file, setting up the parameter information and
		 * information about the bands to monitor. Uses the config file 
		 * from variable configFile to get info from.
		 */
		void readConfigFile();
		
		/**
		 * Dumps parameter information, the bands being monitored, a
		 * summary of the program, a timestamp, etc to the html file
		 * located at $DMTHTMLOUT/index.html
		 * @param time the current time
		 * @param lastData the last time valid data was recieved.
		 *		  If lastData == curTime, then we should be in lock,
		 *		  likewise the inverse is true.
		 */
		void dumpHTMLFile( Time curTime, Time lastData );
	
	private:
		//Parameter information
		int verbosity; //defines amount of info to print out
		string oscFile; //the file defining operating state conditions
		string configFile; //file defining bands to monitor
		string refDataFile; //calibration info for FDEasyCalibrate
		string lockConditions; //conditions that we require
		window_api *window; //window function for PSD calculation
		string windowType; //the name of the window type being used
		string ifo; //the ifo we are monitoring
		string channel; //the channel we are manitoring
		vector<Band *> bands; //the bands that we are monitoring
		
		int timeStride; //the number of seconds between strain calculations
		int numIntervals; //the number of averages in each timestride.
						  //must divide exactly into timestride
		int historyLength; //The number of seconds which data will be kept on
						   //the DMT viewer for
		bool debugMode; //whether we are in debug mode or not

		
		//some other important information
		int tserieslength; //the length of the history array 
						   //(=historyLength/timeStride)
		bool isFirstRun; //if it is the first timestride that we have processed
		float arm_length; //the length of the IFO arm.
		Time startTime; //the time the monitor was started
		Time lastDataTime; //the last time we were in lock
		
		OperStateCondList mOSC; //ensures that we are getting valid data
		PSD *psd; //computes the power spectrum density
		FDEasyCalibrate *mCalibrate; //calibrates data to useful quantities
		Trend trend; //keeps track of all the data trends
		
		double lastFillTime; //last time data was processed (including out of lock)
};

#endif // STRAINBANDSMON_HH included
