//-----------------------StochMon-------------------------------------
//
//   Stochastic Sensitivity Monitor 
//   version: $Id: StochMon.cc 7382 2015-05-24 11:26:59Z john.zweizig@LIGO.ORG $
//   authors: Marc J. Cenac (mjcenac@loyno.edu) 
//   based on SenseMonitor by:
//            Kevin C. Schlaufman (kcs149@psu.edu)
//            Patrick J. Sutton (psutton@ligo.caltech.edu)
//
//----------------------------------------------------------------------


// Notes:
//
// o R_Dat.{T5_stochsens, T15_stochsens} hold (calibrated) stochastic
//    sensitivity data (background detectable in one minute, averaged over
//    5 or 15 minutes).
// -- Output:
// o Report R_Dat.{invomegasq, invomegasq_uncal, ampl, gain, alpha, beta} to trends 
//   and log files always.  Report only .invomegasq to DMTViewer if not tracking 
//   a line.

//---------- Defined constants:
#define DMTViewer_Period 43200  // Time (sec) over which to display stochastic sensitivity histories on DMTViewer.

//---------- definition of monitor program.
#include "StochMon.hh"

#ifndef __CINT__

//---------- Include standard headers
#include <cstdlib>
#include <iomanip>
#include <cmath>
#include <cstdio>
#include <stdint.h>

//---------- Include LIGO-specific headers
#include "Sine.hh" 
#include "Dacc.hh"
#include "html/writer.hh" 
#include "html/Attrib.hh"
#include "html/align.hh"
#include "html/color.hh"
#include "html/document.hh"
#include "html/font.hh"
#include "html/hline.hh"
#include "html/image.hh"
#include "html/label.hh"
#include "html/link.hh"
#include "html/size.hh"
#include "html/style.hh"
#include "html/table.hh"
#include "html/text.hh"
#include "wavearray.hh"
#include "wseries.hh"

#endif               // !def(__CINT__)

//---------- Include monitor-specific headers
#include "PSD.hh"
#include "Integrand.hh"
#include "Integrate.hh"
#include "LLOLHO_Overlap.hh"
#include <stdint.h>


#ifndef __CINT__
//======================================  Generate the main routine.
EXECDAT(StochMon) //Main program C preprocessor macro -- Invoke monitor::MainLoop()
#endif               // !def(__CINT__)
  
  using namespace std;

const char *USAGE_INFO =
"\n"
"StochMon: Stochastic Sensitivity Monitor version: pre-release $Id: StochMon.cc 7382 2015-05-24 11:26:59Z john.zweizig@LIGO.ORG $"
"\n\n"
"Usage:\n\n"
"./StochMon [optional arguments] <IFO_1> <IFO_2>, \n"
"where <IFO_1> and <IFO_2> are a pair of H1, H2, or L1 \n "
"The optional arguments are:\n"
"                 \n"
"-fmax #          Specify maximum frequency to include in sensitivity integration.\n"  
"                 Default 1400Hz.\n"
"                 \n"
"-fmin #          Specify minimum frequency to include in sensitivity integration.\n"
"                 Default  20Hz.\n"
"                 \n"
"-PSD_DumpPeriod  Number of strides  between each dump of a calibrated\n."
"                 DARM_ERR PSD.  Default (every stride). (Use 0 if you don't \n"
"                 want any periodic stochastic sensitivity and calibrated\n."
"                 PSD dumps) \n."
"                 \n"
"-h, --help       Print this usage information.\n"
"                 \n"
"-local           Local mode:  All output files are dumped in local directory\n"
"                 instead of the default directory specified by $DMTHTMLOUT;\n"
"                 plain-text log file renamed <GPS_Start_Time>.log.\n"
"                 \n"
"-logfile <name>  Name plain-text version of log file <name>.  Only works if\n"
"                 -local option is also selected.  Default filename is\n"
"                 <GPS_start_time>.log (-local mode) or\n"
"                 <IFO_1><IFO_2>_StochMon_CumLog.txt (otherwise).\n"
"                 \n"
"-max #           Specify maximum number of strides to process before program\n"
"                 exits.\n"
"                 \n"
"-n #             Specify number of sections to divide data into for \n" 
"                 calculating PSD.  THIS NUMBER MUST DIVIDE 'stride' EVENLY!\n"
"                 Default 15.\n"
"                 \n"
"-OSCfile <file>  Specify the OSC configuration file defining the conditions\n"
"                 for the interferometer to be in lock.  If not specified,\n"
"                 StochMon will attempt to open\n"
"                 $STOCHMON_OSCCONF/StochMon_LockLoss.conf,\n"
"                 then ./StochMon_LockLoss.conf.  StochMon will exit\n"
"                 if no OSC configuration file can be found.\n"
"                 \n"
"-refpsd <file>   Specify file to be used as reference spectrum for P2(f).\n"
"                 \n"
"-screen          Update and error messages are sent to the screen instead of\n"
"                 to the default files <IFO_1><IFO_2>_StochMon_Log.txt and \n"
"                 <IFO_1><IFO_2>_StochMon_Errors.txt.\n"
"                 \n"
"-stride #        Specify length in seconds of data to use for each (1/Omega^2) \n"
"                 estimate.  Default 60.\n"
"                 \n"
"-trend           Write Stochastic Sensitivity (1/Omega^2), relative gain and calibration-line-\n"
"                 amplitude data to trend files.\n"
"                 \n"
"-window <type>   Specify data window type, where <type> is: \n"
"                  hanning for Hanning window,\n"
"                  blackman for Blackman window,\n"
"                  flattop for FlatTop window (NOT RECOMMENDED),\n"
"                  hamming for Hamming window (NOT RECOMMENDED),\n"
"                  square for no window (NOT RECOMMENDED).\n"
"                 Default Hanning.\n"
"                 \n"
"-xmlfile_1 <file>  REQUIRED. Use dynamical calibration based on a reference \n"
"                   calibration file <file>. \n"
"-xmlfile_2 <file>  REQUIRED** only with the double-live version"
"                   \n";


//----- Html Text
const char* const kHtmlTitle = "StochMon  ($Id: StochMon.cc 7382 2015-05-24 11:26:59Z john.zweizig@LIGO.ORG $): Stochastic Background "
			       "Sensitivity Monitor";
const char* const kHtmlAuthor = "Marc J. Cenac (mjcenac@loyno.edu)";
const char* const kHtmlDescription =
   "This monitor calculates one over the square of the Omega parameter "
   "of the faintest constant-Omega stochastic background which could be "
   "detected with a false alarm rate of 5% and a false dismissal rate of 5% "
   "using one minute of data.";
const char* const kHtmlLIGO =
   "Laser Interferometer Gravitational-wave Observatory ";
const char* const kHtmlLIGOLink = "http://www.ligo.caltech.edu/";

//======================================  StochMon constructor.
StochMon::StochMon(int argc, const char *argv[]) 
  : DatEnv(argc, argv), 
    fatal_error(false), prev_endtime(0.0), running_average(0.0),
    running_count(0.0), MaxStride(99999), missing_strides(0), NStride(0), time_since_update(0)
{
  
  //---------- Initialize structures.
  mOutput.error_log = true;
  mOutput.local = false;
  mOutput.screen_output = false;
  mOutput.trend = false;
  mOutput.write_log = true; 
  Run_Par.T = 60.0;
  Run_Par.l_freq = 20.0;
  Run_Par.h_freq = 1400.0;
  Run_Par.stride_dump = 15;
  Run_Par.num_ints = 15;
  Run_Par.CalPSD_file_name = "";
  R_Dat.alpha = 0.0;
  R_Dat.beta = 0.0;
  R_Dat.ampl = 0.0;
  R_Dat.omega = 0.0;
  R_Dat.invomegasq = 0.0;
  R_Dat.T5_stochsens = 0.0;
  R_Dat.T15_stochsens = 0.0;
  for(int i=0; i<5; i++)  R_Dat.T5_array[i] = 0.0;
  for(int i=0; i<15; i++)  R_Dat.T15_array[i] = 0.0;
  
  //---------- Look for command-line arguments.
  //           First check if configuration file is being used.
  //           In this case the command line should read 
  //             StochMon -config <filename>
  //           If file <filename> does not exist or cannot be parsed 
  //           then exit with error message.
  //           If not using config file, make sure we have some arguments and that
  //           the last one does not start with a hyphen.
  if ((argc == 3) && (!strcmp("-config", argv[1]))) {
    Run_Par.config_file_name = argv[2];
    Configure(Run_Par.config_file_name);
  } else {
    Configure(argc, argv); 
  }
  
  //---------- Send monitor name to MonServer.
  std::string MonitorName; 
  if (Run_Par.CalPSD_file_name == "") {
      MonitorName = "StochMon_2L_";
      LiveMode="2L";
  } else {
      MonitorName = "StochMon_1L_";
      LiveMode="1L";
  }
  MonitorName += Run_Par.IFO_1 + Run_Par.IFO_2; 
  MonServer::setServerName(MonitorName.c_str());
  
  //---------- Do some elementary checks on the parameters.
  VerifyParameters(Run_Par,fatal_error);
  
  //---------- Finish setup of monitor.
  if (!fatal_error) {
    //-------- Check to see if running locally.  If not, dump html 
    //         files to $DMTHTMLOUT instead of to local directory.
    //         Plain-text log file is named in ProcessData() method.
    if (!mOutput.local) {
      mOutput.html_dir = getenv("DMTHTMLOUT");
      if (mOutput.html_dir) {
	mOutput.dmtviewer_file_name = string(mOutput.html_dir) + "/";
	mOutput.error_file_name = string(mOutput.html_dir) + "/";
	mOutput.log_file_name = string(mOutput.html_dir) + "/";
	mOutput.revolver_file_name = string(mOutput.html_dir) + "/";
	mOutput.summary_file_name = string(mOutput.html_dir) + "/";
      } else { 
	std::cerr << "StochMon WARNING: Environment variable "
		  << "DMTHTMLOUT not set; switching to\n";
	std::cerr << "                      -local mode (all output "
		  << "files go to the local directory).\n";
	mOutput.local = true;
      }
    }
    mOutput.dmtviewer_file_name += Run_Par.IFO_1 + Run_Par.IFO_2 + "_DMTViewer_Data.txt";
    mOutput.error_file_name += Run_Par.IFO_1 + Run_Par.IFO_2;
    mOutput.log_file_name += Run_Par.IFO_1 + Run_Par.IFO_2;
    mOutput.revolver_file_name += Run_Par.IFO_1 + Run_Par.IFO_2;
    mOutput.revolver_file_name += "_StochMon_Summary.revolver.html";
    mOutput.error_file_name += "_StochMon_Errors.txt";
    mOutput.error_file_link = Run_Par.IFO_1  + Run_Par.IFO_2 + "_StochMon_Errors.txt";
    mOutput.log_file_name += "_StochMon_Log.txt";
    mOutput.log_file_link = Run_Par.IFO_1  + Run_Par.IFO_2 + "_StochMon_Log.txt";  
    if (!mOutput.local) {
      mOutput.cumlog_file_link = Run_Par.IFO_1  + Run_Par.IFO_2 
	                       + "_StochMon_CumLog.txt"; 
    }
    mOutput.summary_file_name += "index.html";
    
    //-------- Divert log and error messages to files, if desired(default).
    if (mOutput.error_log && 
	!freopen(mOutput.error_file_name.c_str(), "w", stderr) ) {
      cerr << "Unable to divert stderr" << endl;
    }
    if (mOutput.write_log && 
	!freopen(mOutput.log_file_name.c_str(), "w", stdout)) {
      cerr << "Unable to divert stdout" << endl;	
    }

    //---------- Set up array that will hold data for DMTViewer history 
    //           TSeries.  Used to insure that fixed length of data is 
    //           displayed.  Initialize data to -1 to indicate StochMon 
    //           was not running previously.
    tserieslength = (int)( DMTViewer_Period / Run_Par.T);
    invomegasq_values = new double[tserieslength];
    omega_values = new double[tserieslength];
    amplvalues = new double[tserieslength];
    alphavalues = new double[tserieslength];
    betavalues = new double[tserieslength];
    //---------- Initialize TSeries using data from previous run, if possible.
    ifstream dmtviewer_data;
    dmtviewer_data.open(mOutput.dmtviewer_file_name.c_str());
    if (dmtviewer_data.fail()) {
      for (int i=0; i<tserieslength; i++) {
	invomegasq_values[i] = -1.0;
	omega_values[i] = -1.0;
	amplvalues[i] = -1.0;
	alphavalues[i] = -1.0;
	betavalues[i] = -1.0;
      }
    } else {
      
      int i=0;
      while (dmtviewer_data >> invomegasq_values[i]) { 
	dmtviewer_data >> omega_values[i];
	dmtviewer_data >> amplvalues[i];
	dmtviewer_data >> alphavalues[i];
	dmtviewer_data >> betavalues[i];
	++i;
      }
      //----- Reset latest values to -2 to indicate that histories have been read from a file.
      invomegasq_values[i-1] = -2.0;
      omega_values[i-1] = -2.0;
      amplvalues[i-1] = -2.0;
      alphavalues[i-1] = -2.0;
      betavalues[i-1] = -2.0;
      //----- Check that our new histories are of the desired length.
      if (i != tserieslength) {
	cerr << "StochMon WARNING:  Stored DMTViewer history from previous run not of ";
	cerr << "required length.  Re-initializing DMTViewer histories.\n";
	for (int i=0; i<tserieslength; i++) {
	  omega_values[i] = -1.0;
	  invomegasq_values[i] = -1.0;
	  amplvalues[i] = -1.0;
	  alphavalues[i] = -1.0;
	  betavalues[i] = -1.0;
	}
      }
    }
    dmtviewer_data.close();
    
    //-------- Set window type and intialize PSD object.
    if (Run_Par.window_name == "hamming") {
      ham = new Hamming;
      psd = new PSD(ham, Run_Par.num_ints);
      
    } else if (Run_Par.window_name == "hanning") {
      han = new Hanning;
      psd = new PSD(han, Run_Par.num_ints);
      
    } else if (Run_Par.window_name == "flattop") {
      flat = new FlatTop;
      psd = new PSD(flat, Run_Par.num_ints);
      
    } else if (Run_Par.window_name == "blackman") {
      black = new Blackman;
      
    } else if (Run_Par.window_name == "square") {
      psd = new PSD("square", Run_Par.num_ints);
      
    } else {
      Run_Par.window_name = "hanning";
      han = new Hanning;
      psd = new PSD(han, Run_Par.num_ints);
    }
    
    //---------- Set the time stride.
    getDacc().setStride(Interval(Run_Par.T));
    
    //---------- Get DARM_ERR channel; decimate to maximum frequency 4096Hz.
    //           (Keep frequencies above 2kHz for the calibrated PSD dumps.)
    getDacc().addChannel((Run_Par.mChannel).c_str(),2);  
    if (Run_Par.CalPSD_file_name == ""){getDacc().addChannel((Run_Par.mChannel_2).c_str(),2);}
    
    
    //---------- Initialize FDEasyCalibrate objects
    //            This should be in monitor constructor (ie, before any data 
    //            is requested) because FDEasyCalibrate adds new channels to
    //            the Dacc
    //            KLUDGE: Hard-wire for generating new alphas, betas.
    //            KLUDGE: Hard-wire requested frequencies.
    mCalibrate = new FDEasyCalibrate(&getDacc(),
				     (Run_Par.xml_file_name_1).c_str(), 
				     true,
				     0.0,  
				     (Run_Par.num_ints)/(Run_Par.T),
				     int((4096.0-0.0)*(Run_Par.T)/(Run_Par.num_ints)+1))
      ;
    if (Run_Par.CalPSD_file_name == ""){ 
    mCalibrate_2 = new FDEasyCalibrate(&getDacc(),
		      		     (Run_Par.xml_file_name_2).c_str(), 
				     true,
			             0.0,  
				     (Run_Par.num_ints)/(Run_Par.T),
				     int((4096.0-0.0)*(Run_Par.T)/(Run_Par.num_ints)+1));
    }
    
    //---------- Initialize OSC & read config file.
    mOsclist = new OperStateCondList(getDacc());
    mOsclist->readConfig((Run_Par.osc_file_name).c_str());
    //    mOsclist->ignoreAllExcept(Run_Par.IFO_1);
    getDacc().setIgnoreMissingChannel(true);


    if (mOutput.trend)
      {
	//---------- Intialize Trend frame.
	mTrend.setIFO((Run_Par.IFO_1).c_str());
	mTrend.setName(MonitorName.c_str());
	mTrend.setType(Trend::kMinute);
	
	//---------- Add trend channels.  Force calibration-line-related 
	//           trends even if not tracking a line (we don't want to 
	//           change the trends that are being written during a science
	//           run, as this causes the trend writer to crash). 
	std::string trendName;
	trendName = Run_Par.IFO_1
	  + ":DMT-STCH_" +  Run_Par.IFO_1 + Run_Par.IFO_2 + LiveMode
	  + "_OMEGA";
	mTrend.addChannel(trendName.c_str());
	trendName = Run_Par.IFO_1
	  + ":DMT-STCH_" +  Run_Par.IFO_1 + Run_Par.IFO_2 + LiveMode
	  + "_INV_OMEGA_SQ";
	mTrend.addChannel(trendName.c_str());
	trendName = Run_Par.IFO_1
	  + ":DMT-STCH_" +  Run_Par.IFO_1 + Run_Par.IFO_2 + LiveMode
	  + "_CAL_LINE_AMP";
	mTrend.addChannel(trendName.c_str());
	
	//---------- Trends to carry alpha, alpha*beta 
	//           (names requested by Inspiral group).
	trendName = Run_Par.IFO_1
	  + ":DMT-STCH_" +  Run_Par.IFO_1 + Run_Par.IFO_2 + LiveMode
	  + "_CAV_FAC";
	mTrend.addChannel(trendName.c_str());
	trendName = Run_Par.IFO_1
	  + ":DMT-STCH_" +  Run_Par.IFO_1 + Run_Par.IFO_2 + LiveMode
	  + "_OLOOP_FAC";
	mTrend.addChannel(trendName.c_str());
	//---------- Tell manager to make list of trended channels.
	mTrend.writeIndex();
      }
    
  }   //---------- !fatal_error
  
  //---------- Flush any lingering data
  fflush(stderr);
  fflush(stdout);
} //-- End of StochMon constructor


//======================================  StochMon destructor.
StochMon::~StochMon() 
{
  //---------- No pointers are assigned if a fatal error is registered.
  if (!fatal_error) {
    //---------- Export DMTViewer data to a file to be read if monitor is restarted.
    //           Name of file set elsewhere; uses mOutput.html_dir if defined.
    ofstream dmtviewer_data;
    dmtviewer_data.open(mOutput.dmtviewer_file_name.c_str());
    for (int i=0; i<tserieslength; i++) {
      dmtviewer_data << invomegasq_values[i] << "  ";
      dmtviewer_data << omega_values[i] << "  ";
      dmtviewer_data << amplvalues[i] << "  ";
      dmtviewer_data << alphavalues[i] << "  ";
      dmtviewer_data << betavalues[i] << "  ";
      dmtviewer_data << endl; 
    } 
    dmtviewer_data.close();
    
    //----- Pointers assigned in StochMon::StochMon().
    //---------- These pointers are always assigned.
    delete[] invomegasq_values;
    delete[] omega_values;
    delete[] amplvalues;
    delete[] alphavalues;
    delete[] betavalues;
    
    //---------- Delete the window pointer (if any).
    if (Run_Par.window_name == "hamming") delete ham;
    if (Run_Par.window_name == "hanning") delete han;
    if (Run_Par.window_name == "flattop") delete flat;
    if (Run_Par.window_name == "blackman") delete black;
    delete psd;
    delete mCalibrate;
    if (Run_Par.CalPSD_file_name == ""){delete mCalibrate_2;}
    delete mOsclist; 
    
    //----- Pointers assigned in StochMon::ProcessData().
    delete history_alpha;
    delete history_beta;
    delete history_invomegasq;
    delete history_omega;
    delete history_ampl; 
    
    //----- Close files.
    if (mOutput.trend) mTrend.close();
    mOutput.cumlog_file.close(); 
  }
  //----- Send "finished happily" message.
  std::cout << "StochMon MESSAGE: StochMon is finished after "
	    << NStride << " strides.\n";
}


//======================================  Frame processing function.
void
StochMon::ProcessData(void) {
  
  //---------- Get current GPS time and check that it is an integer.
  Time time = getDacc().getFillTime();
  if (time.getS() != time.totalS()) {
    cerr << "StochMon WARNING: GPS start time not an integer: ";
    cerr << time.totalS() << endl;
  }
  
  //---------- Check for gap in data.
  if ( (time.totalS() != prev_endtime) && (NStride != 0) ) { 
    char msg[150];
    sprintf(
	    msg,"Missing data between GPS times (%0.3f, %0.3f).\n",
	    prev_endtime,time.totalS()
	    );
    std::cerr << "StochMon ERROR: " << msg;
    missing_strides = int( 1+(time.totalS()-prev_endtime)/Run_Par.T);
  } else { 
    missing_strides = 0;
  }
  prev_endtime = time.totalS();
  prev_endtime += Run_Par.T; 
  
  //---------- Calculate start time of histories to DMTViewer.
  //           Note:  TSeries start time = "time" - "DMTViewer_Period",
  //           where DMTViewer_Period is a defined quantity copied to 
  //           tserieslength. 
  Time history_start = time;
  history_start -= (tserieslength-1)*(Run_Par.T);
  
  //---------- Set cumlog_file_name.  Choices in order of preference are 
  //           (1) "$DMTHTMLOUT/<IFO_1>_StochMon_CumLog.txt" if 
  //               -local option NOT selected (default).
  //           (2) User's choice if -local and -logfile options selected.
  //           (3) Use <GPS_start_time>.log if only -local selected. 
  if (!mOutput.local) {  
    mOutput.cumlog_file_name = string(mOutput.html_dir) + "/" + Run_Par.IFO_1  + Run_Par.IFO_2 +
      "_StochMon_CumLog.txt";
  }
  if ( (mOutput.cumlog_file_name).empty() && !NStride ) {
    char temp_file_name[50]; 
    sprintf(temp_file_name,"%ld",time.getS());
    mOutput.cumlog_file_name = temp_file_name;
    mOutput.cumlog_file_name += ".log";
  }
  
  //---------- Things to do once only.
  if (NStride == 0) {
    //-----------Need some data for initializing certain storage
    //--Doesn't matter if IFO is locked.
    
    //--------Get DARM_ERR data to TSeries.
    ts_asq = getDacc().refData((Run_Par.mChannel).c_str());
    
    //--------Calculate PSD
    psd->generate(PSDed_data_1,ts_asq);
    
    //---------- Output run parameters.  
    //----- Set starttime
    starttime = time.getS();
    
    //----- Open logfile
    mOutput.cumlog_file.open((mOutput.cumlog_file_name).c_str(), std::ios::app);
    
    //----- Dump parameters to html log file and cout. 
    DumpParameters(cout);
    DumpParameters(mOutput.cumlog_file);
    DumpReportHeader(mOutput.cumlog_file);
    
    if (!mOutput.local) {  
      //----- Set up two-week summary file
      std::string summary_name = string(mOutput.html_dir) + "/" + Run_Par.IFO_1  + Run_Par.IFO_2 +
	"_summary.txt";
      mStochSensSumm.init(summary_name, time);
    }
    
    //---------- Write histories to DMTViewer.
    //----- Initialize and subscribe history of
    // 	    (stochastic sensitivity with calibration line).
    history_omega = new TSeries(history_start,Run_Par.T,tserieslength,omega_values);
    std::string DMTViewer_name_prefix
      = Run_Par.IFO_1 + Run_Par.IFO_2 + "_" + LiveMode;
    std::string DMTViewer_name = DMTViewer_name_prefix + " Omega";
    serveData(DMTViewer_name.c_str(), history_omega);
    history_invomegasq = new TSeries(history_start,Run_Par.T,tserieslength,invomegasq_values);

    DMTViewer_name = DMTViewer_name_prefix + " inverse Omega squared";
    serveData(DMTViewer_name.c_str(), history_invomegasq);
    //----- Initialize and subscribe history of
    //      calibration-line amplitude.
    history_ampl = new TSeries(history_start,Run_Par.T,tserieslength,amplvalues);
    DMTViewer_name = DMTViewer_name_prefix + " cal line amplitude";
    serveData(DMTViewer_name.c_str(), history_ampl);
    //----- Initialize and subscribe history of alpha parameter
    history_alpha = new TSeries(history_start,Run_Par.T,tserieslength,alphavalues);
    DMTViewer_name = DMTViewer_name_prefix + " alpha";
    serveData(DMTViewer_name.c_str(), history_alpha);
    //----- Initialize and subscribe history of beta parameter.
    history_beta = new TSeries(history_start,Run_Par.T,tserieslength,betavalues);
    DMTViewer_name = DMTViewer_name_prefix + " beta";
    serveData(DMTViewer_name.c_str(), history_beta);
    
    //---------- Make fake strain_noise_ampl for DMTViewer initialization. 
    strain_noise_ampl = PSDed_data_1;
    DMTViewer_name = DMTViewer_name_prefix + " primary (" + Run_Par.IFO_1
      + ") Strain noise ASD (Hz^-0.5)";
    serveData(DMTViewer_name.c_str(), &strain_noise_ampl);

    const int Max = PSDed_data_1.getNStep();
    float Fstep = PSDed_data_1.getFStep();
    double f_step = PSDed_data_1.getFStep();
    double low_freq = PSDed_data_1.getLowFreq(); 
    //--Open CalPSD file (if specified on command line) and
    //--fill the PSD_2_Array here
    if (Run_Par.CalPSD_file_name != ""){
      ifstream input;
      float Num;
      int npoints = PSDed_data_1.getNStep() + 1;
      float minF = PSDed_data_1.getLowFreq();
      float maxF =  PSDed_data_1.getLowFreq() + (npoints-1)*(PSDed_data_1.getFStep());
      int GPS;
      const int maxLineLength = 200;
      int nPoints = PSDed_data_1.getNStep() + 1;
      char units[maxLineLength];
      char psdline[maxLineLength];
      float* PSD_2_Array = NULL;
      char mark = '#';
      int count = 1;
      
      input.open((Run_Par.CalPSD_file_name).c_str());
      //--Fill the array with data CalPSD file
      while (!input.getline(psdline, maxLineLength).fail() && 
	     psdline[0] == mark) {
	if (sscanf(psdline,"#Min Frequency: %f Hz\n" ,&minF) == 1) {
	  cerr << "Min Freq is " << minF << "\n";
	} else if (sscanf(psdline,"#Max Frequency: %f Hz\n",&maxF) == 1) {
	  cerr << "The Max Freq is " << maxF<< "\n";
	} else if (sscanf(psdline,"#Frequency step: %f Hz\n",&Fstep) == 1) {
	  cerr << "The Freq Step is " << Fstep<< "\n";
	} else if (sscanf(psdline,"#GPS start time of this data: %d \n",&GPS) == 1) {
	  //		  cerr << "The GPS time is " << GPS<< "\n";
	} else if (sscanf(psdline,"#Number of points: %d \n",&nPoints) == 1) {
	  cerr << "The Number of Points is " << nPoints<< "\n";
	} else if (sscanf(psdline,"#Units: %s \n", units) == 1) {
	  //		  cerr << "The units are " << units << "\n";
	}
      } 
      //-- Check metadata 
      int j = 0;
      double max_freq =  PSDed_data_1.getLowFreq() + (npoints-1)*(PSDed_data_1.getFStep());
      double f_step = PSDed_data_1.getFStep();
      cerr << "Number of points (online): " << npoints << "\n";
      cerr << "Frequency step (online): " << f_step << " Hz" << "\n";
      cerr << "Min Frequency (online): " << low_freq << " Hz" << "\n";
      cerr << "Max Frequency (online): " << max_freq << " Hz" << "\n";
      
      
      if (f_step != Fstep){
	cerr << "StochMon WARNING: The frequency step is not the same for both PSDs \n";
	cout << "StochMon WARNING: The frequency step must be the same for both PSDs \n";
	cout << "   Exiting  \n";
	fatal_error = true;
	finish();   
      } 
      if (low_freq < minF){
	double diff = (minF - low_freq);
	j = (int)rint(diff / Fstep);}
      //frac_diff = modf(diff,&int_diff);
      //if (frac_diff > .5) {int_diff = int_diff +1;} 
      
      if (low_freq > minF){
	double diff = (low_freq - minF);
	j = (int)rint(diff / Fstep);
	while((j/Fstep) < count){ //--count is set to one b/c Num already has a first value
	  sscanf(psdline,"%f\n", &Num);//--This skips the initial bins of the input PSD
	  count++;}
      }
      
      // Allocate storage for the array
      PSD_2_Array = new float[nPoints];
      if( PSD_2_Array == NULL ) 
	std::cout << "Failed to allocate memory" << std::endl;
      
      //--Assign first values to array
      int n = 0;
      if (j > 0){
	while (n < j) {
	  PSD_2_Array[n] = atof("NaN");
	  n++;}
      }
      sscanf(psdline,"%f\n", &Num);
      PSD_2_Array[j] =  pow((Num*1.0e18),2);
      
      //--Fill the rest of the array
      while (j < nPoints && !input.getline(psdline,maxLineLength).fail()) { 
	sscanf(psdline,"%f\n", &Num);
	// The datafile contains an ASD in units of
	// strain/rtHz Now we multiply it by 1e18 and square
	// to convert it into a PSD in units of
	// (attostrain)^2/Hz
	PSD_2_Array[j] = pow((Num*1.0e18),2);
	j++;
      }
      
      input.close();
      PSDed_data_2 = PSDed_data_1; //--Fill metadata for PSD2 with copy from PSD1
      PSDed_data_2.setData(PSDed_data_1.getNStep() + 1, PSD_2_Array);// Fill data for PSD2 from reference spectrum
      delete[] PSD_2_Array;
      
    
    } else { 
      //--------Get AS_Q data from second channel to TSeries.
      ts_asq_2 = getDacc().refData((Run_Par.mChannel_2).c_str());
      
      //--------Calculate PSD for second channel
      psd->generate(PSDed_data_2,ts_asq_2);

      //---------- Make fake strain_noise_ampl for DMTViewer initialization. 
      strain_noise_ampl_2 = PSDed_data_2;
      DMTViewer_name = DMTViewer_name_prefix + " secondary (" + Run_Par.IFO_2
	+ ") Strain noise ASD (Hz^-0.5)";
      serveData(DMTViewer_name.c_str(), &strain_noise_ampl_2);   
    }
    
    //--Calculate gamma[] to be passed to integrand()	
    float* gammaArray = NULL;
    gammaArray = new float[Max + 1];
    if (Run_Par.IFO_1[0] == Run_Par.IFO_2[0]) 
      for(int i = 0; i < (Max + 1); i++)
	{
	  gammaArray[i] = 1;
	}
    
    else 
      LLOLHO_Overlap(gammaArray, Max, f_step, low_freq);    
    GammaValues.setData(PSDed_data_1.getNStep(), gammaArray);
    delete[] gammaArray;
    sec_per_stride = Run_Par.T;
    integrand(stoch_igd, PSDed_data_1, PSDed_data_2,sec_per_stride, GammaValues);
    DMTViewer_name
      = DMTViewer_name_prefix + " 1/Omega^2 sensitivity integrand";
    serveData(DMTViewer_name.c_str(), &stoch_igd);
    
  } //-- end if(nstride == 0)
  

    std::string osc_cond1 = Run_Par.IFO_1 + ":Both_arms_locked_strict_cm";
    std::string osc_cond2 = Run_Par.IFO_2 + ":Both_arms_locked_strict_cm";

    //------- Check that IFO_1 is locked before proceeding.
      if (mOsclist->satisfied(osc_cond1.c_str())) {

            //---------- Get AS_Q data to TSeries.
      ts_asq = getDacc().refData((Run_Par.mChannel).c_str());
      
      //---------- Calcuate PSD.
      psd->generate(PSDed_data_1,ts_asq);
          
      //---------- Apply calibration (extracts subsection of PSD 
      //           for which we have calibration info).
      mCalibrate->UpdateResponseFunction();
      R_Dat.alpha = mCalibrate->GetAlpha();
      R_Dat.beta = mCalibrate->GetBeta();
      PSDed_data_1 = mCalibrate->Apply(PSDed_data_1);
      
      //----------KLUDGE: Don't yet have "get" method for retrieving 
      //          line amplitude from FDCalibrate.
      R_Dat.ampl = 0.0;
      
      
      // PSDed_data_1 is now a calibrated power spectral density
      // in units of nm^2/Hz
      // We divide it by the square of the
      // arm length in nanometers to get strain^2/Hz
      // and then by (1e-18)^2 to get (attostrain)^2/Hz
      // This is equivalent to dividing by square of
      // the arm length in gigameters
      
      //----- Specify arm length in Gm for conversion of PSD units
      // from nm to attostrain:
      float arm_length_1_Gm = 1.0;  
      if (Run_Par.IFO_1  == "H2") { 
	arm_length_1_Gm = 2.0e-6;  
      } else if ((Run_Par.IFO_1   == "L1") || (Run_Par.IFO_1  == "H1")) {
	arm_length_1_Gm = 4.0e-6;  
      } else {
	cerr << "StochMon ERROR: IFO_1 " << Run_Par.IFO_1 
	     << " not recognized.  Calibrated PSD files will \n";
	cerr << "                     not have units of strain.\n";
      }
      
      //----- Set up array to hold values of attostrain PSD
      float* PSDvalues = new float[PSDed_data_1.getNStep() + 1];
      
      //----- Get values of calibrated PSD.
      PSDed_data_1.getData(PSDed_data_1.getNStep() + 1, PSDvalues);
      
      for (uint32_t j=0; j<PSDed_data_1.getNStep()+1; j++) {
	PSDvalues[j] /= (arm_length_1_Gm * arm_length_1_Gm);
	//cerr << "PSDvalues[" << j << "] is  "  << PSDvalues[j] << "\n";
      }

      PSDed_data_1.setData(PSDed_data_1.getNStep() + 1, PSDvalues);
      
      delete[] PSDvalues;

      //--------------------------------------------------------------------------------------------------------     

      if (Run_Par.CalPSD_file_name == ""){
	//------- Check that IFO_2 is locked before proceeding.
	if (mOsclist->satisfied(osc_cond2.c_str())) {

	//---------- Get AS_Q data to TSeries.
	ts_asq_2 = getDacc().refData((Run_Par.mChannel_2).c_str());
	
	//---------- Calcuate PSD.
	psd->generate(PSDed_data_2,ts_asq_2);
	
	//---------- Apply calibration (extracts subsection of PSD 
	//           for which we have calibration info).
	mCalibrate_2->UpdateResponseFunction();
	R_Dat.alpha = mCalibrate_2->GetAlpha();
	R_Dat.beta = mCalibrate_2->GetBeta();
	PSDed_data_2 = mCalibrate_2->Apply(PSDed_data_2);
	// PSDed_data_2 is now a calibrated power spectral density
	// in units of nm^2/Hz
	// We divide it by the square of the
	// arm length in nanometers to get strain^2/Hz
	// and then by (1e-18)^2 to get (attostrain)^2/Hz
	// This is equivalent to dividing by square of
	// the arm length in gigameters
	
	//----- Specify arm length in Gm for conversion of PSD units
	// from nm to attostrain:
	float arm_length_1_Gm = 1.0;  
	if (Run_Par.IFO_2  == "H2") { 
	  arm_length_1_Gm = 2.0e-6;  
	} else if ((Run_Par.IFO_2   == "L1") || (Run_Par.IFO_2  == "H1")) {
	  arm_length_1_Gm = 4.0e-6;  
	} else {
	  cerr << "StochMon ERROR: IFO_2 " << Run_Par.IFO_1 
	       << " not recognized.  Calibrated PSD files will \n";
	  cerr << "                     not have units of strain.\n";
	}

	//----- Set up array to hold values of attostrain PSD
	float* PSDvalues_2 = new float[PSDed_data_2.getNStep() + 1];
	
	//----- Get values of calibrated PSD.
	PSDed_data_2.getData(PSDed_data_2.getNStep() + 1, PSDvalues_2);
	
	for (uint32_t j=0; j<PSDed_data_2.getNStep()+1; j++) {
	  PSDvalues_2[j] /= (arm_length_1_Gm * arm_length_1_Gm);
	}

	PSDed_data_2.setData(PSDed_data_2.getNStep() + 1, PSDvalues_2);
	
	delete[] PSDvalues_2;
	}

      }
      //----------------------------------------------------------------------------------------------------------------


      //---------- Compute integrand of sensitivity
      //           integral ([PSD_1*PSD_2*f^6]^-1).
      Run_Par.T = sec_per_stride; 
      integrand(stoch_igd, PSDed_data_1, PSDed_data_2, sec_per_stride,GammaValues);
      
      //---------- Calculate strength of background 
      //           to which detector is sensitive
      R_Dat.invomegasq = integrate(stoch_igd, Run_Par.l_freq, Run_Par.h_freq);
      R_Dat.omega = 1/sqrt(R_Dat.invomegasq);
      
      //---------- Compute strain_noise_ampl from PSDed_data_1, for DMTViewer.
      //           This code effectively duplicates the Calibrated PSD dump code.
      //           I should remove the latter.
      
      //----- Set up array to hold values of calibrated AS_Q PSD.
      float* strainvalues = new float[PSDed_data_1.getNStep() + 1];
      
      //----- Get values of calibrated PSD.
      PSDed_data_1.getData(PSDed_data_1.getNStep() + 1, strainvalues);
      
      //----- PSDed_data_1 is in units of attostrain^2/Hz
      //      We construct from it an ASD in units of strain/rtHz
      for (uint32_t j=0; j<PSDed_data_1.getNStep()+1; j++) {
	strainvalues[j] = (float) 
	  1e-18 * pow((double) (strainvalues[j]),0.5);
	//		cerr << "strainvalues[" << j << "] is  "  << strainvalues[j] << "\n";
      }
      
      //----- Copy data to strain_noise_ampl and clean up.
      strain_noise_ampl = PSDed_data_1;
      strain_noise_ampl.setData(PSDed_data_1.getNStep() + 1, strainvalues);
      delete[] strainvalues;

  //-----------------------------------------------------------------------------------------
      if (Run_Par.CalPSD_file_name == ""){
       //---------- Compute strain_noise_ampl_2 from PSDed_data_2, for DMTViewer.
      //           This code effectively duplicates the Calibrated PSD dump code.
      //           I should remove the latter.
      
      //----- Set up array to hold values of calibrated AS_Q PSD.
      float* strainvalues_2 = new float[PSDed_data_2.getNStep() + 1];
      
      //----- Get values of calibrated PSD.
      PSDed_data_2.getData(PSDed_data_2.getNStep() + 1, strainvalues_2);
      
      //----- PSDed_data_2 is in units of attostrain^2/Hz
      //      We construct from it an ASD in units of strain/rtHz
      for (uint32_t j=0; j<PSDed_data_2.getNStep()+1; j++) {
	strainvalues_2[j] = (float) 
	  1e-18 * pow((double) (strainvalues_2[j]),0.5);
	//		cerr << "strainvalues_2[" << j << "] is  "  << strainvalues_2[j] << "\n";
      }
      
      //----- Copy data to strain_noise_ampl and clean up.
      strain_noise_ampl_2 = PSDed_data_2;
      strain_noise_ampl_2.setData(PSDed_data_2.getNStep() + 1, strainvalues_2);
      delete[] strainvalues_2;
      }
    //-----------------------------------------------------------------------------------------
      
    } else { 
      
      //---------- IFO_1 not Locked; set Sensitivity to zero, don't increment
      //		 4-volume.
      R_Dat.alpha = 0.0;
      R_Dat.beta = 0.0;
      R_Dat.invomegasq = 0.0;
      R_Dat.omega = 0.0;
      R_Dat.ampl = 0.0;	
      
      //----- Set to 1 PSDs to be sent to DMTViewer.
      int nsteps = PSDed_data_1.getNStep();
      float *data = new float[nsteps + 1];
      for (int i=0; i<(nsteps + 1); i++) {
	data[i] = 1;
      }
      strain_noise_ampl.setData(nsteps + 1, data);
      stoch_igd.setData(nsteps + 1, data);
      delete[] data;
      }

    //---------- Shift DMTViewer TSeries data one time step.
    int nshift = 1+missing_strides;
    for (int ii=0; ii<(tserieslength-nshift); ii++) {
      invomegasq_values[ii] = invomegasq_values[ii+nshift];
      omega_values[ii] = omega_values[ii+nshift];
      amplvalues[ii] = amplvalues[ii+nshift];
      alphavalues[ii] = alphavalues[ii+nshift];
      betavalues[ii] = betavalues[ii+nshift];
    } 
    
    //---------- Fill any missing strides with -1.0.
    if (missing_strides) {
      for (int ii=(tserieslength-nshift); ii<(tserieslength-1); ii++) {
	invomegasq_values[ii] = -1.0;
	omega_values[ii] = -1.0;
	amplvalues[ii]  = -1.0;
	alphavalues[ii] = -1.0;
	betavalues[ii]  = -1.0;
      } 
    }
    
    //---------- Store most recent values.
    invomegasq_values[tserieslength-1] = R_Dat.invomegasq;
    omega_values[tserieslength-1] = R_Dat.omega;
    amplvalues[tserieslength-1] = R_Dat.ampl;
    alphavalues[tserieslength-1] = R_Dat.alpha;
    betavalues[tserieslength-1] = R_Dat.beta;
    
    //---------- Export data to DMTViewer. 
    history_invomegasq->setData(history_start, Run_Par.T, invomegasq_values, tserieslength); 
    history_omega->setData(history_start, Run_Par.T, omega_values, tserieslength);    
    history_ampl->setData(history_start, Run_Par.T, amplvalues, tserieslength);
    history_alpha->setData(history_start, Run_Par.T, alphavalues, tserieslength);
    history_beta->setData(history_start, Run_Par.T, betavalues, tserieslength);
    
    //---------- Write the stochastic sensitivity and 4-volume scanned to the log file
    //	     and cout.
    ReportResults(mOutput.cumlog_file, time, Run_Par);
    if (mOutput.screen_output || mOutput.write_log) {
      ReportResults(cout, time, Run_Par);
    }
    
    //---------- Update Trends.
    if (mOutput.trend) {
      std::string trendName;
      trendName = Run_Par.IFO_1
	+ ":DMT-STCH_" +  Run_Par.IFO_1 + Run_Par.IFO_2 + LiveMode
	+ "_OMEGA";
      mTrend.trendData(trendName.c_str(), time, R_Dat.omega);
      trendName = Run_Par.IFO_1
	+ ":DMT-STCH_" +  Run_Par.IFO_1 + Run_Par.IFO_2 + LiveMode
	+ "_INV_OMEGA_SQ";
      mTrend.trendData(trendName.c_str(), time, R_Dat.invomegasq);
      trendName = Run_Par.IFO_1
	+ ":DMT-STCH_" +  Run_Par.IFO_1 + Run_Par.IFO_2 + LiveMode
	+ "_CAL_LINE_AMP";
      mTrend.trendData(trendName.c_str(), time, R_Dat.ampl);
      trendName = Run_Par.IFO_1
	+ ":DMT-STCH_" +  Run_Par.IFO_1 + Run_Par.IFO_2 + LiveMode
	+ "_CAV_FAC";
      mTrend.trendData(trendName.c_str(), time, R_Dat.alpha);
      trendName = Run_Par.IFO_1
	+ ":DMT-STCH_" +  Run_Par.IFO_1 + Run_Par.IFO_2 + LiveMode
	+ "_OLOOP_FAC";
      mTrend.trendData(trendName.c_str(), time, (R_Dat.alpha)*(R_Dat.beta));
      
    }
    
    //---------- Do periodic stochastic sensitivity and calibrated PSD dumps if running in 
    //           background (!local) mode:    
    //           -> two-week summary file  
    //           -> calibrated psd dumped to ascii file
    //           NOTE: Since stride "Run_Par.T" may not exactly divide
    //           the specified update period "Run_Par.stride_dump",
    //	     not all strides will contribute with equal rate
    //	     to a given average.  Eg: If Run_Par.stride_dump is 1000 
    //           and Run_Par.T = 300, the first through 
    //	     fourth strides contribute to first 1000-sec average with
    //	     weights 1, 1, 1, 1/3.  Fourth stride also contributes to
    //	     second 1000-sec average with weight 2/3.
    //           Note also that report time (GPS truncated to last multiple
    //           of Run_Par.stride_dump) is 
    //           guaranteed to lie in the 1000-sec interval reported on.
    //           Times when the IFO_1 is not locked contribute zero to the average.
    if (!mOutput.local && (Run_Par.stride_dump != 0)) {
      time_since_update += (int) Run_Par.T;
      Run_Par.stride_dump *= (int) Run_Par.T;
      if (time_since_update >= Run_Par.stride_dump) {
	
	//---------- 2-week summary file update:
	
	//----- Add current stochastic sensitivity to running average with
	//      weight = fraction of stride coming before (Run_Par.stride_dump)-sec mark.
	running_average += (Run_Par.T - time_since_update + Run_Par.stride_dump) /
	  Run_Par.T * R_Dat.invomegasq;
	running_count += (Run_Par.T - time_since_update + Run_Par.stride_dump)/Run_Par.T;
	running_average /= running_count;
	
	//----- Dump running average to two-week summary file.
	//      Report time to last multiple of Run_Par.stride_dump 
	//      (eg, to last 000 sec for Run_Par.stride_dump = 1000).
	//cout <<  (time.getS() % Run_Par.stride_dump) << endl;
	StochSensDataSummary tmpstatus = {time.getS() - (time.getS() % (int)Run_Par.stride_dump), running_average};
	mStochSensSumm.append(tmpstatus);
	mStochSensSumm.dumpList();
	
	//----- Initialize next running average with whatever is left
	//      of current stride.
	time_since_update -= (int)Run_Par.stride_dump;
	running_average = (time_since_update) / Run_Par.T * R_Dat.invomegasq;
	running_count = (time_since_update)/Run_Par.T;
       
	//---------- Set up array to hold values of calibrated AS_Q PSD.
	float* CalPSDValues = new float[PSDed_data_1.getNStep() + 1];
	char* time_holder = new char[128];

	//----- calibrated psd file dump:
	if (mOsclist->satisfied(osc_cond1.c_str())){
 
	  //---------- Get values of calibrated PSD.
	  PSDed_data_1.getData(PSDed_data_1.getNStep() + 1, CalPSDValues);
	  
	  //---------- Print values to text file
	  TimeStr(ts_asq->getStartTime(), time_holder, "%s.txt");
	  mOutput.calpsd_file_name = time_holder;
	  mOutput.calpsd_file_name = string(mOutput.html_dir) + "/" + Run_Par.IFO_1  + "_CalPSD_GPS_" + mOutput.calpsd_file_name;
	  mOutput.calpsd_file.open(mOutput.calpsd_file_name.c_str());
	  
	  mOutput.calpsd_file << "#Calibrated ASD of DARM_ERR Channel, produced by StochMon  ($Id: StochMon.cc 7382 2015-05-24 11:26:59Z john.zweizig@LIGO.ORG $)." << endl;
	  mOutput.calpsd_file << "#Units: (strain)/Hz^{-0.5}" << endl;
	  mOutput.calpsd_file << "#Number of Averages for PSD: " << Run_Par.num_ints << endl;
	  mOutput.calpsd_file << "#Window: " << Run_Par.window_name << endl;
	  mOutput.calpsd_file << "#GPS start time of this data: " << time.getS() << endl;
	  int npoints = PSDed_data_1.getNStep() + 1; 
	  mOutput.calpsd_file << "#Number of points: " << npoints << endl;
	  mOutput.calpsd_file << "#Frequency step: " << PSDed_data_1.getFStep() << " Hz" << endl;
	  mOutput.calpsd_file << "#Min Frequency: " << PSDed_data_1.getLowFreq() << " Hz" << endl;
	  mOutput.calpsd_file << "#Max Frequency: " << PSDed_data_1.getLowFreq() + (npoints-1)*(PSDed_data_1.getFStep())<< " Hz" << endl;
	}

	//---------- Set up array to hold values of calibrated AS_Q PSD_2.
	float* CalPSDValues_2 = new float[PSDed_data_2.getNStep() + 1];
	char* time_holder_2 = new char[128];

if (Run_Par.CalPSD_file_name == ""){

  //----- calibrated psd file dump:
  if (mOsclist->satisfied(osc_cond2.c_str())){

    //---------- Get values of calibrated PSD_2.
    PSDed_data_2.getData(PSDed_data_2.getNStep() + 1, CalPSDValues_2);
    
    //---------- Print values to text file
    
    TimeStr(ts_asq_2->getStartTime(), time_holder_2, "%s.txt");
    mOutput.calpsd_file_name_2 = time_holder_2;
    mOutput.calpsd_file_name_2 = string(mOutput.html_dir) + "/" + Run_Par.IFO_2  + "_CalPSD2_GPS_" + mOutput.calpsd_file_name_2;
    mOutput.calpsd_file_2.open(mOutput.calpsd_file_name_2.c_str());
    
    mOutput.calpsd_file_2 << "#Calibrated ASD of DARM_ERR Channel, produced by StochMon  ($Id: StochMon.cc 7382 2015-05-24 11:26:59Z john.zweizig@LIGO.ORG $)." << endl;
    mOutput.calpsd_file_2 << "#Units: (strain)/Hz^{-0.5}" << endl;
    mOutput.calpsd_file_2 << "#Number of Averages for PSD: " << Run_Par.num_ints << endl;
    mOutput.calpsd_file_2 << "#Window: " << Run_Par.window_name << endl;
    mOutput.calpsd_file_2 << "#GPS start time of this data: " << time.getS() << endl;
    int npoints_2 = PSDed_data_2.getNStep() + 1; 
    mOutput.calpsd_file_2 << "#Number of points: " << npoints_2 << endl;
    mOutput.calpsd_file_2 << "#Frequency step: " << PSDed_data_2.getFStep() << " Hz" << endl;
    mOutput.calpsd_file_2 << "#Min Frequency: " << PSDed_data_2.getLowFreq() << " Hz" << endl;
    mOutput.calpsd_file_2 << "#Max Frequency: " << PSDed_data_2.getLowFreq() + (npoints_2-1)*(PSDed_data_2.getFStep())<< " Hz" << endl;
  }  
  }
  //----- PSDed_data_1 is in units of attostrain^2/Hz
  //      We dump an ASD in units of strain/rtHz
	  for (uint32_t j=0; j<PSDed_data_1.getNStep()+1; j++) {
	    mOutput.calpsd_file
	      << 1e-18 * pow((double) (CalPSDValues[j]),0.5)
	      << endl;
	  }
	  mOutput.calpsd_file.close();
	  
	  //---------- Clean up.
	  delete[] CalPSDValues;
	  delete[] time_holder;
	
if (Run_Par.CalPSD_file_name == ""){
	  //----- PSDed_data_2 is in units of attostrain^2/Hz
	  //      We dump an ASD in units of strain/rtHz
	  for (uint32_t j=0; j<PSDed_data_2.getNStep()+1; j++) {
	    mOutput.calpsd_file_2
	      << 1e-18 * pow((double) (CalPSDValues_2[j]),0.5)
	      << endl;
	  }
	  mOutput.calpsd_file_2.close();
	  
	  //---------- Clean up.
	  delete[] CalPSDValues_2;
	  delete[] time_holder_2; 
}
      }
   
        } else { 
	
	//----- Add current sensitivity to running average with weight 1.
	running_average += R_Dat.invomegasq;
	running_count   += 1.0;
	}
      
    
    //----------  Calculate average of last five time segments.
    R_Dat.T5_array[NStride % 5] = R_Dat.invomegasq;
    for(int i = 0; i < 5; i++)
      R_Dat.T5_stochsens += R_Dat.T5_array[i];
    R_Dat.T5_stochsens /= 5.0;
    
    //----------  Calculate average of last fifteen time segments.
    R_Dat.T15_array[NStride % 15] = R_Dat.invomegasq;
    for(int i = 0; i < 15; i++)
      R_Dat.T15_stochsens += R_Dat.T15_array[i];
    R_Dat.T15_stochsens /= 15.0;
    
    //---------- Print summary webpages.
    RevolverSummary(time, Run_Par, R_Dat);
    Summary(time, Run_Par, R_Dat);
    
    //---------- Clear old data.
    R_Dat.T15_stochsens = 0.0;
    R_Dat.T5_stochsens = 0.0;
    
    //---------- Check if should finish.
    if (++NStride >= MaxStride) finish();
    
    //----------  Flush any lingering data.
    fflush(stderr);
    fflush(stdout);
  } //--End ProcessData method
  
  
  //====================================== Write revolver.html page
  void
    StochMon::RevolverSummary(const Time& t, Parameter Run_Par,
			      StochSens_Data R_Dat)
    {
      //---------- Html document
      html::document doc (kHtmlTitle);
      char buf[128];
      
      //---------- Title
      html::block titleblk ("center");
      titleblk.lineBreak();
      html::text title (kHtmlTitle);
      html::font fonttitle ("Helvetica");
      title.setFont (fonttitle);
      title.setColor (html::color (0, 0, 128));
      title.setSize (html::size (+4));
      titleblk.addObject (title);
      titleblk.lineBreak();
      titleblk.addObject (html::text ("&nbsp;"));
      doc.addObject (titleblk);
      doc.addObject (html::hline());
      
      //---------- Current status section.
      html::block status_blk ("center");
      status_blk.lineBreak();
      //:KLUDGE: Should specify 1/Omega^2 is with/without cal lines 
      //(if dyncal||refcal||Cal_Line -> yes; else -> no).   
      html::text status_title ("Stochastic Sensitivity (1/Omega^2)/stride");
      status_title.setFont (fonttitle);
      status_title.setSize (html::size (+2));
      status_blk.addObject (status_title);
      status_blk.lineBreak();
      
      //---------- Result table.
      html::table results;
      results.addColumn ("GPS Time");
      results.addColumn ("15-Stride Average Sensitivity (1/Omega^2)");
      results.addColumn ("Next Update Time (UTC)");
      results.addRow();
      results.insertData(0, 0, html::text (TimeStr(t, buf, "%s")));
      results.insertData(0, 1, html::text (R_Dat.T15_stochsens));
      results.insertData(0, 2, html::text (t + Run_Par.T));
      for (int j=0; j<3; ++j) {
        results.refCell(0, j).setAlign ("right");
      }
      results.setBorder(true);
      status_blk.addObject(results);
      doc.addObject(status_blk);
      
      
      //---------- Last update time.
      doc.addObject (html::hline());
      html::block updateblk ("center");
      html::text lasttime ("This page was last updated: ");
      lasttime << TimeStr (t, buf, "%M %d, %Y at %H:%N:%S");
      updateblk.addObject (lasttime);
      updateblk.lineBreak();
      doc.addObject (updateblk);
      doc.addObject (html::hline());
      
      //---------- Write web page to file.
      ofstream out (mOutput.revolver_file_name.c_str());
      if (out) {
        html::writer htmlout (out);
        doc.write (htmlout);
      }
    } //--End RevolverSummary method
  
  
  //====================================== Write summary html page.
  void
    StochMon::Summary(const Time& t, Parameter Run_Par, StochSens_Data R_Dat)
    {   
      //---------- Html document
      html::document doc (kHtmlTitle);
      char buf[128];
      
      //---------- Title
      html::block titleblk ("center");
      titleblk.lineBreak();
      html::text title (kHtmlTitle);
      html::font fonttitle ("Helvetica");
      title.setFont (fonttitle);
      title.setColor (html::color (0, 0, 128));
      title.setSize (html::size (+4));
      titleblk.addObject (title);
      titleblk.lineBreak();
      titleblk.addObject (html::text ("&nbsp;"));
      doc.addObject (titleblk);
      doc.addObject (html::hline());
      
      //---------- Short description
      doc.addObject (html::text (kHtmlDescription));
      doc.addObject (html::hline());
      
      //---------- Last update time.
      html::block firstupdateblk ("center");
      html::text lasttime1 ("This page was last updated: "); 
      lasttime1 << TimeStr (t, buf, "%M %d, %Y at %H:%N:%S %Z");
      firstupdateblk.addObject (lasttime1);
      firstupdateblk.lineBreak();
      html::text lasttime2 ("Next scheduled update time: ");
      lasttime2 << TimeStr (t+Run_Par.T, buf, "%M %d, %Y at %H:%N:%S %Z");
      firstupdateblk.addObject (lasttime2);
      firstupdateblk.lineBreak();
      doc.addObject (firstupdateblk);
      doc.addObject (html::hline());
      
      //---------- Run parameters
      html::block run_param_blk ("center");
      run_param_blk.lineBreak();
      html::text param_title ("Run Parameters");
      param_title.setFont (fonttitle);
      param_title.setSize (html::size (+2));
      run_param_blk.addObject (param_title);
      run_param_blk.lineBreak();
      //---------- Parameters table.
      html::table params; 
      //----- WARNING:  Must make columns first, then rows.
      params.addColumn("");
      params.addColumn("");
      int i=-1;
      params.addRow();
      params.insertData(++i, 0, html::text ("Primary IFO <IFO_1>:"));
      params.insertData(  i, 1, html::text (Run_Par.IFO_1 ));
      params.addRow();
      params.insertData(++i, 0, html::text ("Secondary IFO <IFO_2>:"));
      params.insertData(  i, 1, html::text (Run_Par.IFO_2));
      params.addRow();
      params.insertData(++i, 0, html::text ("Reference PSD file:"));
      params.insertData(  i, 1, html::text (Run_Par.CalPSD_file_name));
      params.addRow();
      params.insertData(++i, 0, html::text ("Primary Channel: "));
      params.insertData(  i, 1, html::text (Run_Par.mChannel));
     if (Run_Par.CalPSD_file_name == ""){ params.addRow();
      params.insertData(++i, 0, html::text ("Secondary Channel: "));
      params.insertData(  i, 1, html::text (Run_Par.mChannel_2));}
      params.addRow();
      params.insertData(++i, 0, html::text ("Stride: "));
      params.insertData(  i, 1, html::text (Run_Par.T));
      params.insertData(  i, 1, html::text (" sec"));
      params.addRow();
      params.insertData(++i, 0, html::text ("Number of Strides Already "
					    "Processed: "));
      params.insertData(  i, 1, html::text (NStride+1));
      params.addRow();
      params.insertData(++i, 0, html::text ("Maximum Number of Strides to "
					    "Process: "));
      params.insertData(  i, 1, html::text (MaxStride));
      params.addRow();
      params.insertData(++i, 0, html::text ("Number of Averages for primary channel PSD: "));
      params.insertData(  i, 1, html::text (Run_Par.num_ints));
      params.addRow();
      params.insertData(++i, 0, html::text ("Window for primary channel PSD: "));
      params.insertData(  i, 1, html::text (Run_Par.window_name));
      params.addRow();
      params.insertData(++i, 0, html::text ("Low Frequency : "));
      params.insertData(  i, 1, html::text (Run_Par.l_freq));
      params.insertData(  i, 1, html::text (" Hz"));
      params.addRow();
      params.insertData(++i, 0, html::text ("High Frequency : "));
      params.insertData(  i, 1, html::text (Run_Par.h_freq));
      params.insertData(  i, 1, html::text (" Hz"));
      params.addRow();
      params.insertData(++i, 0, html::text ("Monitor Start Time (GPS): "));
      params.insertData(  i, 1, html::text (TimeStr(Time(starttime), buf,
						    "%s")));
      params.addRow();
      params.insertData(++i, 0, html::text ("(UTC): "));
      params.insertData(  i, 1, html::text (TimeStr(Time(starttime), buf,
						    "%M %d, %Y at %H:%N:%S")));
      
      //----- WARNING:  It appears that alignments can only be set AFTER the cell
      //	    is filled.
      //      WARNING:  The (html::table).getNRow() method returns 1+(# of rows).
      //	    Beware!
      for (int j=0; j<params.getNRow(); ++j) {  
        params.refCell(j, 0).setAlign("right");
        params.refCell(j, 1).setAlign ("left");
      }
      run_param_blk.addObject(params);
      doc.addObject (run_param_blk);
      
      //---------- Current status section.
      html::block status_blk ("center");
      status_blk.lineBreak();
      //:KLUDGE: Should specify 1/Omega^2 is with/without cal lines 
      //(yes provided using cal_line).
      html::text status_title ("Stochastic Sensitivity (1/Omega^2)/stride");
      status_title.setFont (fonttitle);
      status_title.setSize (html::size (+2));
      status_blk.addObject (status_title);
      status_blk.lineBreak();
      
      //---------- Result table.
      html::table results;
      results.addColumn ("GPS Time");
      results.addColumn ("1-Stride Sensitivity(1/Omega^2)");
      results.addColumn ("5-Stride Average Sensitivity per stride (1/Omega^2)");
      results.addColumn ("15-Stride Average Sensitivity per stride (1/Omega^2)");
      results.addColumn ("Next Update Time (UTC)");
      results.addRow();
      results.insertData(0, 0, html::text (TimeStr(t, buf, "%s")));
      results.insertData(0, 1, html::text (R_Dat.invomegasq));
      results.insertData(0, 2, html::text (R_Dat.T5_stochsens));
      results.insertData(0, 3, html::text (R_Dat.T15_stochsens));
      results.insertData(0, 4, html::text (t + Run_Par.T));
      for (int j=0; j<5; ++j) {
        results.refCell(0, j).setAlign ("right");
      }
      results.setBorder(true);
      status_blk.addObject(results);
      doc.addObject(status_blk);
      
      if (!mOutput.screen_output) {  
        //---------- Insert error message link
        html::block errorinfo ("center");
	
        //---------- Link to error-message log.
	html::link errormessage ("Error Messages", mOutput.error_file_link.c_str() );
	errorinfo.addObject (errormessage);
        errorinfo.lineBreak();
        doc.addObject (errorinfo);
	
        //---------- Insert links to current (this run of monitor) and 
        //           cumulative (all runs) log files.
        html::block output ("center");
	html::link outputlink1 ("Monitor Output: more details", (mOutput.log_file_link).c_str() );
        output.addObject (outputlink1);
        output.lineBreak();
        if (!mOutput.local) {
	  html::link outputlink2 ("Cumulative Monitor Output from all runs", (mOutput.cumlog_file_link).c_str() );
	  output.addObject (outputlink2);
	  output.lineBreak();
        }
        output.lineBreak();
        doc.addObject (output);
      }
      
      //---------- Last update time.
      doc.addObject (html::hline());
      html::block updateblk ("center");
      html::text lasttime ("This page was last updated: ");
      //:KLUDGE:
      lasttime << TimeStr (t, buf, "%M %d, %Y at %H:%N:%S %Z");
      updateblk.addObject (lasttime);
      updateblk.lineBreak();
      
      //---------- Author info.
      updateblk.addObject( html::text ("This monitor was written by ") );
      updateblk.addObject( html::link ("Marc J. Cenac ",
				       "mailto:mjcenac@loyno.edu") );
      updateblk.addObject( html::text ("based on SenseMonitor by ") ); 
      updateblk.addObject( html::link ("Patrick Sutton",
				       "mailto:psutton.ligo.caltech.edu") );
      updateblk.addObject( html::text (" and ") );
      updateblk.addObject( html::link ("Kevin C. Schlaufman",
				       "mailto:kcs149@psu.edu") );
      doc.addObject (updateblk);
      doc.addObject (html::hline());
      
      //---------- LIGO logo.
      html::block logoblk ("div");
      logoblk.addAttr ("align", html::align ("right"));
      logoblk.addObject (html::link (kHtmlLIGO, kHtmlLIGOLink));
      html::image logo;
      logo.setSource ("ligo_logo.gif");
      logo.setWidth ("80");
      logoblk.addObject (logo);
      doc.addObject (logoblk);
      
      //---------- Write web page to file.
      ofstream out (mOutput.summary_file_name.c_str());
      if (out) {
        html::writer htmlout (out);
        doc.write (htmlout);
      }
    } //--End Summary method
  
  
  //====================================== Dump run parameters to output stream.
  void
    StochMon::DumpParameters(std::ostream& out)
    {   
      //----- Dump run parameters.
      out << "# Command Line: " << command_line << endl;
      out << "# Launch Directory: " << pwd_dir <<  endl;
      out << "# Log File: " << mOutput.cumlog_file_name << endl;
      out << "# OSC Configuration File: " << Run_Par.osc_file_name << endl; 
      out << "# Reference Calibration File 1: " << Run_Par.xml_file_name_1 << endl;
      out << "# Reference Calibration File 2: " << Run_Par.xml_file_name_2 << endl;
      out << "# Channel 1 (primary): " << Run_Par.mChannel << endl;
      out << "# Channel 2 (secondary): " << Run_Par.mChannel_2 << endl;
      out << "# Stride: " << Run_Par.T << " sec" << endl;
      out << "# Max Number of Strides: " << MaxStride << " sec" << endl;
      out << "# Number of Averages for PSD: " << Run_Par.num_ints << endl;
      out << "# Low Freq.  : " << Run_Par.l_freq << " Hz" << endl;
      out << "# High Freq. : " << Run_Par.h_freq << " Hz" << endl;
      out << "# Window: " << Run_Par.window_name << endl;
      out << "# Monitor Start Time: " << starttime << endl;
      out << "# " <<endl;
      
    }  //--End DumpParameters
  
  
  //====================================== Dump column headers to output.
  void
    StochMon::DumpReportHeader(std::ostream& out) 
    {
      //----- Dump headers for stochastic sensitivity data that will follow.
      std::cout << "# 1) Data value of 0 means IFO(s) not locked." << endl;
      std::cout << "# 2) Data values of -1, -2 mean monitor not functioning." << endl;
      std::cout << "# " << endl;
      std::cout << "# Start         Stochastic Sensitivity        High-Freq.      Calibration     Calibration" << endl;
      std::cout << "# Time of              FOMs:                  Line Ampl.       Parameter       Parameter" << endl;
      std::cout << "# Segment        Omega        1/Omega^2      (ASQ Counts)       alpha            beta" << endl;
      
    }
  
  
  //====================================== Dump column headers to output.
  void
    StochMon::DumpReportHeader(std::ostream& out, double freq) 
    {
      
      //----- Dump headers for stochastic sensitivity data that will follow.
      out << "# Notes:" << endl;
      out << "# 1) Sensitivity of 0 means IFO_1 not locked." << endl;
      out << "# 2) Sensitivities quoted are adjusted by the" << endl;
      out << "#    calibration-line amplitude only if the -dyncal option is used." << endl;
      out << "# " << endl;
    }
  
  
  //====================================== Read configuration file and pass contents 
  //                                       to command-line parser.
  void
    StochMon::Configure(string& file_name) 
    {
      int NEWargc = 1;
      const char **NEWargv;
      //cout << "Trying to read config file " << file_name << ".\n";
      //---------- Verify that configuration file can be opened.
      ifstream input;
      input.open(file_name.c_str());
      if (input.fail()) {
        std::cerr << "StochMon ERROR: Cannot open configuration file ";
        std::cerr << file_name  << ".  Exiting.\n";
        fatal_error = true;
        finish();
      } else {
        cout << "StochMon MESSAGE: Opened configuration file ";
        cout << file_name << endl;
        //---------- Get number of words in configuration file.
        std::string word;
        while (input >> word) {
	  ++NEWargc;
        }
      }
      input.close();
      NEWargv = new const char*[NEWargc];
      NEWargv[0] = strdup("Configuration File: ");
      //---------- Open file and copy contents to char arrays.
      ifstream inp2;
      inp2.open(file_name.c_str());
      if (inp2.fail()) {
        cout << "Yikes!!!!! Can't open " << file_name << endl;
	return;
      }
      std::string word;
      for (int i=1; i<NEWargc; i++) {
        inp2 >> word;
        NEWargv[i] = strdup(word.c_str());
      }
      inp2.close();
      Configure(NEWargc, NEWargv);

      //---------------------------------- clean up strings
      for (int i=0; i<NEWargc; i++) {
        free(const_cast<char*>(NEWargv[i]));
      }
      free(NEWargv);
    }
    //--End Configure (1) method
  
  //====================================== Parse contents of configuration file or command line.
  void
    StochMon::Configure(int argc, const char *argv[]) 
    {
      
      if ( (argc <= 2) || (argv[argc-1][0] == '-') || (argv[argc-2][0] == '-') ){
        cout << USAGE_INFO << endl;
        fatal_error = true;
        finish();
      } else {
        for (int i=1 ; i < argc-2 ; i++) {
	  std::string argi = argv[i];
	  if (argi == "-max") {
	    MaxStride = strtol(argv[++i], 0, 0);
	  } else if (argi == "-stride") {
	    Run_Par.T = strtod(argv[++i], 0);
	  } else if (argi == "-n") {
	    Run_Par.num_ints = (int)strtod(argv[++i], 0);
	  } else if (argi == "-fmin") {
	    Run_Par.l_freq = strtod(argv[++i], 0);
	  } else if (argi == "-fmax") {
	    Run_Par.h_freq = strtod(argv[++i], 0);
	  } else if (argi == "-PSD_DumpPeriod") {
	    Run_Par.stride_dump = (int)strtod(argv[++i], 0);
	  } else if (argi == "-xmlfile") { // for backwards compatability
	    Run_Par.xml_file_name_1 = argv[++i];
	  } else if (argi == "-xmlfile_1") { 
	    Run_Par.xml_file_name_1 = argv[++i];
	  } else if (argi == "-xmlfile_2") { 
	    Run_Par.xml_file_name_2 = argv[++i];
	  } else if (argi == "-logfile") {
	    mOutput.cumlog_file_name = argv[++i];
	  } else if (argi == "-OSCfile") { 
	    Run_Par.osc_file_name = argv[++i];
	  } else if (argi == "-refpsd") {
	    Run_Par.CalPSD_file_name = argv[++i];
	  } else if(argi == "-screen") {
	    mOutput.screen_output = true;
	    mOutput.write_log = false;
	    mOutput.error_log = false;
	  } else if(argi == "-trend") {
	    mOutput.trend = true;
	  } else if(argi == "-window") {
	    Run_Par.window_name = argv[++i];
	    for(uint32_t ii = 0; ii < (Run_Par.window_name).length(); ii++) {
	      Run_Par.window_name[ii] = tolower(Run_Par.window_name[ii]);
	    }
	  } else if ( (argi == "-h") || (argi == "--help") ) {
	    cout << USAGE_INFO << endl;
	    fatal_error = true;
	    finish();
	  } else if(argi == "-local") {
	    mOutput.local = true;
	  } else if (isDatEnvArg(argi.c_str())) {
	    i++;
	  } else {
	    std::cerr << "StochMon WARNING:  Argument: " << argi;
	    std::cerr << " not recognized." << std::endl;
	    cout << USAGE_INFO << endl;
	  }
      }  //--this ends the loop that parses the command line
    }
  
  //---------- Get command line, for the record.
  for (int i=0; i < argc; i++) {
    std::string arg_i = argv[i];
    command_line += arg_i;        
    command_line += "  ";         
  }
  
  //---------- Get local directory, for the record.
  char *tempdir = getenv("PWD");
  if (tempdir) {
    pwd_dir = string(tempdir) + "/";
  }  
  
  //---------- Get Channel name and IFO_1.
  Run_Par.IFO_1 = argv[argc - 2];
  cout << "StochMon MESSAGE: IFO_1 is " << argv[argc - 2] << endl;
  Run_Par.mChannel = Run_Par.IFO_1; 
  Run_Par.mChannel += ":LSC-DARM_ERR";
  
  //---------- Get IFO_2.
  Run_Par.IFO_2 = argv[argc - 1];
  cout << "StochMon MESSAGE: IFO_2 is " << argv[argc - 1] <<  endl;    
 if (Run_Par.CalPSD_file_name == ""){
  Run_Par.mChannel_2 = Run_Par.IFO_2; 
  Run_Par.mChannel_2 += ":LSC-DARM_ERR";
} 
}  //--End Configure (2) method

//====================================== Run list of checks on parameters.
//:KLUDGE: Since this is a method of the StochMon class, shouldn't need to 
//hand it data members of the same class (Run_Par, fatal_error).
void
StochMon::VerifyParameters(Parameter& Run_Par, bool& fatal_error)
{
  if ( ((int)(Run_Par.T) % Run_Par.num_ints) != 0 ) {
    cout << "StochMon ERROR: The number of averages 'n' must divide the length 'stride'.\n"; 
    cout << "                    Exiting.\n";
    cout << USAGE_INFO << endl;
    fatal_error = true;
    finish();
  } 
  if ( Run_Par.l_freq >= Run_Par.h_freq ) {
    cout << "StochMon ERROR: Low frequency must be less than high ";
    cout << "frequency.  Exiting.\n";
    cout << USAGE_INFO << endl;
    fatal_error = true;
    finish();
  }
  if (Run_Par.window_name == "square") {
    cout << "StochMon WARNING: Square window (ie, no windowing) selected for\n"; 
    cout << "                      power-spectrum estimation.  This is a VERY BAD\n";
    cout << "                      IDEA; you should select anything else.\n";
  }
  if (!fatal_error) {
    if ( !mOutput.local && !((mOutput.cumlog_file_name).empty()) ) {
      cout << "StochMon WARNING: -local option has not been selected; specification\n";
      cout << "                      of log-file name will be ignored.\n";
    }
    
    
    //---------- Get name of OSC configuration files.
    if ( (Run_Par.osc_file_name).empty() ) {
      cout << "StochMon WARNING: No OSC configuration file specified; will look for\n";
      cout << "                      default OSC configuration file.\n";
      //---------- Look for OSC config file in directory specified
      //           in $STOCHMON_OSCCONF instead of in local directory.
      const char* osc_dir = getenv("STOCHMON_OSCCONF");
      if (osc_dir) {
	Run_Par.osc_file_name = string(osc_dir) + "/";
      } else { 
	cout << "StochMon WARNING: Environment variable $STOCHMON_OSCCONF not set;\n"; 
	cout << "                      looking in local directory for default OSC file.\n";
      }
      Run_Par.osc_file_name += "StochMon_LockLoss.conf";
    }
    
  }
}  //--End VerifyParameters method

//====================================== Write current stochastic sensitivity, etc to specified
//                                      output stream.
void
StochMon::ReportResults(std::ostream& out, const Time& time, Parameter& Run_Par)
{
  //---------- Write the stochastic sensitivity, alpha, etc. to the specified 
  //	     output stream.
  out << time.getS();
  out << ' ' << setw(13) << R_Dat.omega;
  out << ' ' << setw(14) << R_Dat.invomegasq;
  out << ' ' << setw(15) << R_Dat.ampl;
  out << ' ' << setw(15) << R_Dat.alpha;
  out << ' ' << setw(15) << R_Dat.beta;
  out << endl;
}







