/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef PSLTOOL_HH
#define PSLTOOL_HH

#include <string>
#include <iosfwd>
#include "PSLChan.hh"

class MonServer;
class TrigClient;
class OperStateCondList;
class Trend;

namespace trig {
    class SegAccountant;
}


//======================================  Band monitor class
/**  PSLTool is an API for the PSLmon tools. 
 *  @author J. Zweizig
 *  @version 1.0; Last modified 9/13/2005
 */
class PSLTool {
public:
    PSLTool(const std::string& name, const PSLChan* chan);
    virtual ~PSLTool(void);

    virtual PSLTool* clone(void) const = 0;
    virtual bool crunch(void) = 0;
    virtual void postConfig(MonServer& v) = 0;
    virtual void reset(void)  = 0;

    //------------------------------------  Accessors
    /** Get the Alarm Enable flag
     */
    const std::string& comment(void) const;

    /** Get the Alarm Enable flag
     */
    bool getAlarmEnable(void) const;

    /** Get the channel name
     */
    const char* getChannel(void) const;

    /**  Get the debug flag value.
     */
    int getDebug(void) const;

    /** Get the enable condition name.
     */
    const char* getEnableCond(void) const;

    /** Get the channel Ifo field.
     */
    std::string getIFO(void) const;

    /** Get the tool name.
     */
    const std::string& getName(void) const;

    /** Get the trigger enable flag.
     */
    bool getTrigEnable(void) const;

    /** Print statistics
     */
    virtual void printStats(std::ostream& o) const = 0;

    /** Print statistics header
     */
    virtual void printStatHdr(std::ostream& o) const = 0;

    //-------------------------------------  Segments and triggers.
    /**  Send a trigger to the trigger manager
     */
    virtual void configSegment(trig::SegAccountant& tc);

    /**  Send a trigger to the trigger manager
     */
    virtual void sendSegment(trig::SegAccountant& tc);

    /**  Send a trigger to the trigger manager
     */
    virtual void sendTrigger(TrigClient& tc);

    /**  Get a constan reference to the channel descriptor
     */
    const PSLChan& refChan(void) const;

    /**   Get a constant reference to the tool name.
     */
    const std::string& refName(void) const;

    //------------------------------------  Mutators
    /**  Set the alarm enable flag.
     */
    void setAlarmEnable(bool yorn);

    /**  Set the alarm enable flag.
     */
    void setComment(const std::string& com);

    /**  Set the debug printout level.
     *  @memo Set debug level.
     *  @param dbg Level of debug printouts.
     */
    void setDebug(int dbg);

    /** Set the enable condition name.
     */
    void setEnableCond(const std::string& cond);

    /** Set the trigger enable flag.
     */
    void setTrigEnable(bool yorn);

    /** Test the enable condition if it's available.
     */
    bool testCondition(OperStateCondList& osc) const;

private:
    /// Tool name
    std::string    mName;

    ///  Tool description
    std::string mComment;

    /// Channel name.
    const PSLChan* mChan;

    ///  Enable condition
    std::string mEnableCond;

    /// Debug flag
    int mDebug;

    ///  Alarm enable flag
    bool mAlarmEnable;

    ///  Trigger enable flag
    bool mTrigEnable;
};


//======================================  Inline functions
#ifndef __CINT__
inline const std::string&
PSLTool::comment(void) const {
    return mComment;
}

inline bool
PSLTool::getAlarmEnable(void) const {
    return mAlarmEnable;
}

inline const char*
PSLTool::getChannel(void) const {
    return mChan->getChannel();
}

inline int 
PSLTool::getDebug(void) const {
    return mDebug;
}

inline const char* 
PSLTool::getEnableCond(void) const {
    return mEnableCond.c_str();
}

inline std::string
PSLTool::getIFO(void) const {
    return mChan->getIFO();
}

inline const std::string&
PSLTool::getName(void) const {
    return mName;
}

inline bool
PSLTool::getTrigEnable(void) const {
    return mTrigEnable;
}

inline const PSLChan& 
PSLTool::refChan(void) const {
    return *mChan;
}

inline const std::string& 
PSLTool::refName(void) const {
    return mName;
}

#endif  // !defined(__CINT__)

#endif  // !defined(PSLBAND_HH)
