#include "PSLOChan.hh"
#include "MonServer.hh"
#include "ReadTrend.hh"
#include "Trend.hh"
#include <stdexcept>
#include <iostream>

using namespace std;

//======================================  Null constructor
OutChan::OutChan(void) 
  : mTrend(0)
{}

//======================================  Channel constructor
OutChan::OutChan(Trend& tr, const string& chan, const string& prefix,
		 const string& suffix)
  : mTrend(&tr)
{
    if (prefix.empty()) setPrefix(tr.getName());
    else                setPrefix(prefix);
    if (!chan.empty())  setChannel(chan, suffix);
}

//======================================  Destructor
OutChan::~OutChan(void) {
}

//======================================  Add the trend channel
void
OutChan::addTrendChannel(void) {
    cout << "Add Channel: " << mTrendChannel << endl;
    if (mTrend && !mTrendChannel.empty()) {
      mTrend->addChannel(mTrendChannel.c_str());
    }
}

//======================================  Fill data
void 
OutChan::fillData(const Time& t, double data, Interval dt) {
    if (mTrend) {
        mTrend->trendData(mTrendChannel.c_str(), t, data);
	if (mHistory.isEmpty()) setHistoryData(t, dt);
    }
    float fdata(data);
    mHistory.fixedAppend(t, dt, &fdata);
    mStats.update(data, dt);
}

//======================================  Define the channel to be served.
void 
OutChan::serveViewerSeries(MonServer& mon, const char* comm) {
    mon.serveData(mViewerChannel.c_str(), &mHistory, comm);
}

//======================================  Set trend and viewer channel names
void 
OutChan::setChannel(const std::string& chan, const std::string& suffix) {
    if (chan[2] != ':' || chan[6] != '-') 
        throw runtime_error("Non-standard channel name");
    mTrendChannel =  chan.substr(0,3);
    mTrendChannel += "DMT-";
    mTrendChannel += mPrefix;
    mTrendChannel += "_";
    mTrendChannel += chan.substr(3,3);
    mTrendChannel += "_";
    mTrendChannel += chan.substr(7);
    mTrendChannel += "_";
    mTrendChannel += suffix;

    mViewerChannel =  chan.substr(0,3);
    mViewerChannel += mPrefix;
    mViewerChannel += "_";
    mViewerChannel += chan.substr(3,3);
    mViewerChannel += "_";
    mViewerChannel += chan.substr(7);
    mViewerChannel += "_";
    mViewerChannel += suffix;
}

//======================================  Set the history data from the 
void 
OutChan::setHistoryData(const Time& tEnd, Interval tStep) {

    //----------------------------------  Check the time increments (defined?)
    Interval dT = mHistory.getMaxLen();
    if (!dT || !tStep) return;
    Time tStart(tEnd - dT);

    //----------------------------------  Get the mean value time series
    const char* pDir = getenv("DMTRENDOUT");
    if (!pDir) pDir = ".";
    ReadTrend rt(pDir);
    TSeries Avg;
    rt.getSeries(mTrendChannel.c_str(), tStart, dT, &Avg);

    //----------------------------------  Fill in data one word at a time.
    int N=Avg.getNSample();
    for (Time t=tStart; t<tEnd; t+=tStep) {
        int ibin = Avg.getBin(t);
	float x = (ibin < N) ? Avg.getDouble(ibin) : 0.0;
	mHistory.fixedAppend(t, tStep, &x);
    }
}

//======================================  Set the dmtviewer series length
void 
OutChan::setHistoryLen(Interval lhist) {
    mHistory.setMaxLen(lhist);
}

//======================================  Pad the history series
void 
OutChan::setHistoryTime(const Time& t0, Interval dT) {
    if (mHistory.isEmpty()) setHistoryData(t0, dT);
    mHistory.padUntil(t0, dT);
}

//======================================  Set the trend channel prefix name
void 
OutChan::setPrefix(const std::string& pfx) {
    mPrefix = pfx.substr(0,4);
    int N = mPrefix.length();
    for (int i=0; i<N; i++) {
        if (mPrefix[i] >= 'a' && mPrefix[i] <= 'z') mPrefix[i] += 'A' -'a';
    }
}

//======================================  Set the trend channel nanme
void
OutChan::setTrendChannel(const std::string& chan) {
    mTrendChannel = chan;
}


//======================================  set the trend object pointer.
void
OutChan::setTrend(Trend& trnd) {
    mTrend = &trnd;
    if (mPrefix.empty()) setPrefix(mTrend->getName());
}

//======================================  Set the viewer channel name.
void 
OutChan::setViewerChannel(const std::string& chan) {
    mViewerChannel = chan;
}

//======================================  Stats constructor
OutChan::stats_type::stats_type(void)
  : _Sumx(0), _Sumxx(0), _SumN(0), _Sumt(0)
{}

//======================================  Calculate sigma
double 
OutChan::stats_type::getSigma(void) const {
    if (!_SumN) return 0;
    double mean  = _Sumx/_SumN;
    double sigma = _Sumxx/_SumN - mean*mean;
    if (sigma > 0) sigma = sqrt(sigma);
    else           sigma = 0.0;
    return sigma;
}

//======================================  Update stats
void 
OutChan::stats_type::update(double data, Interval dt) {
    _Sumx  += data;
    _Sumxx += data*data;
    _Sumt  += dt;
    _SumN  += 1;
}
