/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "PSLGlitch.hh"
#include "TrigRslt.hh"
#include "TrigClient.hh"
#include "MonServer.hh"
#include "DVecType.hh"
#include <iostream>
#include <iomanip>
#include <cmath>
#include <float.h>

using namespace std;

//  SvSize is the number of glitch-lengths of data to save in the data
//  snippet on either side of the glitch limits.
static const double SvSize = 2.0;

//======================================  Glitch finder constructor
PSLGlitch::PSLGlitch(const string& name, const PSLChan* chan) 
  : PSLTool(name, chan), mSigma(4.0), mSize(0), mTime(0.001), mTrigger(0), 
    mStartTime(0), mLastTime(0), mLastTrig(0), mRecentGlitches(0), 
    mRecentTime(0.0), mRecentRate(0.0), mAlarmSet(false), mAlarmThresh(0.0), 
    mAlarmTime(60.0), mNTotal(0), mObserved(0.0), mSumAmpl(0.0), mSumTime(0.0),
    mDtLt1(0), mSigHist("Glitch Amplitude (sigma)", 100, 0.0, 200.0),
    mDurHist("Glitch Duration (s)", 100, 0.0, 0.25),
    mDtHist("Glitch: log(Delta-t)", 120, -6.0, 6.0)
{
    mRateHstry.setHistoryLen(7200.0);
    setComment("PSLGlitch");
}

//======================================  Glitch finder destructor
PSLGlitch::~PSLGlitch(void) {
    reset();
}

//======================================  Clone a Glitch finder
PSLGlitch*
PSLGlitch::clone(void) const {
    return new PSLGlitch(*this);
}

//======================================  Configure the segment accountant
void 
PSLGlitch::confSegment(trig::SegAccountant& sa) {
    if (!mSegID.refName().empty()) sa.addSegment(mSegID, comment(), 1);
}

//======================================  Look for a glitch
bool 
PSLGlitch::crunch(void) {

    //----------------------------------  Get the TSeries. Bail if empty
    if (!refChan().isValid()) return false;
    const TSeries* ts = &refChan().refTSeries();
    int   nData = ts->getNSample();

    //----------------------------------  Set start, end times.
    bool isFiltered = !mFilter.null();
    Time tsStart = ts->getStartTime();
    if (mLastTime != tsStart) {
        mStartTime = tsStart;
        mLastTrig = Time(0);
	if (isFiltered) mFilter.reset();
    }
    mLastTime = ts->getEndTime();

    //----------------------------------  Get TSeries statistics
    const PSLChan::chan_data* pF=0;
    double avg   = 0;
    double sigma = 0;
    if (isFiltered) {
        TSeries* tn = new TSeries(mFilter(*ts));
	tn->Convert(DVecType<PSLChan::chan_data>::getDataType());
	Interval runt = tsStart - mStartTime;
	if (mFilter.getSettle() > runt) {
	   tn->eraseStart(mFilter.getSettle() - runt);
	}
	ts = tn;
	nData = ts->getNSample();
	if (!nData) return false;

	pF = static_cast<const PSLChan::chan_data*>(ts->refData());
	const PSLChan::chan_data* pt = pF;
	for (int i=0; i<nData; ++i) {
	    PSLChan::chan_data x = *pt++;
	    avg   += x;
	    sigma += x * x;
	}
	avg /= double(nData);
	sigma = sigma/double(nData) - avg*avg;
	if (sigma > 0.0) sigma = sqrt(sigma);
	else             sigma = 0.0;
    } else {
        pF = static_cast<const PSLChan::chan_data*>(ts->refData());
        avg   = refChan().getAverage();
	sigma = refChan().getSigma();
    }

    if (sigma > FLT_MIN) {
	PSLChan::chan_data delt = mSigma*sigma;
	if (delt < mSize) delt = mSize;
	if (getDebug() > 2) cout << "Glitch: " << getName() << " avg/sigma= " 
				 << avg  << " / " << sigma << " delta= " 
				 << delt <<  endl;

	//------------------------------  Get the number of deviations.
	bool   inGlitch = false;
	int    tSamples = int(mTime/ts->getTStep() + 0.5); // #samples in mTime
	Time   tStart, tMax;
	int    tLeft(1);
	double gMax = 0.0;
	for (int i=0 ; i<nData ; i++) {
	    PSLChan::chan_data x = pF[i] - avg;
	    if (x < -delt || x >= delt) {
	        double ampl = fabs(x);
		tLeft = tSamples;
		if (!inGlitch) {
		    tStart = ts->getBinT(i);
		    inGlitch = true;
		    if (getDebug() > 2) cout << "Start Glitch: " << getName() 
					     << endl;
		    gMax = ampl;
		    tMax = tStart;
		} else if (ampl > gMax) {
		    gMax = ampl;
		    tMax = ts->getBinT(i);
		}
	    } else if (inGlitch && --tLeft == 0) {
	        Time tStop = ts->getBinT(i);
		saveIt(tStart, tStop, tMax, sigma, gMax, ts);
		inGlitch = false;
		gMax = 0.0;
	    }
	}

	//------------------------------  Process glitch at end of a stride
	if (inGlitch) {
	    Time tStop = ts->getEndTime();
	    saveIt(tStart, tStop, tMax, sigma, gMax, ts);
	}

	//------------------------------  Alarm processing
	if (mAlarmThresh > 0.0) {
	    mRecentGlitches += getCount();
	    mRecentTime     += ts->getInterval();
	    if (mRecentTime >= mAlarmTime) {
	        mRecentRate     = mRecentGlitches / double(mRecentTime);
		mAlarmSet       = mRecentRate > mAlarmThresh;
		mRecentGlitches = 0;
		mRecentTime     = 0.0;
	    }
	}
    }

    //----------------------------------  Update stats, Return.
    if (mObserved == Interval(0.0)) dumpConfig(cout);
    statUpdate(refChan().refTSeries().getStartTime(), 
	       refChan().refTSeries().getInterval());
    mObserved  += ts->getInterval();
    if (isFiltered) delete ts;
    return !mList.empty();
}

//======================================  Dump the glitch tool configuration
std::ostream& 
PSLGlitch::dumpConfig(std::ostream& out) const {
    out << endl;
    out << "Glitch tool    " << getName() << " configuration" << endl;
    out << "Channel name:            " << getChannel() << endl;
    out << "Filter name:             " << mFilter.getName() << endl;
    out << "Filter delay(s):         " << mFilter.getDelay() << endl;
    out << "Filter settling time(s): " << mFilter.getSettle() << endl;
    out << "Enable OSC condition:    " << getEnableCond() << endl;
    out << "Trend channel name:      ";
    if  (!*mRateHstry.getTrendChannel()) {
        out << "-- None --" << endl;
    } else {
        out << mRateHstry.getTrendChannel() << endl;
    }
    out << "Threshold (sigma):       " << mSigma << endl;
    out << "Threshold (ampl):        " << mSize << endl;
    out << "Characteristic time:     " << mTime << endl;
    out << "Triggers/stride (max):   " << mTrigger << endl;
    return out << endl;
}

//======================================  post-configuration 
void 
PSLGlitch::postConfig(MonServer& v) {
    if (getDebug() > 1) cout << "Serving glitch: " << getName() << endl;
    mRateHstry.serveViewerSeries(v, "Glitch Rate History");
    mRateHstry.addTrendChannel();

    string gName = mRateHstry.getViewerChannel();
    string::size_type l = gName.rfind("_RATE");
    if (l < string::npos) gName.erase(l);
    else                  gName = getName();

    v.serveData(gName.c_str(),              &mSvLast,  "Glitch series");
    v.serveData((gName+"_SigHist").c_str(), &mSigHist, "Significance Histo");
    v.serveData((gName+"_DurHist").c_str(), &mDurHist, "Duration Histo");
    v.serveData((gName+"_DtHist").c_str(),  &mDtHist,  "delta-t Histo");
    if (getDebug()) dumpConfig(cout);
}

//======================================  Save the glitch
void 
PSLGlitch::sendSegment(trig::SegAccountant& sa) {
    typedef unsigned long gps_type;
    typedef std::pair<gps_type, gps_type> segnode;
    typedef std::vector<segnode> segvec;
    segvec seg_list;

    if (mSegID.refName().empty()) return;

    int N=mList.size();
    for (int i=0; i<N; ++i) {
	gps_type sStart=mList[i]._time.getS();
	gps_type sEnd=(mList[i]._time + mList[i]._dt).getS() + 1;
	if (sEnd > mLastTime.getS()) sEnd = mLastTime.getS();
	for (segvec::iterator j=seg_list.begin(); j != seg_list.end(); ++j) {
	    //-------------------------------- Test for overlap...
	    if (sStart <= j->second && sEnd >= j->first) {
		if (sStart < j->first)  j->first  = sStart;
		if (sEnd   > j->second) j->second = sEnd;
		sEnd = sStart = 0;
		break;
	    }

	    //-------------------------------- Insert if before
	    if (sEnd < j->first) {
		seg_list.insert(j, segnode(sStart, sEnd));
		sEnd = sStart = 0;
		break;
	    }
	}
	if (sEnd != sStart) {
	    seg_list.push_back(segnode(sStart, sEnd));
	}
    }

    //----------------------------------  Merge overlapping segment
    int nSeg=seg_list.size();
    int j=0;
    for (int i=1; i<nSeg; i++) {
	if (seg_list[i].first <= seg_list[j].second) {
	    if (seg_list[j].second < seg_list[i].second) {
		seg_list[j].second = seg_list[i].second;
	    } else {
		j++;
		if (j != i) seg_list[j] = seg_list[i];
	    }
	}
    }
    j++;
    if (j < nSeg) {
	seg_list.erase(seg_list.begin()+j, seg_list.end());
	nSeg = j;
    }

    //----------------------------------  Add segments in range of data
    Time tStart = refChan().refTSeries().getStartTime();
    for (int i=0; i<nSeg; ++i) {
	Time sStart(seg_list[i].first);
	if (tStart < sStart) {
	    sa.set_segment(mSegID, tStart, sStart, false);
	}
	Time sEnd(seg_list[i].second);
	sa.set_segment(mSegID, sStart, sEnd, true);
	tStart = sEnd;
    }
    if (tStart < mLastTime) {
	sa.set_segment(mSegID, tStart, mLastTime, false);
    }
}

//======================================  Save the glitch
void 
PSLGlitch::saveIt(const Time& tStart, const Time& tStop, const Time& tMax, 
		  double sig, double gMax, const TSeries* ts) {

    //----------------------------------  Build a glitch descriptor
    Interval dt = tStop - tStart;
    SaveGlitch svg(tStart - mFilter.getDelay(), dt, sig, gMax);
    svg._peak = tMax - tStart;

    //----------------------------------  Insert the glitch at the right place
    int N = mList.size();
    for (int i=0; i<N; ++i) if (gMax > mList[i]._ampl) N=i;
    mList.insert(mList.begin()+N, svg);

    //----------------------------------  Save largest amplitude time series
    if (!N) {
	Time t0   = tStart - SvSize * mTime;
	mSvLast   = ts->extract(t0, dt + (2*SvSize - 1.0)*mTime );
    }

    //----------------------------------  Histogram stuff
    mSigHist.Fill(gMax/sig);
    mDurHist.Fill(double(dt));
    if (mLastTrig != Time(0)) {
        Interval Delta = tMax - mLastTrig;
        mDtHist.Fill(log(double(Delta)));
	if (Delta < Interval(1.0)) mDtLt1++;
    }

    //----------------------------------  Keep statistics
    mSumAmpl += gMax;
    mSumTime += dt;
    if (getDebug() > 2) cout << "Found Glitch: " << getName() << " Start/Stop "
			     << tStart << " / " << tStop << endl;
    mLastTrig = tMax;
}

//======================================  Send Triggers
void 
PSLGlitch::sendTrigger(TrigClient& tc) {
   int nGlitch = getCount();
   if (mTrigger < nGlitch) nGlitch = mTrigger;
   if (!nGlitch) return;

   //-----------------------------------  Send the required number of triggers.
   for (int j=0; j<nGlitch; ++j) {
       const SaveGlitch* glix = &mList[j];
       if (getDebug()) cout << "Glitch found, channel: " << getName() <<endl;
       trig::TrigRslt t("Glitch", getName().c_str(), 0);
       t.setTime(glix->_time);
       t.setPriority(trig::p_error);
       t.setDisposition(trig::d_metaDB);
       t.setDuration(glix->_dt);
       t.setPeakOffset(glix->_peak);
       t.setIntensity(glix->_ampl);
       t.setSignificance(glix->_ampl/glix->_sigma);
       t.setIfos(getIFO().c_str());
       int rc = tc.sendTrigger(t);
       if (rc && rc!=12) cout << "Error "<< rc << " logging Glitch "
			      << getName() << " trigger" << endl;
   }
}

//=======================================  Set the segment name
void 
PSLGlitch::setSegment(const std::string& seg) {
    string name(seg);
    string ifo;
    int version = 1;
    string::size_type inx = name.find(':');
    if (inx != string::npos) {
	ifo = name.substr(0, inx);
	name.erase(0, inx+1);
    }
    inx = name.find(':');
    if (inx != string::npos) {
	version = strtol(name.c_str()+inx+1, 0, 0);
	name.erase(inx);
    }
    mSegID = trig::SegAccountant::seg_id(name, version);
    if (!ifo.empty()) mSegID.setIfo(ifo);
}

//======================================  Reset
void 
PSLGlitch::reset(void) {
    mList.clear();
    mAlarmSet = false;
}

//======================================  Update statistics and rate history.
void 
PSLGlitch::statUpdate(const Time& t0, Interval dT) {
    int    N    = getCount();
    double delt = double(dT);
    if (delt != 0.0) mRateHstry.fillData(t0, double(N)/delt, dT);
    mNTotal += N;
}

//======================================  Print statistics
void 
PSLGlitch::printStats(ostream& out) const {
    // out << setw(20) << getName();
    out << left << setw(20) << getChannel();
    if (mFilter.null()) out << setw(15) << "-";
    else                  out << setw(15) << mFilter.getName();
    if (!*getEnableCond()) out << setw(30) << "-";
    else                   out << setw(30) << getEnableCond();
    out.setf(ios::right);
    out << setw(8)  << setprecision(4) << mSigma;
    out << setw(8)  << mNTotal;
    out << setw(10) << setprecision(4) << getAvgRate();
    out << setw(10) << setprecision(6) << getAvgAmpl();
    out << setw(10) << setprecision(4) << getAvgTime();
    out << setw(10) << setprecision(4) << getFracLt1();
    out << endl;
}

void 
PSLGlitch::printStatHdr(ostream& out) const {
    // out << setw(20) << "Tool Name";
    out.setf(ios::left);
    out << setw(20) << "Channel Name";
    out << setw(15) << "Filter";
    out << setw(30) << "Condition";
    out.setf(ios::right);
    out << setw(8)  << "Thres";
    out << setw(8)  << "Total";
    out << setw(10) << "Rate";
    out << setw(10) << "Amplitude";
    out << setw(10) << "Duration";
    out << setw(10) << "Dt<1/All";
    out << endl;
}

//======================================  Get largest deviation (in sigma).
double
PSLGlitch::getSigma(unsigned int i) const {
    if (i < mList.size()) return 0.0;
    else                  return mList[i]._ampl/ mList[i]._sigma;
}

//======================================  Set the alarm parameters.
void
PSLGlitch::setAlarmParams(double thresh, Interval aTime) {
    mAlarmThresh = thresh;
    if (aTime > Interval(0.0)) mAlarmTime = aTime;
}

//======================================  Set filter
void 
PSLGlitch::setFilter(const PSLfilter& f) {
    mFilter = f;
}

//======================================  Set size threshold
void 
PSLGlitch::setSigmaT(double Size) {
    mSigma = Size;
}

//======================================  Set size threshold
void 
PSLGlitch::setSize(double Size) {
    mSize = Size;
}

//======================================  Set snippet time
void
PSLGlitch::setTime(Interval time) {
    mTime = time;
}
