#ifndef PSLCHAN_HH
#define PSLCHAN_HH

#include "PSLfilter.hh"
#include "FSeries.hh"
#include "TSeries.hh"
#include <string>

class DaccAPI;

/*  PSLChan holds the channel time series, spectrum and statistics. Each 
 *  time the channel data are read in the PSLchan class gets a pointer to 
 *  the time series, converts the data to 'chan_data' type and performs
 *  the calibration transformation (bias and scale). Strides may optionally
 *  be forced to overlap by setting the Overlap time. If the overlap is not
 *  zero, the resulting time series length is t_stride + t_overlap, and the
 *  first t_overlap seconds of the time series repeat the last t_overlap 
 *  seconds of the previous time series. A spectrum is calculated and stored 
 *  on request.
 */
class PSLChan {
public:
  typedef float chan_data;

public:

  /** Null channel constructor
    */
  PSLChan(void);

  /** Named channel constructor.
    */
  explicit PSLChan(const std::string& name);

  /** Destructor.
    */
  virtual ~PSLChan(void);

  /**  Test whether channel was read out from the data stream.
    *  @memo Test whether channel exists.
    */
  bool exists(void) const;

  /**  Return the channel name associated with this PSLChan as a constant
    *  character pointer.
    *  @memo Get channel name.
    */
  const char* getChannel(void) const;

  /**  Get the average value of the time series.
   */
  double getAverage(void) const;

  /**  Get the time series sigma.
    */
  double getSigma(void) const;

  /**  Get the IFO name from the starting characters of the channel name.
    */
  std::string getIFO(void) const;

  /**  Return the overlap time (in seconds)
    */
  Interval getOverlap(void) const;

  /**  Return if the current time series is contiguous with the last
    *  time series.
    */
  bool isContig(void) const;

  /**  Return if the current time series is valid.
    *
    */
  bool isValid(void) const;

  /**  Test whether buffered data are sufficient for requested overlap.
    */
  bool noOverlap(void) const;

  /**  Return a constant reference to the time series.
   */
  const TSeries& refTSeries(void) const;

  /**  Return a constant reference to a frequency series.
   */
  const FSeries& refFSeries(void) const;

  /**  Set an offset to be applied to the Time series.
   */
  void setBias(double bias);

  /** Set the overlap time.
   */
  void setOverlap(Interval dt);

  /**  Set the window.
    */
  void setPipe(const PSLfilter& f);

  /**  Set the optional scale factor.
    */
  void setScale(double scale);

  /**  Calculate the statistics (mean, sigme) of the input time series.
    */
  void setStats(void) const;

  /**  Set up channel data when a new stride arrives.
    */
  void start(void);

  /**  Clean up the channel at the end of a stride.
    */
  void finish(void);

  /**  Associate a data accessor (DaccAPI class object) with this channel.
    */
  void setup(DaccAPI& In);

private:
  std::string   mChannel;
  DaccAPI*      mDacc;
  TSeries*      mTS;
  PSLfilter     mPipe;
  double        mBias;
  double        mScale;
  Interval      mOverlap;

  bool             mTSerOK;
  mutable bool     mSpecOK;
  mutable bool     mStatOK;
  mutable TSeries  mTSeries;
  mutable FSeries  mFS;
  mutable double   mAverage;
  mutable double   mSigma;
  Time             mLastData;
  Time             mEndStride;
};

//======================================  Inline methods.
inline Interval
PSLChan::getOverlap(void) const {
    return mOverlap;
}

inline bool
PSLChan::isContig(void) const {
    return mTSeries.getStartTime() + mOverlap == mLastData;
}

inline bool
PSLChan::isValid(void) const {
    return mTSerOK;
}

inline const TSeries& 
PSLChan::refTSeries(void) const {
    return mTSeries;
}

#endif  // !defined(PSLCHAN_HH)
