/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef WPIPE_WTILE_HH
#define WPIPE_WTILE_HH

#include <string>
#include <vector>
#include <iosfwd>
#include <memory>
#include "wtypes.hh"

namespace containers {
   class DFT;
}

// WTILE Tile space of time, frequency, and Q for discrete Q transform 
// analysis
//
// WTILE covers the specified range of time, frequency, and Q with the minimum
// number of measurement tiles such that the fractional energy loss encountered
// by an arbitrary minimum uncertainty signal never exceeds the requested maximum
// mismatch.  WTILE is typically called once for a particular set of search
// parameters.  The resulting tiling structure is then used in subsequent calls
// to QTRANSFORM on different data segments.
//
// usage: tiling(timeRange, integrationTime, frequencyRange, sampleFrequency,
//               maximumMismatch, highPassCutoff, lowPassCutoff,
//               whiteningDuration, transientFactor);
//
//  timeRange          duration of analysis
//  qRange             range of Q to search
//  frequencyRange     range of frequency to search
//  sampleFrequency    sample frequency of input data
//  maximumMismatch    fractional loss in squared signal energy due to mismatch
//  highPassCutoff     cutoff frequency for high pass filter
//  lowPassCutoff      cutoff frequency for low pass filter
//  whiteningDuration  duration of whitening filter
//  transientFactor    ratio of transient duration to whitening filter duration
//
//  tiling             output discrete Q transform tiling structure
//
// To define the targeted signal space, WTILE takes as arguments the duration,
// frequency band, and Q range of the search.  The desired duration should be
// specified as a single scalar number, while the desired frequency range and Q
// range should both be two component vectors that specify the minimum and
// maximum frequency and Q for the analysis.
//
// This signal space cannot be specified arbitrarily.  In order to avoid
// frequency domain aliasing, WTILE enforces a minimum permissible Q of sqrt(11)
// and a maximum permissible analysis frequency that depends on Q.  In order to
// ensure a sufficient number of statistically independent tiles in each
// frequency row, WTILE also enforces a minimum permissible analysis frequency
// that depends on Q.  For convenience, the maximum permissible frequency range
// may be obtained for each Q plane by specifying a frequency range of [].
// Alternatively, the minimum permissible frequency may be obtained by specifying
// a lower limit of 0 Hz and the maximum permissible frequency by specifiying an
// upper limit of Inf Hz.
//
// WTILE also reports recommended filter parameters for data conditioning that
// include cutoff frequencies for high pass and low pass filters and whitening
// filter duration.  It also reports a recommended duration to ignore at both
// beginning and end of the transform due to filter transients.  This transient
// duration is simply the product of the whitening filter duration and a user
// specified factor.  If no transient factor is specified, a default value of
// four is used.  The user may also override the suggested data conditioning
// filter parameters by providing alternative parameters as arguments to WTILE.
//
// The output Q transform tiling structure contains the following fields.
//
//   id                    identification string for structure
//   duration              duration of data under analysis
//   minimumQ              minimum Q of search
//   maximumQ              maximum Q of search
//   minimumFrequency      minimum frequency of search
//   maximumFrequency      maximum frequency of search
//   sampleFrequency       sample frequency of the data under analysis
//   maximumMismatch       maximum fractional energy loss due to signal mismatch
//   numberOfPlanes        number of Q planes in analysis
//   qs                    vector of Qs
//   planes                cell array of plane structures
//   numberOfTiles         total number of tiles in analysis
//   numberOfIndependents  total number of statistically independent tiles
//   numberOfFlops         total number of flops in analysis
//   highPassCutoff        cutoff frequency for high pass filter
//   lowPassCutoff         cutoff frequency for low pass filter
//   whiteningDuration     duration of whitening filter
//   transientDuration     duration of filter transients to supress

namespace wpipe {

   //
   // The rows field is a vector of row structures, one for each frequency in
   // the plane, which contain the following fields.
   //
   class qrow {
   public:
      qrow(void);
      ~qrow(void);
      /**
       *  Initialize a q-transform row at the specified (\a q, \a f)
       *  \param q Central q values of for ths row.
       *  \param f Central frequency for this row
       *  \param timeRange Analysis stride length (in seconds) 
       *  \param nyquistFrequency Half of sample rate.
       *  \param fStep Hal-widt of frequency band.
       *  \param mismatchStep Mismatch step
       */
      void init(double q, double f, double timeRange, double nyquistFrequency,
		double fStep, double mismatchStep);
      std::ostream& display(std::ostream& out) const;
      TSeries tileCoeffs(const containers::DFT& data) const;

      double frequency;            /// frequency of row
      double duration;             /// tile duration for coincidence testing
      double bandwidth;            /// tile bandwidth for coincidence testing
      double timeStep;             /// tile time step for integration
      double frequencyStep;        /// tile frequency step for integration
      int    numberOfTiles;        /// number tiles in frequency row
      double numberOfIndependents; /// # statistically independent tiles in row
      double numberOfFlops;        /// number of flops to compute frequency row
   private:
      typedef std::shared_ptr<containers::DFT> winptr_type;
   private:
      int    zeroPadLength;        /// number of zeros to append to window data
      winptr_type window;          /// window vector
   };

   typedef std::vector<qrow> qrow_vect;

   //
   // The planes field is a vector of plane structures, one for each Q, as
   // follows.
   //
   /**  The %qplane class contains information about a single Q-transform plane.
    */
   struct qplane {
      qplane(void);
      ~qplane(void);
      /**
       *  The init function initializes the qplane structure. Specifically, it
       *  will
       *  <ul>
       *    <li> Set the q value.
       *    <li> calculate the plane normalization factor.
       *    <li> Calculate a logarithmically spaced frequency grid
       *    <li> allocate and construct a qrow instance for each frequency point
       *    <li> initialize summary counters.
       *  </ul>
       *  \param q0 Plane q value
       *  \param timeRange Number of data seconds processed
       *  \param minimumAllowableIndependents Minimum independents
       *  \param nyquistFrequency Half the sample frequency
       *  \param reqFmin Minimum frequency to be covered
       *  \param reqFmax Maximum frequency to be covered
       *  \param mismatchStep Phase mismatch per step
       */ 
      void init(double q0, double timeRange, long minimumAllowableIndependents,
		double nyquistFrequency, double reqFmin, double reqFmax,
		double mismatchStep);

      /**  Constant reference to a requested qrow.
       */
      const qrow& row(int i) const;

      /**  Print the contents of the qplane structure to the specified 
       *  STL output stream.
       */
      std::ostream& display(std::ostream& out) const;

      /**  Calculate a minimum whitening duration for this plane from
       *  dt = Q / (2 fMin)
       */
      double defaultWhiteningDt(void) const;
 
      double q;                     /// Q of plane
      double minimumFrequency;      /// Q dependent minimum frequency of search
      double maximumFrequency;      /// Q dependent maximum frequency of search
      double normalization;         /// Q dependent normalization factor
      int    numberOfRows;          /// number of frequency rows in plane
      qrow_vect rows;               /// cell array of row structures
      int    numberOfTiles;         /// number of tiles in plane
      int    numberOfIndependents;  /// #statisticly independent tiles in plane
      double numberOfFlops;         /// number of flops to compute plane
   };

   //===================================  Inline methods
   inline const qrow&
   qplane::row(int i) const {
      return rows[i];
   }

   inline double 
   qplane::defaultWhiteningDt(void) const {
      return q / (2 * minimumFrequency);
   }

   typedef std::vector<qplane>        qplane_vec;
   typedef qplane_vec::const_iterator const_qplane_iter;

   //
   // See also QCONDITION, QTRANSFORM, QTHRESHOLD, QSELECT, QEXAMPLE, and QPIPELINE.

   // Original version by: Shourov K. Chatterji (shourov@ligo.mit.edu)
   class wtile {
   public:
      wtile(void);

      wtile(double timeRange, const dble_vect& qRange, 
	    const dble_vect& frequencyRange, double sampleFrequency, 
	    double maximumMismatch, double highPassCutoff=-1.0, 
	    double lowPassCutoff=-1.0, double whiteningDuration=-1.0, 
	    double transientFactor=4.0, int minAllowableIndependents=50,
	    int debug=0);
      ~wtile(void);
      void init(double timeRange, const dble_vect& qRange, 
		const dble_vect& frequencyRange, double sampleFrequency, 
		double maximumMismatch, double highPassCutoff, 
		double lowPassCutoff, double whiteningDuration, 
		double transientFactor, int minAllowableIndependents,
                int debugLevel);

      int debug(void) const;
      std::ostream& display(std::ostream& out) const;
      double duration(void) const;
      bool empty(void) const;
      const qplane& end_plane(void) const;

      /**  Get the structure type identifier string.
        */
      const std::string& id(void) const;

      /**  Configured hi-pass filter cutoff frequency for this channel group.
        *  \brief High-pass cutoff
	*  \return Signal conditioning minimum frequency. 
        */
      double highPassCutoff(void) const;

      /**  independentsRate evaluates to the approximate number of independent 
        *  tiles divided by the duration, times an empirically detemined 
	*  correction factor (1.5). This number is used to calculate the false
	*  alarm probability.
        *  \brief Calculate the independent tile rate.
        *  \return Independent tile rate.
        */
      double independentsRate(void) const;

      /**  Find the configured Q value nearest the specified test value.
        *  \brief nearest Q value.
	*  \param qTest Test q value. 
	*  \return Configured q value closest to the specified test Q.
	*/ 
      double nearest_q(double qTest) const;

      /**  Find the number of the plane whose Q value is fractionally closest 
        *  to the specified test value. The closeness is measured by the 
	*  absolute value of the logarithm of the ratio of the two numbers.
	*  \brief Plane with closest Q.
	*  \param qTest Test q-value
	*  \return Number of plane with the closest Q.
	*/
      size_t nearest_plane(double qTest) const;
      long numberOfIndependents(void) const;
      long numberOfPlanes(void) const;
      const qplane& planes(int i) const;
      double sampleFrequency(void) const;
      double transientDuration(void) const;

      /**  Calculate a normalized energy threshold from a specified target 
        *  rate.
	*  \brief Calculate event threshold
	*  \param rate Target rate (Hz).
	*  \return Normalizedd enegy threshold.
	*/
      double threshold_from_rate(double rate) const;

      /**  Get the whitening duration needed for this tiling space.
        *  \brief Whitening duration.
	*  \return Duration in seconds.
	*/
      double whiteningDuration(void) const;

   private:
      double defaultHighPassCutoff(void) const;
      double defaultLowPassCutoff(void) const;
      double defaultWhiteningDuration(void) const;

   private:
      std::string _id;                /// identification string for structure
      double      _duration;          /// duration of data under analysis
      double      _minimumQ;          /// minimum Q of search
      double      _maximumQ;          /// maximum Q of search
      double      _minimumFrequency;  /// minimum frequency of search
      double      _maximumFrequency;  /// maximum frequency of search
      double      _sampleFrequency;   /// sample rate of the data under analysis
      double      _maximumMismatch;   /// maximum allowed fractional energy loss 
      ///   due to signal mismatch
      int         _numberOfPlanes;    /// number of Q planes in analysis
      qplane_vec  _planes;            /// vector of plane structures
      int         _numberOfTiles;     /// total number of tiles in analysis
      int         _numberOfIndependents; /// number of independent tiles
      double      _numberOfFlops;     /// total number of flops in analysis
      double      _highPassCutoff;    /// cutoff frequency for high pass filter
      double      _lowPassCutoff;     /// cutoff frequency for low pass filter
      double      _whiteningDuration; /// duration of whitening filter
      double      _transientDuration; /// Filter transient suppression duration
      int         _debugLevel;        /// debug printout level.
   };

   //====================================  Inline methods.
   inline int
   wtile::debug(void) const {
      return _debugLevel;
   }

   inline double
   wtile::duration(void) const {
      return _duration;
   }

   inline bool
   wtile::empty() const {
      return _planes.empty();
   }

   inline const qplane& 
   wtile::end_plane(void) const {
      return _planes.back();
   }

   inline double
   wtile::highPassCutoff(void) const {
      return _highPassCutoff;
   }

   inline const std::string&
   wtile::id(void) const {
      return _id;
   }

   inline double
   wtile::independentsRate(void) const {
      return 1.5 * _numberOfIndependents / _duration;
   }

   inline long
   wtile::numberOfPlanes(void) const {
      return _numberOfPlanes;
   }

   inline long
   wtile::numberOfIndependents(void) const {
      return _numberOfIndependents;
   }


   inline double
   wtile::nearest_q(double qTest) const {
      return _planes[nearest_plane(qTest)].q;
   }

   inline const qplane& 
   wtile::planes(int i) const {
      return _planes[i];
   }

   inline double
   wtile::sampleFrequency(void) const {
      return _sampleFrequency;
   }
 
   inline double
   wtile::transientDuration(void) const {
      return _transientDuration;
   }

   inline double
   wtile::whiteningDuration(void) const {
      return _whiteningDuration;
   }

} // namespace 

#endif // !defined(WPIPE_WTILE_HH)
