/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "wstream_chan.hh"
#include "wcondition.hh"
#include "weventlist.hh"
#include "wtransform.hh"
#include "woutput.hh"

#include "matlab_fcs.hh"
#include "MultiDacc.hh"
#include "DVecType.hh"
#include "lcl_array.hh"
#include "OperStateCondList.hh"

//  constant dInf;
static const double dInf = 1.0 / 0.0;

using namespace wpipe;

//======================================  Construct a channel streaming block
wstream_chan::wstream_chan(const wparameters& par, int dbugLvl) 
   : pars(par), debugLevel(dbugLvl), minimumTimeShift(0), maximumTimeShift(0),
     oscList(0)
{

   //-----------------------------------  Define tile for Q-transform space
   wlog(debugLevel, 1, "tiling search space");
   double fSample = pars.sampleFrequency;
   tiling.init(pars.blockDuration, pars.qRange, pars.frequencyRange, 
	       fSample, pars.maximumMismatch, 
	       pars.highPassCutoff, pars.lowPassCutoff, 
	       pars.whiteningDuration, pars.transientFactor, 
	       pars.minAllowableIndependents, debugLevel);

   if (debugLevel >=2) {
      cout << "  whiteningDuration:   " << tiling.whiteningDuration() << endl;
      cout << "  transientDuration:   " << tiling.transientDuration() << endl;
   }

   if (debugLevel > 4) tiling.display(cout);

   //----------------------------------  get max, min time shift (sample
   //                                    alignment forced in pars)
   wlog(debugLevel, 1, "portioning segment");
   if (!pars.timeShifts.empty()) {
      maximumTimeShift = pars.timeShifts[0];
      minimumTimeShift = pars.timeShifts[0];
      size_t N = pars.timeShifts.size();
      for (size_t i=1; i<N; i++) {
	 if (pars.timeShifts[i] > maximumTimeShift) {
	    maximumTimeShift = pars.timeShifts[i];
	 } else if (pars.timeShifts[i] < minimumTimeShift) {
	    minimumTimeShift = pars.timeShifts[i];
	 }
      }
   }

   //-----------------------------------  Look for osc overlap
   size_t nOscConds = pars.oscConditions.size();
   osc_index.resize(nOscConds, -1);
   for (size_t i=0; i<nOscConds; ++i) {
      string cname = pars.oscConditions[i];
      if (cname.empty() || cname == "NONE") {
	 continue;
      }
      osc_index[i] = i;
      for (size_t j=0; j<i; ++j) {
	 if (osc_index[j] < 0) continue;
	 if (cname == pars.oscConditions[j]) {
	    osc_index[i] = j;
	    break;
	 }
      }
      if (debugLevel >= 2) {
	 cout << "OSC condition " << i << " index " << osc_index[i]
	      << " conditionName: " << cname << endl;
      }
   }

   //-----------------------------------  Look for state data overlap
   size_t nStateChans = pars.numberOfStates();
   state_index.resize(nStateChans, -1);
   for (size_t i=0; i<nStateChans; ++i) {
      string sname = pars.stateNames[i];
      int mask = int(pars.stateMasks[i]);
      if (!mask || sname.empty() || sname == "NONE") {
	 continue;
      }
      state_index[i] = i;
      for (size_t j=0; j<i; ++j) {
	 if (state_index[j] < 0) continue;
	 if (sname == pars.stateNames[j] && mask == pars.stateMasks[j]) {
	    state_index[i] = j;
	    break;
	 }
      }
      if (debugLevel >= 2) {
	 cout << "state " << i << " index " << state_index[i] << " channel: "
	      << sname << " mask: " << mask << endl;
      }
   }
}

//======================================  wstream_chan destructor.
wstream_chan::~wstream_chan(void) {
   if (oscList) delete oscList;
   oscList = 0;
}

//======================================  book a channel in th appropriate mdacc
void
wstream_chan::bookInput(MultiDacc& md, const string& channel,
			const string& frameType) const {
   //--------------------------------  Split the frame:structure type.
   string str_type;
   string fr_type = frameType;
   size_t cinx = fr_type.find(':');
   if (cinx != string::npos) {
      str_type = fr_type.substr(cinx+1);
      fr_type.erase(cinx);
   }

   //--------------------------------  Get the stream id.
   int id = md.frame_type(fr_type);
   if (id < 0) {
      cerr << "Channel: " << channel << " frame type: " << fr_type
	   << " not matched to an input stream. id=0 assumed." << endl;
      id = 0;
   }
   
   //--------------------------------  Add channel according to type
   if (str_type == "simdata") {	
      md.addSimulated(channel.c_str(), id);
   }
   else {
      md.addChannel(channel.c_str(), id);
   }
}

//======================================  Request all useful channels.
void
wstream_chan::addChannels(MultiDacc& md) {
   
   
   //-----------------------------------  Add data channels
   size_t nChannel = pars.channelNames.size();
   for (size_t i=0; i<nChannel; ++i) {
      bookInput(md, pars.channelNames[i], pars.frameTypes[i]);
   }

   //-----------------------------------  Add OSC channels
   if (!pars.oscFile.empty()) {
      int id = md.frame_type(pars.oscFrameType);
      if (id < 0) {
	 cerr << "Unable to find oscFrameType: " << pars.oscFrameType
	      << ". Using default frame stream." << endl;
	 id = 0;
      }
      oscList = new OperStateCondList(*(md.getDacc(id)));
      // oscList->setDebug(debugLevel);
      oscList->readConfig(pars.oscFile.c_str());
   }
   
   //-----------------------------------  Add state channels
   size_t nStates  = pars.stateNames.size();
   for (size_t i=0; i<nStates; ++i) {
      bookInput(md, pars.stateNames[i], pars.stateTypes[i]);
   }

   //-----------------------------------  Add request channels
   size_t nInject  = pars.injectionNames.size();
   for (size_t i=0; i < nInject; ++i) {
      const string& name = pars.injectionNames[i];
      if (!name.empty() && name != "NONE"){
	 bookInput(md, name, pars.injectionTypes[i]);
      }
   }

   //-----------------------------------  Print channel summary
   if (debugLevel > 1) md.list(cout);
}

//======================================  Add output files.
void
wstream_chan::addOutputFiles(woutput& wo, const std::string& fileTime,
			     const std::string& outdir) const {
   wo.addMany(pars.channelNames, fileTime, outdir,  pars.triggerFormat, 
	      pars.triggerTypes, pars.triggerFiles);
}

//======================================  Read injection data
void
wstream_chan::inject(MultiDacc& md) {
   size_t nInject = pars.injectionChannels();
   if (!nInject) return;

   wlog(debugLevel, 1, "  reading injection data");

   //-----------------------------------  Veto missing, inconsistent injections
   size_t nChan = pars.numberOfChannels();
   for (size_t chan=0; chan<nChan; chan++) {
      string& name = pars.injectionNames[chan];
      if (name.empty()) continue;

      TSeries injectionData = *md.refData(name.c_str());

      if (injectionData.empty()) {
	 error("error reading injection data");
      }

      if (injectionData.getTStep() != input_data[chan].getTStep()) {
	 error("inconsistent detector and injection data");
      }

      // Scale injection data and add it in.
      injectionData    *= pars.injectionFactors[chan];
      input_data[chan] += injectionData;
      if (debugLevel >= 1) {
	 cout <<  "    " << name << ":        loaded x " 
	      << pars.injectionFactors[chan] << endl;
      }
   }  // end of loop over channels.
}

//======================================  Calculate minimum overlap
double 
wstream_chan::minimumOverlap(void) const {

   //-----------------------------------  determine minimum block overlap
   double minimumBlockOverlap = 2 * tiling.transientDuration() + 
                                pars.extraBlockOverlap;

   if (debugLevel >=2) {
      cout << "  minimumBlockOverlap: " << minimumBlockOverlap << endl;
   }
      
   //-----------------------------------  Segment livetime lost to time shifts
   //double timeShiftLoss = maximumTimeShift - minimumTimeShift;

   //-----------------------------------  Round overlap up to nearest second
   double blockOverlap = floor(minimumBlockOverlap);
   if (blockOverlap < minimumBlockOverlap) blockOverlap += 1.0;
   return blockOverlap;
}

//======================================  Perform processing
void 
wstream_chan::process(const woutput& outputFiles, const Time& blockStartTime, 
		      Interval blockDuration, Interval blockOverlap) {
   size_t   nChans  = pars.numberOfChannels();
   double   fSample = pars.sampleFrequency;
   dble_vect coordinate;

#ifndef NO_COHERENT
   //////////////////////////////////////////////////////////////////////
   //                    transform sky coordinate                      //
   //////////////////////////////////////////////////////////////////////

   // if (the skyPosition parameter is not empty, determine coordinate
   if (!pars.skyPosition.empty()) {

      // gps center time of block
      Time blockCenterTime = blockStartTime + (blockDuration / 2);
      if (debugLevel >= 2) {
	 cout << "  block center time:       " << blockCenterTime << endl;
      }

      // convert sky coordinate into geocentric coordinate
      wconvertskycoordinates coordinate(pars.skyPosition, blockCenterTime, 
					pars.skyCoordinateSystem, "geocentric");
      if (debugLevel >= 1) {
	 cout <<  "  sky coordinate:          [" << coordinate[1] << " " 
	      << coordinate[2] << endl;
      }
   }  // if (!skyposition.empty())
#endif  

   //-----------------------------------  pre-process data
   tser_vect data;
   str_vect  channelNames;
   data.reserve(nChans);
   channelNames.reserve(nChans);

   size_t    nValid = 0;
   for (size_t chan=0; chan < nChans; chan++) {
      if (!validChannels[chan]) continue;

      data.push_back(input_data[chan].extract(blockStartTime, blockDuration));
      channelNames.push_back(strrep(pars.channelNames[chan], ";", ":"));

      if (debugLevel) {
	 cerr << "Channel " << channelNames[nValid] << " start: " 
	      << data[nValid].getStartTime() << " end: "
	      << data[nValid].getEndTime() << endl;
      }

      // test for missing detector data
      if (data[nValid].empty()) error("error reading detector data");
      nValid++;
   }
   if (!nValid) {
      wlog(debugLevel, 1, "  No valid channels found");
      return;
   }

   //-----------------------------------  resample data
   wlog(debugLevel, 1, "  resampling data");
   tser_vect resdata = resample.wresample(data, fSample, validChannels);
   if (debugLevel) {
      cerr << "Resampled channel " << channelNames[0] << " start: " 
	   << resdata[0].getStartTime() << " end: "
	   << resdata[0].getEndTime() << endl;
   }

   //------------------------------------  Condition data
   wlog(debugLevel, 1, "  conditioning data");
   wcondition conditioner(resdata, tiling, pars.doubleWhiten);

   //-----------------------------------  Transform and threshold
   Time thresholdReferenceTime = blockStartTime + pars.blockDuration*0.5;
   dble_vect thresholdTimeRange(2, 0);
   thresholdTimeRange[1] = 0.5 * (blockDuration - blockOverlap);
   thresholdTimeRange[0] = -thresholdTimeRange[1];
   dble_vect thresholdFrequencyRange;
   dble_vect thresholdQRange;

   // The original version combined the transformation and triggering.
   // I don't understand why this would be necessary except for memory 
   // management. this could be done in c++ but it certainly isn't a 
   // critical problem. If it should prove to be important at a later 
   // time, the channel-by-channel transform+thresholding can be 
   // implemented using the single channel transform and thresholding 
   // classes.
   wlog(debugLevel, 1, "  transforming data");
   dft_vect condDft, coeffs;
   conditioner.whitenedDFT(condDft);
   conditioner.coefficientDFT(coeffs);

   wtransform transforms(condDft, tiling, pars.outlierFactor, 
			 pars.analysisMode, channelNames, coeffs, 
			 coordinate);

   //  Select tiles using event (and veto) threshold(s) calculated to 
   //  give a specified rate for white noise.
   //
   //  The threshold is set to yield the specified false event rate when 
   //  applied to all available frequencies and Qs of a white noise signal.
   //  It is not modified to account for restricted f, Q ranges. It is also
   //  only a rough estimate, and the result false event rate may vary 
   //  significantly depending on the quality of the data.
   //
   wlog(debugLevel, 1, "  thresholding data");
   weventstack triggers, triggersThresholded;
   double eventThreshold = pars.eventThreshold;
   if (!eventThreshold) {
      eventThreshold = tiling.threshold_from_rate(pars.falseEventRate);
   }
   double vetoThreshold = dInf;
   if (pars.vetoThreshold > 0) {
      vetoThreshold = pars.vetoThreshold;
   } else if (pars.falseVetoRate > 0) {
      vetoThreshold = tiling.threshold_from_rate(pars.falseVetoRate);
   }

   triggersThresholded.wthreshold(transforms, tiling, 
				  eventThreshold, thresholdReferenceTime, 
				  thresholdTimeRange, thresholdFrequencyRange, 
				  thresholdQRange, pars.maximumSignificants, 
				  pars.analysisMode, vetoThreshold, 
				  pars.uncertaintyFactor, 
				  pars.correlationFactor, debugLevel);
   if (debugLevel >= 1) {
      triggersThresholded.status(cout);
   }
   if (debugLevel > 3) triggersThresholded.display(cout);

   //-----------------------------------  select significant triggers
   wlog(debugLevel, 1, "  selecting significant triggers");

   // excluding overlapping significant tiles
   triggersDownselected.wselect(triggersThresholded, 
				pars.durationInflation,
				pars.bandwidthInflation, 
				pars.maximumTriggers, 
				debugLevel);

   // replace triggers
   triggers = triggersDownselected;

   // report downselected tile numbers
   if (debugLevel >= 1) {
      triggers.status(cout);
   }

   //-----------------------------------  Veto inconsistent triggers
#ifndef NO_COHERENT
   // if veto is requested and analysis mode is coherent
   if (pars.applyVeto && pars.analysisMode == "coherent") {
      wlog(debugLevel, 1, "  applying veto");

      //-----------------------------  Apply null stream veto
      eventstack triggersVetoed;
      triggersVetoed.wveto(triggers, pars.durationInflation, 
			   pars.bandwidthInflation, 
			   vetoDurationFactor, vetoBandwidthFactor, 
			   maximumConsistents, debugLevel);

      // replace triggers
      triggers = triggersVetoed;

      // report consistent tile numbers
      if (debugLevel >= 1) {
	 triggersVetoed.status(cout);
      }
   } // end test for apply veto
#endif // !defined(NO_COHERENT)

   //-----------------------------------  Cluster triggers
   if (pars.applyClustering || pars.bayesian) {
      wlog(debugLevel, 1, "  applying clustering");

      weventstack triggersClustered = triggers;

      //---------------------------------  Density clustering
      string method = pars.clusterMethod;
      if (method == "density") {
	 clusters.wcluster(triggersClustered, pars.clusterRadius, 
			   pars.clusterDensity, pars.clusterSingles, 
			   pars.distanceMetric, pars.durationInflation, 
			   pars.bandwidthInflation, debugLevel);
      }

      //-----------------------------  Hierarchical clustering
      else if (method == "hierarchical") {
	 clusters.wcluster(triggersClustered, pars.clusterLinkage, 
			   pars.clusterCriterion, pars.clusterThreshold, 
			   pars.distanceMetric, pars.durationInflation,
			   pars.bandwidthInflation, debugLevel);
      }

      //--------------------------------  Invalid clustering method
      else {
	 error(string("Unknown cluster method: ") + method);
      }

      //--------------------------------  Replace triggers
      triggers = triggersClustered;

      //--------------------------------  Report cluster numbers
      if (debugLevel >= 1) {
	 clusters.status(cout);
      }
   } // end test for apply clustering

#ifndef NO_COINCIDENCE
   //-----------------------------------  Coincident triggers
   bool coincidentsFound = false;

   //-----------------------------------  apply coincidence if requested
   if ((pars.coincidenceNumber > 1) && (nValid > 1)) {
      wlog(debugLevel, 1, "  coinciding triggers");
	 
      // apply coincidence
      wcoincide triggersCoincident(triggers, 
				   pars.coincidenceNumber, 
				   pars.durationInflation,
				   pars.bandwidthInflation, 
				   pars.maximumCoincidents, 
				   debugLevel);

      // replace triggers
      triggers = triggersCoincident;
  
      // cull coincident clusters based on coincident triggers
      if (exist("clusters", "var")) {
	 wcoincidecluster clustersCoincident(triggersCoincident, 
					     clusters, debugLevel);
      
	 // replace clusters
	 clusters = clustersCoincident;
      }

      //--------------------------------  Note if any coincidents were found
      if (!triggersCoincident.empty()) {
	 coincidentsFound = true;
      }

      //--------------------------------  Report coincidence numbers
      if (debugLevel >= 1) {
	 for (size_t chanNumber=0; chanNumber < triggersCoincident.size();
	      chanNumber++) {
	    cout <<  "    " << triggersCoincident[chanNumber].channelName 
		 << ": " << triggersCoincident[chanNumber].size() 
		 << " coincidents" << endl;
	 }
      }

      //--------------------------------  Write coincident triggers
      if (outputFiles.enabled("COINCIDE")) {
	 wlog(debugLevel, 1, "  writing coincident triggers");

	 // write triggers or clusters
	 if (pars.applyClustering && pars.writeClusters) {
	    (clustersCoincident.writeEvents(outputFiles["COINCIDE"], 
					    pars.triggerFields, 
					    pars.triggerFormat);
	 }
	 else {
	    triggersCoincident.writeEvents(outputFiles["COINCIDE"], 
					   pars.triggerFields, 
					   pars.triggerFormat);
	 }
      }
   }    // end test for apply coincidence

   ///////////////////////////////////////////////////////////////////
   //                    EVENT FOLLOWUP
   ///////////////////////////////////////////////////////////////////

   // apply followup if requested and coincident tiles 
   // and clusters have been found
   if (coincidentsFound) {

      //--------------------------------  Create the output event structure
      event.id = "Discrete Q-transform event structure";
      event.network = buildNetworkString(channelNames);
      event.blockStartTime = blockStartTime;
      event.blockStopTime = blockStopTime;
      event.livetime = pars.blockDuration - 2*tiling.transientDuration;

      event.outputFields = {"network", "blockStartTime",
			    "blockStopTime", "livetime"};

      //--------------------------------  Find candidate event cluster
      wlog(debugLevel, 1, "  finding candidate cluster");

      wfindcandidate candidateCluster(clustersCoincident, 
				      triggersClustered,
				      eventTime, pars, debugLevel);

      event.time = candidateCluster.clusterTime;
      event.frequency = candidateCluster.clusterFrequency;
      event.duration = candidateCluster.clusterDuration;
      event.bandwidth = candidateCluster.clusterBandwidth;

      event.outputFields = {event.outputFields{:}, "time", 
			    "frequency", "duration", "bandwidth"};

#ifndef NO_FOLLOWUP
      //--------------------------------  Follow-up Analyses
      if (pars.bayesian || pars.xCoherentCheck || pars.maxEnt){
	 wlog(debugLevel, 1, "  Event followups:");
	    
	 [event, skymap] = 
	    wfollowups(event, data, coefficients, tiling, 
		       blockStartTime, coordinate, candidateCluster, 
		       channelNames, pars, outputFiles, 
		       debugLevel);
      }
#endif // !defined( NO_FOLLOWUP)

      //--------------------------------  Write event
      wlog(debugLevel, 1, "  writing event");
      wwriteevents(event, outputFiles.EVENTS, event.outputFields, 
		   pars.triggerFormat);
   } // end if (coincidentsFound)
#endif // !defined(NO_COINCIDENCE)
}

//======================================  Read data and state information.
void 
wstream_chan::read_data(MultiDacc& md, const Time& startTime, double duration) {
   size_t nChans = pars.numberOfChannels();
   if (!nChans) return;
   if (validChannels.size() < nChans) {
      error("validChannels not initialized");
   }

   //-----------------------------------  Evaluate osc conditions
   size_t nOscConds = pars.oscConditions.size();
   if (nOscConds) wlog(debugLevel, 1, "  getting OSC conditions");
   else           wlog(debugLevel, 1, "  no OSC conditions");
   for (size_t i=0; i<nOscConds; i++) {
      int inx = osc_index[i];
      if (inx < 0) {
	 if (debugLevel > 1) {
	    cout << "   channel[" << i << "] no osc requested." << endl;
         }
	 validChannels[i] = true;
      }

      //--------------------------------  Check OSC list is defined.
      else if (!oscList) {
	 error("OSC list not created!");
      }

      //--------------------------------  Was condition already evaluated?
      else if (inx < int(i)) {
	 if (debugLevel > 1) {
	    cout << "   channel[" << i << "] state already tested." << endl;
         }
	 validChannels[i] = validChannels[inx];
      }

      //--------------------------------  Fill data with the evaluated condition
      else {
	 Time tFill = md.getFillTime();
	 Interval stride = md.getFillStride();
	 size_t nsample = size_t(stride / pars.oscStride);
	 DVectI idata;
	 if (!oscList->defined(pars.oscConditions[inx])) {
	    cerr << "*** Error *** OSC condition: " << pars.oscConditions[inx]
		 << " is not defined!" << endl;
	 }
	 idata.replace_with_zeros(0, idata.size(), nsample);
	 if (oscList->satisfied(pars.oscConditions[inx])) idata += 1;
	 if (debugLevel > 1) {
	    cout << "OSC condition: " << pars.oscConditions[inx]
		 << " satisfied = " << idata[0] << " for gps " << tFill.getS()
		 << "[" << double(stride) << "]" << endl;
	 }

	 TSeries ts(tFill, Interval(pars.oscStride), idata);

	 //--------------------------------  Remove data before this stride.
	 Time tStart = osc_data[i].getStartTime();
	 if (!osc_data[i].empty() && tStart < startTime) {
	    Interval tDelete = startTime - tStart;
	    osc_data[i].eraseStart(tDelete);
	 }

	 //--------------------------------  Set the osc data series if no data
	 if (osc_data[i].empty()) {
	    osc_data[i] = ts;
	 }
	 //--------------------------------  Otherwise append
	 else if (osc_data[i].Append(ts)) {
	    error("Unaligned OSC time-series");
	 }
	 if (osc_data[i].getMinimum() == 1.0 &&
	     osc_data[i].getMaximum() == 1.0) {
	    validChannels[i] = true;
	 } else {
	    validChannels[i] = false;
	 }
      }
   }
   
   //-----------------------------------  Evaluate state data
   size_t nStateChans = pars.numberOfStates();
   std::vector<bool> state_valid(nStateChans, true);
   if (nStateChans) wlog(debugLevel, 1, "  getting state conditions");
   else             wlog(debugLevel, 1, "  no state conditions");
   for (size_t i=0; i<nStateChans; i++) {
      int inx = state_index[i];
      //--------------------------------  Test whether state was requested
      if (inx < 0) {
	 if (debugLevel > 1) {
	    cout << "   channel[" << i << "] no state requested." << endl;
         }
      }

      //--------------------------------  Was state already evaluated?
      else if (inx < int(i)) {
	 if (debugLevel > 1) {
	    cout << "   channel[" << i << "] state already tested." << endl;
         }
	 state_valid[i] = state_valid[inx];
      }

      //--------------------------------  Test specified state
      else {
	 string sname = pars.stateNames[i];
	 int mask = int(pars.stateMasks[i]);

	 //--------------------------------  Check for available data.
	 TSeries* ts = md.refData(sname.c_str());
	 if (!ts) {
	    cerr << "Data not available for channel: " << sname[i] << endl;
	    md.list(cerr);
	    throw runtime_error("wchunk: Error reading state channel");
	 }

	 //--------------------------------  Remove data before this stride.
	 Time tStart = state_data[i].getStartTime();
	 if (!state_data[i].empty() && tStart < startTime) {
	    Interval tDelete = startTime - tStart;
	    state_data[i].eraseStart(tDelete);
	 }

	 if (state_data[i].empty()) {
	    state_data[i] = *ts;
	    state_data[i].Convert(DVector::t_int);
	 } else {
	    state_data[i].Append(*ts);
	 }
	 
	 //--------------------------------  Quick check for missing or invalid 
	 //                                  state data
	 TSeries stateData=state_data[i].extract(startTime, duration);
	 if (stateData.empty()) {
	    state_valid[i] = false;
	    if (pars.errorOnStateError) {
	       error("Error reading detector state");
	    }
	 }

	 //--------------------------------  Loop over all state data and test
	 //                                  that mask bits are on.
	 else {
	    int N = stateData.getNSample();
	    const int* state_chan = 
	       dynamic_cast<const DVecType<int>*>(stateData.refDVect())->refTData();
	    for (int j=0; j<N; j++) {
	       if ((state_chan[j] & mask) != mask) {
		  if (debugLevel >= 2) {
		     cout <<  "    " << sname 
			  << ":        detector not in requested state" <<endl; 
		  }
		  state_valid[i] = false;
		  break;
	       }
	    }
	    if (state_valid[i] && debugLevel >= 2) {
               cout <<  "    " << sname << ":        state is valid "
                    << "(" << mask << " bits on)" << endl;
            }
  
	 }
      }
      validChannels[i] = validChannels[i] && state_valid[i];
   }

   //-----------------------------------  Read in the data
   for (size_t i=0; i<nChans; i++) {
      //--------------------------------  Check for available data.
      TSeries* ts = md.refData(pars.channelNames[i].c_str());
      if (!ts) {
	 cerr << "Data not available for channel: " << pars.channelNames[i]
	      << endl;
	 md.list(cerr);
	 throw runtime_error("wchunk: Error reading data");
      }
      Time tStart = input_data[i].getStartTime();
      if (!input_data[i].empty() && tStart < startTime) {
	 Interval tDelete = startTime - tStart;
	 input_data[i].eraseStart(tDelete);
      }

      //--------------------------------  Scale data and append to input buffer
      double dFact = pars.dataFactors[i];
      if (input_data[i].empty() || 
	  ts->getStartTime() > input_data[i].getEndTime()) {
	 input_data[i] = *ts;
	 if (dFact != 1.0) input_data[i] *= dFact;
      } else if (dFact != 1.0) {
	 TSeries scaled_ts = *ts;
	 scaled_ts *= dFact;
	 input_data[i].Append(scaled_ts);
      } else {
	 input_data[i].Append(*ts);
      }

      //--------------------------------  Report data loaded
      if (debugLevel >= 1) {
	 cout <<  "    " << pars.channelNames[i] << ":        loaded x "
	      <<  dFact << endl;
      }
   }
}

//======================================  Reset, ready for start of next chunk
void
wstream_chan::start_chunk(void) {
   validChannels.clear();
   size_t nChan = pars.numberOfChannels();
   validChannels.resize(nChan, true);
   if (input_data.size() != nChan) input_data.resize(nChan);

   //-----------------------------------   Check state data series
   size_t nState = pars.numberOfStates();
   if (nState > nChan) {
      error("Invalid number of state channels");
   }   
   if (state_data.size() != nState) state_data.resize(nState);

   //-----------------------------------   Check state data series
   size_t nOscCond = pars.oscConditions.size();
   if (nOscCond > nChan) {
      error("Invalid number of state channels");
   }
   osc_data.clear();
   if (nOscCond) osc_data.resize(nOscCond);
}

//======================================  Thread entry point
void*
wstream_chan::thread_entry(void) {
   return 0;
}

//======================================  Reset, ready for start of next chunk
void 
wstream_chan::validChans(str_vect& processedChannels) const {
   size_t nChan = pars.numberOfChannels();
   for (size_t i=0; i<nChan; i++) {
      if (validChannels[i]) processedChannels.push_back(pars.channelNames[i]);
   }
}
