/* -*- mode: c++; c-basic-offset: 3; -*- */
// WSEARCH Top level function for the Omega Pipeline burst search
//
// WSEARCH applies the discrete Q transform to search for statistically
// significant transient events in data from interferometric
// gravitational-wave detectors.
//
// usage: wsearch(startTime, stopTime, parameterFile, frameCacheFile, 
//                outputDirectory, debugLevel)
//
//   startTime           gps start time of analysis
//   stopTime            gps stop time of analysis
//   parameterFile       parameter file
//   frameCacheFile      readframedata formatted frame cache file
//   outputDirectory     directory to write results
//   debugLevel          verboseness level of debug output
//
// If the specified stop time is less than the specified start time, it is
// instead assumed to be the duration of the analysis.  Non-integer start
// and stop times are truncated to the nearest integer.
//
// If no output directory is specified, WSEARCH places the resulting trigger
// and event files in a subdirectory of the current directory, named after 
// the specified segment:
//
//   segments/
//     <startTime>-<stopTime>/
//       livetime.txt
//       <channelName1>.txt
//       <channelName2>.txt
//       
//
// If an outputDirectory is specified, then all output is written to that
// directory instead.
//
// If no parameter file or frame cache file is specified, WSEARCH looks
// for the files parameters.txt or framecache.txt in the current directory.
//
// The specified debugLevel controls the amount of detail in the output log.
// A debugLevel of unity is assumed by default.
//
// See also WBLOCK, WEVENT

// Authors:
// Shourov K. Chatterji <shourov@ligo.caltech.edu>
// Leo C. Stein <lstein@ligo.mit.edu>
// Jameson Graef Rollins <jrollins@phys.columbia.edu>

#include "wtypes.hh"
#include "wtile.hh"
#include "wblock.hh"
#include "wframecache.hh"
#include "woutput.hh"
#include "wparameters.hh"
#include "matlab_fcs.hh"

#include "EggTimer.hh"
#include "ParseLine.hh"

#include <iostream>
#include <iomanip>
#include <sstream>
#include <cstdlib>

//-->  The next three lines are needed if you are going to generate triggers.
//     The descriptive title in PIDTITLE should the monitor function.
#define PIDCVSHDR "$Id$"
#define PIDTITLE  "DMT Omega event search"
#include "ProcIdent.hh"

using namespace wpipe;
using namespace std;

//==================================  Search function prototype
void
wsearch(const Time& start, const Time& stop, 
	const std::string& params, const std::string& cache,
	const std::string& outdir, int debugLevel);


//==================================  Main function
void
syntax(void) {
   cerr << "Command line syntax: " << endl;
   cerr << "  wsearch <start-time> <stop-time> <par-file> <frame-cache>	\\"
	<< endl;
   cerr << "          <temp-dir> <debug-level>" << endl;
}


//==================================  Main function
//
//  Command line:
//    wsearch <start-time> <stop-time> <par-file> <frame-cache> 
//            <temp-dir> <debug-level>
//
int
main(int argc, const char* argv[]) {
   if (argc < 5) {
      cerr << "insufficient parameters:" << endl;
      syntax();
      return 1;
   }
   Time start(strtol(argv[1], 0, 0));
   Time stop(strtol(argv[2], 0, 0));
   string parFile = argv[3];
   string frameCache = argv[4];
   string outDir;
   if (argc > 5) outDir = argv[5];
   int debugLevel = 1;
   if (argc > 6) debugLevel = strtol(argv[6], 0, 0);

   try {
      wsearch(start, stop, parFile, frameCache, outDir, debugLevel);
   } catch (std::exception& e) {
      cerr << "Caught exception: " << e.what() << endl;
   }
}

//==================================  Event search function.
void
wsearch(const Time& start, const Time& stop, 
	const std::string& params, const std::string& cache,
	const std::string& outdir, int debugLevel) {

   /////////////////////////////////////////////////////////////////////////////
   //                           start analysis timer                          //
   /////////////////////////////////////////////////////////////////////////////

   // start time of analysis
   EggTimer analysisTimer;

   /////////////////////////////////////////////////////////////////////////////
   //                         parse command line arguments                    //
   /////////////////////////////////////////////////////////////////////////////
   // apply default arguments
   string parameterFile = params;
   if (params.empty()) {
      parameterFile = "parameters.txt";
   }

   string frameCacheFile = cache;
   if (frameCacheFile.empty()) {
      frameCacheFile = "framecache.txt";
   }

   string outputDirectory = outdir;

   // truncate to integer start and stop times
   Time startTime(start.getS());
   Time stopTime(stop.getS());

   // segment duration
   Interval segmentDuration = stopTime - startTime;

   // string start and stop times
   ostringstream oss;
   oss << startTime.getS();
   string startTimeString = oss.str();
   oss.str("");
   oss << stopTime.getS();
   string stopTimeString = oss.str();
   oss.str("");
   oss << setprecision(10) << segmentDuration;
   string segmentDurationString = oss.str();
   oss.str("");

   // create output path if (not specified
   if (outputDirectory.empty()) {
      oss << "./segments/" << startTimeString << "-" << stopTimeString;
      outputDirectory = oss.str();
      oss.str("");
      // path to segment output directory
   }

   // check if specified path is absolute
   else if (outputDirectory[0] != '/') {
      // make explicitly relative to current directory if (not absolute
      outputDirectory = string("./") + outputDirectory;
   }

   /////////////////////////////////////////////////////////////////////////////
   //                              write header                               //
   /////////////////////////////////////////////////////////////////////////////

   // report analysis name, time and command line arguments
   if (debugLevel >= 1) {
      cout << ">> Omega Search analysis" << endl;
      cout << "Run by " << getenv("USER") << " on " << datestr(0, 29)
	   << " at " << datestr(0, 13) << endl;
      cout << "  analysis start time:     " << startTimeString << endl;
      cout << "  analysis stop time:      " << stopTimeString << endl;
      cout << "  parameter file:          " << parameterFile << endl;
      cout << "  framecache file:         " << frameCacheFile << endl;
      cout << "  output directory:        " << outputDirectory << endl;
      cout << "  debug level:             " << debugLevel << endl;
   }

   /////////////////////////////////////////////////////////////////////////////
   //                               read parameters                           //
   /////////////////////////////////////////////////////////////////////////////

   // report status
   wlog(debugLevel, 1, "reading parameter file");

   // get parameter defaults
   wparameters defpars;
   defpars.set_defaults();
   string rcfile = subst_env("$HOME/.wstreamrc");
   if (exist(rcfile, "file")) {
      ifstream inrc(rcfile.c_str());
      if (inrc.is_open()) {
	 wlog(debugLevel, 1, string("Read default parameters from: ") + rcfile);
	 defpars.read_params(inrc, debugLevel);
	 if (debugLevel > 3) defpars.display(cout);
      }
   }
   
   rcfile = subst_env(".wstreamrc");
   if (exist(rcfile, "file")) {
      ifstream inrc(rcfile.c_str());
      if (inrc.is_open()) {
	 wlog(debugLevel, 1, string("Read default parameters from: ") + rcfile);
	 defpars.read_params(inrc, debugLevel);
	 if (debugLevel > 3) defpars.display(cout);
      }
   }

   // read parameters
   wparameters pars(parameterFile, defpars, debugLevel);
   pars.display(cout);

   /////////////////////////////////////////////////////////////////////////////
   //                              tile signal space                          //
   /////////////////////////////////////////////////////////////////////////////

   // report status
   wlog(debugLevel, 1, "tiling search space");

   // generate Q transform tiling
   double fSample = pars.sampleFrequency;
   wtile tiling(pars.blockDuration, pars.qRange, pars.frequencyRange, 
		pars.sampleFrequency, pars.maximumMismatch, 
		pars.highPassCutoff, pars.lowPassCutoff, 
		pars.whiteningDuration, pars.transientFactor, 
		pars.minAllowableIndependents, debugLevel);

   if (debugLevel >=2) {
      cout << "  whiteningDuration:   " << tiling.whiteningDuration() << endl;
      cout << "  transientDuration:   " << tiling.transientDuration() << endl;
   }

   /////////////////////////////////////////////////////////////////////////////
   //                              partition segment                          //
   /////////////////////////////////////////////////////////////////////////////

   // report status
   wlog(debugLevel, 1, "portioning segment");

   //----------------------------------  get max, min time shift (sample
   //                                    alignment forced in pars)
   double maximumTimeShift(0), minimumTimeShift(0);
   if (!pars.timeShifts.empty()) {
      maximumTimeShift = pars.timeShifts[0];
      minimumTimeShift = pars.timeShifts[0];
      size_t N = pars.timeShifts.size();
      for (size_t i=1; i<N; i++) {
	 if (pars.timeShifts[i] > maximumTimeShift) {
	    maximumTimeShift = pars.timeShifts[i];
	 } else if (pars.timeShifts[i] < minimumTimeShift) {
	    minimumTimeShift = pars.timeShifts[i];
	 }
      }
   }

   // determine minimum block overlap
   double minimumBlockOverlap = 2 * tiling.transientDuration()
      + pars.extraBlockOverlap;

   if (debugLevel >=2) {
      cout << "  minimumBlockOverlap: " << minimumBlockOverlap << endl;
   }

   // total segment livetime loss due to time shifts
   double timeShiftLoss = maximumTimeShift - minimumTimeShift;

   // if (segment duration is less than block duration
   if (double(segmentDuration) - timeShiftLoss < pars.blockDuration) {
      // write error to log file
      error("segment too short to analyze");
   }

   // number of blocks in segments
   size_t numberOfBlocks = size_t(ceil((double(segmentDuration) - timeShiftLoss 
					- minimumBlockOverlap) 
				       / (pars.blockDuration 
					  - minimumBlockOverlap)));

   // block overlap in seconds between blocks if more than one block.
   // Overlap is calculated so that the blocks are uniformly spaced so
   // that the start of the first block corresponds to the start of the
   // segment and the end of the last block corresponds to the end of the
   // segment.
   double blockOverlap = 0;
   if (numberOfBlocks > 1) {
      blockOverlap = (pars.blockDuration * numberOfBlocks
		      + timeShiftLoss - double(segmentDuration))
	 / double(numberOfBlocks - 1);
   }

   // initial blocks to analyze
   bool_vect blocksToAnalyze(numberOfBlocks, true);
   bool_vect analyzedBlocks(numberOfBlocks,  false);

   // name of livetime file
   string livetimeFile = outputDirectory + "/livetime.txt";

   // report segment information
   if (debugLevel >= 1) {
      cout << "  segment duration:    " << segmentDuration << " seconds"<< endl;
      cout << "  time shift loss:     " << timeShiftLoss   << " seconds"<< endl;
      cout << "  block duration:      " << pars.blockDuration << " seconds" 
	   << endl;
      cout << "  block overlap:       " << blockOverlap    << " seconds"<< endl;
      cout << "  number of blocks:    " << numberOfBlocks  << endl;
   }

   /////////////////////////////////////////////////////////////////////////////
   //                           create results directory                      //
   /////////////////////////////////////////////////////////////////////////////

   // check if (output directory exists
   if (!exist(outputDirectory,"dir")) {

      // report status
      wlog(debugLevel, 1, "creating output directory");

      // create output directory
      string shellcmd = "mkdir -p ";
      shellcmd += outputDirectory;
      int sysrc = system(shellcmd.c_str());
      if (sysrc == -1) {
	 cerr << "failed to fork command shell!" << endl;
      } else if (sysrc != 0) {
	 cerr << "failed to execute shell command: " << shellcmd << endl;
      }
   }

   else {

      /////////////////////////////////////////////////////////////////////////
      //                         check for analyzed blocks                   //
      /////////////////////////////////////////////////////////////////////////

      // report status
      wlog(debugLevel, 1, "identifying previously analyzed blocks");

      // read list of previously analyzed blocks
      ParseLine pl(livetimeFile.c_str());
      size_t nAnalyzed = 0;
      if (pl.isOpen()) {
	 while (pl.getLine() >= 0) {
	    if (pl.getCount() == 0) continue;
	    size_t blkid = pl.getInt(0);
	    if (blkid < numberOfBlocks) {
	       analyzedBlocks[blkid] = true;
	       nAnalyzed++;
	    }
	 }
      }

      // report number of previously analyzed blocks
      if (debugLevel >= 1) {
	 cout << "  analyzed blocks:     " << nAnalyzed << "blocks" << endl;
      }
   }

   ////////////////////////////////////////////////////////////////////////////
   //                              output file paths                         //
   ////////////////////////////////////////////////////////////////////////////

   // time string
   string fileTimeString = startTimeString + "-" + segmentDurationString;

   // generate output file path
   woutput outputFiles(pars.channelNames, fileTimeString, outputDirectory, 
		       pars.triggerFormat, pars.triggerTypes, 
		       pars.triggerFiles);

   ////////////////////////////////////////////////////////////////////////////
   //                            read frame file cache                       //
   ////////////////////////////////////////////////////////////////////////////
   wlog(debugLevel, 1, "reading frame cache");

   // read frame file cache
   wframecache frameCache(frameCacheFile);
   wblock block;

   ////////////////////////////////////////////////////////////////////////////
   //                              begin block loop                          //
   ////////////////////////////////////////////////////////////////////////////

   // initialize error flag
   int errorFlag = 0;

   //  Get a boundary on an even number of nanoseconds (for time specification)
   double boundary = gcd(fSample, 1000000000);

   // initialize coordinate variable
   dble_vect coordinate;
   // begin loop over blocks
   for (size_t blockNumber=0; blockNumber < numberOfBlocks; blockNumber++) {
      if (!blocksToAnalyze[blockNumber]) continue;

      // start time of block analysis
      EggTimer blockTimer;

      // block start time offset
      double blockDelta = double(blockNumber)*(pars.blockDuration-blockOverlap)
	                + maximumTimeShift;

      // force start time alignment with data samples
      //blockDelta = floor(blockDelta * fSample) / fSample;
      blockDelta = floor(blockDelta * boundary) / boundary;

      Time blockStartTime = startTime + blockDelta;

      // stop time of block
      Time blockStopTime = blockStartTime + pars.blockDuration;

      // report status
      if (debugLevel >= 1) {
	 cout << "analyzing block " << blockNumber+1 << " of " << numberOfBlocks
	      << endl; 
      }

      /////////////////////////////////////////////////////////////////////////
      //                            block analysis                           //
      /////////////////////////////////////////////////////////////////////////

      // try block and catch any errors
      str_vect channelNames;
      try {
	 block.process(blockStartTime, tiling, Time(0), pars, frameCache, 
		       outputFiles, debugLevel, channelNames);
      }

      catch (std::exception& e) {

	 if (debugLevel >= 1) {
	    cerr << "ERROR: " << e.what() << endl;
	 }

	 // note error, display, and reset
	 errorFlag = errorFlag + 1;

	 // continue to next block if (any errors found
	 continue;
      }

      /////////////////////////////////////////////////////////////////////////
      //                          report block time                          //
      /////////////////////////////////////////////////////////////////////////

      // report status
      double blockClockTime = blockTimer.elapsed();
      if (debugLevel >= 1) {
	 cout << "  block complete" << endl;
	 cout << "    elapsed time:          " << blockClockTime << " seconds"
	      << endl;

      }

      /////////////////////////////////////////////////////////////////////////
      //                            write livetime                           //
      /////////////////////////////////////////////////////////////////////////

      // report status
      wlog(debugLevel, 1, "  writing livetime");

      // open livetime file for appending
      FILE* livetimeFileFID = fopen(livetimeFile.c_str(), "at");

      // test for error
      if (livetimeFileFID == NULL) {
	 error("cannot open livetime file for writing");
      }

      // analyzed channels string
      string analyzedChannels;
      for (size_t channelNumber=0; channelNumber < channelNames.size(); 
	   channelNumber++) {
	 if (channelNumber != 0) analyzedChannels += ",";
	 analyzedChannels += channelNames[channelNumber];
      }

      // write block livetime to livetime file
      fprintf(livetimeFileFID, "%04zd %#020.9f %#020.9f %s %5.2f\n", 
	      blockNumber, 
	      (blockStartTime + tiling.transientDuration()).totalS(), 
	      (blockStopTime  - tiling.transientDuration()).totalS(), 
	      analyzedChannels.c_str(), 
	      blockClockTime);

      // close livetime file
      fclose(livetimeFileFID);

      /////////////////////////////////////////////////////////////////////////
      //                          end block loop                             //
      /////////////////////////////////////////////////////////////////////////

      // end loop over blocks
   }

   ////////////////////////////////////////////////////////////////////////////
   //                              write footer                              //
   ////////////////////////////////////////////////////////////////////////////

   // report status
   if (debugLevel >= 1) {
      cout << "segment complete" << endl;
      cout << "  elapsed time:            " << analysisTimer.elapsed() 
	   << " seconds" << endl;
   }

   ////////////////////////////////////////////////////////////////////////////
   //                                 return                                 //
   ////////////////////////////////////////////////////////////////////////////

   // exit with error status
   if (errorFlag) error("analysis had errors.");
}
