/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef WPIPE_WOUTPUT_HH
#define WPIPE_WOUTPUT_HH
#include "wtypes.hh"

class Time;
#include "frame_name.hh"
#include <map>
#include <iosfwd>

namespace wpipe {

   /**  The %wouttype class holds a list of channels names and a file path 
     *  for each channel. The file paths are generated for each channel
     *  named in the constructor. There are two forms of constructor, and 
     *  and update methods. In the first form, the time is given as a string
     *  and the file path is generated in the form:
     *  \verbatim
         <dir>/<ifo>-<type>-<time-string>.<fmt> \endverbatim
     *
     *  In the second form, the user may specify an arbitrtary pattern string
     *  which is then used to generate a file path. See the frame_name() 
     *  function for the path generation rules.
     */
   class wouttype {
   public:
      /**  Construct an empty output type instance.
        *  \brief Defalt constructor
	*/
      wouttype(void);

      /**  Construct a new format output type
        *  \brief Construct a type 2 output file type.
	*  \param channels Vector of channel name strings covered by this
	*         output format specifier.
	*  \param time     %Time to be used in the traditional format.
	*  \param duration Duration of the file.
	*  \param prefix   File prefix, e.g. "H-Omega_Triggers"
	*  \param dir      Directory name pattern string. This may contain 
	*                  frame_name escape characters to include gps times.
	*  \param fmt      Output file format ("xml" or "txt").
	*  \param type     Output data type (e.g. DOWNSELECT, CLUSTERS, EVENTS)
	*/
      wouttype(const str_vect& channels, const Time& time, int duration, 
	       const std::string& prefix, const std::string& dir,
	       const std::string& fmt, const std::string& type);

      /**  Destroy an output type descriptor.
       */
      ~wouttype(void);
      
      /**  Add channels to an output-type descriptor.
       */
      void addChannels(const str_vect& channels, const Time& time, int dt=-1);

      void addChannels(const str_vect& channels, const std::string& time);

      /**  Print out a single output type descriptor.
       */
      std::ostream& display(std::ostream& out) const;

      /**  Look for the output type corresponding to a channel name
       */
      size_t find(const std::string& chan) const;
      const std::string& path(const std::string& chan) const;
      const std::string& path(size_t i) const;
      size_t size(void) const;
      void update(const Time& time, int dt);
   private:
      std::string gen_path(const std::string& chan, 
			   const std::string& time) const;
      std::string gen_path(const std::string& chan, 
			   const Time& time, int dt) const;
   private:
      str_vect channelNames;
      str_vect _paths;
      frame_name  pattern;
      std::string trig_type;
   };

   /**  Maintain a list of output file names defined for multiple output 
     *  files defined for this analysis.
     */
   class woutput {
   public:
      woutput(void);
      woutput(const str_vect& channels, const std::string& time, 
	      const std::string& dir, const std::string& fmt, 
	      const str_vect& types, const str_vect& pfx);
      ~woutput(void);
      void addMany(const str_vect& channels, const std::string& time, 
		   const std::string& dir, const std::string& fmt, 
		   const str_vect& types, const str_vect& pfx);
      void addtype(const str_vect& channels, const Time& time, int dt,
		   const std::string& dir, const std::string& fmt, 
		   const std::string& type, const std::string& prefix);
      std::ostream& display(std::ostream& out) const;
      bool enabled(const std::string& name) const;
      const wouttype& operator[](const std::string& name) const;
      wouttype& operator[](const std::string& name);
      void set_ifo(const std::string& ifo);
      void update(const std::string& time, const str_vect& types);
      void update(const str_vect& types, const Time& t0, int dt);
    private:
      typedef std::map<std::string, wouttype> output_map;
      typedef output_map::value_type     omap_value;
      typedef output_map::iterator       omap_iter;
      typedef output_map::const_iterator const_omap_iter;
      std::string site_pfx;
      output_map out_map;
   };

   //======================================  Inline accessors
   inline const std::string&
   wouttype::path(size_t i) const {
      return _paths[i];
   }

   inline const std::string&
   wouttype::path(const std::string& chan) const {
      static std::string null;
      size_t i = find(chan);
      if (i == size()) return null;
      return _paths[i];
   }

   inline size_t
   wouttype::size(void) const {
      return _paths.size();
   }

   inline bool
   woutput::enabled(const std::string& name) const {
      return out_map.find(name) != out_map.end();
   }

   inline const wouttype&
   woutput::operator[](const std::string& name) const {
      return out_map.find(name)->second;
   }

   inline wouttype&
   woutput::operator[](const std::string& name) {
      return out_map.find(name)->second;
   }

}  // namespace wpipe

#endif // !defined(WPIPE_WOUTPUT_HH)
