/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "wfigure.hh"
#include "TCanvas.h"
#include "DVecType.hh"
#include "Hanning.hh"
#include "WelchPSD.hh"
#include <iostream>
#include <fstream>
#include <sstream>
#include <cstdlib>

#define ASYNCH_THUMB_BUILD 1

#ifdef   ASYNCH_THUMB_BUILD
#include <signal.h>
#include "PConfig.h"
#include <unistd.h>
#endif

using namespace wpipe;
using namespace std;


#ifdef ASYNCH_THUMB_BUILD
//======================================  Asynchronous shell command execution.
static int
frog(const string& s) {
#if defined (P__DARWIN)
    sigignore(SIGCHLD);
#else
    sigignore(SIGCLD);
#endif

    pid_t pid = fork();
    if (pid != 0) {
	if (pid == -1) perror("fork failed");
	else           pid = 0;
	return pid;
    }
    string a(s);
    a += char(0);
    const char* argv[32];
    argv[0] = 0;
    char* p = &a[0];
    for (int i=0; i<31 && *p; ) {
        while (*p == ' ') *p++ = 0;
	if (*p) argv[i] = p;
	while (*p && *p != ' ') p++;
	argv[++i] = 0;
    }
    exit(execvp(argv[0], const_cast<char* const*>(argv)));
}
#endif

//======================================  figure constructor
wfigure::wfigure(void) 
    : _wx(21), _wy(15), _format(".png")
{
}

//======================================  clear figure
void 
wfigure::clear(void) {
    if (graphic()) {
	_plot.new_plot();
    } else {
	_data.clear();
    }
}

//======================================  Convert data to requested units
double
wfigure::convertUnits(double NormE) const {
    if (_zUnits.empty() || _zUnits == "NormE") {
	return NormE;
    }
    else if (_zUnits == "SNR") {
	double snrsq = NormE * 2.0 - 1.0;
	if (snrsq > 0) return sqrt(snrsq);
	else           return 0; 
    }
    else {
	cerr << "Undefined z (color) axis units." << endl;
	return NormE;
    }
}

//======================================  Convert TSeries to requested units
void 
wfigure::convertUnits(TSeries& tsNormE) const {
    if (_zUnits.empty() || _zUnits == "NormE") {
	return;
    }
    else if (_zUnits == "SNR") {
	DVectD& dvec = *dynamic_cast<DVectD*>(tsNormE.refDVect());
	size_t N = dvec.size();
	for (size_t i=0; i<N; i++) {
	    double snrsq = dvec[i] * 2.0 - 1.0;
	    if (snrsq > 0.0) dvec[i] = sqrt(snrsq);
	    else             dvec[i] = 0; 
	}
    }
    else {
	cerr << "Undefined z (color) axis units." << endl;
    }
}

//======================================  test for graphic output
bool
wfigure::graphic(void) const {
    return (_format != ".txt"  &&
	    _format != ".json" &&
	    _format != ".xml"  &&
	    _format != ".h5");
}

//======================================  open figure
void 
wfigure::open(void) {
    if (graphic()) {
	_plot.set_canvas(new TCanvas("w-pipeline"), true);
    }
}

//======================================  set file format
void 
wfigure::read_palette(const std::string& palette) {
    _plot.read_palette(palette);
}

//======================================  set canvas size
void 
wfigure::set_size(int nw, int nh) {
    _wx = nw;
    _wy = nh;
}

//======================================  set smoothing
void
wfigure::set_smooth(int n, const string& opt) {
    _plot.set_smooth(n, opt);
}

//======================================  set color palette
void 
wfigure::set_palette(const std::string& palette) {
    _plot.set_palette(palette);
}

//======================================  set z-axis (color scale) units.
void 
wfigure::set_zunits(const std::string& units) {
    _zUnits = units;
}


//======================================  set file format
void 
wfigure::set_format(const std::string& fmt) {
    _format = fmt;
    if (_format.empty()) {
	_format = ".png";
    }
    else if (_format[0] != '.') {
	_format.insert(0, ".");
    }
}

//======================================  plot an amplitude spectrogram
void 
wfigure::asdspectrogram(const TSeries& data, Interval dT,
			const dble_vect& fRange,
			const std::string& channelName) {
    Hanning hw;
    WelchPSD w(Interval(1.0), 0.50, &hw);
    asdspectrogram(data, dT, fRange, channelName, w);
}

//======================================  plot an amplitude spectrogram
void 
wfigure::asdspectrogram(const TSeries& data, Interval dT, 
			const dble_vect& fRange, const string& title,
			const WelchPSD& psdest) {
    Interval tStep = data.getTStep();
    double   fNy   = 0.5/double(tStep);
    double fMax = fRange[1];
    if (fMax > fNy) fMax = fNy;
    
    _plot.set_color(600);
    string yttl = "Frequency [Hz]";
    _plot.ylabel(yttl);
    double tink = _plot.xTimeScale(dT, "Time");
    _plot.title(title);
    _plot.spectrogram(data, dT, fRange[0], fMax, &psdest);
}

//======================================  plot a time series
void 
wfigure::wtimeseries(const TSeries& ts, const Time& centerTime, 
		     Interval before, Interval after, 
		     const std::string& channelName) {
    Time tStart(centerTime + before);
    Interval dT = after - before;
    const TSeries tsPlot(ts.extract(tStart, dT));
    string units = tsPlot.getUnits();

    //----------------------------------  Make a plot for graphic output type
    if (graphic()) {
	_plot.set_color(600);
	string yttl = "Amplitude";
	if (!units.empty()) {
	    yttl += " [";
	    yttl += units + "]";
	}
	_plot.ylabel(yttl);
	double tink = _plot.xTimeScale(dT, "Time");
	_plot.plot(*(tsPlot.refDVect()), before/tink, 
		   ts.getTStep()/tink, channelName);
    }

    //----------------------------------  Use plot_data otherwise.
    else {
	_data.set_title(ts.getName(), "Generated by wpipe::plot_data");
	_data.add_param("eventTime", centerTime.totalS(), "");
	_data.add_coord("timeOffset", "s");
	_data.add_coord("data", units);
	DVectD tsdata(*tsPlot.refDVect());
	size_t N = tsdata.size();
	_data.reserve(N);
	dble_vect dv(2,0);
	for (size_t i=0; i<N; i++) {
	    dv[0] = double(tsPlot.getBinT(i) - centerTime);
	    dv[1] = tsdata[i];
	    _data.fill_row(dv);
	}
    }
}

#if 0
//======================================  plot a spectrogram
void 
wfigure::wspectrogram(const qTransform& transform, const wtile& tiling, 
		      const Time& centerTime, const dble_vect& timeRange,
		      const dble_vect& frequencyRange, const dble_vect& qRange,
		      const dble_vect& plotNormalizedEnergyRange, 
		      long horizontalResolution) {
}

//======================================  plot an eventgram
void 
wfigure::weventgram(const weventlist& significants, const wtile& tiling, 
		    const Time& centerTime, const dble_vect& tRange,
		    const dble_vect& plotFrequencyRange, 
		    double plotDurationInflation, 
		    double plotBandwidthInflation, 
		    const dble_vect& plotNormalizedEnergyRange) {
}
#endif

//======================================  print figure
void 
wfigure::wprintfig(const std::string& figureBase, bool thumb) const {
    std::string file = figureBase + _format;
    if (!graphic()) {
	_data.write(file, _format);
	return;
    }
    _plot.set_size(_wx, _wy);
    _plot.print(file);
    if (thumb) {
	string thumb = figureBase + ".thumb.png";
	ostringstream cmd;
	cmd << "convert -format png -resize 320x240 -strip -depth 8 " 
	    << file << " " << thumb;
#ifdef ASYNCH_THUMB_BUILD
	if (frog(cmd.str()))
#else
	    if (system(cmd.str().c_str()))
#endif
		{
		    cerr << "failed to create thumbnail with command: " << cmd.str()
			 << endl;
		}
    }
}

//======================================================================
//
//        plot_data class
//
//======================================================================

//======================================  Default (empty) constructor
plot_data::plot_data(void) {
}
 
//======================================  Constructor
plot_data::plot_data(const std::string& name, const std::string& comment)
    :  _name(name), _comment(comment)
{}

//======================================  Add a parameter
void 
plot_data::add_param(const std::string& name, double value,
		     const std::string& units) {
    _param_name.push_back(name);
    _param_units.push_back(units);
    _param_value.push_back(value);
}

//======================================  Constructor
void 
plot_data::add_coord(const std::string& name, const std::string& units) {
    _titles.push_back(name);
    _units.push_back(units);
}

//======================================  Clear the data
void 
plot_data::clear(void) {
    _name.clear();
    _comment.clear();
    _param_name.clear();
    _param_value.clear();
    _param_units.clear();
    _titles.clear();
    _units.clear();
    _data.clear();
}

//======================================  Insert one row of data.
void 
plot_data::fill_row(const dble_vect& data) {
    size_t N = data.size();
    if (_titles.size() != N) {
	throw runtime_error("plot_data: dimension mismatch");
    }
    _data.insert(_data.end(), data.begin(), data.end());
}

//======================================  Reserve space for N rows
void 
plot_data::reserve(size_t nRows) {
    _data.reserve(nRows * _titles.size());
}
 
//======================================  Set the data title
void
plot_data::set_title(const std::string& name, const std::string& comment) {
    _name = name;
    _comment = comment;
}

//======================================  write out the plot data
void 
plot_data::write(const std::string& file, const std::string& format) const {
    if (format == ".txt") {
	ofstream out(file.c_str());
	out << "#" << endl;
	out << "#  Plot name: " << _name << endl;
	out << "#  Comment:   " << _comment << endl;
	out << "#" << endl;
	out << "#  Parameters:" << endl;
	for (size_t i=0; i < _param_name.size(); i++) {
	    out << "#  " << _param_name[i] << " = " << _param_value[i];
	    if (!_param_units[i].empty()) out << " [" << _param_units[i] << "]";
	    out << endl;
	}
	out << "#" << endl;
	out << "#  Columns:" << endl;
	size_t N =  _titles.size();
	for (size_t i=0; i < N; i++) {
	    out << "#  " << i+1 << "  " << _titles[i];
	    if (!_units[i].empty()) out << " [" << _units[i] << "]";
	    out << endl;
	}
	out << "#" << endl;
	for (size_t i=0; i < _data.size(); i+=N) {
	    for (size_t j=0; j<N; j++) {
		if (j) out << "  ";
		out << _data[i+j];
	    }
	    out << endl;
	}
    }

    //==================================  Unknown output format.
    else {
	string msg = "plot_data: Unknown data format: ";
	msg += format;
	throw runtime_error(msg);
    }
}
