/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef WPIPE_PARAM_LIST_HH
#define WPIPE_PARAM_LIST_HH
#include "wtypes.hh"
#include <map>
#include <iosfwd>
#include <stdexcept>

class Time;

namespace wpipe {

  /**  The param_list class is used by the dmt wpipe programs to read 
    *  read and maintain a list of parameters. The parameter values are 
    *  parsed as for matlab inputs.
    *  \version 2.0; $Id$
    *  \author John Zweizig <john.zweizig@ligo.org>
    */
   class param_list {
   public:
      class par_def {
      public:
	 /**  Enumerated list of parameter data types.
	   */
 	 enum par_type {
	    tNull,
	    tBool,        ///< Boolean data type
	    tInt,         ///< Integer data type
	    tLong,
	    tDouble,      ///< Double data type
	    tString,      ///< String data type
	    tTime,        ///< %Time data type
	    tDoubleVect,  ///< Numeric (double) vector data type
	    tStringVect   ///< String vector data type
	 };

	 /** Parameter value pointer union
	   */
 	 union par_pointer {
	    void*     v;
	    bool*     b;
	    char*     c;
	    uint32_t* i;
	    uint64_t* l;
	    double*   d;
	    std::string* s;
	    Time*     t;
	    dble_vect* dv;
	    str_vect* sv;
	 };

	 par_def(void);
    	 par_def(par_type t, void* p);
	 ~par_def(void);
	 void clear(void);
	 std::string display(void) const;
	 const void* par_addr(void) const;
	 void set(const void* val);
	 void set(const std::string& val);
	 par_type type(void) const;
      private:
	 par_type    _type;
	 par_pointer _addr;
      };

      /// Parameter map type definition
      typedef std::map<std::string, par_def> par_map_type;
      /// Parameter map iterator type definition
      typedef par_map_type::iterator par_map_iter;
      /// Parameter map cont iterator type definition
      typedef par_map_type::const_iterator const_par_map_iter;

   public:
      /**  Construct an empty parameter list.
        *  \brief Default constructor.
	*/
      param_list(void) {}

      /**  Destroy the parameter list.
        *  \brief Destructor.
	*/
      ~param_list(void) {}

      /**  Add a boolean parameter and set it to the default value.
        *  \brief Add a boolean parameter.
	*  \param name New parameter name string.
	*  \param var  Initial parameter boolean value.
	*/
      void add_param(const std::string& name, bool& addr);
      void add_param(const std::string& name, bool& addr, const bool& value);

      /**  Add an integer parameter and set it to the default value.
        *  \brief Add an integer parameter.
	*  \param name New parameter name string.
	*  \param var  Initial parameter integer value.
	*/
      void add_param(const std::string& name, int32_t& addr);
      void add_param(const std::string& name, int32_t& addr,
		     const int32_t& value);

      void add_param(const std::string& name, int64_t& addr);
      void add_param(const std::string& name, int64_t& addr,
		     const int64_t& value);

      /**  Add a double parameter and set it to the default value.
        *  \brief Add a double parameter.
	*  \param name New parameter name string.
	*  \param var  Initial parameter double value.
	*/
      void add_param(const std::string& name, double& addr);
      void add_param(const std::string& name, double& addr,
		     const double& value);

      /**  Add a string parameter and set it to the default value.
        *  \brief Add a string parameter.
	*  \param name New parameter name string.
	*  \param var  Initial parameter string value.
	*/
      void add_param(const std::string& name, std::string& addr);
      void add_param(const std::string& name, std::string& addr,
		     const std::string& value);

      /**  Add a time parameter and set it to the default value.
        *  \brief Add a time parameter.
	*  \param name New parameter name string.
	*  \param var  Initial parameter time value.
	*/
      void add_param(const std::string& name, Time& addr);
      void add_param(const std::string& name, Time& addr, const Time& value);

      /**  Add a double vector parameter and set it to the default value.
        *  \brief Add a double vector parameter.
	*  \param name New parameter name string.
	*  \param var  Initial parameter double vector value.
	*/
      void add_param(const std::string& name, dble_vect& addr);
      void add_param(const std::string& name, dble_vect& addr,
		     const dble_vect& value);

      /**  Add a string vector parameter and set it to the default value.
        *  \brief Add a string vector parameter.
	*  \param name New parameter name string.
	*  \param var  Initial parameter string vector value.
	*/
      void add_param(const std::string& name, str_vect& addr);
      void add_param(const std::string& name, str_vect& addr,
		     const str_vect& value);

      /** Internal parameter addition fmethod.
        */
      void add_param(const std::string& name, par_def::par_type t, void* addr);

      /**  Assign a parameter to a value as specified by a string of the form:
        *  \c "\<param\>=\<value\>". If the parameter doesn't exist, an 
	*  exception is thrown.
	*  \brief Assign a parameter
	*  \param str Assignment strin.
	*/
      void assign_param(const std::string& str, const char* delim="=");

      /**  Reset the value of the specified parameter as appropriate to the
        *  parameter data type.
	*  \brief Clear a named parameter.
	*  \param name Parameter name string.
	*/
      void clear_param(const std::string& name);

      
      /**  Reset the values of all parameters as appropriate to the
        *  parameter data type.
	*  \brief Clear all parameters.
	*/
       void clear_all(void);
      
      /**  Copy the values of all identical (name and type) parameters in 
        *  the argument list into the current list.
        *  \brief Copy parameters.
	*  \param pars Parameter list to be copied.
	*/
      void copy(const param_list& pars);
		

      /**  Print a list of parameters to the specified ostream. Each parameter 
        *  is listed on a single line with the parameter name followed by the 
	*  parameter value. Each line can optionally be prefixed by a constant 
	*  string as specified by the \a pfx argument.
	*  \brief Display parameter list.
	*  \param out Ouput stream to which the parameter list is displayed.
	*  \param pfx Optional prefix string for each line.
	*  \return Output stream reference.
	*/
      virtual std::ostream& display(std::ostream& out,
				    const std::string& pfx="") const;

      /**  Test whether a named parameter is defined.
        *  \brief Test if parameter exists.
	*  \param name Parameter name string.
	*  \return True if the parameter isdefined.
	*/
      bool exists(const std::string& name) const;

      /** Copy contents of a parameter list.
       */
      param_list& operator=(const param_list& rhs);

      par_def& ref_par(const std::string& name);
      const par_def& ref_par(const std::string& name) const;

      /**  Set the named parameter to the value as specified by the \a value
        *  argument. Vector values strings are parsed as paer matlab.
	*  \brief Set the named parameter to a value specified by a string.
	*  \param name  Name of parameter to be set.
	*  \param value Value string
	*/
      void set_param(const std::string& name, const std::string& value);
   private:
      /**  Don't let ayone do a copy... The pointers will be all wrong.
        */
      param_list(const param_list& x) {}
   public:
      par_map_type  _map;
   };

   //======================================  Inline methods
   inline param_list::par_def::par_type 
   param_list::par_def::type(void) const {
      return _type;
   }
   
   inline const void* 
   param_list::par_def::par_addr(void) const {
      return _addr.v;
   }

   inline bool
   param_list::exists(const std::string& name) const {
      return _map.find(name) != _map.end();
   }

   inline param_list::par_def&
   param_list::ref_par(const std::string& name) {
      par_map_iter p = _map.find(name);
      if (p == _map.end()) {
	 throw std::runtime_error(std::string("Undefined parameter: ") + name);
      }
      return p->second;
   }

   inline const param_list::par_def&
   param_list::ref_par(const std::string& name) const {
      const_par_map_iter p = _map.find(name);
      if (p == _map.end()) {
	 throw std::runtime_error(std::string("Undefined parameter: ") + name);
      }
      return p->second;
   }

}

#endif // !defined(PARAM_LIST_HH)
