/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "param_list.hh"
#include "matlab_fcs.hh"
#include <iomanip>
#include <sstream>

using namespace std;
using namespace wpipe;

//======================================  Construct a parameter definition
param_list::par_def::par_def(void)
   : _type(tNull)
{
   _addr.v = 0;   
}

param_list::par_def::par_def(par_type t, void* p)
   : _type(t)
{
   _addr.v = p;
}

//======================================  Destroy a parameter definition
param_list::par_def::~par_def(void) {
}

//======================================  Set a parameter from a string
void
param_list::assign_param(const std::string& str, const char* delim) {
   string tmp = deblank(str);
   size_t ieql = tmp.find_first_of(delim);
   if (ieql == string::npos) {
      throw runtime_error("assign_param: No delimiter found in string");
   }
   string param = deblank(tmp.substr(0, ieql));
   if (!exists(param)) {
      throw runtime_error(string("assign_param: Unknown parameter name:")
			  + param);
   }
   string value = deblank(tmp.substr(ieql+1));
   set_param(param, value);
}

//======================================  Set a parameter from a string
void 
param_list::par_def::clear(void) {
   switch (_type) {
   case tNull:
      throw std::runtime_error("Unable to clear undefined parameter");
      break;
   case tBool:
      *_addr.b = false;
      break;
   case tInt:
      *_addr.i = 0;
      break;
   case tLong:
      *_addr.l = 0;
      break;
   case tDouble:
      *_addr.d = 0.0;
      break;
   case tString:
      _addr.s->clear();
      break;
   case tTime:
      *_addr.t = Time(0);
      break;
   case tDoubleVect:
      _addr.dv->clear();
      break;
   case tStringVect:
      _addr.sv->clear();
      break;
   }
}

//======================================  Display parameter value
void 
param_list::copy(const param_list& pars) {
   for (const_par_map_iter i=pars._map.begin(); i != pars._map.end(); i++) {
      par_map_iter j = _map.find(i->first);
      if (j != _map.end() && j->second.type() == i->second.type()) {
	 j->second.set(i->second.par_addr());
      }
   }
}

//======================================  Display parameter value
std::string 
param_list::par_def::display(void) const {
   ostringstream out;
   switch (_type) {
   case tNull:
      out << "Parameter not defined";
      break;
   case tBool:
      out << string(boolstr_tf(*_addr.b));
      break;
   case tInt:
      out << *_addr.i;
      break;
   case tLong:
      out << *_addr.l;
      break;
   case tDouble:
      if (*_addr.d == long(*_addr.d)) out << long(*_addr.d);
      else                            out << *_addr.d;
      break;
   case tString:
      out << "\"" << *_addr.s << "\"";
      break;
   case tTime:
      out << fixed << setprecision(3) << _addr.t->totalS();
      break;
   case tDoubleVect:
      wpipe::display(*_addr.dv, out);
      break;
   case tStringVect:
      wpipe::display(*_addr.sv, out);
      break;
   default:
      throw std::runtime_error("Invalid parameter type code");
   }
   return out.str();
}

//======================================  Set a parameter to a binary value.
void 
param_list::par_def::set(const void* val) {
   par_pointer value;
   value.v = const_cast<void*>(val);
   switch (_type) {
   case tNull:
      throw std::runtime_error("Unable to set undefined parameter");
      break;
   case tBool:
      *_addr.b = *value.b;
      break;
   case tInt:
      *_addr.i = *value.i;
      break;
   case tLong:
      *_addr.l = *value.l;
      break;
   case tDouble:
      *_addr.d = *value.d;
      break;
   case tString:
      *_addr.s = *value.s;
      break;
   case tTime:
      *_addr.t = *value.t;
      break;
   case tDoubleVect:
      *_addr.dv = *value.dv;
      break;
   case tStringVect:
      *_addr.sv = *value.sv;
      break;
   }
}

//======================================  Set a parameter from a string
void 
param_list::par_def::set(const std::string& val) {
   const char* p = val.c_str();
   double d = 0;
   switch (_type) {
   case tNull:
      throw std::runtime_error("Unable to set undefined parameter");
      break;
   case tBool:
      if (*p >= '0' && *p <= '9') {
	 *_addr.b = (strtol(p, 0, 0) != 0);
      } else if (tolower(val) == "true" || tolower(val) == "yes") {
	 *_addr.b = true;
      } else {
	 *_addr.b = false;
      }
      break;
   case tInt:
      *_addr.i = strtol(p, 0, 0);
      break;
   case tLong:
      *_addr.l =  strtol(p, 0, 0);
      break;
   case tDouble:
      *_addr.d =  strtod(p, 0);
      break;
   case tString:
      *_addr.s =  unquote(val);
      break;
   case tTime:
      d = strtod(p, 0);
      *_addr.t =  Time(long(d), long((d - long(d)) * 1e9 + 0.5));
      break;
   case tDoubleVect:
      *_addr.dv = wpipe::eval(val);
      break;
   case tStringVect:
      *_addr.sv = wpipe::eval_str(val);
      break;
   }
}

//===================================  Add bool parameter
void 
param_list::add_param(const std::string& name, bool& addr) {
   add_param(name, par_def::tBool, &addr);
}

//===================================  Add bool parameter
void 
param_list::add_param(const std::string& name, bool& addr, const bool& value) {
   add_param(name, par_def::tBool, &addr);
   _map[name].set(&value); 
}

//===================================  Add int parameter
void 
param_list::add_param(const std::string& name, int32_t& addr) {
   add_param(name, par_def::tInt, &addr);
}

//===================================  Add int parameter
void 
param_list::add_param(const std::string& name, int32_t& addr,
		      const int32_t& value)  {
   add_param(name, par_def::tInt, &addr);
   _map[name].set(&value); 
}

//===================================  Add long parameter
void 
param_list::add_param(const std::string& name, int64_t& addr) {
   add_param(name, par_def::tLong, &addr);
}

//===================================  Add long parameter
void 
param_list::add_param(const std::string& name, int64_t& addr,
		      const int64_t& value)  {
   add_param(name, par_def::tLong, &addr);
   _map[name].set(&value); 
}

//===================================  Add double parameter
void 
param_list::add_param(const std::string& name, double& addr)  {
   add_param(name, par_def::tDouble, &addr);
}

//===================================  Add double parameter
void 
param_list::add_param(const std::string& name, double& addr,
		      const double& value)  {
   add_param(name, par_def::tDouble, &addr);
   _map[name].set(&value); 
}

//===================================  Add double parameter
void 
param_list::add_param(const std::string& name, std::string& addr) {
   add_param(name, par_def::tString, &addr);
}

//===================================  Add string parameter
void 
param_list::add_param(const std::string& name, std::string& addr,
		      const std::string& value) {
   add_param(name, par_def::tString, &addr);
   _map[name].set(&value); 
}

//===================================  Add double parameter
void 
param_list::add_param(const std::string& name, Time& addr) {
   add_param(name, par_def::tTime, &addr);
}

//===================================  Add string parameter
void 
param_list::add_param(const std::string& name, Time& addr, const Time& value) {
   add_param(name, par_def::tTime, &addr);
   _map[name].set(&value); 
}

//===================================  Add double vector parameter
void 
param_list::add_param(const std::string& name, dble_vect& addr) {
   add_param(name, par_def::tDoubleVect, &addr);
}

//===================================  Add double vector parameter
void 
param_list::add_param(const std::string& name, dble_vect& addr,
		      const dble_vect& value) {
   add_param(name, par_def::tDoubleVect, &addr);
   _map[name].set(&value); 
}

//===================================  Add string vector parameter
void 
param_list::add_param(const std::string& name, str_vect& addr) {
   add_param(name, par_def::tStringVect, &addr);
}

//===================================  Add string vector parameter
void 
param_list::add_param(const std::string& name, str_vect& addr,
		      const str_vect& value) {
   add_param(name, par_def::tStringVect, &addr);
   _map[name].set(&value); 
}

void 
param_list::add_param(const std::string& name, par_def::par_type t,
		      void* addr) {
   par_def p(t, addr);
   par_map_iter iter = _map.find(name);
   if (iter != _map.end()) iter->second = p;
   else                    _map.insert(par_map_type::value_type(name, p));
}

//===================================  Clear a parameter
void 
param_list::clear_param(const std::string& name) {
   ref_par(name).clear();
}

void 
param_list::clear_all(void) {
   for (par_map_iter p=_map.begin(); p!=_map.end(); p++) {
      p->second.clear();
   }
}

//==================================== Display parameter list
std::ostream&
param_list::display(ostream& out, const std::string& pfx) const {
   size_t max_name_size = 0;
   for (const_par_map_iter p=_map.begin(); p!=_map.end(); p++) {
      if (p->first.size() > max_name_size) max_name_size = p->first.size();
   }

   for (const_par_map_iter p=_map.begin(); p!=_map.end(); p++) {
      size_t npad = max_name_size + 2 - p->first.size();
      out << pfx << p->first << ":" << string(npad, ' ')
	  << p->second.display() << endl;
   }
   return out;
}

param_list&
param_list::operator=(const param_list& rhs) {
   copy(rhs);
   return *this;
}

void 
param_list::set_param(const string& name, const string& value) {
   ref_par(name).set(value);
}
