/***************************************************************************
    File        : RandomRanrotW.cpp
    Description : implements the Random number generator of type RANROT-W
 ---------------------------------------------------------------------------
    Begin       : Fri Jan 2 2004
    Author(s)   : Roberto Grosso
 ***************************************************************************/


#include "RandomRanrotW.h"

// Defines
// If your system doesn't have a rotate function for 32 bits integers,
// then use the definition below. If your system has the _lrotl function 
// then remove this.
// #ifdef LINUX_GCC
inline unsigned int _lrotl (unsigned int x, int r) { return (x << r) | (x >> (sizeof(x)*8-r)); }
// #endif

void
gwd::RandomRanrotW::DetectComputerArchitecture()
{
  // detect computer architecture
  randbits[2] = 0; randp1 = 1.0;
  if (randbits[2] == 0x3FFF) mArchitecture = EXTENDEDPRECISIONLITTLEENDIAN;
  else if (randbits[1] == 0x3FF00000) mArchitecture = LITTLEENDIAN;
  else if (randbits[0] == 0x3FF00000) mArchitecture = BIGENDIAN;
  else mArchitecture = NONIEEE;
}


unsigned long int
gwd::RandomRanrotW::BRandom()
{
  // generate next random number
  unsigned long int y, z;
  // generate next number
  z = _lrotl(randbuffer[p1][0], R1) + randbuffer[p2][0];
  y = _lrotl(randbuffer[p1][1], R2) + randbuffer[p2][1];
  randbuffer[p1][0] = y; randbuffer[p1][1] = z;
  // rotate list pointers
  if (--p1 < 0) p1 = KK - 1;
  if (--p2 < 0) p2 = KK - 1;
  // perform self-test
  if (randbuffer[p1][0] == randbufcopy[0][0] &&
    memcmp(randbuffer, randbufcopy[KK-p1], 2*KK*sizeof(unsigned long int)) == 0)
  {
      // self-test failed
      if ((p2 + KK - p1) % KK != JJ)
      {
        // note: the way of printing error messages depends on system
        // In Windows you may use FatalAppExit
        gwd::Warning("Random number generator not initialized");
      }
      else
      {
        gwd::Warning("Random number generator returned to initial state");
      }
      exit(1);
  }
  randbits[0] = y;
  randbits[1] = z;
  return y;
}


long double
gwd::RandomRanrotW::Random()
{
  // returns a random number between 0 and 1.
  unsigned long int z = BRandom();  // generate 64 random bits
  switch (mArchitecture)
  {
    case EXTENDEDPRECISIONLITTLEENDIAN:
      // 80 bits floats = 63 bits resolution
      randbits[1] = z | 0x80000000;
      break;
    case LITTLEENDIAN:
      // 64 bits floats = 52 bits resolution
      randbits[1] = (z & 0x000FFFFF) | 0x3FF00000;
      break;
    case BIGENDIAN:
      // 64 bits floats = 52 bits resolution
      randbits[0] = (randbits[0] & 0x000FFFFF) | 0x3FF00000;
      break;
    case NONIEEE: default:
      // not a recognized floating point format. 32 bits resolution
      return (double)z * (1./((double)(unsigned long int)(-1L)+1.));
  }

  return randp1 - 1.0;
}


int
gwd::RandomRanrotW::IRandom(int min, int max)
{
  // get integer random number in desired interval
  int iinterval = max - min + 1;
  if (iinterval <= 0) return 0x80000000;  // error
  int i = int(iinterval * Random());      // truncate
  if (i >= iinterval) i = iinterval-1;
  return min + i;
}


void
gwd::RandomRanrotW::RandomInit (unsigned long int seed)
{
  // this function initializes the random number generator.
  int i, j;

  // make random numbers and put them into the buffer
  for (i=0; i<KK; i++)
  {
    for (j=0; j<2; j++)
    {
      seed = seed * 2891336453UL + 1;
      randbuffer[i][j] = seed;
    }
  }
  // set exponent of randp1
  randbits[2] = 0; randp1 = 1.0;
  // initialize pointers to circular buffer
  p1 = 0;  p2 = JJ;
  // store state for self-test
  memcpy (randbufcopy, randbuffer, 2*KK*sizeof(unsigned long int));
  memcpy (randbufcopy[KK], randbuffer, 2*KK*sizeof(unsigned long int));
  // randomize some more
  for (i=0; i<31; i++) BRandom();
}

