/***************************************************************************
    File        : Numerics.h
    Description : Implements some usefuls inline functions
 ---------------------------------------------------------------------------
    Begin       : Fri Jul 27 2001
    Copyright   : (C) 2001 by Roberto Grosso
                  All rights reserved
 ***************************************************************************/


/*!
 * \file Numerics.h
 *  Contains simple and basic numerical operations and functions which are not included 
 *  in the standards C99 or C++ or which are reimplemented here in order to achieve
 *  plattform independence between Linux and Windows, e.g. the macros
 *  isnan() in Linux and _isnan() in Windows
 */


#ifndef __NUMERICS_H
#define __NUMERICS_H

// Libs
#include <vector>
#include <complex>
#include <cmath>
#include <cfloat>
#include <limits>
#include <numeric>
#include <algorithm>


// Project files
#include "Types.h"

/*! \file Numerics.h 
 * Contains some useful numerical constants and simple inline functions.
 */ 

namespace gwd {


  //! Compiler dependent epsilon.
  const double epsilon = std::numeric_limits<double>::epsilon();
  const double eps = epsilon;

  //! A long representation of PI intended to be valid also for high presicion long double.
  const double PI    = 3.14159265358979323846264338327950288419716939937510582097494459230781640628620899862803482534211706798214;

  //! The velocity of Ligth in [m/s].
  const double VelocityOfLight = 2.99792458e8;

  //! The Earth radious in [m].
  const double EarthRadius = 6.370137e6;

  //! Computes the square of a number of type T.
  //template<typename T> inline T
  //square(const T a) { return a*a; }
  //! Computes the square of an int number.
  inline int
  square(const int a) { return a*a; }
  //! Computes the square of a float number.
  inline float
  square(const float a) { return a*a; }
  //! Computes the square of a double number.
  inline double
  square(const double a) { return a*a; }
  //! Computes the square of a complex float number.
  inline std::complex<float>
  square(const std::complex<float> a) { return (a*a); }
  //! Computes the square of a complex double number number.
  inline std::complex<double>
  square(const std::complex<double> a) { return (a*a); }

  //! Computes the square of all components of a vector in-place.
  template<typename vector> inline void
  square(vector& in)
  {
    // in-place
    typedef typename vector::size_type SizeType;
    //typedef Vector::size_type SizeType;
    const SizeType vSize = in.size();
    for (SizeType nn = 0; nn < vSize; nn++) in[nn] = square(in[nn]);
  }

  //! Computes the square of all components of a vector.
  template<typename vector> inline void
  square(const vector& in,vector& out)
  {
    typedef typename vector::size_type SizeType;
    const SizeType vSize = in.size();
    for (SizeType nn = 0; nn < vSize; nn++) out[nn] = square(in[nn]);
  }

  //! Computes the absolute square of z, i.e. z.z* , with z* the complex conjugate.
  inline double
  AbsoluteSquare(const std::complex<double> a)
  {
    return (a.real()*a.real()+a.imag()*a.imag());
  }

  //! Multiply a vector with a scalar
  template<typename T> void
  Multiply(std::vector<T>& v,const T val)
  {
    typename std::vector< T >::iterator p = v.begin();
    while(p != v.end())*p++ *= val;
  }

  //! Multiply a vector with another vector componentwise.
  template<typename T> void
  Multiply(std::vector<T>& v,std::vector<T> w)
  {
    typename std::vector<T>::iterator p = v.begin();
    typename std::vector<T>::iterator q = w.begin();
    while(p != v.end())*p++ *= *q++;
  }

  //! Zero pad a vector for signal processing.
  template<typename T> void
  ZeroPad(std::vector<T>& v,const unsigned int pos)
  {
    std::fill(v.begin()+pos,v.end(),T(0));
  }

  //! Computes the inner product of two vectors.
  template<typename T> T
  InnerProduct(std::vector<T>& a,std::vector<T>& b)
  {
    return (std::inner_product(a.begin(),a.end(),b.begin(),T(0)));
  }

  //! Computes the absolute value of a number of type T.
  template<typename T>
  inline T Abs(T val) 
  {
    return ((val >= 0) ? val : -val);
  }
  
  //! Finds the largest element in absolute value in a specified range.
  inline double MaxAbsoluteValue(const gwd::Vector& vt)
  {
    Vector::const_iterator p = vt.begin();
    double val = fabs(*p);
    while(p != vt.end())
    {
      double temp = fabs(*p++);
      val = (val > temp) ? val : temp;
    }
    return val;
  }

  template<class T> inline const T
  Max(const T &a, const T &b)
  {
    return b > a ? (b) : (a);
  }

  template<class T> inline const T
  Min(const T &a, const T &b)
  {
    return b < a ? (b) : (a);
  }

  template<class T> inline const T
  Sign(const T &a, const T &b)
  {
    return b >= 0 ? (a >= 0 ? a : -a) : (a >= 0 ? -a : a);
  }

  template<class T> inline void
  Swap(T &a, T &b)
  {
    T dum=a; a=b; b=dum;
  }

  template<class T> inline void
  Shift(T& a,T& b,T& c,T d) { a = b; b = c; c = d; }
  
  //! Computes the modulo between two integer numbers. This function
  //! does only make sense for numbers which can also be negative.
  inline int
  mod(int n, int m) { return ( (n >= 0) ? (n % m) : ((m - (-n % m)) % m) ); }

  //! Computes the indices for a symmetric boundary condition. This is
  //! e.g. the case when processing images or re-ordering the components
  //! after a fft.
  template<typename T> inline T
  sym(T n, T m)
  {
    T m2 = (m << 1);
    T tp = ((n >= 0) ? (n % m2) : ((m2 - (-n % m2)) % m2) );
    return ((tp < m) ? tp : (m2 - tp -1));
  }

  //! Factorize to integer numbers.
  inline void
  factorize(int a,int b,int& aa,int& bb)
  {
    int atemp = a;
    int btemp = b;
    int a2=0;
    int b2=0;
  
    while (!(1 & a)) { a = a>>1; a2++; }
    while (!(1 & b)) { b = b>>1; b2++; }
    while (a != b)
    {
      if (b>a) { int tmp = b; b = a; a = tmp; }
      a -= b;
      while (!(1 & a)) a >>= 1;
    }
  
    // largest common factor.
    int lcf = ( a*(2<<((a2<b2? a2:b2)-1)) );
    aa = atemp / lcf;
    bb = btemp / lcf;
  }


  //! Check if an integer number is a power of two, i.e. m = 2^n.
  template<typename T> inline bool
  IsPowOf2(T NN)
  {
    bool flag = false;
    T locSize = NN;
    T locBit = 1;
    while (locSize)
    {
      if (NN&locBit)
      {
        if (flag) return false;
        else flag = true;
      }
      locBit <<= 1;
      locSize >>= 1;
    }
    return true;
  }

  //! Computes the largest power of 2 smaller than N.
  template<typename T> inline void
  FloorPowerOf2(const T aa,T& p2,T& NN)
  {
    NN = static_cast<T>(ceil(log((double)aa)/log(2.)));
    p2 = 2<<(NN-1);
    if ( aa < p2)
    {
      NN--;
      p2 = 2<<(NN-1);
    }
  }
  
  //! Computes the smallest power of 2 larger than N.
  template<typename T> inline void
  CeilPowerOf2(const T aa,T& p2,T& NN)
  {
    NN = static_cast<T>(ceil(log((double)aa)/log(2.)));
    p2 = 2<<(NN-1);
    if ( aa > p2)
    {
      NN++;
      p2 = 2<<(NN-1);
    }
  }

  //! Check if an integer number is even.
  inline bool
  IsEven(const int nn) { return !(nn&1); }
  inline bool
  IsEven(const unsigned int nn) { return !(nn&1); }

  //! Check if an integer number is odd.
  inline bool
  IsOdd(const int nn)  { return  (nn&1); }
  inline bool
  IsOdd(const unsigned int nn)  { return  (nn&1); }

  // Floating point numbers, Window version.
  #ifdef _MSC_VER
  //! Check if a floating point number is a NaN.
  inline int
  IsNaN(const double x) { return _isnan(x); }
  //! Check if a floating point number is finite, i.e. is not infinite.
  inline int
  IsFinite(const double x) { return _finite(x); }
  // Floating point numbers, Linux version
  #elif LINUX_GCC
  //! Check if a floating point number is a NaN.
  inline int
  //IsNaN(const double x) { return std::isnan(x); }
  IsNaN(const double x) { return isnan(x); }
  //! Check if a floating point number is finite, i.e. it is not infinite and not NaN.
  //inline int
  //IsFinite(const double x) { return std::isfinite(x); }
  //IsFinite(const double x) { return isfinite(x); }
  #endif

  //! Comute the neares integer.
  inline double
  NearestInt(const double a)
  {
    double res = fabs(a - std::floor(a));
    if (res <= 0.5)
      return std::floor(a);
    else
      return (std::ceil(a));
  }
} // namespace gwd


#endif // __NUMERIC_H
