/***************************************************************************
    File        : NoiseFloorMonitor.h
    Description : Median-based noise floor tracker MNFT
 ---------------------------------------------------------------------------
    Begin       : Tue March 30 2005
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#ifndef __NOISE_FLOOR_MONITOR_H
#define __NOISE_FLOOR_MONITOR_H


/**  @name NoiseFloorMonitor
  *  @memo Median-based noise floor tracker (MNFT): robust estimation of
  *  noise floor drifts in interferometric data.
  *
  *  @author Roberto Grosso; Brownsville, Texas
  *  @version 2.0; Last modified May 14, 2006
  */

// Libs
#include <iostream>
#include <sstream>
#include <fstream>
#include <list>
#include <string>
#include <vector>
#include <cstdlib>
#include <limits>
#include <algorithm>

// Control numerical fp values
// #include <ieeefp.h>


// Project files
#include "DatEnv.hh"
#include "MonServer.hh"
#include "Interval.hh"
#include "TSeries.hh"
#include "Time.hh"
#include "Trend.hh"
#include "OperStateCondList.hh"

// HTML output
#include "html/color.hh"
#include "html/document.hh"
#include "html/table.hh"
#include "html/size.hh"
#include "html/text.hh"
#include "html/writer.hh"
#include "html/hline.hh"
#include "html/link.hh"

// NoiseFloor files
#include "Clock.h"
#include "Types.h"
#include "Utilities.h"
#include "Exception.h"
#include "Numeric.h"
#include "Parameter.h"
#include "LowPassFilter.h"
#include "BandPassFilter.h"
#include "HighPassFilter.h"
#include "NotchFilter.h"
#include "WhiteningFilter.h"
#include "RunningMedian.h"
#include "FIRFilter.h"
#include "Multirate.h"
#include "Convolution.h"
#include "NormalDistribution.h"
#include "Crosscorrelation.h"

// Channel
#include "DetectorChannel.h"
#include "FrequencyBand.h"


using namespace std;

/** The Noise Floor Monitor tracks slow variations or 'drifts' in the noise 
 ** floor (that component of interferometric data that is left behind after 
 ** the lines and obvious transients are subtracted).  The algorithm 
 ** implemented by the Monitor is based on the technique described in the 
 ** paper Median based noise floor tracker : robust estimation of noise floor
 ** drifts in interferometric data. S. Mukherjee, Classical and Quantum
 ** Gravity, 20, S925-S936,2003.
 ** 
 ** Monitor's Output
 ** The output of the monitor is a time series representing the running
 ** median estimate of the second moment. Every time, the output crosses the
 ** three sigma threshold, a trigger is recorded and the data is inferred to
 ** have departed from the stationary behaviour. In each time stride, 60
 ** seconds of raw output are analyzed. Analysis is done separately in four
 ** frequency bands : 0-20 Hz, 20-100 Hz, 100-200 Hz and 200-2048 Hz. The
 ** behavior of the data in these four frequency bands can be seen in the
 ** DMT viewer and is updated every minute. In addition to this, a 12 hour
 ** data trend plot is also available.
 ** 
 ** The result files record four quantities : the minimum, maximum, rms of
 ** the output and percentage of the data points crossing the threshold in a
 ** one hour period. Those segments that record more than 25% threshold
 ** crossing need further attention and are useful for construction of data
 ** quality flags.
 **
 **/
class NoiseFloorMonitor : public DatEnv, MonServer {
public:
  //! Constructor
  NoiseFloorMonitor(int argc, const char *argv[]);
  //! Destructor
  virtual ~NoiseFloorMonitor(void);

  // Methods
  //! Method called by MainLoop.
  void ProcessData(void);
  //! Handle Message signal.
  void Attention(void);
private:
  //! Detector site, HA for Hanford and LA for Livingstone.
  std::string mDetectorSite;
  //! Name of main channel.
  std::vector<std::string> mMainChannel;
  //! No. of channels to be processed.
  unsigned int mNoChannels;
  //! Name of the Channel being processed.
  std::vector<std::string> mChannelName;
  //! Channel sampling rate.
  gwd::Vector mChannelSampleRate;
  //! Channel data.
  std::vector<gwd::DetectorChannel> mDetectorChannel;
  
  //! Count the no. of time stride read.
  unsigned long mCounterTimeStrides;
  //! Length on the time stride.
  Interval mTStride;
  //! Noise floor trends.
  Trend mTrend;
  //! Current time
  Time mCurrentTime;
  //! Time stride in sec.
  double mTStrideSize;

  //! Number of frequency bands.
  int mNoFreqBands;
  //! The frequency bands.
  gwd::Vector mFreqBand;
  //! The sampling rate for each frequency band.
  gwd::Vector mSampleRateFreqBand;
  //! The mean values of the Gaussian noise for each frequency band.
  gwd::Vector mMeanValue;
  //! The standard deviation of the Gaussian noise for each frequency band.
  gwd::Vector mStdDev;
  //! Upper threshold for the frequency bands.
  std::vector<TSeries> mUpperThreshold;
  //! Lower threshold for the frequency band.
  std::vector<TSeries> mLowerThreshold;
  //! Upper 2*sigma threshold for the frequency band..
  std::vector<TSeries> mUpper2SigmaThreshold;
  //! Lower 2*sigma threshold for the frequency band..
  std::vector<TSeries> mLower2SigmaThreshold;  

  //! No. of loops before updating the whitening filter.
  unsigned int mUpdateWhiteFlt;
  //! Size in sec. of the Running median filter.
  double mWindowSizeRunningMedian;
  //! Signal's sampling rate in Hz.
  double mSignalSampleRate;
  //! Processing sampling rate.
  double mSampleRate;
  //! The oversampling factor.
  double mOverSamplingFactor;
  
  //! List of running status messages.
  std::list<std::string*> mStatusMessage;
  //! List with warning messages
  std::list<std::string*> mWarningMessage;
  //! List with error messages
  std::list<std::string*> mErrorMessage;
  
  // Private Methods
  //! Estimate Threshold for Noise floor via Monte Carlo simulation with Gaussian white noise.
  void EstimateThreshold();

  //! Prepare data for dmtview plot: down sample.
  void PrepareDataForPlot(gwd::Vector& in,std::vector<float>& ou);
  //! Send empty plot to the DMT-viewer for non locked detector.
  void EmptyPlot();
  //! Send thresholds plot to DMT-viewer.
  void ThresholdPlot();
  
  //! Operation State Condition List
  OperStateCondList *mOSC;
  //! Lock condition name
  std::string mLockConditionName;
  //! set lock conditon of detector's arms.
  bool mLockConditionFlag;
  
  //! Set Trend, write trend data into file.
  void SetNoiseFloorTrend();
  //! Name of channel with cross correlation value in trend file.
  std::vector<std::string> mTrendNameCrossCorr;
  //! Name of channel with max noise floor value in trend file.
  std::vector<std::string> mTrendNameNoiseFloor;
  
  //! Write into html document
  std::string mHTMLStatusFile;
  std::string mHTMLErrorFile;
  void DumpHTMLStatusPage();
  void DumpHTMLErrorPage();
  
  //! Estimate the mean value and the standard deviation.
  void Estimator(const gwd::Vector& ts,double& mean,double& stdDev);
};



#endif     //  __NOISE_FLOOR_MONITOR_H
