/***************************************************************************
    File        : DetectorChannel.cpp
    Description : Implements the NoiseFloorMonitor
 ---------------------------------------------------------------------------
    Begin       : Fri April 14 2006
    Author(s)   : Roberto Grosso
 ***************************************************************************/


#include "DetectorChannel.h"

//======================================  NoiseFloorMonitor constructor
gwd::DetectorChannel::DetectorChannel( )
{
  //----------------------------------  init class variables
  //--------------------------------------------------------------------
  mEmpty = false;
  mTStride = 0.;

  mOrderNotchFilter = 0;
  mOrderWhiteFilter = 0;
  mNoOfFreqPsd      = 0;

  mSizePsd                 = 0.;
  mRMSizeForWhiteningFlt   = 0.;
  mWindowSizeRunningMedian = 0.;
  mSampleRate              = 0.;
  mOverSamplingFactor      = 1.0;
  
  mNoOfLines = 0;
}


//======================================  NoiseFloorMonitor destructor
gwd::DetectorChannel::~DetectorChannel( )
{
  // nothing to be destroyed!
}


bool gwd::DetectorChannel::Init(const std::string& channelname,const double chsamplerate,gwd::Parameter& param,const Vector& freqs,const Vector& freqbnds)
{
  //----------------------------------  get the singleton
  //-------------------------------------------------------------------------------
  gwd::Singleton* single = gwd::Singleton::exemplar();

  //----------------------------------  read frequencies for the Notch filter
  //-------------------------------------------------------------------------
  mNoOfLines             = (int) freqs.size();
  if (mNoOfLines)
  {
    mFreqNotchFilter       = freqs;
    mFreqBndWidthNotFilter = freqbnds;
  }
  
  //----------------------------------  set the time stride.
  mTStride = Interval(param.TimeStride());
  //getDacc().setStride(param.TimeStride());
  mTStrideSize = mTStride.GetSecs();

  //----------------------------------  copy parameters
  //--------------------------------------------------------------------
  mOrderNotchFilter = param.OrderNotchFilter();
  mOrderWhiteFilter = param.OrderWhiteFilter();
  mNoOfFreqPsd      = param.NoOfFreqPSD();
  mSizePsd          = param.SizeForPSD();
  mRMSizeForWhiteningFlt   = param.RunningMedianSizeForWhiteningFilter();
  mWindowSizeRunningMedian = param.WindowSizeRunningMedian();
  mChannelName   = channelname;

  mFltOrder        = param.FltOrder();
  mUpperCutoffFreq = param.UpperCutoffFreq();
  mBandFltOrder    = param.BandFltOrder();
  mUpdateWhiteFlt  = param.UpdateWhiteFilter();
  mNoFreqBands     = param.NoFreqBands();

  //----------------------------------  oversampling factor for resampling and decimation
  //-------------------------------------------------------------------------------------
  mOverSamplingFactor = param.OverSamplingFactor();

  //----------------------------------  sample rate is equal twice of cutoff frequency (Nyquist)
  //--------------------------------------------------------------------------------------------
  mSampleRate = 2.*mUpperCutoffFreq;

  //----------------------------------  sample rate must be less than or equal to signal sampling rate
  //--------------------------------------------------------------------------------------------------
   mSignalSampleRate = chsamplerate;
  if (mSampleRate > mSignalSampleRate)
  {
    mSampleRate = mSignalSampleRate;
    mUpperCutoffFreq = mSampleRate / 2.;
  }

  //----------------------------------  copy frequencies for the Notch filter
  //-------------------------------------------------------------------------
  if (mNoOfLines > 0)
  {
    mFreqNotchFilter.resize(freqs.size());
    mFreqBndWidthNotFilter.resize(freqbnds.size());
    std::copy(freqs.begin(),freqs.end(),mFreqNotchFilter.begin());
    std::copy(freqbnds.begin(),freqbnds.end(),mFreqBndWidthNotFilter.begin());
  }

  //----------------------------------  compute total filter delay
  //--------------------------------------------------------------------
  // cutoff filter operates at mSignalSampleRate
  mFilterDelay = 0.0;
  if (fabs(mSampleRate - mSignalSampleRate) > 1.0) // cutoff required
    mFilterDelay = static_cast<double>(mFltOrder)/mSignalSampleRate;
  // add delay due to whitening filter at mSampleRate
  mFilterDelay += static_cast<double>(mOrderWhiteFilter)/mSampleRate;
  // add delay due to Notch filter at mSampleRate
  if (mNoOfLines)
    mFilterDelay += static_cast<double>(mOrderNotchFilter)/mSampleRate;
  // add delay due to low-pass or band-pass or high-pass filter at mSampleRate
  mFilterDelay += static_cast<double>(mBandFltOrder)/mSampleRate;

  //----------------------------------  compute buffer for time continuous processing
  //---------------------------------------------------------------------------------
  // Create a buffer which is as long as the filter dealy
  // and the running median filter plus a time shift. The
  // running median cut seconds from the tail of the time
  // series.
  mBufferTimeShift = 0.5; // in sec
  unsigned int buffSize = static_cast<unsigned int>((mBufferTimeShift + mFilterDelay + mWindowSizeRunningMedian)*mSignalSampleRate);
  buffer.resize(buffSize);
  // set buffer values to white noise
  gwd::NormalDistribution wn;
  for(unsigned int nn = 0; nn < buffSize; nn++) buffer[nn] = wn.Normal(0,1);
  //std::fill(buffer.begin(),buffer.end(),double(0));

  // Check that buffer size is shorter the time stride
  if ((mBufferTimeShift + mFilterDelay + mWindowSizeRunningMedian) > mTStrideSize)
  {
    std::string strError = std::string("time stride shorter than running median window size and filter delay\n")
                         + std::string("set time stride >= running median window + filter delay + 1 sec.");
                         
    single->AppendMessage(strError);
    std::cout << strError << std::endl
              << mBufferTimeShift << "  " <<  mFilterDelay << "  " << mWindowSizeRunningMedian << "  " <<  mTStrideSize << std::endl;
    return false;
  }

  //----------------------------------  check if time stride is large enough
  //--------------------------------------------------------------------
  const unsigned int wSize    = (unsigned int) (mRMSizeForWhiteningFlt);
  const unsigned int segSize  = (unsigned int) (mSampleRate * mSizePsd) + wSize;
  const unsigned int tshift   = (unsigned int) ((5. + mFilterDelay) * mSampleRate); // shift by 5 secs. + filter delay
  const unsigned int tsSize   = (unsigned int) (mTStrideSize*mSampleRate);
  if ((tshift+segSize) >= tsSize)
  {
    std::cerr << "Channel: " << mChannelName << std::endl;
    std::cerr << mRMSizeForWhiteningFlt << "  " << mSampleRate << "  " << mSizePsd << std::endl;
    std::cerr << (tshift+segSize) << "  " << tsSize << std::endl;
    std::string strError = std::string("The size if sec. of the time series used for calculating the psd is too large\n")
                         + std::string("The size should be psd size  < TimeStrid - 5 sec.");
    single->AppendMessage(strError);
    return false;
  }

  //---------------------------------- compute no. of frequency bands
  //--------------------------------------------------------------------
  for (int ii = 0; ii < mNoFreqBands; ii++)
  {
    double freqBand = param.FreqBand(ii);
    if (freqBand >= mUpperCutoffFreq)
    {
      mNoFreqBands = ii+1;
      break;
    }
  }  

  //---------------------------------- compute frequecy pass filters
  //--------------------------------------------------------------------
  gwd::LowPassFilter  lpf;
  gwd::BandPassFilter bpf;
  gwd::HighPassFilter hpf;

  //----------------------------------  compute the cutoff filter
  //--------------------------------------------------------------------
  if (fabs(mSampleRate - mSignalSampleRate) > 1.0) // cutoff required
    lpf.ComputeFilter(mSignalSampleRate,mUpperCutoffFreq,mFltOrder,mCutoffFlt);

  //----------------------------------  compute data for the frequency
  //----------------------------------  bands here.
  //--------------------------------------------------------------------
  mNoiseFloor.resize(mNoFreqBands);
  //----------------------------------  low-pass band.
  //--------------------------------------------------------------------
  mNoiseFloor[0].type = LOW_PASS;
  mNoiseFloor[0].mLowFreq  = double(0);
  mNoiseFloor[0].mHighFreq = param.FreqBand(0);
  lpf.ComputeFilter(mSampleRate,mNoiseFloor[0].mHighFreq,mBandFltOrder,mNoiseFloor[0].mLowPassFlt);
  mNoiseFloor[0].mSampleRate = 2.*mOverSamplingFactor*mNoiseFloor[0].mHighFreq;
  mNoiseFloor[0].mDecimate   = static_cast<int>(std::floor(mSampleRate/mNoiseFloor[0].mSampleRate));
  if (mNoiseFloor[0].mDecimate < 1) mNoiseFloor[0].mDecimate = 1;
  mNoiseFloor[0].mSampleRate = mSampleRate / static_cast<double>(mNoiseFloor[0].mDecimate);
  //----------------------------------  band-pass band.
  //--------------------------------------------------------------------
  for (int ii = 1; ii < (mNoFreqBands-1); ii++)
  {
    // Band parameters
    mNoiseFloor[ii].type = BAND_PASS;
    mNoiseFloor[ii].mLowFreq  = param.FreqBand(ii-1);
    mNoiseFloor[ii].mHighFreq = param.FreqBand(ii);
    // Filters
    bpf.ComputeFilter(mSampleRate,mNoiseFloor[ii].mHighFreq,mNoiseFloor[ii].mLowFreq,mBandFltOrder,mNoiseFloor[ii].mBandPassFlt);
    lpf.ComputeFilter(mSampleRate,mNoiseFloor[ii].mHighFreq - mNoiseFloor[ii].mLowFreq,mBandFltOrder,mNoiseFloor[ii].mLowPassFlt);
    // Sample rate and decimation factor
    mNoiseFloor[ii].mSampleRate = 2.*mOverSamplingFactor*mNoiseFloor[ii].mHighFreq;
    mNoiseFloor[ii].mDecimate   = static_cast<int>(std::floor(mSampleRate/mNoiseFloor[ii].mSampleRate));
    if (mNoiseFloor[ii].mDecimate < 1) mNoiseFloor[ii].mDecimate = 1;
    mNoiseFloor[ii].mSampleRate = mSampleRate / static_cast<double>(mNoiseFloor[ii].mDecimate);
  }

  //----------------------------------  high-pass band.
  //--------------------------------------------------------------------
  // Band parameters
  mNoiseFloor[mNoFreqBands-1].type = HIGH_PASS;
  mNoiseFloor[mNoFreqBands-1].mLowFreq  = param.FreqBand(mNoFreqBands-2);
  mNoiseFloor[mNoFreqBands-1].mHighFreq = mUpperCutoffFreq;
  // Filters
  hpf.ComputeFilter(mSampleRate,mNoiseFloor[mNoFreqBands-1].mLowFreq,mBandFltOrder,mNoiseFloor[mNoFreqBands-1].mHighPassFlt);
  lpf.ComputeFilter(mSampleRate,mNoiseFloor[mNoFreqBands-1].mHighFreq - mNoiseFloor[mNoFreqBands-1].mLowFreq,mBandFltOrder,mNoiseFloor[mNoFreqBands-1].mLowPassFlt);
  // Sample rate and decimation factor
  mNoiseFloor[mNoFreqBands-1].mSampleRate = 2.*mOverSamplingFactor*mNoiseFloor[mNoFreqBands-1].mHighFreq;
  mNoiseFloor[mNoFreqBands-1].mDecimate   = static_cast<int>(std::floor(mSampleRate/mNoiseFloor[mNoFreqBands-1].mSampleRate));
  if (mNoiseFloor[mNoFreqBands-1].mDecimate < 1) mNoiseFloor[mNoFreqBands-1].mDecimate = 1;
  mNoiseFloor[mNoFreqBands-1].mSampleRate = mSampleRate / static_cast<double>(mNoiseFloor[mNoFreqBands-1].mDecimate);
  
  //----------------------------------  compute Notch filter
  //--------------------------------------------------------------------
  if (mNoOfLines > 0)
  {
    gwd::NotchFilter nf;
    nf.ComputeFilter(mSampleRate,mOrderNotchFilter,mFreqNotchFilter,mFreqBndWidthNotFilter,mNotchFilter);
  }
  
  //----------------------------------  compute initial Whiten filter
  //--------------------------------------------------------------------
  mWhiteningFilter.resize(mOrderWhiteFilter);
  gwd::NormalDistribution normal;
  for(unsigned int nn = 0; nn < mOrderWhiteFilter; nn++)
  {
    mWhiteningFilter[nn] = normal.Normal(0.,1.);
    mWhiteningFilter[nn] = gwd::square(mWhiteningFilter[nn]);
  }
 
  //----------------------------------  Calculate plot size for the twelve hs plot
  //-----------------------------------------------------------------------------
  for (int ii = 0; ii < mNoFreqBands; ii++)
  {
    unsigned int size12h = (unsigned int)(43200./mTStride.GetSecs());
    mNoiseFloor[ii].m12hsPlot.resize(size12h,double(0));
  }
  

  // Channel initialized
  return true;
} // Constructor


//=================================================================================================
//=================================================================================================
//======================================  Frame processing function.
//=================================================================================================
bool
gwd::DetectorChannel::ProcessData(const TSeries* ts,const Time& currentTime,const bool lockConditionFlag,const unsigned long counter)
{
  // track the processing status
  bool flag = true;
  // Set an error string for processing
  std::string errStr = mChannelName + " DetectorChannel::ProcessData(): ";
  //----------------------------------  get the singleton
  //-------------------------------------------------------------------------------
  gwd::Singleton* single = gwd::Singleton::exemplar();

  //----------------------------------  Get channel data: time series
  //--------------------------------------------------------------------
  bool tsFlag = true;
  mEmpty = false;
  Vector inTS;
  //----------------------------------  Get current time
  //--------------------------------------------------------------------
  mCurrentTime = currentTime;

  //----------------------------------  Check if data is correct
  //--------------------------------------------------------------------
  if (ts == 0)
  {
    tsFlag = false;
    mEmpty = true;
    std::string message = errStr + "time series pointer is NULL";
    single->AppendMessage(message);
  }
  else if (ts->isEmpty())
  {
    tsFlag = false;
    mEmpty = true;
    std::string message = errStr + "time series is empty";
    single->AppendMessage(message);
  }
  else if (std::string(ts->getName()) != mChannelName)
  {
    tsFlag = false;
    mEmpty = true;
    std::string message = errStr + "data pointer to wrong channel, channel name and data pointer do not coincide!";
    single->AppendMessage(message);
  }

  if (tsFlag)
  { 
    //----------------------------------  remember no. of samples
    //--------------------------------------------------------------------
    const unsigned int tsNSample = ts->getNSample();
    //----------------------------------  get sampling rate
    //--------------------------------------------------------------------
    //double samplerate = 1./ ts->getTStep().GetSecs();
    //std::cout << " ... sample rate: " << samplerate << std::endl;
    
    //----------------------------------  set buffer
    //--------------------------------------------------------------------
    inTS.resize(buffer.size()+tsNSample);
    std::copy(buffer.begin(),buffer.end(),inTS.begin());
    Vector::iterator p = inTS.begin()+buffer.size();
    for (unsigned int ii = 0; ii < tsNSample; ii++) *p++ = ts->getDouble(ii);
    std::copy(inTS.end()-buffer.size(),inTS.end(),buffer.begin());

    // check if time series is empty
    if (inTS.empty())
    {
      tsFlag = false;
      single->AppendMessage("DetectorChannel::ProcessData(): can't create input time series from detector data");
    }
  } // if (tsFlag)

  // Main Routine: compute noise floor
  if (tsFlag && lockConditionFlag)
  {
    //----------------------------------  Resample and FIR filtering
    //--------------------------------------------------------------------
    gwd::FIRFilter fir;
    gwd::Multirate mrt;
    
    //----------------------------------  compute band limited data:
    //----------------------------------  16kHz -> 4kHz sample rate
    //--------------------------------------------------------------------
    Vector locTS;
    if (fabs(mSampleRate - mSignalSampleRate) > 1.0) // cutoff required
    {
      int factor = (int)gwd::NearestInt(mSignalSampleRate/mSampleRate);
      if (!mrt.Decimate(factor,mCutoffFlt,inTS,locTS))
      {
        std::string message = errStr + "can't decimate data down to cutoff frequency.";
        single->AppendMessage(message);
        flag = false;
      }
    }
    else // no cutoff required
      locTS = inTS;
    
    //----------------------------------  clear data and free memory
    //--------------------------------------------------------------------
    inTS.clear();

    //----------------------------------  compute the whitening filter
    //--------------------------------------------------------------------
    // Sun does not like unsigned int, have to use ints!!!!
    unsigned int a = (unsigned int) counter;
    unsigned int b = (unsigned int) mUpdateWhiteFlt;
    if ((a%b) == 0)
    {
      gwd::WhiteningFilter wf;
      const unsigned int wSize    = (unsigned int) (mRMSizeForWhiteningFlt);
      const unsigned int segSize  = (unsigned int) (mSampleRate * mSizePsd) + wSize;
      const unsigned int tshift   = (unsigned int) (5. * mSampleRate)+ buffer.size(); // shift by 5 secs. + buffer size

      //----------------------------------  create a vector with the time series
      //--------------------------------------------------------------------
      Vector locSig(segSize);
      for (unsigned int nn = 0; nn < segSize; nn++)
        locSig[nn] = locTS[tshift+nn];

      //----------------------------------  Compute the whiten filter
      //--------------------------------------------------------------------
      if (!wf.ComputeFilter(mOrderWhiteFilter,mNoOfFreqPsd,wSize,locSig,mWhiteningFilter))
      {
        std::string message = errStr + "can't compute whitening filter";
        single->AppendMessage(message);
        flag = false;
      }
    }

    //----------------------------------  Whitening
    //--------------------------------------------------------------------
    Vector wtTS;
    if (flag) flag = fir.Filter(locTS,mWhiteningFilter,wtTS);
    if (!flag)
    {
      std::string message = errStr + "can't whiten data.";
      single->AppendMessage(message);
      flag = false;
    }
    //----------------------------------  clear data and free memory
    //--------------------------------------------------------------------
    locTS.clear();

    //----------------------------------  line removal
    //--------------------------------------------------------------------
    Vector lrTS; // line removed signal
    if (mNoOfLines > 0)
    {
      if (flag) flag = fir.Filter(wtTS,mNotchFilter,lrTS);
      if (!flag)
      {
        std::string message = errStr + "can't eliminate lines with Notch filter.";
        single->AppendMessage(message);
        flag = false;
      }
    }
    else
      lrTS = wtTS;

     //----------------------------------  clear data and free memory
    //--------------------------------------------------------------------
    wtTS.clear();
    
    //----------------------------------  Compute the frequency bands
    //--------------------------------------------------------------------
    //----------------------------------  low-pass band
    //--------------------------------------------------------------------
    if (flag) 
      flag = mrt.Decimate(mNoiseFloor[0].mDecimate,mNoiseFloor[0].mLowPassFlt,lrTS,mNoiseFloor[0].mNoiseFloor);
    if (!flag) {
      std::string message = errStr + "can't create the low-pass band.";
      single->AppendMessage(message);
      flag = false;
    }

    //----------------------------------  process band-pass data
    //--------------------------------------------------------------------
    Vector ss;
    double bandwidth;
    for (int ii = 1; ii < (mNoFreqBands-1); ii++)
    {
      bandwidth = mNoiseFloor[ii].mHighFreq - mNoiseFloor[ii].mLowFreq;
      fir.Filter(lrTS,mNoiseFloor[ii].mBandPassFlt,ss);                                                           // band-pass filter
      MixWithOscilator(bandwidth/2.,mSampleRate,ss);                                                              // mix with oscillator
      if (flag)
        flag = mrt.Decimate(mNoiseFloor[ii].mDecimate,mNoiseFloor[ii].mLowPassFlt,ss,mNoiseFloor[ii].mNoiseFloor);   // decimate
      if (!flag) {
        std::string message = errStr + "can't create one of the pass band.";
        single->AppendMessage(message);
        flag = false;
      }
      ss.clear();
    }
    //----------------------------------  process high-pass band
    //--------------------------------------------------------------------
    bandwidth = mNoiseFloor[mNoFreqBands-1].mHighFreq - mNoiseFloor[mNoFreqBands-1].mLowFreq;
    fir.Filter(lrTS,mNoiseFloor[mNoFreqBands-1].mHighPassFlt,ss);
    MixWithOscilator(bandwidth/2.,mSampleRate,ss);
    if (flag)
      flag = mrt.Decimate(mNoiseFloor[mNoFreqBands-1].mDecimate,mNoiseFloor[mNoFreqBands-1].mLowPassFlt,ss,mNoiseFloor[mNoFreqBands-1].mNoiseFloor);
    if (!flag) {
      std::string message = errStr + "can't create the high-pass band.";
      single->AppendMessage(message);
      flag = false;
    }
    ss.clear();
     //----------------------------------  clear data and free memory
    //--------------------------------------------------------------------
    lrTS.clear();
    
    //----------------------------------  Compute the Running Median
    //--------------------------------------------------------------------
    for (int ii = 0; ii < mNoFreqBands; ii++)
    {
      // Band's sampling rate
      double SampleRate = mNoiseFloor[ii].mSampleRate;
      
      //----------------------------------  select the time stride
      //--------------------------------------------------------------------
      Vector tstride;
      unsigned int tsShift;
      unsigned int tsSize;
      const double szTStride = mTStride.GetSecs();
      tsShift = static_cast<unsigned int>((mFilterDelay+mBufferTimeShift)*SampleRate);
      tsSize  = static_cast<unsigned int>((szTStride+mWindowSizeRunningMedian)*SampleRate);
      tstride.resize(tsSize);
      for (unsigned int jj = 0; jj < tsSize; jj++) tstride[jj] = mNoiseFloor[ii].mNoiseFloor[tsShift+jj];
      mNoiseFloor[ii].mNoiseFloor.resize(tsSize);
      std::copy(tstride.begin(),tstride.end(),mNoiseFloor[ii].mNoiseFloor.begin());
      tstride.clear();

    
      //----------------------------------  transform to Normal(0,1) 
      //----------------------------------  square values 
      //--------------------------------------------------------------------
      int nfSize = mNoiseFloor[ii].mNoiseFloor.size();
      double mean;
      double stdDev;
      Estimator(mNoiseFloor[ii].mNoiseFloor,mean,stdDev);
      
      for (int nn = 0; nn < nfSize; nn++) mNoiseFloor[ii].mNoiseFloor[nn] = (mNoiseFloor[ii].mNoiseFloor[nn] - mean)/stdDev;
      for (int nn = 0; nn < nfSize; nn++) mNoiseFloor[ii].mNoiseFloor[nn] = gwd::square(mNoiseFloor[ii].mNoiseFloor[nn]);
    
      //----------------------------------  running median
      //--------------------------------------------------------------------
      gwd::RunningMedian rm;
      unsigned int rmwz;
      Vector rmBand;
      rmwz = (unsigned int)(mWindowSizeRunningMedian*mNoiseFloor[ii].mSampleRate);
      if (flag) 
        flag = rm.ComputeMedians(rmwz,mNoiseFloor[ii].mNoiseFloor,rmBand);
      if (!flag) {
        std::string message = errStr + "can't compute median for band data.";
        single->AppendMessage(message);
        flag = false;
      }
      mNoiseFloor[ii].mNoiseFloor.clear();
      mNoiseFloor[ii].mNoiseFloor.resize(rmBand.size());
      std::copy(rmBand.begin(),rmBand.end(),mNoiseFloor[ii].mNoiseFloor.begin());
      //mNoiseFloor[ii].mNoiseFloor = rmBand;
      rmBand.clear();
    
      //---------------------------------- shift data to zero mean
      //--------------------------------------------------------------------
      Estimator(mNoiseFloor[ii].mNoiseFloor,mean,stdDev);
      for (size_t nn = 0; nn < mNoiseFloor[ii].mNoiseFloor.size(); nn++) mNoiseFloor[ii].mNoiseFloor[nn] = (mNoiseFloor[ii].mNoiseFloor[nn] - mean);
      
      //---------------------------------- check for NaN
      //--------------------------------------------------------------------
      // for (size_t nn = 0; nn < mNoiseFloor[ii].mNoiseFloor.size(); nn++) 
      // {
        // if (gwd::IsNaN(mNoiseFloor[ii].mNoiseFloor[nn]))
        // {
          // std::cout << "ERROR: NoiseFloor is NaN" << std::endl;
          // break;
        // }
      // }
    }
    
    //----------------------------------  send data to dmt-viewer
    // ----------------------------------  if there are problems do better an empty plot
    //--------------------------------------------------------------------
    if (flag) TimeSeriesPlot();
    else EmptyPlot();

    //----------------------------------  write data to file for debugging
    //--------------------------------------------------------------------
    //WriteData((unsigned int)mCurrentTime.getS());

  } // LockConditionFlag
  else
  {
    //----------------------------------  Send empty plot to the dmt-viewer
    //----------------------------------------------------------------------------------
    EmptyPlot();
  } // LockConditionFlag
 
  
  //----------------------------------  get processing time
  //--------------------------------------------------------------------
  // clk.stop();
  // clk.print(str.c_str());

  return true;
}


//=================================================================================================
//=================================================================================================
//====================================== Prepare data for plot
//=================================================================================================
void
gwd::DetectorChannel::PrepareDataForPlot(Vector& in,std::vector<float>& out)
{
  //
  // The dmtviewer can't plot more than 10000 samples at a time
  //###################################################################
  const unsigned int maxNoOfSamples = 10000;

  // obtain the time stride to be plotted
  unsigned int inSize;
  unsigned int ouSize;

  // Check no. of samples
  inSize = (unsigned int) in.size();
  ouSize = inSize;
  // Distinguish two cases:
  //  1. input signal is larger than output signal
  //     here resampling is necessary
  //  2. the input signal is shortar than maxNoOfSamples
  //     simply copy input into output
  if (ouSize < maxNoOfSamples)
  {
    out.resize(ouSize);
    for (unsigned int ii = 0; ii < ouSize; ii++) out[ii] = in[ii];
    return;
  }
  else
  {
    // can't write more samples than maxNoOfSamples
    ouSize = maxNoOfSamples;
    // Time intervals
    const double dt = mTStride / static_cast<double>(inSize - 1);
    const double DT = mTStride / static_cast<double>(ouSize);
    // get min-max
    double inMax = *(std::max_element(in.begin(),in.end()));
    double inMin = *(std::min_element(in.begin(),in.end()));

    // Check if signal has a constant value
    if (fabs(inMin - inMax) <= std::numeric_limits<float>::epsilon( ))
    {
      out.resize(ouSize);
      for (unsigned int ii = 1; ii < ouSize; ii++) out[ii] = in[ii];
    }
    else
    {
      // Set output data
      out.resize(ouSize);
      out[0] = in[0];
      for (unsigned int ii = 1; ii < ouSize; ii++)
      {
        // Find time in [0,mTStride]
        double t = static_cast<double>(ii)*DT;
        unsigned int jj = (unsigned int)floor(t/dt);
        double delta = t - static_cast<double>(jj)*dt;
        // Linear interpolate value
        out[ii] = static_cast<float>(((in[jj+1]-in[jj]) / dt)*delta + in[jj]);
      }

      // Scale the signal to min-max interval
      float outMax = *(std::max_element(out.begin(),out.end()));
      float outMin = *(std::min_element(out.begin(),out.end()));
      float alpha;
      float beta;
      if (fabs(outMin - outMax) <= std::numeric_limits<float>::epsilon( ))
      {
        alpha = 1.;
        beta  = 0.;
      }
      else
      {
        alpha = (outMax - outMin) / (inMax - inMin);
        beta  = (inMax*outMin - inMin*outMax) / (inMax - inMin);
      }

      for (unsigned int ii = 0; ii < ouSize; ii++)
      {
        out[ii] = alpha*out[ii] + beta;
      }
    }
    // finish!
    return;
  }
}

//=================================================================================================
//=================================================================================================
//====================================== DMT-viewer: Empty plot for non locked state
//=================================================================================================
void
gwd::DetectorChannel::EmptyPlot()
{
  //----------------------------------  Plot empty time series for non locked state
  //----------------------------------------------------------------------------------
  Time t0 = mCurrentTime;
  Interval dt;
  std::vector<float> ouTS(100,double(0));
  dt = mTStride / static_cast<double>(ouTS.size()-1);
  for (int ii = 0; ii < mNoFreqBands; ii++)
  {
    mNoiseFloor[ii].mTS.setData(t0, dt, &ouTS[0], ouTS.size());
    mNoiseFloor[ii].mNoiseFloor.clear();
  }
  //----------------------------------  Process twelve hours plots
  //----------------------------------------------------------------------------------
  // For the twelve hour plot use 12hs = 43200 sec
  Time t12 = mCurrentTime - Interval(43200.);
  Interval dt12;
  //----------------------------------  12hs Plot
  //----------------------------------------------------------------------------------
  for (int ii = 0; ii < mNoFreqBands; ii++)
  {
    mNoiseFloor[ii].m12hsPlot.pop_front();
    mNoiseFloor[ii].m12hsPlot.push_back(double(0));
    ouTS.resize(mNoiseFloor[ii].m12hsPlot.size());
    std::copy(mNoiseFloor[ii].m12hsPlot.begin(),mNoiseFloor[ii].m12hsPlot.end(),ouTS.begin());
    dt12 = mTStride / static_cast<double>(ouTS.size()-1);
    mNoiseFloor[ii].mTS12hs.setData(t12, dt12, &ouTS[0], ouTS.size());
  }
}

//=================================================================================================
//=================================================================================================
//====================================== DMT-viewer: ime series plot
//=================================================================================================
void
gwd::DetectorChannel::TimeSeriesPlot( )
{
  //----------------------------------  send data to dmt-viewer
  //--------------------------------------------------------------------
  for (int ii = 0; ii < mNoFreqBands; ii++)
  {
    Time t0 = mCurrentTime;
    Interval dt;
    std::vector<float> ouTS;
    PrepareDataForPlot(mNoiseFloor[ii].mNoiseFloor,ouTS);
    dt = mTStride / static_cast<double>(ouTS.size()-1);
    mNoiseFloor[ii].mTS.setData(t0, dt, &ouTS[0], ouTS.size());
  }
}


//##########################################################################################################
//=================================================================================================
//===================================== Auxiliary functions
//=================================================================================================
//##########################################################################################################

//=================================================================================================
//=================================================================================================
//================ Estimate mean value and standard deviation of a time series
//=================================================================================================
void gwd::DetectorChannel::Estimator(const Vector& ts,double& mean,double& stdDev)
{
  const int lcSize = (int) ts.size();
  mean = double(0);
  for (int ii = 0; ii < lcSize; ii++)
  {
    mean += ts[ii];
  }

  mean /= static_cast<double>(lcSize);

  stdDev = double(0);
  for (int ii = 0; ii < lcSize; ii++)
  {
    stdDev += gwd::square(ts[ii]-mean);
  }

  stdDev /= static_cast<double>(lcSize-1);
  stdDev = std::sqrt(stdDev);

}

//=================================================================================================
//=================================================================================================
//=================================================================================================
//=================================================================================================
inline void gwd::DetectorChannel::MixWithOscilator(const double carrierFreq,const double sampleRate,Vector& signal)
{
  double tt = double(0);
  double dt = 1./sampleRate;
  double factor = 2.*gwd::PI*carrierFreq;
  for (int ii = 0; ii < (int)signal.size(); ii++)
  {
    signal[ii] *= std::cos(factor*tt);
    tt += dt;
  }
}


//=================================================================================================
//=================================================================================================
void 
gwd::DetectorChannel::WriteData(const unsigned int& starttime)
{
  for (int ii = 0; ii < mNoFreqBands; ii++)
  {
    std::ostringstream tempStr;
    if (mChannelName == "H1:LSC-DARM_ERR")
    {
      tempStr << "H1-LSC-DARM_ERR-" << starttime << "-Band-" << ii << ".bin";
    }
    else if (mChannelName == "H2:LSC-DARM_ERR")
    {
      tempStr << "H2-LSC-DARM_ERR-" << starttime << "-Band-" << ii << ".bin";
    }
    else
      tempStr << mChannelName.substr(3,mChannelName.size()-3)<< "-" << starttime << "-Band-" << ii << ".bin";
    
    std::ofstream ofd(tempStr.str().c_str(),std::ios::binary);
    int size = (int)mNoiseFloor[ii].mNoiseFloor.size();
    ofd.write((char*)&size,sizeof(int));
    ofd.write((char*)&(mNoiseFloor[ii].mNoiseFloor[0]),(std::streamsize)(size*sizeof(double)));
    ofd.close();
  }
}
