/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    File LockLoss.cc
//    Trigger on loss of lock.
//    $Id: LockLoss.cc 7126 2014-06-27 09:14:07Z john.zweizig@LIGO.ORG $

#include "LockLoss.hh"

#ifndef __CINT__

#include <iostream>
#include <iomanip>
#include <fstream>
#include <string>
#include <stdlib.h>
#include <cstring>
#include <vector>

#include "TSeries.hh"
#include "FSeries.hh"
#include "Dacc.hh"
#include "DatEnv.hh"
#include "TrigRslt.hh"

// The next three lines must appear, and in this sequence
#define  PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/LockLoss/LockLoss.cc 7126 2014-06-27 09:14:07Z john.zweizig@LIGO.ORG $"
#define  PIDTITLE  "Watch for loss and acquiring of lock"
#include "ProcIdent.hh"

//
// Generate the main routine.
//
EXECDAT(LockLoss) ;
#endif // !def(__CINT__)

using namespace std;

typedef unsigned long ulong_t;

LockLoss::LockLoss(int argc, const char *argv[]) 
    : DatEnv(argc, argv),
      mDebug(0),
      mMaxFrame(999999),
      mTimeXInLock(0,0),
      mTimeYInLock(0,0),
      mTimeBothInLock(0,0),
      mTotalRunTime(0,0),
      mXLiveTimeRatio(0.),
      mYLiveTimeRatio(0.),
      mBothLiveTimeRatio(0.),
      mXLiveTimeRatio_10min(0.),
      mYLiveTimeRatio_10min(0.),
      mBothLiveTimeRatio_10min(0.),
      mXLiveTimeRatio_1hr(0.),
      mYLiveTimeRatio_1hr(0.),
      mBothLiveTimeRatio_1hr(0.),
      mXLiveTimeRatio_4hr(0.),
      mYLiveTimeRatio_4hr(0.),
      mBothLiveTimeRatio_4hr(0.),
      mXLiveTimeRatio_8hr(0.),
      mYLiveTimeRatio_8hr(0.),
      mBothLiveTimeRatio_8hr(0.),
      mXLiveTimeRatio_12hr(0.),
      mYLiveTimeRatio_12hr(0.),
      mBothLiveTimeRatio_12hr(0.),
      mOsclist(getDacc(), 0),
      mTstampFormat("%Y-%m-%d %H:%N:%S %Z"),
      mName("LockLoss_"),
      mXLock(-1,mMaxLength),
      mYLock(-1,mMaxLength),
      mBothLock(-1,mMaxLength),
      mXLock1min(-1,mMaxLength1min),
      mYLock1min(-1,mMaxLength1min),
      mBothLock1min(-1,mMaxLength1min),
      mLockState(-1,mMaxLength1min),
      mXLock6hr(-1,mMaxLength6hr),
      mYLock6hr(-1,mMaxLength6hr),
      mBothLock6hr(-1,mMaxLength6hr),
      mXLock12hr(-1,mMaxLength12hr5),
      mYLock12hr(-1,mMaxLength12hr5),
      mBothLock12hr(-1,mMaxLength12hr5),
      mTimeXInLock_10minTW(0,mMaxLength10min),
      mTimeYInLock_10minTW(0,mMaxLength10min),
      mTimeBothInLock_10minTW(0,mMaxLength10min),
      mTimeXInLock_1hrTW(0,mMaxLength1hr),
      mTimeYInLock_1hrTW(0,mMaxLength1hr),
      mTimeBothInLock_1hrTW(0,mMaxLength1hr),
      mTimeXInLock_4hrTW(0,mMaxLength4hr),
      mTimeYInLock_4hrTW(0,mMaxLength4hr),
      mTimeBothInLock_4hrTW(0,mMaxLength4hr),
      mTimeXInLock_8hrTW(0,mMaxLength8hr),
      mTimeYInLock_8hrTW(0,mMaxLength8hr),
      mTimeBothInLock_8hrTW(0,mMaxLength8hr),
      mTimeXInLock_12hrTW(0,mMaxLength12hr),
      mTimeYInLock_12hrTW(0,mMaxLength12hr),
      mTimeBothInLock_12hrTW(0,mMaxLength12hr),
      mTimeBothInLock_1000sTW(0,mMaxLength1000s),
      mXLockHist("X-Arm Lock segment histogram", 24*12, 0., 60.*24.,
                 "X-Arm Lock segment length - minutes", "Frequency"),
      mYLockHist("Y-Arm Lock segment histogram", 24*12, 0., 60.*24.,
                 "Y-Arm Lock segment length - minutes", "Frequency"),
      mBothLockHist("Both-Arms Lock segment histogram", 24*12, 0., 60.*24.,
                    "Both-Arms Lock segment length - minutes", "Frequency")
{
    if (argc < 3) {
        cerr << "Must specify IFO name (H1, H2, or L1)" << endl;
        cerr << "Usage: LockLoss -ifo ifoname [-debug n] " << endl;
        std::string eObj = "Must specify IFO name (H1, H2, or L1)";
        throw runtime_error(eObj);
    }

    bool ifoset_p = false;
        
    for (int i = 0; i < argc; ++i) {
        if (strcmp(argv[i], "-debug") == 0) {
            ++i;
            mDebug = atoi(argv[i]);
        }

        if (strcmp(argv[i], "-ifo") == 0) {
            ++i;

            if (i > argc-1 || argv[i] == NULL) {
                std::string eObj = "Must specify IFO name (H1, H2, or L1)";
                cerr << eObj << endl;
                throw runtime_error(eObj);
            }

            mIFOname = argv[i];
            ifoset_p = true;
            
            if (mIFOname != "H1" && mIFOname != "H2" && mIFOname != "L1") {
                std::string eObj = "Must specify IFO name (H1, H2, or L1)";
                cerr << eObj << endl;
                throw runtime_error(eObj);
            }
        }
    }

    if (ifoset_p == false) {
        std::string eObj = "Must specify IFO name (H1, H2, or L1)";
        cerr << eObj << endl;
        throw runtime_error(eObj);
    }

    mName += mIFOname;
    if (mDebug == 0) {
        MonServer::setServerName(mName.c_str());
    } else {
        mName += "_DEBUG";
        MonServer::setServerName(mName.c_str());
    }

    // set debug level on OSC list
    mOsclist.setDebug(0);
    
    // Write to $DMTHTMLOUT or $GDSAPACHE/monitor_reports/mName/
    char *htmldir = getenv("DMTHTMLOUT");
    if (!htmldir) {
        cerr << mName << ": warning -- environment variable DMTHTMLOUT "
             << "not set. Trying GDSAPACHE." << endl;
        
        htmldir = getenv("GDSAPACHE");
        if (!htmldir) {
            cerr << mName << ": warning -- environment variable GDSAPACHE "
                 << "not set. Writing log and duty cycle files to "
                 << "current directory" << endl;
        } else {
            cerr << mName << ": Writing log and duty cycle files to "
                 << "$GDSAPACHE/monitor_reports/" << mName << "/"
                 << endl;

            mHtmlDir  = htmldir;
            mHtmlDir += "/monitor_reports/" + mName + "/";
        }
    } else {
        cerr << mName << ": Writing log and duty cycle files to "
             << htmldir << endl;

        mHtmlDir = htmldir;
        mHtmlDir += "/";
    }
    
    mLogfileName   = mHtmlDir + "LockLoss_Log.txt";
    mDispfileName  = mHtmlDir + "LockLoss_Uptime.html";
    mIndexPageName = mHtmlDir + "index.html";
    //char *alarmHtmlDir = getenv("GDSAPACHE");
    //mLockLoss_alarmDesc = alarmHtmlDir;
    std::string urldir;
    if (mIFOname == "H1" || mIFOname == "H2")
        urldir = "http://blue.ligo-wa.caltech.edu/gds/Alarms/" + mName;
    else
        urldir = "http://london.ligo-la.caltech.edu/gds/Alarms/" + mName;
    mLockLoss_alarmDesc = urldir + "_alarmDesc.html";

    if (mDebug > 2) {
        cerr << mName << ": Log file -- " << mLogfileName << endl;
        cerr << mName << ": Display file -- " << mDispfileName << endl;
        cerr << mName << ": Index file -- " << mIndexPageName << endl;
    }

    writeIndexPage();

    if (mDebug > 2)
        std::cerr << mName << ": done writing index page" << std::endl;

    // Open log with append mode
    mLogfile.open(mLogfileName.c_str(), ios::app);

    if (!mLogfile) {
        cerr << mName << ": could not open logfile " << mLogfileName  << endl;
        cerr << "          reverting to stdout" << endl;
        mLog = &std::cout;
    } else {
        mLog = &mLogfile;
    }

    // Set alarm client name
    mAlarmClient.setName(mName.c_str());

    // Set alarm data and handles
    Interval timeout(20.0);
    int      severity(1);

    AlarmData tmp_ad1(mName, "Lock_lost", timeout, severity,
                      mLockLoss_alarmDesc);
    mLockLoss_ad = tmp_ad1;

    lmsg::error_t rc;
    rc = mAlarmClient.defineAlarm(mLockLoss_ad);
    if (rc != 0) {
        // failure
        *mLog << mName << ": error defining alarm: Lock lost: return code "
              << rc << std::endl;
    }

    Time now = getDacc().getFillTime();
    *mLog << "Creating LockLoss object at time " << now << endl;

    //
    // Set the time stride
    //
    float tStep(1.0);
    getDacc().setStride(tStep);




    std::string tmpnam = mIFOname + ":X-Arm Lock segment histogram";
    mXLockHist.SetTitle(tmpnam.c_str());
    
    tmpnam = mIFOname + ":X-Arm Lock segment length -minutes";
    mXLockHist.SetXLabel(tmpnam.c_str());
    mXLockHist.SetNLabel("Frequency");

    tmpnam = mIFOname + ":Y-Arm Lock segment histogram";
    mYLockHist.SetTitle(tmpnam.c_str());
    
    tmpnam = mIFOname + ":Y-Arm Lock segment length -minutes";
    mYLockHist.SetXLabel(tmpnam.c_str());
    mYLockHist.SetNLabel("Frequency");

    tmpnam = mIFOname + ":Both-Arms Lock segment histogram";
    mBothLockHist.SetTitle(tmpnam.c_str());
    
    tmpnam = mIFOname + ":Both-Arms Lock segment length -minutes";
    mBothLockHist.SetXLabel(tmpnam.c_str());
    mBothLockHist.SetNLabel("Frequency");
    
    // Initialize frame counter
    mFrameCount = 0;

    // Read in configuration file
    char *ifocstr = getenv("DMTIFOS");
    // char *dmtparcstr = getenv("DMTPARS");

    if (!ifocstr) {
        std::string eObj =
            "LockLoss: Don't know if we're at Hanford or Livingston. ";
        eObj += "Set the environment variable DMTIFOS and restart.";
        std::cerr << eObj << std::endl;
        throw runtime_error(eObj);
    } else {
        std::string ifos(ifocstr);
        // std::string parPath(dmtparcstr);

        // Make sure requested IFO is available at this site
        if (ifos.find(mIFOname) == std::string::npos) {
            std::string eObj = "Specified IFO (";
            eObj += mIFOname;
            eObj += ") is not available at this site";
            std::cerr << eObj << std::endl;
            throw runtime_error(eObj);
        } else {
            if (mDebug > 0) {
                // read config file from current directory
                mOsclist.readConfig("LockLoss.conf");
            } else {
                // read central config file
	        std::string oscFile;
		if (getenv("DMTPARS")) oscFile = getenv("DMTPARS");
		else           oscFile = std::string(getenv("HOME")) + "/pars";
		oscFile += "/LockLoss.conf";
                mOsclist.readConfig(oscFile.c_str());
            }
        }
    }

    // Dump all conditions that we don't care about
    if (mDebug > 0)
        cerr << mName << ": dumping unneeded IFOs" << endl;
   
    mOsclist.ignoreAllExcept(mIFOname);

    if (mDebug > 1)
        mOsclist.chanUtilInfo();

    // Once config file is read, query OSCs for channel names so we
    // know the IFO name with which to tag our triggers.  Do this
    // even though the mIFOname is set, i.e. don't trust it.

    std::string ifoname;
    ifoname = mOsclist[mIFOname + ":sgl_X_arm_thres"]->channel();

    if (mDebug > 2) {
        cerr << mName << ": ifoname = " << ifoname << endl;
        cerr << "          substr = " << ifoname.substr(0,2) << endl;
    }
    
    if (mIFOname != ifoname.substr(0,2)) {
        cerr << "IFO name specified on command line does not match IFO "
             << "name in config file" << endl;
        std::string eObj = "IFO name on cmdline does not match IFO name in config file";
        throw runtime_error(eObj);
    } 

    if (mDebug > 0) {
        cerr << mName << ": mIFOname = " << mIFOname << endl;
    }


    setTSWindowStuff();

    // once the filenames for the TSWindow objects have been set,
    // we can restore them
    restoreAllTSWindows();
    
    //
    // Serve data to DMTviewer
    //
    serveData(mXLock.name().c_str(), &mXts);
    serveData(mYLock.name().c_str(), &mYts);
    serveData(mBothLock.name().c_str(), &mBts);
    
    serveData(mXLock6hr.name().c_str(), &mXts6hr);
    serveData(mYLock6hr.name().c_str(), &mYts6hr);
    serveData(mBothLock6hr.name().c_str(), &mBts6hr);

    serveData(mXLock12hr.name().c_str(), &mXts12hr);
    serveData(mYLock12hr.name().c_str(), &mYts12hr);
    serveData(mBothLock12hr.name().c_str(), &mBts12hr);
    serveData((mIFOname+":BothArmsLocked12hExport").c_str(), &mBts12hrExp);
    
    serveData(mXLockHist.GetTitle(), &mXLockHist);
    serveData(mYLockHist.GetTitle(), &mYLockHist);
    serveData(mBothLockHist.GetTitle(), &mBothLockHist);

    //
    // Set up trending
    //

    // channel names
    //mTrendChanNames["X"] = "X-arm lock status";
    //mTrendChanNames["Y"] = "Y-arm lock status";
    //mTrendChanNames["Both"] = "Both-arm lock status";
    //mTrendChanNames["X"] = mIFOname + ":LSC-XARM_LOCK";
    //mTrendChanNames["Y"] = mIFOname + ":LSC-YARM_LOCK";
    //mTrendChanNames["Both"] = mIFOname + ":LSC-BOTHARM_LOCK";
    // Trend channels need to be named IFO:PREFIX-DESCRIPTION
    // mTrendChanNames["X"] = mIFOname + ":LSC-XARM_LOCK";
    // mTrendChanNames["Y"] = mIFOname + ":LSC-YARM_LOCK";
    // mTrendChanNames["Both"] = mIFOname + ":LSC-BOTHARMS_LOCK";
    // mTrendChanNames["SV_Op_Go"] = mIFOname + ":SV-Operator_Go";
    // mTrendChanNames["SV_Conlog"] = mIFOname + ":SV-Conlog_NoChange";
    // mTrendChanNames["SV_PSL"] = mIFOname + ":SV-PSL_OK";
    // mTrendChanNames["SV_CM"] = mIFOname + ":SV-Common_Mode";
    // mTrendChanNames["SV_ISC"] = mIFOname + ":SV-ISC_OK";
    // mTrendChanNames["SV_AutoLocked"] = mIFOname + ":SV-Auto_Locked";
    // mTrendChanNames["SV_NoExc"] = mIFOname + ":SV-No_Excitation";
    // mTrendChanNames["SV_MC_Locked"] = mIFOname + ":SV-MC_Locked";
    // mTrendChanNames["SV_ScienceMode"] = mIFOname + ":SV-Science_Mode";
    // mTrendChanNames["SV_IFO_UP"] = mIFOname + ":SV-IFO_UP";
    // mTrendChanNames["LockState"] = mIFOname + ":LSC-LOCK_STATE";

    // KR 7/28/05 renamed trend channel names 
    // mTrendChanNames["X"] = mIFOname + ":DMT-LOCKLOSS_XarmLOCK";
    // mTrendChanNames["Y"] = mIFOname + ":DMT-LOCKLOSS_YarmLOCK";
    // mTrendChanNames["Both"] = mIFOname + ":DMT-LOCKLOSS_BOTHarmsLOCK";
    // mTrendChanNames["LockState"] = mIFOname + ":DMT-LOCKLOSS_StateVEC";

    mTrendChanNames["X"] = mIFOname + ":DMT-LCKL_XARM_LOCKED";
    mTrendChanNames["Y"] = mIFOname + ":DMT-LCKL_YARM_LOCKED";
    mTrendChanNames["Both"] = mIFOname + ":DMT-LCKL_BOTH_ARMS_LOCKED";
    mTrendChanNames["LockState"] = mIFOname + ":DMT-LCKL_STATE_VECTOR";

    // KR 10/29/05 Added missing 'T' to 'DMTTREMDOUT'
    // Trend index file is $DMTTRENDOUT/channel.cfg
    char *tdir = getenv("DMTTRENDOUT");
    std::string trendDir;
    if (!tdir) {
      //KR 7/28/05 changed trends root directory name
      // trendDir = "/export/DMTtrends/";
        trendDir = "/dmt/";
        trendDir += mName;
        trendDir += "/";
        cout << "trendDir now defined to be " << trendDir << endl;
    } else {
        trendDir = tdir;
        cout << "trendDir already defined to be " << trendDir << endl;
    }

    if (mDebug > 2) {
        std::cout << "trendDir = " << trendDir << std::endl;
    }

    // ONLY WRITE INDEX ON THE FIRST TIME EVER THIS PROGRAM IS RUN!
    char indexfile[512];
    sprintf(indexfile, "%s/channel.cfg", trendDir.c_str());
    // trend frame
    mTrend.setIFO(mIFOname.c_str());
    mTrend.setName(mName.c_str());
    mTrend.setType(Trend::kMinute);
    //mTrend.setFrameLen(60);       // 60 min = 1 hr per frame
    mTrend.setFrameCount(1);
    mTrend.setAutoUpdate(false);

    if (mDebug > 2) {
        std::cout << "indexfile = " << indexfile << std::endl;
    }

    // add trend channels
    cout << "Adding X trend channel: " << mTrendChanNames["X"] << endl;
    mTrend.addChannel(mTrendChanNames["X"].c_str());
    cout << "Adding Y trend channel: " << mTrendChanNames["Y"] << endl;
    mTrend.addChannel(mTrendChanNames["Y"].c_str());
    cout << "Adding Both trend channel: " << mTrendChanNames["Both"] << endl;
    mTrend.addChannel(mTrendChanNames["Both"].c_str());
    cout << "Adding State trend channel: " << mTrendChanNames["LockState"] << endl;
    mTrend.addChannel(mTrendChanNames["LockState"].c_str());

    if (mDebug > 0)
        std::cout << "Writing index file " << indexfile << std::endl;

    mTrend.writeIndex(indexfile);

    // serve up trends to DMTviewer
    const std::string ntag = " min. trend - 1hr";
    std::string dmtviewerName = mTrendChanNames["X"] + ntag;
    serveData(dmtviewerName.c_str(),
              &(mTrend.find(mTrendChanNames["X"].c_str()).refAvgSeries()),
              "LockLoss: X-arm lock status minute trend");

    dmtviewerName = mTrendChanNames["Y"] + ntag;
    serveData(dmtviewerName.c_str(),
              &(mTrend.find(mTrendChanNames["Y"].c_str()).refAvgSeries()),
              "LockLoss: Y-arm lock status minute trend");
    
    dmtviewerName = mTrendChanNames["Both"] + ntag;
    serveData(dmtviewerName.c_str(),
              &(mTrend.find(mTrendChanNames["Both"].c_str()).refAvgSeries()),
              "LockLoss: Both-arms lock status minute trend");

    dmtviewerName = mTrendChanNames["LockState"] + ntag;
    serveData(dmtviewerName.c_str(),
              &(mTrend.find(mTrendChanNames["LockState"].c_str()).refAvgSeries()),
	      "LockLoss: Lock State minute trend");

    if (mDebug > 0) {
        std::cout << mName << ": OSC info" << std::endl;
        OperStateCondList::const_iterator iter = mOsclist.begin();
        for ( ; iter != mOsclist.end(); ++iter) {
            std::cout << (*iter).first << ": ";
            (*iter).second->printInfo();
        }

        // TSWindow objects info
        cerr << "mYLock12hr.saveFileName() = " << mYLock12hr.saveFileName()
             << endl;

        cerr << mName << ": exiting constructor" << endl;
    }
}

LockLoss::~LockLoss() 
{
    mTrend.close();

    saveAllTSWindows();
    
    Time now = getDacc().getFillTime();
    *mLog << "LockLoss is finished at time: " << now << endl;
    *mLog << endl << "***********************************************" 
          << endl << endl;

    mLogfile.close();
}

//
// XArmLockedActions()
//
void LockLoss::XArmLockedActions(const Time &now)
{
    mTimeXInLock += getDacc().getStride();

    mTimeXInLock_10minTW.append((double)(getDacc().getStride()), now);
    mTimeXInLock_1hrTW.append((double)(getDacc().getStride()), now);

    // The next three windows all have the same time interval,
    // which we will use as the update interval
    if (now.getS() % (int)(mTimeXInLock_4hrTW.dt()) == 0) {
        mTimeXInLock_4hrTW.append((double)(mTimeXInLock_4hrTW.dt()), now);
        mTimeXInLock_8hrTW.append((double)(mTimeXInLock_8hrTW.dt()), now);
        mTimeXInLock_12hrTW.append((double)(mTimeXInLock_8hrTW.dt()), now);
    }
        
    mXLock.append(1, now);
    mXLock1min.append(1, now);

    // update the 6hr series only every 5 seconds
    if (now.getS() % 5 == 0)
        mXLock6hr.append(1, now);

    if (now.getS() % 6 == 0)
        mXLock12hr.append(1, now);
        
    if (mFrameCount == 1)   // make sure we don't have a false alarm
        mXacquiredP = false;
} // END: XArmLockedActions()


//
// XArmNotLockedActions()
// 
void LockLoss::XArmNotLockedActions(const Time &now)
{
    mTimeXInLock_10minTW.append(0., now);
    mTimeXInLock_1hrTW.append(0., now);
        
    // The next three windows all have the same time interval,
    // which we will use as the update interval
    if (now.getS() % (int)(mTimeXInLock_4hrTW.dt()) == 0) {
        mTimeXInLock_4hrTW.append(0., now);
        mTimeXInLock_8hrTW.append(0., now);
        mTimeXInLock_12hrTW.append(0., now);
    }

    mXLock.append(0, now);
    mXLock1min.append(0, now);

    // update the 6hr series only every 5 seconds
    if (now.getS() % 5 == 0)
        mXLock6hr.append(0, now);

    if (now.getS() % 6 == 0)
        mXLock12hr.append(0, now);
} // END: XArmNotLockedActions()


//
// YArmLockedActions()
//
void LockLoss::YArmLockedActions(const Time &now)
{

    if (mDebug > 0) 
        cout << "YArmLockedActions(): time = " << now.getS() 
             << "; time % 5 = " << now.getS() % 5 << endl;

    mTimeYInLock += getDacc().getStride();

    mTimeYInLock_10minTW.append((double)(getDacc().getStride()), now);
    mTimeYInLock_1hrTW.append((double)(getDacc().getStride()), now);

    // The next three windows all have the same time interval,
    // which we will use as the update interval
    if (now.getS() % (int)(mTimeYInLock_4hrTW.dt()) == 0) {
        mTimeYInLock_4hrTW.append((double)(mTimeYInLock_4hrTW.dt()), now);
        mTimeYInLock_8hrTW.append((double)(mTimeYInLock_8hrTW.dt()), now);
        mTimeYInLock_12hrTW.append((double)(mTimeYInLock_12hrTW.dt()), now);
    }

    mYLock.append(1, now);
    mYLock1min.append(1, now);
        
    // update the 6hr and 12hr series only every 5 seconds
    if (now.getS() % 5 == 0)
        mYLock6hr.append(1, now);

    if (now.getS() % 6 == 0)
        mYLock12hr.append(1, now);
        
    if (mFrameCount == 1)
        mYacquiredP = false;
} // END: YArmLockedActions()


//
// YArmNotLockedActions()
//
void LockLoss::YArmNotLockedActions(const Time &now)
{
    mTimeYInLock_10minTW.append(0., now);
    mTimeYInLock_1hrTW.append(0., now);

    if (now.getS() % (int)(mTimeYInLock_4hrTW.dt()) == 0) {
        mTimeYInLock_4hrTW.append(0., now);
        mTimeYInLock_8hrTW.append(0., now);
        mTimeYInLock_12hrTW.append(0., now);
    }
        
    mYLock.append(0, now);
    mYLock1min.append(0, now);

    // update the 6hr series only every 5 seconds
    if (now.getS() % 5 == 0)
        mYLock6hr.append(0, now);

    if (now.getS() % 6 == 0)
        mYLock12hr.append(0, now);
} // END: YArmNotLockedActions()

//
// For the BothArmsLocked conditions, need to incorporate
// different values in the TSWindow for the different 
// Lock Modes, i.e. run mode, common mode, etc.
//
// Talk with Dave Barker, 2002-Aug-19
//   value            meaning
//   1                mode cleaner only locked (both-arms not locked)
//   2                both-arms locked
//   3                both-arms locked + common mode
//   4                both-arms locked + Op Go button on
//
//   The Op Go MEDM button is constrained by the state vector bits:
//   it can only be turned on if ALL state vector bits are on

//
// BothArmsLockedActions()
//
void LockLoss::BothArmsLockedActions(const Time &now)
{
    mTimeBothInLock += getDacc().getStride();

    mTimeBothInLock_10minTW.append((double)(getDacc().getStride()), now);
    mTimeBothInLock_1hrTW.append((double)(getDacc().getStride()), now);
    mTimeBothInLock_1000sTW.append(static_cast<double>(getDacc().getStride()),
                                   now);
        
    // The next three windows all have the same time interval,
    // which we will use as the update interval
    if (now.getS() % (int)(mTimeBothInLock_4hrTW.dt()) == 0) {
        mTimeBothInLock_4hrTW.append((double)(mTimeBothInLock_4hrTW.dt()),
                                     now);
        mTimeBothInLock_8hrTW.append((double)(mTimeBothInLock_8hrTW.dt()),
                                     now);
        mTimeBothInLock_12hrTW.append((double)(mTimeBothInLock_12hrTW.dt()),
                                      now);
    }
        
    mBothLock.append(1, now);
    mBothLock1min.append(1, now);

    // update the lock state trend every stride
    if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_op_go")) {
        if (mDebug>0) cout << "Appending 4 to mLockState" << endl;
        mLockState.append(4, now);
    } else if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_common_mode")) {
        if (mDebug>0) cout << "Appending 3 to mLockState" << endl;
        mLockState.append(3, now);
    } else {
        if (mDebug>0) cout << "Appending 2 to mLockState" << endl;
        mLockState.append(2, now);
    }

    // update the 6hr series only every 5 seconds
    if (now.getS() % 5 == 0) {
        if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_op_go")) {
            mBothLock6hr.append(4, now);
            if (mDebug > 0)
                cout << "Appending 4" << endl;
        } else if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_common_mode")) {
            mBothLock6hr.append(3, now);
            if (mDebug > 0)
                cout << "Appending 3" << endl;
        } else {
            mBothLock6hr.append(2, now);
            if (mDebug > 0)
                cout << "Appending 2" << endl;
        } 
    }

    // update the 12hr series every 6 seconds
    if (now.getS() % 6 == 0) {
        if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_op_go")) {
            mBothLock12hr.append(4, now);
        } else if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_common_mode")) {
            mBothLock12hr.append(3, now);
        } else {
            mBothLock12hr.append(2, now);
        }
    }
        
    if (mFrameCount == 1)
        mBacquiredP = false;
} // END: BothArmsLockedActions()


//
// BothArmsNotLockedActions()
//
void LockLoss::BothArmsNotLockedActions(const Time &now)
{
    mTimeBothInLock_10minTW.append(0., now);
    mTimeBothInLock_1hrTW.append(0., now);
    mTimeBothInLock_1000sTW.append(0., now);
        
    // The next three windows all have the same time interval,
    // which we will use as the update interval
    if (now.getS() % (int)(mTimeBothInLock_4hrTW.dt()) == 0) {
        mTimeBothInLock_4hrTW.append(0., now);
        mTimeBothInLock_8hrTW.append(0., now);
        mTimeBothInLock_12hrTW.append(0., now);
    }

    mBothLock.append(0, now);
    mBothLock1min.append(0, now);

    // it could be that the ModeCleaner is in lock
    
    // update lock state every stride
    if (mOsclist.satisfied(mIFOname + ":Mode_Cleaner_locked")) {
        if (mDebug>0) cout << "Appending 1 to mLockState" << endl;
        mLockState.append(1, now);
    } else {
        if (mDebug>0) cout << "Appending 0 to mLockState" << endl;
        mLockState.append(0, now);
    }

    // update the 6hr series only every 5 seconds, and the 12hr series
    // every 6 seconds
    if (mOsclist.satisfied(mIFOname + ":Mode_Cleaner_locked")) {
        if (now.getS() % 5 == 0)
            mBothLock6hr.append(1, now);
        
        if (now.getS() % 6 == 0)
            mBothLock12hr.append(1, now);
    } else {
        if (now.getS() % 5 == 0)
            mBothLock6hr.append(0, now);
        
        if (now.getS() % 6 == 0)
            mBothLock12hr.append(0, now);
    }
} // END: BothArmsNotLockedActions()


//
// Main loop
//
void LockLoss::ProcessData(void)
{
    // check stride alignment -- copied from PSLmon
    // this is attempt at fixing problem with zeros at start of trend frames
    Time tEnd = getDacc().getCurrentTime();
    int tSec = tEnd.getS();
    int iStride = int(getDacc().getStride());

    if (mDebug > 2) {
        cerr << mName << ": tEnd = " << tEnd << endl;
        cerr << mName << ": tSec = " << tSec << endl;
        cerr << mName << ": iStride = " << iStride << endl;
    }

    if ((tSec % iStride) != 0 || tEnd.getN() != 0) {
        tSec += iStride - (tSec % iStride);
        *mLog << mName << ": Skipping to " << tSec << " GPS." << endl;
        getDacc().seek(Time(tSec));
        return;
    }

    bool printed_p = false;
    size_t ndata     = 0;
    double totalTime = 0.;

    if (mDebug > 2) {
        cerr << mName << ": entered ProcessData()" << endl;
        mXLock.printInfo();
        mYLock.printInfo();
        mBothLock.printInfo();
        mXLock6hr.printInfo();
        mYLock6hr.printInfo();
        mBothLock6hr.printInfo();
    }
    
    ++mFrameCount;

    Time now = getDacc().getFillTime();

    //
    // Things to do on the first pass through ProcessData()
    //
    if (mFrameCount == 1) {
        mLockSumm.init(mHtmlDir, now, mDebug);
        initTSWindows(now);
    }

    // Form timestamp
    TimeStr(now, mTimestamp, mTstampFormat.c_str());

    if (mDebug > 0 || now.getS() % mLiveTimeUpdateInterval == 0) {
        *mLog << mName << ": Interval " << mFrameCount 
              << ", GPS = " << now << " (" << mTimestamp << ")" << endl;
        printed_p = true;
    }

    if (mDebug > 0) {
        if (mOsclist.satisfied(mIFOname + ":X_arm_locked"))
            *mLog << mName << ": X_arm_locked" << endl;

        if (mOsclist.satisfied(mIFOname + ":Y_arm_locked"))
            *mLog << mName << ": Y_arm_locked" << endl;

        if (mOsclist.satisfied(mIFOname + ":Both_arms_locked"))
            *mLog << mName << ": Both_arms_locked" << endl;

        printed_p = true;
    }

    mXacquiredP    =
        mOsclist.satisfied(mIFOname + ":X_arm_lock_acquired");
    mXlostP        =
        mOsclist.satisfied(mIFOname + ":X_arm_lock_lost");
    mYacquiredP    =
        mOsclist.satisfied(mIFOname + ":Y_arm_lock_acquired");
    mYlostP        =
        mOsclist.satisfied(mIFOname + ":Y_arm_lock_lost");
    mBacquiredP =
        mOsclist.satisfied(mIFOname + ":Both_arms_lock_acquired");
    mBlostP     =
        mOsclist.satisfied(mIFOname + ":Both_arms_lock_lost");

    //
    // Compute duty cycle
    //
    mTotalRunTime += getDacc().getStride();

    if (mDebug) {
        cerr << "mTotalRunTime   = " << mTotalRunTime << endl;
        cerr << "mTimeXInLock    = " << mTimeXInLock << endl;
        cerr << "mTimeYInLock    = " << mTimeYInLock << endl;
        cerr << "mTimeBothInLock = " << mTimeBothInLock << endl;

        cerr << "X_arm_locked = "
             << mOsclist.watchedQuantity(mIFOname + ":X_arm_locked")
             << endl;
        cerr << "Y_arm_locked = "
             << mOsclist.watchedQuantity(mIFOname + ":Y_arm_locked")
             << endl;

        cerr << "sgl_X_arm_thres = "
             << mOsclist.watchedQuantity(mIFOname + ":sgl_X_arm_thres")
             << endl;
        cerr << "sgl_Y_arm_thres = "
             << mOsclist.watchedQuantity(mIFOname + ":sgl_Y_arm_thres")
             << endl;

        cerr << "SV_Operator_Go = "
             << mOsclist.watchedQuantity(mIFOname + ":SV_Operator_Go")
             << "; satisfied = " << mOsclist[mIFOname + ":SV_Operator_Go"]->satisfied()
             << endl;

        cerr << "SV_Conlog_NoChange = "
             << mOsclist.watchedQuantity(mIFOname + ":SV_Conlog_NoChange")
             << "; satisfied = " << mOsclist[mIFOname + ":SV_Conlog_NoChange"]->satisfied()
             << endl;

        cerr << "SV_PSL_OK = "
             << mOsclist.watchedQuantity(mIFOname + ":SV_PSL_OK")
             << "; satisfied = " << mOsclist[mIFOname + ":SV_PSL_OK"]->satisfied()
             << endl;

        cerr << "SV_CommonMode = "
             << mOsclist.watchedQuantity(mIFOname + ":SV_CommonMode")
             << "; satisfied = " << mOsclist[mIFOname + ":SV_CommonMode"]->satisfied()
             << endl;

        cerr << "SV_ISC = "
             << mOsclist.watchedQuantity(mIFOname + ":SV_ISC")
             << "; satisfied = " << mOsclist[mIFOname + ":SV_ISC"]->satisfied()
             << endl;

        cerr << "SV_AutoLocked = "
             << mOsclist.watchedQuantity(mIFOname + ":SV_AutoLocked")
             << "; satisfied = " << mOsclist[mIFOname + ":SV_AutoLocked"]->satisfied()
             << endl;

        cerr << "SV_NoExcitation = "
             << mOsclist.watchedQuantity(mIFOname + ":SV_NoExcitation")
             << "; satisfied = " << mOsclist[mIFOname + ":SV_NoExcitation"]->satisfied()
             << endl;

        cerr << "SV_AllOnSkipExc = "
             << mOsclist.watchedQuantity(mIFOname + ":SV_AllOnSkipExc")
             << "; satisfied = " << mOsclist[mIFOname + ":SV_AllOnSkipExc"]->satisfied()
             << endl;

        cerr << endl;
    }

    if (mOsclist.satisfied(mIFOname + ":X_arm_locked") == true)
        XArmLockedActions(now);
    else
        XArmNotLockedActions(now);
    
    if (mOsclist.satisfied(mIFOname + ":Y_arm_locked") == true)
        YArmLockedActions(now);
    else
        YArmNotLockedActions(now);

    if (mOsclist.satisfied(mIFOname + ":Both_arms_locked") == true)
        BothArmsLockedActions(now);
    else
        BothArmsNotLockedActions(now);

    // JGZ: Eliminate mean computation for lock status trends.
    // Update trend data; Trend class takes care of putting data in at the
    // right time
    // mXLock1min.fillVector(mXLockData);
    // mYLock1min.fillVector(mYLockData);
    // mBothLock1min.fillVector(mBothLockData);
    // mLockState.fillVector(mLockStateData);
 
    // mTrend.trendData(mTrendChanNames["X"].c_str(), now, mean(mXLockData));
    // mTrend.trendData(mTrendChanNames["Y"].c_str(), now, mean(mYLockData));
    // mTrend.trendData(mTrendChanNames["Both"].c_str(), now,
    //                  mean(mBothLockData));

    mTrend.trendData(mTrendChanNames["X"].c_str(), now, mXLock1min.getLatest());
    mTrend.trendData(mTrendChanNames["Y"].c_str(), now, mYLock1min.getLatest());
    mTrend.trendData(mTrendChanNames["Both"].c_str(), now,
                     mBothLock1min.getLatest());

    //  KR: Eliminated mean computation for state vector - Feb 20, 2006

    //    mTrend.trendData(mTrendChanNames["LockState"].c_str(), now, 
    //                 mean(mLockStateData));
    int ivalue = mLockState.getLatest();
    double fvalue = (double)ivalue;
    if (mDebug > 0) {
      cout << "Appending fvalue= " << fvalue << " to LockState minute trend" << endl; 
    }
    mTrend.trendData(mTrendChanNames["LockState"].c_str(), now, 
                     fvalue);

    if (mDebug > 2) {
        // cerr << "mean(mXLockData) = " << mean(mXLockData) << endl; 
        // cerr << "mean(mYLockData) = " << mean(mYLockData) << endl;
        // cerr << "mean(mBothLockData) = " << mean(mBothLockData) << endl;
        cerr << "X-Lock state = " << mXLock1min.getLatest() << endl; 
        cerr << "Y-Lock state = " << mYLock1min.getLatest() << endl;
        cerr << "Both-Lock state = " << mBothLock1min.getLatest() << endl;
    }

    if (mDebug > 2) {
        cerr << "total run time = " << mTotalRunTime.GetSecs() << endl;
        cerr << "update interval = " << mLiveTimeUpdateInterval << endl;
        cerr << "update? " << now.getS() % mLiveTimeUpdateInterval << endl;
        cerr << endl;
    }

    if (mTotalRunTime.GetSecs() > 0 &&
        now.getS() % mLiveTimeUpdateInterval == 0) {

        if (mDebug > 0) {
            cout << "ProcessData: about to write status page" << endl;
        }

        mXLiveTimeRatio = (double)(mTimeXInLock.GetSecs()) /
            (double)(mTotalRunTime.GetSecs());
        mYLiveTimeRatio = (double)(mTimeYInLock.GetSecs()) /
            (double)(mTotalRunTime.GetSecs());
        mBothLiveTimeRatio = (double)(mTimeBothInLock.GetSecs()) /
            (double)(mTotalRunTime.GetSecs());

        *mLog << mName << ": Cumulative duty cycle for X arm:     " 
              << mXLiveTimeRatio * 100.  << " %" << endl;
        *mLog << mName << ": Cumulative duty cycle for Y arm:     " 
              << mYLiveTimeRatio * 100.  << " %" << endl;
        *mLog << mName << ": Cumulative duty cycle for Both arms: " 
              << mBothLiveTimeRatio * 100.  << " %" << endl;

        // Open display file with overwrite mode
        ofstream dispfile(mDispfileName.c_str(), ios::out);

        if (!dispfile) {
            cerr << mName << ": could not open dispfile "
                 << mDispfileName  << endl;
            cerr << "          reverting to stdout" << endl;
            mDisplay = &std::cout;
        } else {
            mDisplay = &dispfile;
        }
        (*mDisplay).precision(3);

        //
        // Write out livetime web page
        //
        htmlTopOfPage();
        
        if (mDebug > 2) {
            cerr << mName << ": computing 10 min live time" << endl;
        }

        //
        // compute 10 minute duty cycle
        //
        *mDisplay << "<tr>" << endl;
        if (mTotalRunTime.GetSecs() < 600) {
            // write out to display file
            htmlStatusNA("10 minute");
        } else {
            totalTime = 600.;

            // X
            mXLiveTimeRatio = 0.;
            ndata = mTimeXInLock_10minTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mXLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }

            if (mDebug > 1) {
                cerr << mName << ": mTotalRunTime.GetSecs() = "
                     << mTotalRunTime.GetSecs() << endl;
                cerr << mName << ": totalTime = " << totalTime << endl;
                cerr << mName << ": (10 min) mXLiveTimeRatio = "
                     << mXLiveTimeRatio << endl;
            }

            mXLiveTimeRatio /= totalTime;

            // Y
            mYLiveTimeRatio = 0.;
            ndata = mTimeYInLock_10minTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mYLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }

            if (mDebug > 1) {
                cerr << mName << ": (10 min) mYLiveTimeRatio = "
                     << mYLiveTimeRatio << endl;
            }
        
            mYLiveTimeRatio /= totalTime;

            // Both
            mBothLiveTimeRatio = 0.;
            ndata = mTimeBothInLock_10minTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mBothLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mBothLiveTimeRatio /= totalTime;
        

            if (mDebug > 2) {
                cerr << mName << ": done computing 10 min live time" << endl;
            }

            // write out to display file
            htmlStatus("10 minute");
        }

        // END 10 min livetime

        //
        // compute 1 hour duty cycle
        //
        if (mTotalRunTime.GetSecs() < 3600) {
            // write out to display file
            htmlStatusNA("1 hour");            
        } else {
            totalTime = 3600.;

            if (mDebug > 1)
                cerr << mName << ": totalTime = " << totalTime << endl;

            // X
            mXLiveTimeRatio = 0.;
            ndata = mTimeXInLock_1hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mXLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mXLiveTimeRatio /= totalTime;

            // Y
            mYLiveTimeRatio = 0.;
            ndata = mTimeYInLock_1hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mYLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mYLiveTimeRatio /= totalTime;

            // Both
            mBothLiveTimeRatio = 0.;
            ndata = mTimeBothInLock_1hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mBothLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mBothLiveTimeRatio /= totalTime;

            // write out to display file
            htmlStatus("1 hour");
        }

        *mDisplay << "</tr>" << endl;
        
        // END 1 hr duty cycle


        //
        // compute 4 hour duty cycle
        //
        *mDisplay << "<tr>" << endl;
        if (mTotalRunTime.GetSecs() < 14400) {
            // write out to display file
            htmlStatusNA("4 hour");
        } else {
            totalTime = 14400.;
            if (mDebug > 1)
                cerr << mName << ": totalTime = " << totalTime << endl;
        

            // X
            mXLiveTimeRatio = 0.;
            ndata = mTimeXInLock_4hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mXLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mXLiveTimeRatio /= totalTime;

            // Y
            mYLiveTimeRatio = 0.;
            ndata = mTimeYInLock_4hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mYLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mYLiveTimeRatio /= totalTime;

            // Both
            mBothLiveTimeRatio = 0.;
            ndata = mTimeBothInLock_4hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mBothLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mBothLiveTimeRatio /= totalTime;

            // write out to display file
            htmlStatus("4 hour");
        }        
        
        // END 4 hr duty cycle

        //
        // compute 8 hour duty cycle
        //
        if (mTotalRunTime.GetSecs() < 28800) {
            // write out to display file
            htmlStatusNA("8 hour");
        } else {
            totalTime = 28800.;
            if (mDebug > 1)
                cerr << mName << ": totalTime = " << totalTime << endl;
        

            // X
            mXLiveTimeRatio = 0.;
            ndata = mTimeXInLock_8hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mXLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mXLiveTimeRatio /= totalTime;

            // Y
            mYLiveTimeRatio = 0.;
            ndata = mTimeYInLock_8hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mYLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mYLiveTimeRatio /= totalTime;

            // Both
            mBothLiveTimeRatio = 0.;
            ndata = mTimeBothInLock_8hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mBothLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mBothLiveTimeRatio /= totalTime;

            // write out to display file
            htmlStatus("8 hour");
        }

        *mDisplay << "</tr>" << endl;
        // END 8 hr duty cycle

        //
        // compute 12 hour duty cycle
        //
        *mDisplay << "<tr>" << endl;
        if (mTotalRunTime.GetSecs() < 43200) {
            // write out to display file
            htmlStatusNA("12 hour", 2);
        } else {
            totalTime = 43200.;
            if (mDebug > 1)
                cerr << mName << ": totalTime = " << totalTime << endl;
        
            // X
            mXLiveTimeRatio = 0.;
            ndata = mTimeXInLock_12hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mXLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mXLiveTimeRatio /= totalTime;

            // Y
            mYLiveTimeRatio = 0.;
            ndata = mTimeYInLock_12hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mYLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mYLiveTimeRatio /= totalTime;

            // Both
            mBothLiveTimeRatio = 0.;
            ndata = mTimeBothInLock_12hrTW.fillVector(mLiveTimeData);

            if (ndata > mLiveTimeData.size()) {
                unknownError(ndata);
            }

            for (size_t i = 0; i < ndata; ++i) {
                mBothLiveTimeRatio += mLiveTimeData[i];  // total live time
                // in seconds
            }
        
            mBothLiveTimeRatio /= totalTime;

            // write out to display file
            htmlStatus("12 hour", 2);
        }
        *mDisplay << "</tr>" << endl;
        // END 12 hr duty cycle
        
        *mDisplay << "</table></center>" << endl;
        
        htmlFoot();
        
        dispfile.close();

        if (mDebug > 0) {
            cout << "ProcessData: done writing status page" << endl;
        }

        printed_p = true;
    }

    // Compute 1000 sec duty cycle (for the summary
    // status web page)
    if (mDebug > 2) {
        std::cerr << mName << ": total run time = " << mTotalRunTime.GetSecs()
                  << "; now = " << now.getS() << "; remainder = "
                  << now.getS() % 1000
                  << std::endl;
    }
    
    if (mTotalRunTime.GetSecs() > 0 && now.getS() % 1000 == 0) {
        if (mDebug > 2) {
            std::cerr << mName << ": computing 1000s duty cycle" << std::endl;
        }
        
        totalTime = 1000.;
        mBothLiveTimeRatio = 0.;
        ndata = mTimeBothInLock_1000sTW.fillVector(mLiveTimeData);

        if (ndata > mLiveTimeData.size()) {
            unknownError(ndata);
        }

        for (size_t i = 0; i < ndata; ++i) {
            // total live time in seconds
            mBothLiveTimeRatio += mLiveTimeData[i];
        }

        mBothLiveTimeRatio /= totalTime;
        lockstatus_t tmpstatus = {now.getS(), float(mBothLiveTimeRatio)};
        mLockSumm.append(tmpstatus);
        mLockSumm.dumpList();
        // END 1000 sec duty cycle
    }
    

    if (mXacquiredP) {
        mXLockStartTime = now;
        sendTrig("X_arm_lock_acquired", mIFOname, now);
        *mLog << mName << ": X arm lock ACQUIRED" << endl;
        printed_p = true;
    }

    if (mXlostP) {
        mXTimeInLock = now - mXLockStartTime;
        mXLockHist.Fill((float)(mXTimeInLock.GetSecs())/60.);
        sendTrig("X_arm_lock_lost", mIFOname, now);
        *mLog << mName << ": X arm lock LOST" << endl;
        *mLog << mName << ": X Lock segment length = "
              << (float)(mXTimeInLock.GetSecs())/60.
              << " mins" << endl;
        printed_p = true;
    }

    if (mYacquiredP) {
        mYLockStartTime = now;
        sendTrig("Y_arm_lock_acquired", mIFOname, now);
        *mLog << mName << ": Y arm lock ACQUIRED" << endl;
        printed_p = true;
    }

    if (mYlostP) {
        mYTimeInLock = now - mYLockStartTime;
        mYLockHist.Fill((float)(mYTimeInLock.GetSecs())/60.);
        sendTrig("Y_arm_lock_lost", mIFOname, now);
        *mLog << mName << ": Y arm lock LOST" << endl;
        *mLog << mName << ": Y Lock segment length = "
              << (float)(mYTimeInLock.GetSecs())/60.
              << " mins" << endl;
        printed_p = true;
    }

    if (mBacquiredP) {
        mBothLockStartTime = now;
        sendTrig("Both_arms_lock_acquired", mIFOname, now);
        *mLog << mName << ": Both arms lock ACQUIRED" << endl;
        printed_p = true;
    }

    if (mBlostP) {
        // set alarm
        lmsg::error_t rc = mAlarmClient.setAlarm(mLockLoss_ad, mLockLoss_ah);
        if (rc)
            *mLog << mName << ": Error setting alarm for LockLoss: erro code "
                  << rc << std::endl;
        else
            *mLog << mName << ": Alarm Lock lost successfully set"
                  << std::endl;
        mBothTimeInLock = now - mBothLockStartTime;
        mBothLockHist.Fill((float)(mBothTimeInLock.GetSecs())/60.);
        sendTrig("Both_arms_lock_lost", mIFOname, now);
        *mLog << mName << ": Both arms lock LOST" << endl;
        *mLog << mName << ": Both Lock segment length = "
              << (float)(mBothTimeInLock.GetSecs())/60.
              << " mins" << endl;
        printed_p = true;
    }

    // A transition occurs whenever a lock acquired or a lock lost
    // event occurs
    mTransitionP = mXacquiredP || mXlostP ||
        mYacquiredP || mYlostP || mBacquiredP || mBlostP;
    
    if (mFrameCount == 1) {
        *mLog << "Starting LockLoss with following lock states:" << endl;
        if (mOsclist[mIFOname + ":X_arm_locked"]->satisfied())
            *mLog << "    X arm locked" << endl;
        else
            *mLog << "    X arm not locked" << endl;

        if (mOsclist[mIFOname + ":Y_arm_locked"]->satisfied())
            *mLog << "    Y arm locked" << endl;
        else
            *mLog << "    Y arm not locked" << endl;
        
        if (mOsclist[mIFOname + ":Both_arms_locked"]->satisfied())
            *mLog << "    Both arms locked" << endl;
        else
            *mLog << "    Both arms not locked" << endl;

        printed_p = true;
    }

    if (mDebug > 2) {
        cerr << "mTransitionP = " << mTransitionP << endl;
        cerr << "mFrameCount  = " << mFrameCount << endl;
        cerr << mName << ": GPS = " << now << " (" << mTimestamp << ")" 
             << endl;
        *mLog << mName << ": GPS = " << now << " (" << mTimestamp << ")" 
              << endl;
    }
        
    if (mTransitionP == true || mFrameCount == 1) {
        *mLog << mName << ": GPS = " << now << " (" << mTimestamp << ")" 
              << endl;
        printed_p = true;
    }

    //
    // Update data for DMT viewer
    //
    mXLock.fillTSeries(mXts);
    mYLock.fillTSeries(mYts);
    mBothLock.fillTSeries(mBts);
    mXLock6hr.fillTSeries(mXts6hr);
    mYLock6hr.fillTSeries(mYts6hr);
    mBothLock6hr.fillTSeries(mBts6hr);
    mXLock12hr.fillTSeries(mXts12hr);
    mYLock12hr.fillTSeries(mYts12hr);
    mBothLock12hr.fillTSeries(mBts12hr);
    mBts12hrExp = mBts12hr.decimate(6);

    if (mDebug > 2) {
        cerr << mName << ": mXLock.startTime()    = " << mXLock.startTime()
             << endl;
        cerr << mName << ": mYLock.startTime()    = " << mYLock.startTime()
             << endl;
        cerr << mName << ": mBothLock.startTime() = " << mBothLock.startTime()
             << endl;
        cerr << mName << ": mXts.getStartTime()   = " << mXts.getStartTime()
             << endl;
        cerr << mName << ": mYts.getStartTime()   = " << mYts.getStartTime()
             << endl;
        cerr << mName << ": mBts.getStartTime()   = " << mBts.getStartTime()
             << endl;
        cerr << mName << ": mXts.getNSample()     = " << mXts.getNSample()
             << endl;
        cerr << mName << ": mYts.getNSample()     = " << mYts.getNSample()
             << endl;
        cerr << mName << ": mBts.getNSample()     = " << mBts.getNSample()
             << endl;

        cerr << mName << ": mXLock6hr.startTime()    = " << mXLock6hr.startTime()
             << endl;
        cerr << mName << ": mYLock6hr.startTime()    = " << mYLock6hr.startTime()
             << endl;
        cerr << mName << ": mBothLock6hr.startTime() = "
             << mBothLock6hr.startTime()
             << endl;
        cerr << mName << ": mXts6hr.getStartTime()   = "
             << mXts6hr.getStartTime()
             << endl;
        cerr << mName << ": mYts6hr.getStartTime()   = "
             << mYts6hr.getStartTime()
             << endl;
        cerr << mName << ": mBts6hr.getStartTime()   = "
             << mBts6hr.getStartTime()
             << endl;
        cerr << mName << ": mXts6hr.getNSample()     = " << mXts6hr.getNSample()
             << endl;
        cerr << mName << ": mYts6hr.getNSample()     = " << mYts6hr.getNSample()
             << endl;
        cerr << mName << ": mBts6hr.getNSample()     = " << mBts6hr.getNSample()
             << endl;
    }

    if (printed_p == true) 
        *mLog << endl;

    // update trend data
    mTrend.Update();

    // save on I/O time. since the saveAllTSWindows is also done in
    // the destructor, this is only for "checkpointing"
    if (mFrameCount > 10 && (now.getS() % 600) == 0)
        saveAllTSWindows();

    return;
} /* END: ProcessData() */


void LockLoss::sendTrig(string trigname, string ifoname, Time now)
{
    trig::TrigRslt tout("Lock transition",trigname.c_str(),0);
    tout.setIfos(ifoname.c_str());
    tout.setTime(now);

    if (mDebug > 0) {
        *mLog << mName << ": Debug trigger of subtype " << trigname
              << " at GPS " << now << endl;
    } else {
        tout.setDisposition(trig::d_metaDB + trig::d_alarm);
        *mLog << mName << ": Sending trigger of subtype " << trigname
              << " at GPS " << now << endl;
    
        lmsg::error_type rc = sendTrigger(tout);

        if (rc)
            *mLog << mName << ": Error sending trigger: error code "
                  << rc << endl;
        else
            *mLog << mName << ": Trigger successfully sent" << endl;
    }

}  /* END: sendTrig() */


void LockLoss::htmlHead(void) const
{
    *mDisplay << "<!doctype html public \"-//W3C//DTD HTML 4.0 Transitional//EN\""
              << endl;
    *mDisplay << "\"http://www.w3.org/TR/REC-html40/loose.dtd\">"
              << endl;

    *mDisplay << "<html><head>" << endl;
    *mDisplay << "<!-- dwchin@umich.edu -->" << endl;
    *mDisplay << "<meta http-equiv=\"Refresh\" content=\"" << mHTMLupdateInterval
              << "\">" << endl;
    *mDisplay << "<style type=\"text/css\">" << endl;
    *mDisplay << "   <!-- body { color: black; background: white; " << endl;
    *mDisplay << "               font-family: Helvetica, Arial, sans-serif; " << endl
              << "               font-weight: normal; } " << endl
              << "        h1   { font-weight: bold; }" << endl
              << "        h2   { font-weight: bold; }" << endl
              << "        h3   { font-weight: bold; }" << endl
              << "        h4   { font-weight: bold; }" << endl
              << "        h5   { font-weight: bold; }" << endl
              << "    -->" << endl;
    *mDisplay << "</style>" << endl;
    *mDisplay << "<title>LockLoss: Current Status for " << mIFOname;
    if (mIFOname == "H1") {
        *mDisplay << " (Hanford 4k)";
    } else if (mIFOname == "H2") {
        *mDisplay << " (Hanford 2k)";
    } else if (mIFOname == "L1") {
        *mDisplay << " (Livingston 4k)";
    } else {
        // should never reach here since mIFOname is checked before
        std::string eObj = "Unknown IFO: ";
        eObj += mIFOname;
        cerr << eObj << endl;
        throw runtime_error(eObj);
    }
    *mDisplay << "</title>" << endl;

    *mDisplay << "</head>" << endl;
    
    *mDisplay << "<body><h1>Current Lock Status for " << mIFOname;
    if (mIFOname == "H1") {
        *mDisplay << " (Hanford 4k)";
    } else if (mIFOname == "H2") {
        *mDisplay << " (Hanford 2k)";
    } else if (mIFOname == "L1") {
        *mDisplay << " (Livingston 4k)";
    } else {
        // should never reach here since mIFOname is checked before
        std::string eObj = "Unknown IFO: ";
        eObj += mIFOname;
        cerr << eObj << endl;
        throw runtime_error(eObj);
    }
    *mDisplay << "</h1>" << endl;

}

void LockLoss::htmlFoot(void) const
{
    *mDisplay << "<p>&nbsp;</p><hr>" << endl;
    *mDisplay << "Contact: <a href=\"mailto:dwchin@umich.edu\">David Chin <tt>&lt;dwchin@umich.edu&gt;</tt> 734-730-1274</a>"
              << endl;
    *mDisplay << "</body></html>" << endl;
}

void LockLoss::htmlTopOfPage(void)
{
    htmlHead();

    (*mDisplay).setf(ios::fixed, ios::floatfield);
    (*mDisplay).precision(2);
    
    *mDisplay << "<h3>Last update: " << mTimestamp << "</h3>" << endl;
    *mDisplay << "<p>LockLoss (" << mIFOname << ") has been running "
              << "for " << mTotalRunTime.GetSecs() << " seconds ("
              << (double)(mTotalRunTime.GetSecs())/3600. << " hrs)</p>" << endl;
    *mDisplay << "<p><font size=\"-1\"><em>Updates every 60 seconds ";
    *mDisplay << "($Revision: 7126 $)</em></font></p>" << endl;

    *mDisplay << "<p><div style=\"font-weight: bold;\">Note:</div> "
              << "<table border=\"1\" width=\"60%\"><tr><td><b>DMTviewer TSeries "
              << "value</b></td>"
              << "<td><b>Meaning</b></td></tr> "
              << "<tr><td>-1</td><td>No data available</td></tr>"
              << "<tr><td>0</td><td>Mode Cleaner not locked</td></tr>"
              << "<tr><td>1</td><td>Mode Cleaner locked, but not BothArms</td></tr>"
              << "<tr><td>2</td><td>Mode Cleaner locked, BothArms locked</td></tr>"
              << "<tr><td>3</td><td>Mode Cleaner locked, BothArms locked, "
              << "Common Mode</td></tr>"
              << "<tr><td>4</td><td>Mode Cleaner locked, BothArms locked, Common Mode, "
              << "Operator Go on (<i>i.e.</i> Run Mode)</td></tr></table>" << endl;

    *mDisplay << "<p>&nbsp;</p>" << endl;

    *mDisplay << "<center><table border=\"1\">" << endl;
    *mDisplay << "<tr>" << endl;
    if (mOsclist.satisfied(mIFOname + ":X_arm_locked")) 
        *mDisplay << "<td valign=\"middle\"><h3>X arm <font color=\"#00a0a0\">LOCKED</font>"
                  << "</h3></td> " << endl;
    else
        *mDisplay << "<td valign=\"middle\"><h3>X arm <font color=\"#ff0000\">NOT LOCKED</font>"
                  << "</h3></td> " << endl;
        
    if (mOsclist.satisfied(mIFOname + ":Y_arm_locked")) 
        *mDisplay << "<td valign=\"middle\"><h3>Y arm <font color=\"#00a0a0\">LOCKED</font>"
                  << "</h3></td> " << endl;
    else
        *mDisplay << "<td valign=\"middle\"><h3>Y arm <font color=\"#ff0000\">NOT LOCKED</font>"
                  << "</h3></td> " << endl;
    *mDisplay << "</tr></table>" << endl;

    *mDisplay << "<p>&nbsp;</p>" << endl;

    *mDisplay << "<center><table border=\"1\">" << endl;
    *mDisplay << "<tr><td valign=\"middle\"><h4>X arm in lock for " 
              << mTimeXInLock.GetSecs() << " s</h4></td>" << endl;
    *mDisplay << "<td valign=\"middle\"><h4>Y arm in lock for " 
              << mTimeYInLock.GetSecs() << " s</h4></td></tr>" << endl;
    *mDisplay << "<tr><td colspan=\"2\" valign=\"middle\" align=\"center\"><h4>Both arms in lock for " 
              << mTimeBothInLock.GetSecs() << " s</h4></td></tr>" << endl;
    *mDisplay << "</table></center>" << endl;

    *mDisplay << "<p>&nbsp;</p>" << endl;

    *mDisplay << "<table border=\"1\">" << endl;

    *mDisplay << "<tr><td colspan=\"2\" align=\"center\">" << endl;
    *mDisplay << "<h3>Cumulative duty cycle for "
              << "<font color=\"#008040\">X arm</font>: "
              << "<font size=\"+2\" color=\"#008040\">"
              << mXLiveTimeRatio * 100. << " %</font></h3>" << endl;
    *mDisplay << "<h3>Cumulative duty cycle for "
              << "<font color=\"#000080\">Y arm</font>: "
              << "<font size=\"+2\" color=\"#000080\">"
              << mYLiveTimeRatio * 100. << " %</font></h3>" << endl;
    *mDisplay << "<h3>Cumulative duty cycle for "
              << "<font color=\"#800000\">Both arms</font>: "
              << "<font size=\"+2\" color=\"#800000\">"
              << mBothLiveTimeRatio * 100. << " %</font></h3>" << endl;
    *mDisplay << "</td></tr>" << endl;
}
    
void LockLoss::htmlStatusNA(const char *timeIntervalStr,
                            const int colspan) const
{
    *mDisplay << "<td colspan=\"" << colspan << "\">" << endl;
    *mDisplay << "<h3>" << timeIntervalStr << " duty cycle for "
              << "<font color=\"#008040\">X arm</font>: "
              << "<font size=\"+2\" color=\"#008040\">"
              << "N/A</font></h3>" << endl;
    *mDisplay << "<h3>" << timeIntervalStr << " duty cycle for "
              << "<font color=\"#000080\">Y arm</font>: "
              << "<font size=\"+2\" color=\"#000080\">"
              << "N/A</font></h3>" << endl;
    *mDisplay << "<h3>" << timeIntervalStr << " duty cycle for "
              << "<font color=\"#800000\">Both arms</font>: "
              << "<font size=\"+2\" color=\"#800000\">"
              << "N/A</font></h3>" << endl;
    *mDisplay << "</td>" << endl;
}


void LockLoss::htmlStatus(const char *timeIntervalStr,
                          const int colspan) const
{
    // write out to display file
    *mDisplay << "<td colspan=\"" << colspan << "\">" << endl;
    *mDisplay << "<h3>" << timeIntervalStr << " duty cycle for "
              << "<font color=\"#008040\">X arm</font>: "
              << "<font size=\"+2\" color=\"#008040\">"
              << mXLiveTimeRatio * 100. << " %</font></h3>" << endl;
    *mDisplay << "<h3>" << timeIntervalStr << " duty cycle for "
              << "<font color=\"#000080\">Y arm</font>: "
              << "<font size=\"+2\" color=\"#000080\">"
              << mYLiveTimeRatio * 100. << " %</font></h3>" << endl;
    *mDisplay << "<h3>" << timeIntervalStr << " duty cycle for "
              << "<font color=\"#800000\">Both arms</font>: "
              << "<font size=\"+2\" color=\"#800000\">"
              << mBothLiveTimeRatio * 100. << " %</font></h3>" << endl;
    *mDisplay << "</td>" << endl;
}


void LockLoss::unknownError(const size_t ndata) const
{
    /* // gcc 2.95.1 doesn't have sstream
      std::ostringstream eObj;
    eObj << mName << ": Unknown error: ndata > mLiveTimeData.size()"
         << ": ndata = " << ndata
         << "; mLiveTimeData.size() = " << mLiveTimeData.size()
         << endl;
    */
    char tmpstr[256];
    sprintf(tmpstr, 
	    "%s: Unknown error: ndata > mLiveTimeData.size(): "
	    "ndata = %zd; mLiveTimeData.size() = %zd", 
	    mName.c_str(), ndata, mLiveTimeData.size());
    std::string eObj(tmpstr);
    cerr << eObj << endl;
    throw runtime_error(eObj);
}


//
// LockLoss::printIndexPage()
//
void LockLoss::writeIndexPage(void)
{
    mIndexPage.open(mIndexPageName.c_str(), ios::out);
    
    if (!mIndexPage) {
        char tmpstr[256];
        sprintf(tmpstr,
                "%s: Error opening index.html file for writing. Aborting.",
                mName.c_str());
        std::string eObj(tmpstr);
        std::cerr << eObj << std::endl;
        throw runtime_error(eObj);
    } else {
        mIndexPage << "<!doctype html public \"-//w3c//dtd html 4.0 transitional//en\">"
                   << std::endl;
        mIndexPage << "<html><head> " << std::endl;
        mIndexPage << "<!-- dwchin@umich.edu -->" << std::endl;
        mIndexPage << "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\">";
        mIndexPage << "<style type=\"text/css\">" << std::endl;
        mIndexPage << "   <!-- body { color: black; background: white; "
                   << "               font-family: Helvetica, Arial, sans-serif; "
                   << "               font-weight: normal; } "
                   << "        h1   { font-weight: bold; }"
                   << "        h2   { font-weight: bold; }"
                   << "        h3   { font-weight: bold; }"
                   << "        h4   { font-weight: bold; }"
                   << "        h5   { font-weight: bold; }"
                   << " -->" << std::endl
                   << "</style>" << std::endl;
        mIndexPage << "<title>" << mName << "</title>" << std::endl;
        mIndexPage << "</head><body><center>" << std::endl;
        mIndexPage << "<h2>Global summary for " << mName
                   << " DMT Monitor running at ";

        if (mIFOname == "H1" || mIFOname == "H2")
            mIndexPage << "Hanford";
        else if (mIFOname == "L1")
            mIndexPage << "Livingston";
        else
            mIndexPage << "Timbuctoo";
             
        mIndexPage << "</h2></center>" << std::endl;

        mIndexPage << "<h3>" << mIFOname << " Interferometer (";

        if (mIFOname == "H1" || mIFOname == "L1")
            mIndexPage << "4";
        else if (mIFOname == "H2")
            mIndexPage << "2";

        mIndexPage << "km)</h3>" << std::endl;

        mIndexPage << "<ul>";
        mIndexPage << "<li><a href=\"LockLoss_Uptime.html\">";
        mIndexPage << "Livetime summary page</a></li>";
        mIndexPage << "<li><a href=\"LockLoss_Log.txt\">";
        mIndexPage << "Raw log file</a></li>";
        mIndexPage << "</ul>";
        mIndexPage << "(Time series of lock histories and histograms of ";
        mIndexPage << "lock lengths may be seen via the GDS&nbsp;DMT Viewer)";
        mIndexPage << "<p>&nbsp;</p>" << std::endl;
	mIndexPage << "<pre>" << std::endl;
	mIndexPage << "       1.  Program Name: \n"
                   << "       LockLoss\n\n"
                   << "       2.  Authors:\n"
                   << "       David Chin &lt;dwchin@umich.edu&gt;\n"
                   << "       Keith Riles &lt;kriles@umich.edu&gt;\n"
                   << "       University of Michigan\n\n"
                   << "       3.  Purpose:\n\n"
                   << "       Watch for acquisition and loss of lock. Based on thresholds\n"
                   << "       in normalized transmitted light through the ends of the X \n"
                   << "       and Y arms.\n\n"
                   << "       4.  Outputs:\n\n"
                   << "       A. Log and status web page\n\n"
                   << "       Base directory: $GDSAPACHE/monitor_reports/LockLoss_XX\n"
                   << "       browsable as http://server/gds/monitor_reports/LockLoss_XX\n"
                   << "       where XX is one of {H1, H2, L1},\n"
                   << "       server is one of blue.ligo-wa.caltech.edu\n"
                   << "                        london.ligo-la.caltech.edu   \n\n"
                     << "       Let the base directory path or URL be assigned to BASEDIR.\n"
                     << "       Log file:  $BASEDIR/LockLoss_Log.txt\n"
                     << "       Status file: $BASEDIR/LockLoss_Uptime.html\n\n"
                   << "       B. Graphical output\n\n"
                     << "       i) Time series data (DMTviewer)\n\n"
                       << "       LockLoss produces several time series of data of the \n"
                       << "       lock state of the interferometer.  The time series also\n"
                       << "       includes different values based on the value of particular\n"
                       << "       sets of bits in the State Vector.  These time series data\n"
                       << "       are viewable using the DMTviewer.\n"
                     << "       \n"
                       << "       There are time series of different durations, the most useful\n"
                       << "       (and most used) one being the 12 hour time series.\n"
                     << "       \n"
                     << "       \n"
                     << "       ii) Lock segment histogram (DMTviewer)\n"
                     << "       \n"
                       << "       LockLoss also produces a histogram of lock stretches.  Again,\n"
                       << "       these are viewable using DMTviewer.\n"
                     << "       \n"
                     << "       iii) Trends (Data Viewer)\n"
                     << "       \n"
                       << "       Minute trends of the same time series data are generated,\n"
                       << "       as well.  Data Viewer is used to plot these in almost real time.\n"
                     << "       \n"
                       << "       Trend channel names are:\n"
                     << "       \n"
                         << "       XX:DMT-LOCKLOSS_XarmLOCK\n"
                         << "       XX:DMT-LOCKLOSS_YarmLOCK\n"
                         << "       XX:DMT-LOCKLOSS_BOTHarmsLOCK\n"
                         << "       XX:DMT-LOCKLOSS_StateVEC\n"
            << "       \n"
                       << "       The XarmLOCK, YarmLOCK, and BOTHarmsLOCK trends are\n"
                       << "       self-descriptive.  The StateVEC trend is a trend of\n"
                       << "       the StateVector data as displayed in the Control Room.\n"
                     << "       \n"
                  << "       C. Alarms\n"
                  << "       \n"
                     << "       LockLoss generates one alarm:\n"
                    << "       \n"
                        << "       Lock_lost\n"
                        << "       \n"
                     << "       (See http://server/gds/Alarms/)\n"
                        << "       \n"
                  << "       D. Meta-db triggers\n"
                     << "       \n"
                     << "       Triggers are generated in the meta-database.  The triggers are:\n"
         
                       << "       Trigger type:  \"Lock transition\"\n"
                       << "       Trigger subtypes:\n"
           
                         << "       i)    X_arm_lock_acquired\n"
                         << "       ii)   X_arm_lock_lost\n"
                         << "       iii)  Y_arm_lock_acquired\n"
                         << "       iv)   Y_arm_lock_lost\n"
                         << "       v)    Both_arms_lock_acquired\n"
                         << "       vi)   Both_arms_lock_lost\n"

               << "       5.  Usage Notes:\n"
                   << "       Not meant to be run stand-alone by general users.\n"
            << "       $Id: LockLoss.cc 7126 2014-06-27 09:14:07Z john.zweizig@LIGO.ORG $\n"

            << "       </pre>\n"
        << "<p>&nbsp;</p><hr><p>Contact: "
                   << "<a href=\"mailto:dwchin@umich.edu\">David Chin &lt;dwchin@umich.edu&gt;</a>"
	            << "       </body>\n"
            << "       </html>\n";
        mIndexPage << "</body></html>" << std::endl;
    }

    mIndexPage.close();
}
// END: printIndexPage()


void LockLoss::setTSWindowStuff(void)
{
    
    //
    // Initialize and set TSWindow parameters and histogram parameters
    //
    Interval dt(1, 0);

    // strings for savefile names
    std::string chartstr = mHtmlDir + mIFOname + "_StripChart";
    std::string timeinlockstr = mHtmlDir + mIFOname + "_TimeInLock";

    if (mDebug > 0) {
        cout << "TSWindow: chartstr = " << chartstr << std::endl;
    }
        
    mXLock.setDebug(mDebug);
    mXLock.setDt(dt);
    mXLock.setName(mIFOname + ":X-arm_locked");
    mXLock.setSaveFileName(chartstr + "_X_2hr.dat");

    if (mDebug > 0) {
        cout << "TSWindow: mXLock.saveFileName = " << mXLock.saveFileName() << std::endl;
    }

    mYLock.setDebug(mDebug);
    mYLock.setDt(dt);
    mYLock.setName(mIFOname + ":Y-arm_locked");
    mYLock.setSaveFileName(chartstr + "_Y_2hr.dat");

    mBothLock.setDebug(mDebug);
    mBothLock.setDt(dt);
    mBothLock.setName(mIFOname + ":Both_arms_locked");
    mBothLock.setSaveFileName(chartstr + "_Both_2hr.dat");

    mTimeXInLock_10minTW.setDebug(0);
    mTimeXInLock_10minTW.setDt(dt);
    mTimeXInLock_10minTW.setSaveFileName(timeinlockstr + "_X_10min.dat");
    mTimeYInLock_10minTW.setDebug(0);
    mTimeYInLock_10minTW.setDt(dt);
    mTimeYInLock_10minTW.setSaveFileName(timeinlockstr + "_Y_10min.dat");
    mTimeBothInLock_10minTW.setDebug(0);
    mTimeBothInLock_10minTW.setDt(dt);
    mTimeBothInLock_10minTW.setSaveFileName(timeinlockstr + "_Both_10min.dat");

    mTimeXInLock_1hrTW.setDebug(0);
    mTimeXInLock_1hrTW.setDt(dt);
    mTimeXInLock_1hrTW.setSaveFileName(timeinlockstr + "_X_1hr.dat");
    mTimeYInLock_1hrTW.setDebug(0);
    mTimeYInLock_1hrTW.setDt(dt);
    mTimeYInLock_1hrTW.setSaveFileName(timeinlockstr + "_Y_1hr.dat");
    mTimeBothInLock_1hrTW.setDebug(0);
    mTimeBothInLock_1hrTW.setDt(dt);
    mTimeBothInLock_1hrTW.setSaveFileName(timeinlockstr + "_Both_1hr.dat");
    mTimeBothInLock_1000sTW.setDebug(0);
    mTimeBothInLock_1000sTW.setDt(dt);
    mTimeBothInLock_1000sTW.setSaveFileName(timeinlockstr + "_Both_1000s.dat");

    // 5 second interval
    dt.SetS(5);
    dt.SetN(0);
    
    mXLock6hr.setDebug(0);
    mXLock6hr.setDt(dt);
    mXLock6hr.setName(mIFOname + ":X-arm_locked_6hr");
    mXLock6hr.setSaveFileName(chartstr + "_X_6hr.dat");

    mYLock6hr.setDebug(0);
    mYLock6hr.setDt(dt);
    mYLock6hr.setName(mIFOname + ":Y-arm_locked_6hr");
    mYLock6hr.setSaveFileName(chartstr + "_Y_6hr.dat");

    mBothLock6hr.setDebug(0);
    mBothLock6hr.setDt(dt);
    mBothLock6hr.setName(mIFOname + ":Both_arms_locked_6hr");
    mBothLock6hr.setSaveFileName(chartstr + "_Both_6hr.dat");

    mTimeXInLock_4hrTW.setDebug(0);
    mTimeXInLock_4hrTW.setDt(dt);
    mTimeXInLock_4hrTW.setSaveFileName(timeinlockstr + "_X_4hr.dat");
    mTimeYInLock_4hrTW.setDebug(0);
    mTimeYInLock_4hrTW.setDt(dt);
    mTimeYInLock_4hrTW.setSaveFileName(timeinlockstr + "_Y_4hr.dat");
    mTimeBothInLock_4hrTW.setDebug(0);
    mTimeBothInLock_4hrTW.setDt(dt);
    mTimeBothInLock_4hrTW.setSaveFileName(timeinlockstr + "_Both_4hr.dat");

    mTimeXInLock_8hrTW.setDebug(0);
    mTimeXInLock_8hrTW.setDt(dt);
    mTimeXInLock_8hrTW.setSaveFileName(timeinlockstr + "_X_8hr.dat");
    mTimeYInLock_8hrTW.setDebug(0);
    mTimeYInLock_8hrTW.setDt(dt);
    mTimeYInLock_8hrTW.setSaveFileName(timeinlockstr + "_Y_8hr.dat");
    mTimeBothInLock_8hrTW.setDebug(0);
    mTimeBothInLock_8hrTW.setDt(dt);
    mTimeBothInLock_8hrTW.setSaveFileName(timeinlockstr + "_Both_8hr.dat");
    
    mTimeXInLock_12hrTW.setDebug(0);
    mTimeXInLock_12hrTW.setDt(dt);
    mTimeXInLock_12hrTW.setSaveFileName(timeinlockstr + "_X_12hr.dat");
    mTimeYInLock_12hrTW.setDebug(0);
    mTimeYInLock_12hrTW.setDt(dt);
    mTimeYInLock_12hrTW.setSaveFileName(timeinlockstr + "_Y_12hr.dat");
    mTimeBothInLock_12hrTW.setDebug(0);
    mTimeBothInLock_12hrTW.setDt(dt);
    mTimeBothInLock_12hrTW.setSaveFileName(timeinlockstr + "_Both_12hr.dat");

    // 6 second interval
    dt.SetS(6);
    dt.SetN(0);
    
    mXLock12hr.setDebug(0);
    mXLock12hr.setDt(dt);
    mXLock12hr.setName(mIFOname + ":X-arm_locked_12hr");
    mXLock12hr.setSaveFileName(chartstr + "_X_12hr.dat");

    mYLock12hr.setDebug(0);
    mYLock12hr.setDt(dt);
    mYLock12hr.setName(mIFOname + ":Y-arm_locked_12hr");
    mYLock12hr.setSaveFileName(chartstr + "_Y_12hr.dat");

    mBothLock12hr.setDebug(0);
    mBothLock12hr.setDt(dt);
    mBothLock12hr.setName(mIFOname + ":Both_arms_locked_12hr");
    mBothLock12hr.setSaveFileName(chartstr + "_Both_12hr.dat");
}

void LockLoss::initTSWindows(const Time & now)
{

    if (mDebug > 0) 
        std::cout << "LockLoss::initTSWindows() Entered: now = " 
                  << now << std::endl;
    // have to put the starting time back for each TSWindow

    mXLock.fillMaybe(now-Interval(7200,0));
    mYLock.fillMaybe(now-Interval(7200,0));
    mBothLock.fillMaybe(now-Interval(7200,0));

    mXLock1min.fillMaybe(now-Interval(60,0));
    mYLock1min.fillMaybe(now-Interval(60,0));
    mBothLock1min.fillMaybe(now-Interval(60,0));

    mLockState.fillMaybe(now-Interval(60,0));

    mXLock6hr.fillMaybe(now-Interval(6*60*60,0));
    mYLock6hr.fillMaybe(now-Interval(6*60*60,0));
    mBothLock6hr.fillMaybe(now-Interval(6*60*60,0));

    mXLock12hr.fillMaybe(now-Interval(12*60*60,0));
    mYLock12hr.fillMaybe(now-Interval(12*60*60,0));
    mBothLock12hr.fillMaybe(now-Interval(12*60*60,0));

    //mTimeXInLock_10minTW.fillMaybe(now-Interval(10*60,0));
    //mTimeYInLock_10minTW.fillMaybe(now-Interval(10*60,0));
    //mTimeBothInLock_10minTW.fillMaybe(now-Interval(10*60,0));

    //mTimeXInLock_1hrTW.fillMaybe(now-Interval(60*60,0));
    //mTimeYInLock_1hrTW.fillMaybe(now-Interval(60*60,0));
    //mTimeBothInLock_1hrTW.fillMaybe(now-Interval(60*60,0));

    //mTimeXInLock_4hrTW.fillMaybe(now-Interval(4*60*60,0));
    //mTimeYInLock_4hrTW.fillMaybe(now-Interval(4*60*60,0));
    //mTimeBothInLock_4hrTW.fillMaybe(now-Interval(4*60*60,0));

    //mTimeXInLock_8hrTW.fillMaybe(now-Interval(8*60*60,0));
    //mTimeYInLock_8hrTW.fillMaybe(now-Interval(8*60*60,0));
    //mTimeBothInLock_8hrTW.fillMaybe(now-Interval(8*60*60,0));

    //mTimeXInLock_12hrTW.fillMaybe(now-Interval(12*60*60,0));
    //mTimeYInLock_12hrTW.fillMaybe(now-Interval(12*60*60,0));
    //mTimeBothInLock_12hrTW.fillMaybe(now-Interval(12*60*60,0));

    //mTimeBothInLock_1000sTW.fillMaybe(now-Interval(1000,0));
}


void LockLoss::saveAllTSWindows(void)
{
    // I wish I had used a hash_map here, so I could iterate

    // Don't need to save the 1min TSWindows because those contain
    // data that's trended and written to Frames, anyway

    mXLock.save();    mYLock.save();    mBothLock.save();
    
    mXLock6hr.save();    mYLock6hr.save();    mBothLock6hr.save();
    
    mXLock12hr.save();    mYLock12hr.save();    mBothLock12hr.save();
    
    //mTimeXInLock_10minTW.save();
    //mTimeYInLock_10minTW.save();
    //mTimeBothInLock_10minTW.save();
    
    //mTimeXInLock_1hrTW.save();
    //mTimeYInLock_1hrTW.save();
    //mTimeBothInLock_1hrTW.save();
    
    //mTimeXInLock_4hrTW.save();
    //mTimeYInLock_4hrTW.save();
    //mTimeBothInLock_4hrTW.save();

    //mTimeXInLock_8hrTW.save();
    //mTimeYInLock_8hrTW.save();
    //mTimeBothInLock_8hrTW.save();

    //mTimeXInLock_12hrTW.save();
    //mTimeYInLock_12hrTW.save();
    //mTimeBothInLock_12hrTW.save();

    //mTimeBothInLock_1000sTW.save();
}


void LockLoss::restoreAllTSWindows(void)
{
    // I wish I had used a hash_map here, so I could iterate

    mXLock.restore();    mYLock.restore();    mBothLock.restore();
    
    mXLock6hr.restore();    mYLock6hr.restore();    mBothLock6hr.restore();
    
    mXLock12hr.restore();    mYLock12hr.restore();    mBothLock12hr.restore();
    
    //mTimeXInLock_10minTW.restore();
    //mTimeYInLock_10minTW.restore();
    //mTimeBothInLock_10minTW.restore();
    
    //mTimeXInLock_1hrTW.restore();
    //mTimeYInLock_1hrTW.restore();
    //mTimeBothInLock_1hrTW.restore();
    
    //mTimeXInLock_4hrTW.restore();
    //mTimeYInLock_4hrTW.restore();
    //mTimeBothInLock_4hrTW.restore();

    //mTimeXInLock_8hrTW.restore();
    //mTimeYInLock_8hrTW.restore();
    //mTimeBothInLock_8hrTW.restore();

    //mTimeXInLock_12hrTW.restore();
    //mTimeYInLock_12hrTW.restore();
    //mTimeBothInLock_12hrTW.restore();

    //mTimeBothInLock_1000sTW.restore();
}

// ********************************************************
// ********************************************************
// ********************************************************

//
// Here begins LockSummary stuff
//

//
// LockSummary::LockSummary
//
void LockSummary::init(const std::string &htmldir,
                       const Time &now,
                       int dbg)
{
    mDebug = dbg;
    
    mFilename = htmldir + "/summary.txt";

    if (mDebug > 0) {
        std::cerr << "LockSummary: entered init()" << std::endl;
        std::cerr << "        htmldir = " << htmldir << std::endl;
        std::cerr << "        now     = " << now << std::endl;
        std::cerr << "        mFilename = " << mFilename << std::endl;
    }

    // set the current time back to the last '000,
    // ditto the start time
    ulong_t nowRound = now.getS() - (now.getS() % 1000);
    ulong_t startTime = nowRound - (mLength - static_cast<ulong_t>(1))*mDT;
    
    if (mDebug > 2) {
        std::cerr << "nowRound = " << nowRound << std::endl;
        std::cerr << "startTime = " << startTime << std::endl;
    }

    mFile.open(mFilename.c_str(), ios::in);
    if (!mFile) {
        // the summary file does not exist; fill in the list with -1's
        // and create the file for output

        if (mDebug > 2) {
            std::cerr << "LockSummary: summary file does not exist. "
                      << "Creating..." << std::endl;
        }
        
        mFile.close();

        lockstatus_t tmpstate;
        for (ulong_t i = 0; i < mLength; ++i) {
            tmpstate.gpstime  = startTime + i*mDT;
            tmpstate.livefrac = -1.;

            mLockList.push_back(tmpstate);
        }
            
        dumpList();
    } else {
        // the summary file exists; need to read it and restore
        // history;
        // there are two possibilities:
        //   1. the start of the summary file is what the start time
        //      should be
        //   2. the start of the summary file has "dropped off the
        //      screen", i.e. it is further back in the past than
        //      we need.  This also includes the situation where ALL
        //      the data in the summary file is out of date.  (Thanks
        //      to Patrick Sutton for catching this.)

        if (mDebug > 2) {
            std::cerr << "LockSummary: summary file exists. "
                      << "Reading..." << std::endl;
        }

        // read file
        lockstatus_t tmpstate;
        while (true) {
            mFile >> tmpstate.gpstime >> tmpstate.livefrac;
            if (!mFile.eof())
                mLockList.push_back(tmpstate);
            else
                break;
        }

        // done reading. close file
        mFile.close();

        ulong_t fileStartTime = mLockList.front().gpstime;
        if (mDebug > 2) {
            std::cerr << "LockSummary: read "
                      << mLockList.size() << " items" << std::endl;
            std::cerr << "LockSummary: file start time = "
                      << fileStartTime << std::endl;
        }

        // pop off data that is too old
        while (!mLockList.empty() && mLockList.front().gpstime < startTime) {
            mLockList.pop_front();
        }

        if (mDebug > 2) {
            std::cerr << "LockSummary: after popping old summaries"
                      << std::endl;
            std::cerr << "      LENGTH = " << mLockList.size() << std::endl;
            std::cerr << "      START  = " << mLockList.front().gpstime
                      << std::endl;
        }

        // if the summary list is too short, pad the missing data,
        // and write out the file
        if (mLockList.size() < mLength) {

            if (mDebug > 2) {
                std::cout << "OCH! DATA IS OLD!" << std::endl;
                std::cout << "BACK = " << mLockList.back().gpstime
                          << ", " << mLockList.back().livefrac << std::endl;
                ulong_t gapLength = mLength -
                    static_cast<ulong_t>(mLockList.size());
                std::cout << "GAP = " << gapLength << std::endl;
            }

            lockstatus_t tmpstate;
            while (mLockList.back().gpstime < nowRound) {
                tmpstate.gpstime = mLockList.back().gpstime + mDT;
                tmpstate.livefrac = -1.;
                mLockList.push_back(tmpstate);
            }

            if (mDebug > 2) {
                std::cout << "LockSummary: After padding: " << std::endl;
                std::cout << "   LENGTH = " << mLockList.size() << std::endl;
                std::cout << " END TIME = " << mLockList.back().gpstime
                          << std::endl;
                std::cout << "LockSummary: DUMPING..." << std::endl;
            }
            
            // write out list
            dumpList();
        }
    }

    if (mDebug > 0) {
        std::cerr << "LockSummary: length = " << mLockList.size() << std::endl;

        if (mDebug > 3) {
            std::cerr << "LockSummary: list:" << std::endl;
            std::list<lockstatus_t>::const_iterator iter = mLockList.begin();
            for (; iter != mLockList.end(); ++iter)
                std::cerr << "\tgps = " << (*iter).gpstime
                          << "; frac = " << (*iter).livefrac << std::endl;
        }
    }
}

//
// LockSummary::~LockSummary()
//
LockSummary::~LockSummary(void)
{
    if (mFile.is_open()) {
        mFile.close();
    }
}

//
// LockSummary::append()
//
void LockSummary::append(lockstatus_t state)
{
    if (mDebug > 2) {
        std::cerr << "LockSummary::append(): appending {"
                  << state.gpstime << ", "
                  << state.livefrac << "}" << std::endl;
        std::cerr << "                       length = "
                  << mLockList.size() << std::endl;
    }
    mLockList.pop_front();
    mLockList.push_back(state);
}


//
// LockSummary::dumpList()
//
void LockSummary::dumpList(void)
{
    // make sure output file is open
    if (!mFile || !mFile.is_open()) {
        if (mDebug) cerr << "LockSummary::dumpList(): summary status file is "
			 << " not open: opening" << endl;

        mFile.open(mFilename.c_str(), ios::out);

        // set output format
        mFile.setf(ios::scientific);
        mFile.precision(4);
    }

    if (mDebug > 2) {
        std::cerr << "LockSummary: dumpList(): LENGTH = "
                  << mLockList.size() << std::endl;
    }
    
    std::list<lockstatus_t>::const_iterator iter = mLockList.begin();
    for (; iter != mLockList.end(); ++iter)
        mFile << (*iter).gpstime << "\t" << (*iter).livefrac << std::endl;

    mFile.close();
}


