//  -*- C++ -*-
//    File: LightMon.hh
// $Id: LightMon.hh 6527 2011-11-19 04:05:17Z john.zweizig@LIGO.ORG $
#ifndef LIGHTMON_HH
#define LIGHTMON_HH

#include <sys/types.h>
#include <iosfwd>
#include <iostream>
#include <fstream>
// #include <sstream> // bloody gcc-2.95.1 doesn't have sstream
#include <stdexcept>
#include <vector>
#include <deque>
#include <string>
#include <map>
#include <list>

#include "OperStateCondList.hh"
#include "DatEnv.hh"
#include "MonServer.hh"
#include "TrigRslt.hh"
#include "TrigClient.hh"
#include "Segment.hh"
#include "Interval.hh"
#include "TSWindow.hh"
#include "Histogram1.hh"
#include "Trend.hh"
#include "AlarmClient.hh"
#include "AlarmData.hh"

/** @name LightMon Monitor
 *
 *  @memo {\tt LightMon} monitors for lock acquisition and lock loss.
 * It relies on an
 * \URL[OperStateCondList]{../../../dmtlib/osc3/oscdoc/index.html}
 * object with conditions defined in the configuration file {\tt
 * LockLoss.conf} (see an example \URL[here]{../LockLoss.conf}).  The
 * configuration file defines the channels of interest, and
 * appropriate thresholds.
 *
 *  @version 2.1
 *  @author  David Chin <dwchin@umich.edu>; K. Riles <kriles@umich.edu>
 *   
 *
 */

using namespace std;
using namespace trig;

//
// The following templates are for computing statistics
//
template <class T> static double sum(const std::vector<T> &data)
{
    double retval = 0.;

    typename std::vector<T>::const_iterator iter = data.begin();
    for (; iter != data.end(); ++iter)
        retval += (double)*iter;

    return retval;
}


template <class T> static double mean(const std::vector<T> &data)
{
    return (sum(data) / (double)(data.size()));
}

template <class T> static double variance(const std::vector<T> &data)
{
    double avg = mean(data);
    std::vector<T> delta = data;
    typename std::vector<T>::iterator iter = delta.begin();
    for (; iter != delta.end(); ++iter)
        *iter -= avg;
    return (sum(delta) / (double)(data.size() - 1));
}

template <class T> static double stdDev(const std::vector<T> &data)
{
    return sqrt(variance(data));
}

//
//    LightMon monitor class
//
class LightMon
    : public DatEnv,
      public MonServer,
      private TrigClient
{
public:
    /// Constructor
    LightMon(int argc, const char *argv[]);

    /// Destructor
    ~LightMon();

    /** Main routine: processes data to determine if lock has been
     * acquired or lost, computes livetime ratio, generates triggers.
     */
    void ProcessData();

    /** Interrupt for DMT viewer
     */
    void Attention() { MonServer::Attention(); }

private:
    /// Debug level
    int               mDebug;

    /// Max no. of frames to process
    size_t            mMaxFrame;

    /// How many frames already processed
    size_t            mFrameCount;

    /// Predicates for acquisition and loss of lock
    bool              mBacquiredP;
    bool              mBlostP;
    bool              mTransitionP;

    /// Start time of current out_of_lock segment
    Time mOUT_OF_LOCK_start;
    Time mIN_LOCK_start;
    bool mOUT_OF_LOCK_already_true;
    bool mSendTriggers;

    Time mFirstNow;

    /// Pre-lockloss interval lengths    
    int mN_prelockloss;
    int mNsec_prelockloss[10]; 
    Time mLastFlushStart_prelockloss[10];
    Time mLastFlushStop_prelockloss[10];
    int mLastFlushState_prelockloss[10];
    Time mLastFlushStart_prelockloss_on[10];
    Time mLastFlushStop_prelockloss_on[10];

    Time mLastFlushStart_outoflock;
    Time mLastFlushStop_outoflock;
    int mLastFlushState_outoflock;
    Time mLastFlushStart_outoflock_on;
    Time mLastFlushStop_outoflock_on;


    /// Lightdip fractional drops
    int mN_lightdip;
    float mFract_lightdip[20];
    Time mLastFlushStart_lightdip[20];
    Time mLastFlushStop_lightdip[20];
    int mLastFlushState_lightdip[20];
    Time mLastFlushStart_lightdip_on[20];
    Time mLastFlushStop_lightdip_on[20];

    /// Total time this monitor has been running
    Interval          mTotalRunTime;

    /// Print out HTML header
    void              htmlHead(void) const;

    /// Print out HTML footer
    void              htmlFoot(void) const;

    /// Print out top of HTML status page
    void              htmlTopOfPage(Time now);

    /// Send a trigger to the Trigger Manager
    void              sendTrig(string trigname, string ifoname, Time now);

    /// Send a segment to the Trigger Manager
    void              sendSeg(Segment s, Time now);

    /// List of Operational State Conditions to define when we are in lock
    OperStateCondList mOsclist;

    /// Name of logfile
    std::string   mLogfileName;
    
    /// Log file
    std::ofstream mLogfile;

    /// Log stream
    std::ostream *mLog;

    /// Name of livetime display file
    std::string   mDispfileName;

    /// Livetime display html file
    std::ostream *mDisplay;

    /// Index page
    std::ofstream mIndexPage;

    /// Time stamp string
    char mTimestamp[64];

    /// Time stamp format
    std::string mTstampFormat;

    /// Time stride computing live time
    static const int mLiveTimeUpdateInterval = 60;

    /// Update interval for HTML livetime page, and livetime info (in seconds)
    static const int mHTMLupdateInterval = 60;

    /// Length of TSindows below
    static const size_t mMaxLength      = 7200;  // 2hrs @ 1sec stride
    static const size_t mMaxLength6hr   = 4320;  // 6hrs, @ 5sec stride
    static const size_t mMaxLength12hr5 = 7200;  // 12hrs, @ 6sec stride
    static const size_t mMaxLength10min = 600;   // 10min @ 1sec
    static const size_t mMaxLength1hr   = 3600;  // 1hr @ 1sec
    static const size_t mMaxLength4hr   = 14400/5; // 4hr @ 5sec
    static const size_t mMaxLength8hr   = 28800/5; // 8hr @ 5sec
    static const size_t mMaxLength12hr  = 43200/5; // 12hr @ 5sec
    static const size_t mMaxLength1min  = 60;      // 1min @ 1sec
    static const size_t mMaxLength1000s = 1000;    // 1000sec @ 1sec

    /// Name of IFO
    std::string mIFOname;

    /// Name of monitor
    std::string mName;

    /// Time series of lock state
    TSeries mBts;

    /// 6hr time series of lock state to line up with Ed Daw's
    /// seismic monitor
    TSeries mBts6hr;

    /// 12hr time series of lock state for shift summary
    TSeries mBts12hr;
    TSeries mBts12hrExp;

    //  Time series of arm powers
    TSeries mArmPowerTS;
    TSeries mArmPowerNoDipTS;

    //
    // Windows on the time series
    //

    // Cumulative window (updated every second)
    TSWindow<int> mBothLock;

    // 1 minute window (for trending only)
    TSWindow<int> mBothLock1min;

    TSWindow<int> mLockState;

    // The 6hr windows will be updated at 5 second intervals
    TSWindow<int> mBothLock6hr;

    // The 12hr windows will be updated at 6 second intervals
    TSWindow<int> mBothLock12hr;

    TSWindow<double> mArmPower;
    TSWindow<double> mArmPowerNoDip;

    TSeries* Xnow;
    TSeries* Ynow;

    /// Array to store temporary livetime data
    std::vector<double>  mLiveTimeData;

    /// Length of current lock stretch (if in lock)
    Interval   mBothTimeInLock;

    /// Minute trends
    Trend      mTrend;

    // /// Vectors to contain data for trends
    // std::vector<int>    mBothLockData;
    // std::vector<int>    mLockStateData;

    /// Trend channel names
    map<std::string, std::string> mTrendChanNames;

    /// Directory for HTML files
    std::string    mHtmlDir;

    /// Actions to take when Both arms are in lock
    void BothArmsLockedActions(const Time &now);

    /// Actions to take when Both arms are not in lock
    void BothArmsNotLockedActions(const Time &now);

    /// Returns smaller of two objects
    template<class T> T min(T a, T b)
        {
            return (a < b) ? a : b;
        };

    /// Returns larger of two objects
    template<class T> T max(T a, T b)
        {
            return (a > b) ? a : b;
        };

    void setTSWindowStuffMod(void);

    void initTSWindowsMod(const Time &now);

    void saveAllTSWindowsMod(void);

    void restoreAllTSWindowsMod(void);
};


#endif     //  LIGHTMON_HH

