#ifndef HISTCOMPR_HH
#define HISTCOMPR_HH

#include "DatEnv.hh"
#include "MonServer.hh"
#include "Histogram1.hh"
#include "OperStateCondList.hh"
#include "TrigClient.hh"
#include "autopipe.hh"
#include <list>
#include <string>
#include <iosfwd>

//======================================  Interactive mode
// #define SHOWHISTO
#ifdef  SHOWHISTO
class TCanvas;
#endif

/**  @name HistCompr
  *  The HistCompr monitor compiles histograms from a list of channels.
  *  It then compares these histograms to a set of standard histograms 
  *  and generates a trigger if the histograms differ significantly. 
  *  Both the histogram and the reference are served to the DMT viewer
  *  by way of the monitor data protocol.
  *
  *  HistCompr performs the following steps.
  *  \begin{enumerate}
  *  \item {\bf Read reference histograms from the specified directory.}
  *  The histogram is read from the file with the same name as the histogram.
  *  If a reference directory is not specified, or if there is no file with
  *  the appropriate name, the is step is omitted.
  *  \item {\bf Fill the histogram.} The histogram is filled for the length  
  *  of time specified by the #-time# command line argument. The histogram 
  *  is served to any client while the data are being filled.
  *  \item {\bf Compare to reference.}
  *  When the histogram has been filled for the specified time, the 
  *  Histogram data are compared to the reference histogram (if specified).
  *  If the resulting chi-square is greater than the limit specified in the 
  *  configuration file, and the #-trig# argument was specified on the 
  *  command line, a #HistCompr# trigger is generated. If the chi-square
  *  exceeds the limit and the #-plot# argument was specified, The histogram
  *  is plotted on the DISPLAY monitor. In any case the histogram is cleared
  *  and the filling process is repeated.
  *  \item {\bf Save histograms.}
  *  On termination, if the #-savdir# argument was specified, all 
  *  Histograms are saved in the specified directory. The histogram names
  *  are used as the file names.
  *  \end{enumerate}
  *
  *  HistCompr can therefore be used in the following modes:
  *  \begin{itemize}
  *  \item {\bf Background histogrammer:} If the #-refdir# and #-savdir# 
  *        arguments are omitted, HistCompr will collect all histograms 
  *        specified in the configuration file. These may be interrogated
  *        using the monitor data message protocol (e.g. with the DMT viewer).
  *  \item {\bf Collect reference histograms:} If the #-savdir# argument is
  *        specified, HistCompr will collect histograms and save them in the
  *        directory specified by the #-savdir# argument on termination.
  *        The saved histograms are named appropriately to be used as 
  *        references for subsequent trigger runs.
  *  \item {\bf Trigger generation:} If the #-refdir# argument is specified, 
  *        HistCompr will periodically compare the collected histograms to 
  *        the reference histogram found in the specified directory. If the 
  *        resulting chi-square is greater than the limit value specified in 
  *        the configuration file, and if the #-trig# argument was specified, 
  *        HistCompr grnerates a trigger with an ID of "HistCompr" and a 
  *        sub-ID of the histogram name.
  *  \end{itemize}
  *
  *  {\bf {\Large Running HistCompr}}
  *
  *  The HistCompr command line is as follows:
  *  \begin{verbatim}
HistCompr [-conf <config-file>] [-time <reset-time>] [-savedir <save-dir>] \
          [-refdir <reference-dir>] [-osc <osc-config>] [-plot] [-trig]
     \end{verbatim}
  *
  *  The arguments have the following meaning:
  *  \begin{tabular}{rr}
  *  <config-File> & Configuration file name \\
  *  <reset-time> & All histograms are reset after the specified time 
  *  has elapsed. \\
  *  <save-dir> & Directory into which to save the histograms. \\
  *  <reference-dir> & Directory from which the reference histograms are 
  *   to be fetched. \\
  *  <osc-config> & Configuration file for an Operating State Condition
  *   list to be used to qualify data for histogramming.  \\
  *  -plot & Call Root plotter for every histogram failing the comparison. \\
  *  -trig & Enable trigger generation.
  *  \end{tabular}
  *
  *  {\bf {\Large Configuration File}}
  *
  *  The HistCompr configuration file specifies which channels are to be
  *  histogrammed, the binning and options. Each line within a 
  *  configuration file specifies a histogram to be made. The contents of 
  *  a configuration file line are as follows:
  *
  *  \begin{itemize}
  *  \item Filter directive:
  *
  *  \begin{center}
  *  \begin{tabular}{rr}
  *  p0 & Filter name. \\
  *  p1 & Type of filter (BaseLine, IIRFilter or MultiPipe). \\
  *  p2-N & Type specific arguments. \\
  *  \end{tabular}
  *
  *  Baseline Suppression Filters
  *  \begin{tabular}{rr}
  *  p2 & Time constant in seconds [1.0]. \\
  *  \end{tabular}
  *
  *  IIR Filters
  *  \begin{tabular}{rr}
  *  p2 & Sample rate \\
  *  p3 & List of s-plane poles \\
  *  p4 & List of s-plane zeros \\
  *  \end{tabular}
  *
  *  MultiPipe Filters
  *  \begin{tabular}{rr}
  *  p2-N & List of filter names to be used to form a pipeline \\
  *  \end{tabular}
  *
  *  \end{center}
  *
  *  \item Histogram directive:
  *
  *  \begin{center}
  *  \begin{tabular}{rr}
  *  p0 & Histogram name. \\
  *  p1 & Name of chanel to be histogrammed. \\
  *  p2 & Number of histogram bins. \\
  *  p3 & low edge of first bin. \\
  *  p4 & High edge of last bin. \\
  *  p5 & Number of Chi-Square trigger threshold \\
  *  -base & subtract baseline. \\
  *  -while <cond> & Requisite OSC condition.
  *  \end{tabular}
  *  \end{center}
  *
  *  \item Parameter directives contain a parameter name followed by the 
  *        value it is to assume.
  *
  *  \begin{center}
  *  \begin{tabular}{rr}
  *  ServerName & The name under which HistCompr will register [HistCompr] 
  *  \end{tabular}
  *  \end{center}
  *  \end{itemize}
  *
  *  {\bf {\Large HistCompr Chi-Square Calculation}}
  *
  *  The $\chi^2$ is calculated by normalizing both the Histogram and the 
  *  reference to one and using the following formula
  *
  *  \[ \chi^2 = \Sigma_{i=0}^{N+1} { {( x_i - r_i )^2 } \over { r_i ( 1 - r_i ) / N_x } } \]
  *
  *  The numbers $x_i$ and $r_i$ are the fractional occupation of the 
  *  $i^{th}$ histogram or reference bin respectively. N is the number 
  *  of bins in the histogram. Note that the overflow and underflow counts
  *  are treated identically to the histogram bins. $N_x$ is the total 
  *  number of histogram entries. This formula assumes (among other things)
  *  that the reference histogram is an exact representation of the desired
  *  distribution. Any occurance of a non-zero histogram bin with a zero
  *  reference would give an undefined chi-square. To prevent this, the
  *  error on the difference is redefined to be $\sigma_r^2 = 1/N_r$ when
  *  this case occurs.
  *
  *  @memo Gather histograms and compare to standard distributions.
  *  @author J.Zweizig
  *  @version 1.1; October 5, 2009
  */
//@{
//@}

class Histx {
public:
  Histx(const std::string& name, const std::string& channel, 
	int nBin, double lo, double hi, double lim);
  ~Histx(void);
  const char* getName(void) const;
  const char* getChannel(void) const;
  const char* getCondition(void) const;
  double getChiSquare(void) const;
  double getLimit(void) const;
  int getNEntries(void) const;
  const Histogram1& getHistogram(void) const;
  const Histogram1& getLong(void) const;
  const Histogram1& getReference(void) const;
  const Histogram1& getShort(void) const;
  const Histogram1& getStride(void) const;

  void clear(void);
  void clearStride(void);
  void fillData(double x, double wt=1.0);
  void fillData(const TSeries& t, double wt=1.0);
  bool fetchReference(const std::string& fileName);
  void increment(void);

  static void printHdr(std::ostream& out);
  void printStats(std::ostream& out) const;

  bool saveHistogram(const std::string& fileName) const;
  void setBaseLineSubtraction(bool tf);
  void setCondition(const std::string& cond);
  void setFilter(const auto_pipe& p);
  void setFilterName(const std::string& name);

  void updateShort(void);
  void updateLong(void);

private:
  std::string mName;
  std::string mChannel;
  double      mLimit;
  Histogram1  mHisto;
  Histogram1  mRefer;
  Histogram1  mStride;
  Histogram1  mShort;
  Histogram1  mLongAcc;
  Histogram1  mLong;
  bool        mBaseLine;
  std::string mCondition;
  auto_pipe   mPipe;
  std::string mFilterName;
};

//=====================================  Histx inline methods.
inline double
Histx::getLimit(void) const {
    return mLimit;
}

inline const Histogram1& 
Histx::getHistogram(void) const {
    return mHisto;
}

inline const Histogram1& 
Histx::getLong(void) const {
    return mLong;
}

inline const Histogram1& 
Histx::getReference(void) const {
    return mRefer;
}

inline const Histogram1& 
Histx::getShort(void) const {
    return mShort;
}

inline const Histogram1& 
Histx::getStride(void) const {
    return mStride;
}

//====================================== HistCompr Monitor class
class HistCompr : public DatEnv, MonServer, TrigClient {
public:
  HistCompr(int argc, const char* argv[]);
  ~HistCompr(void);
  void ProcessData(void);
  void Attention(void);
  void clear(void);
  void fetch(const std::string& dir);
  void printStats(void) const;
  void readConfig(void);
  void save(const std::string& dir) const;

private:
  typedef std::list<Histx> HistList;
  typedef HistList::iterator       hist_iter;
  typedef HistList::const_iterator const_hist_iter;

private:
  std::string mRefDir;
  std::string mSaveDir;
  std::string mServer;
  HistList    mList;
  Interval    mAvgTime;
  Interval    mLongTime;
  Time        mTimeOut;
  Time        mLongTimeOut;
  OperStateCondList mOSC;
  std::string mConfig;
  bool        mTrig;

#ifdef SHOWHISTO
  TCanvas*    mCanvas;
  bool        mPlot;
#endif

  typedef std::map<std::string, auto_pipe> FiltrList;
  typedef FiltrList::iterator  FiltrIterator;
  FiltrList mFiltr;
};

#endif // HISTCOMPR_HH
