//=============================================================================
//Class Declaration: GainSummary.hh version 2.0 (Aug 14 2002)
//=============================================================================


#ifndef GainSummary_HH
#define GainSummary_HH

#include "Time.hh"
#include <fstream>
#include <list>

/** Helper struct for class GainSummary.
  * @memo Helper struct for class GainSummary.
  */
struct GainDataSummary
{
        Time::ulong_t gpstime;
        double ugf;
};


/** Helper class for GainMon.
  * GainSummary produces two-week summary data files containing 
  * the ugf estimates and other monitor output averaged over 
  * 1000 second intervals.
  * @author 
  * @memo Produces two-week summary data files.
  */
class GainSummary
{
public:
    /** GainSummary constructor.
      * @memo Constructor.
      */
    GainSummary(void) {};

    /** GainSummary destructor.
      * @memo Destructor.
      */
    ~GainSummary(void);

    /// What the hell does this do?
    void init(const std::string &htmldir,
              const Time &now);

    /// Append data point to end of list.
    void append(GainDataSummary state);

    /// Dump data to file
    void dumpList(void);

private:
    //  length of list
    static const long int mLength = 1210;

    //  Summary "sampling" interval
    static const long int mDT = 1000;

    // list of ugf data for last 2 weeks
    typedef std::list<GainDataSummary> Gain_List;
    typedef Gain_List::const_iterator list_iter;
    Gain_List mGainList;

    //  status file name
    std::string        mFilename;

    //  status file stream
    std::fstream       mFile;

    //  time this object was created
    Time               mCreationTime;
};


//=============================================================================
//Class Member Definitions
//=============================================================================


void GainSummary::init(const std::string &summary_name,
                        const Time &now)
{
//  mFilename = htmldir + "/summary.txt";
    mFilename = summary_name;

    // set the current time back to the last '000,
    // ditto the start time
    long int nowRound = now.getS() - (now.getS() % mDT);
    long int startTime = nowRound - ( mLength - 1 ) * mDT;

    std::ifstream File(mFilename.c_str(), std::ios::in);
    if ( !File.is_open( ) ) {
        // the summary file does not exist; fill in the list with -1's
        // and create the file for output

        GainDataSummary tmpstate;
        for (long int i = 0; i < mLength; ++i) {
            tmpstate.gpstime  = startTime + i*mDT;
            tmpstate.ugf = -1.;

            mGainList.push_back(tmpstate);
        }

        dumpList();
    } else {
        // the summary file exists; need to read it and restore
        // history;
        // there are two possibilities:
        //   1. the start of the summary file is what the start time
        //      should be
        //   2. the start of the summary file has "dropped off the
        //      screen", i.e. it is further back in the past than
        //      we need.

        // read file
        GainDataSummary tmpstate;
        while ( File.is_open( ) ) {
            File >> tmpstate.gpstime >> tmpstate.ugf;
            if (File.good() && !File.eof()) {
                mGainList.push_back(tmpstate);
	    } else {
	        // done reading. close file
                File.close( );
	    }
        }

        // long int fileStartTime = mGainList.front().gpstime;

        // pop off data that is too old
        while (!mGainList.empty() && mGainList.front().gpstime < startTime) {
            mGainList.pop_front();
        }

        // if the summary list is too short, pad the missing data,
        // and write out the file
        // WARNING:  This assumes any gaps in the data are at the 
	// most recent end, and not, say, in the middle.  Those gaps 
	// will not be filled.
	if ((int)(mGainList.size()) <(int)(mLength)) {

            GainDataSummary tmpstate;
            while (mGainList.back().gpstime < nowRound) {
                tmpstate.gpstime = mGainList.back().gpstime + mDT;
                tmpstate.ugf = -1.;
                mGainList.push_back(tmpstate);
            }

            // write out list
            dumpList();
        }
    }
}

//
// GainSummary::~GainSummary()
//
GainSummary::~GainSummary(void)
{
  //    if (mFile.is_open()) {
  //      mFile.close();
  //  }
}

//
// GainSummary::append()
//
void GainSummary::append(GainDataSummary state)
{
    mGainList.pop_front();
    mGainList.push_back(state);
}


//
// GainSummary::dumpList()
//
void GainSummary::dumpList(void)
{
    // make sure output file is open
    std::ofstream File(mFilename.c_str(), std::ios::out);
    
    // set output format
    File.setf(std::ios::scientific);
    File.precision(4);
    if (!File.good() || !File.is_open()) {
        std::cerr << "GainSummary::dumpList(): summary status file is "
                  << " not open: opening" << std::endl;
	return;
    }

    for (list_iter iter = mGainList.begin(); iter != mGainList.end(); ++iter)
        File << (*iter).gpstime << "\t" << (*iter).ugf << std::endl;

    File.close();
}

#endif     //  GainSummary_HH
