/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//   Unity Gain Frequency Monitor for the DARM Loop 
//   authors: Kevin C. Schlaufman (kcs149@psu.edu)
//            Patrick J. Sutton (psutton@ligo.caltech.edu)
//            Aaron Rogan (arogan@wsunix.wsu.edu)
//
///////////////////////////////////////////////////////////////////////////

//----- Defined constants:
//---------- Time (sec) over which to display range histories on DMTViewer.
#define DMTViewer_Period 43200  


//----- Include this first
#include "GainMon.hh"


#ifndef __CINT__

//----- Include standard headers
#include <stdlib.h>
#include <iomanip>

//----- Include LIGO-specific headers
#include "Dacc.hh"
#include "html/writer.hh" 
#include "html/Attrib.hh"
#include "html/align.hh"
#include "html/color.hh"
#include "html/document.hh"
#include "html/font.hh"
#include "html/hline.hh"
#include "html/image.hh"
#include "html/label.hh"
#include "html/link.hh"
#include "html/size.hh"
#include "html/style.hh"
#include "html/table.hh"
#include "html/text.hh"

#endif               // !def(__CINT__)

//----- Include monitor-specific headers
#include "Integrand.hh"
#include "Integrate.hh"
#include "Range.hh"


#ifndef __CINT__
//======================================  Generate the main routine.
EXECDAT(GainMon)
#endif               // !def(__CINT__)

  using namespace std;

const char *USAGE_INFO =
"\n"
"GainMon: Uninty Gain Frequency (UGF) for the DARM Loop version 1.0 (2005.10.10)"
"\n\n"
"Usage:\n\n"
"./GainMon [optional arguments] <IFO>\n\n"
"where <IFO> is one of H1, H2, or L1 and the optional arguments are:\n"
"                \n"
"-fmax #         Specify maximum frequency to include in UGF estimate.\n"  
"                Default 1400Hz.\n"
"                \n"
"-fmin #         Specify minimum frequency to include in UGF estimate.\n"
"                Default  20Hz.\n"
"                \n"
"-h, --help      Print this usage information.\n"
"                \n"
"-local          Local mode:  All output files are dumped in local directory\n"
"                instead of the default directory specified by $DMTHTMLOUT;\n"
"                plain-text log file renamed <GPS_Start_Time>.log.\n"
"                \n"
"-logfile <name> Name plain-text version of log file <name>.  Only works if\n"
"                -local option is also selected.  Default filename is\n"
"                <GPS_start_time>.log (-local mode) or\n"
"                <IFO>_GainMon_CumLog.txt (otherwise).\n"
"                \n"
"-max #          Specify maximum number of strides to process before program\n"
"                exits.\n"
"                \n"
"-n #            NOT FUNCTIONAL.  Currently set to 1.\n"
"                Specify number of sections to divide data into for \n" 
"                calculating PSD.  THIS NUMBER MUST DIVIDE 'stride' EVENLY!\n"
"                Default 15.\n"
"                \n"
"-OSCfile <file> REQUIRED.  Specify the OSC configuration file defining the\n"
"                various operating conditions for the interferometer. \n"
"                (If not specified, GainMon will attempt to open\n"
"                $GAINMON_OSCCONF/GainMon_LockLoss.conf,\n"
"                then ./GainMon_LockLoss.conf.)  GainMon will exit\n"
"                if no OSC configuration file can be found.\n"
"                \n"
"-OSCcond <cond> Specify the OSC (operating state condition) for which \n"
"                SenseMonitor will compute UGF estimates.  Default is \n"
"                SV_IFO_UP.  This condition must be defined in the file \n"
"                specified by -OSCfile.\n"
"                \n"
"-readalphabeta  Use stored calibrations from reference calibration file\n"
"                instead of generating new alpha, beta on-the-fly.      \n"
"-channel <name> set the channel name\n"
"-screen         Update and error messages are sent to the screen instead of\n"
"                to the default files <IFO>_GainMon_Log.txt and \n"
"                <IFO>_GainMon_Errors.txt.\n"
"                \n"
"-stride #       Specify length in seconds of data to use for each range \n"
"                estimate.  Default 60.\n"
"                \n"
"-trend          Write range and calibration data to trend files.\n"
"                \n"
"-window <type>  Specify data window type, where <type> is: \n"
"                  hanning for Hanning window,\n"
"                  blackman for Blackman window,\n"
"                  flattop for FlatTop window (NOT RECOMMENDED),\n"
"                  hamming for Hamming window (NOT RECOMMENDED),\n"
"                  square for no window (NOT RECOMMENDED).\n"
"                Default Hanning.\n"
"                \n"
"-xmlcal <file>  REQUIRED. Use dynamical calibration based on a reference \n"
"                calibration file <file>. \n"
"                \n";


//----- Html Text
const char* const kHtmlTitle = "GainMon (version 1.0): Unity Gain Frequency Monitor";
const char* const kHtmlAuthor = "Kevin Schlaufman (kcs149@psu.edu), "
			        "Patrick Sutton (psutton@ligo.caltech.edu) and Aaron Rogan (arogan@wsunix.wsu.edu)";
const char* const kHtmlDescription =
   "This monitor calculates the approximate unity gain frequency for the DARM control loop.";
const char* const kHtmlLIGO =
   "Laser Interferometer Gravitational-wave Observatory ";
const char* const kHtmlLIGOLink = "http://www.ligo.caltech.edu";


//======================================  GainMon constructor.
GainMon::GainMon(int argc, const char *argv[]) 
: DatEnv(argc, argv), 
fatal_error(false), prev_endtime(0.0), running_average(0.0),
running_count(0.0), MaxStride(99999), missing_strides(0), NStride(0), time_since_update(0)
{

    //---------- Initialize structures.
    mOutput.error_log = true;
    mOutput.local = false;
    mOutput.screen_output = false;
    mOutput.trend = false;
    mOutput.write_log = true; 
    Run_Par.genCalParam = true;
    Run_Par.T = 60.0;
    Run_Par.l_freq = 20.0;
    Run_Par.h_freq = 1400.0;
    Run_Par.num_ints = 15;
    //---------- Time (sec) in lock between each dump of a calibrated PSD.
    Run_Par.CalPSD_Dump_Period=0;  
    G_Dat.alpha = 0.0;
    G_Dat.beta = 0.0;
    G_Dat.ampl = 0.0;
    G_Dat.ugf = 0.0;
    G_Dat.T5_gain = 0.0;
    G_Dat.T15_gain = 0.0;
    for(int i=0; i<5; i++)  G_Dat.T5_array[i] = 0.0;
    for(int i=0; i<15; i++)  G_Dat.T15_array[i] = 0.0;

    //---------- Look for command-line arguments.
    //           First check if configuration file is being used.
    //           In this case the command line should read 
    //           GainMon -config <filename>
    //           If file <filename> does not exist or cannot be parsed 
    //           then exit with error message.
    //           If not using config file, make sure we have some arguments and that
    //           the last one does not start with a hyphen.
    if ((argc == 3) && (!strcmp("-config", argv[1]))) {
        Run_Par.config_file_name = argv[2];
        Configure(Run_Par.config_file_name);
    } else {
        Configure(argc, argv); 
    }

    //---------- Send monitor name to MonServer.
    std::string MonitorName = "GainMon_";
    MonitorName += Run_Par.IFO;
    MonServer::setServerName(MonitorName.c_str());

    //---------- Do some elementary checks on the parameters.
    VerifyParameters(Run_Par,fatal_error);

    //---------- Finish setup of monitor.
    if (!fatal_error) {
        //-------- Check to see if running locally.  If not, dump html 
        //         files to $DMTHTMLOUT instead of to local directory.
	//         Plain-text log file is named in ProcessData() method.
        if (!mOutput.local) {
	    mOutput.html_dir = getenv("DMTHTMLOUT");
            if (mOutput.html_dir) {
                mOutput.dmtviewer_file_name = string(mOutput.html_dir) + "/";
		mOutput.error_file_name = string(mOutput.html_dir) + "/";
		mOutput.log_file_name = string(mOutput.html_dir) + "/";
		mOutput.revolver_file_name = string(mOutput.html_dir) + "/";
		mOutput.summary_file_name = string(mOutput.html_dir) + "/";
	    } else { 
		std::cerr << "GainMon WARNING: Environment variable "
			  << "DMTHTMLOUT not set; switching to\n";
		std::cerr << "                      -local mode (all output "
			  << "files go to the local directory).\n";
		mOutput.local = true;
	    }
	}
        mOutput.dmtviewer_file_name += Run_Par.IFO + "_DMTViewer_Data.txt";
        mOutput.error_file_name += Run_Par.IFO;
        mOutput.log_file_name += Run_Par.IFO;
        mOutput.revolver_file_name += Run_Par.IFO;
        mOutput.revolver_file_name += "_GainMon_Summary.revolver.html";
        mOutput.error_file_name += "_GainMon_Errors.txt";
	mOutput.error_file_link = Run_Par.IFO + "_GainMon_Errors.txt";
        mOutput.log_file_name += "_GainMon_Log.txt";
	mOutput.log_file_link = Run_Par.IFO + "_GainMon_Log.txt";  
        if (!mOutput.local) mOutput.cumlog_file_link = Run_Par.IFO + "_GainMon_CumLog.txt"; 
	mOutput.summary_file_name += "index.html";

        //-------- Divert log and error messages to files, if desired(default).
        if (mOutput.error_log && 
	    !freopen(mOutput.error_file_name.c_str(), "w", stderr) ) {
	    cout << "Unable to open stderr file" << endl;
	}
        if (mOutput.write_log &&
	    !freopen(mOutput.log_file_name.c_str(), "w", stdout) ) {
	    cerr << "Unable to open stdout file" << endl;
	}

        //---------- Set up array that will hold data for DMTViewer history 
        //           TSeries.  Used to insure that fixed length of data is 
        //           displayed.  Initialize data to -1 to indicate SenseMonitor 
        //           was not running previously.
        tserieslength = (int)( DMTViewer_Period / Run_Par.T);
        calvalues = new double[tserieslength];
        amplvalues = new double[tserieslength];
        alphavalues = new double[tserieslength];
        alphabetavalues = new double[tserieslength];
        //---------- Initialize TSeries using data from previous run, if possible.
        ifstream dmtviewer_data;
        dmtviewer_data.open(mOutput.dmtviewer_file_name.c_str());
        if (dmtviewer_data.fail()) {
            for (int i=0; i<tserieslength; i++) {
                calvalues[i] = -1.0;
                amplvalues[i] = -1.0;
                alphavalues[i] = -1.0;
                alphabetavalues[i] = -1.0;
            }
        } else {
            //:KLUDGE:  This could go wrong if there is a long time gap between runs of program.
            int i=0;
            while (i<tserieslength && dmtviewer_data >> calvalues[i]) { 
                dmtviewer_data >> amplvalues[i];
                dmtviewer_data >> alphavalues[i];
                dmtviewer_data >> alphabetavalues[i];
                ++i;
            }
            //----- Reset latest values to -2 to indicate that histories have been read from a file.
            calvalues[i-1] = -2.0;
            amplvalues[i-1] = -2.0;
            alphavalues[i-1] = -2.0;
            alphabetavalues[i-1] = -2.0;
            //----- Check that our new histories are of the desired length.
            if (i != tserieslength) {
                cerr << "SenseMonitor WARNING:  Stored DMTViewer history from previous run not of ";
                cerr << "required length.  Re-initializing DMTViewer histories.\n";
                for (int i=0; i<tserieslength; i++) {
                    calvalues[i] = -1.0;
                    amplvalues[i] = -1.0;
                    alphavalues[i] = -1.0;
                    alphabetavalues[i] = -1.0;
                }
            }
        }
        dmtviewer_data.close();

        //-------- Set window type and intialize PSD object.
        if (Run_Par.window_name == "hamming") {
	    Hamming ham;
            psd = new PSD(&ham, Run_Par.num_ints);
        } else if (Run_Par.window_name == "hanning") {
	    Hanning han;
	    psd = new PSD(&han, Run_Par.num_ints);
        } else if (Run_Par.window_name == "flattop") {
	    FlatTop flat;
            psd = new PSD(&flat, Run_Par.num_ints);
        } else if (Run_Par.window_name == "blackman") {
	    Blackman black;
            psd = new PSD(&black, Run_Par.num_ints);
        } else if (Run_Par.window_name == "square") {
            psd = new PSD("square", Run_Par.num_ints);
        } else {
            Run_Par.window_name = "hanning";
	    Hanning han;
            psd = new PSD(&han, Run_Par.num_ints);
        }

        //---------- Set the time stride.
        getDacc().setStride(Interval(Run_Par.T));
	setStrideAlignment(60, 0.0);

        //---------- Get input channel; decimate to maximum frequency 4096Hz.
	//           (Keep frequencies above 2kHz for the calibrated PSD dumps.)
        getDacc().addChannel((Run_Par.mChannel).c_str(),2);

        //---------- Initialize FDEasyCalibrate object.  
        //           This should be in monitor constructor (ie, before any data 
        //           is requested) because FDEasyCalibrate adds new channels to
        //           the Dacc. 
        //           KLUDGE: Hard-wire requested frequencies to [0,4096]Hz.
        mCalibrate = new FDEasyCalibrate(
                            &getDacc(),
                            (Run_Par.xml_file_name).c_str(),
                            Run_Par.genCalParam,
                            0.0,
                            (Run_Par.num_ints)/(Run_Par.T),
                            ((int) ((4096.0-0.0)*(Run_Par.T)/(Run_Par.num_ints)+1)) 
                     );

        //----- Check that calibration file applies to channel we are analysing.
        if ( !(Run_Par.mChannel == mCalibrate->getChannel()) ) {
            std::cerr << "GainMon ERROR:  Reference calibration file does not apply to desired channel." << std::endl;
            std::cerr << "Reference calibration is for: " << mCalibrate->getChannel() << std::endl;
            std::cerr << "GainMon is analysing: " << Run_Par.mChannel << std::endl;
            std::cerr << "Exiting.\n";
            std::cout << "GainMon ERROR:  Reference calibration file does not apply to desired channel." << std::endl;
            std::cout << "Reference calibration is for: " << mCalibrate->getChannel() << std::endl;
            std::cout << "GainMon is analysing: " << Run_Par.mChannel << std::endl;
            std::cout << "Exiting.\n";
            exit(707);
        }

	//---------- Initialize OSC & read config file.
        mOsclist = new OperStateCondList(getDacc());
        mOsclist->readConfig((Run_Par.osc_file_name).c_str());
        mOsclist->ignoreAllExcept(Run_Par.IFO);

	if (mOutput.trend)
	{
	    //---------- Intialize Trend frame.
	    mTrend.setIFO(Run_Par.IFO.c_str());
	    mTrend.setName(MonitorName.c_str());
            mTrend.setType(Trend::kMinute);

	    //---------- Add trend channels.  Force calibration-line-related 
	    //           trends even if not tracking a line (we don't want to 
	    //           change the trends that are being written during a science
	    //           run, as this causes the trend writer to crash). 
	    std::string trendName;
	    trendName = Run_Par.IFO + ":DMT-GAIN_UGF_HZ";
	    mTrend.addChannel(trendName.c_str());
	    trendName = Run_Par.IFO + ":DMT-GAIN_CAL_LINE_AMPL_ASQ";
	    mTrend.addChannel(trendName.c_str());
	    //---------- Trends to carry alpha, alpha*beta 
	    //           (names requested by Inspiral group).
	    trendName = Run_Par.IFO + ":DMT-GAIN_CAL_CAV_FAC";
	    mTrend.addChannel(trendName.c_str());
	    trendName = Run_Par.IFO + ":DMT-GAIN_CAL_OLOOP_FAC";
	    mTrend.addChannel(trendName.c_str());
            //---------- Tell manager to make list of trended channels.
            mTrend.writeIndex();
	}

    }   //---------- !fatal_error

    //---------- Flush any lingering data
    fflush(stderr);
    fflush(stdout);
}


//======================================  GainMon destructor.
GainMon::~GainMon() 
{
    //---------- No pointers are assigned if a fatal error is registered.
    if (!fatal_error) {
        //---------- Export DMTViewer data to a file to be read if monitor is restarted.
        //           Name of file set elsewhere; uses mOutput.html_dir if defined.
        ofstream dmtviewer_data;
        dmtviewer_data.open(mOutput.dmtviewer_file_name.c_str());
        for (int i=0; i<tserieslength; i++) {
            dmtviewer_data << calvalues[i] << "  ";
            dmtviewer_data << amplvalues[i] << "  ";
            dmtviewer_data << alphavalues[i] << "  ";
            dmtviewer_data << alphabetavalues[i] << "  ";
            dmtviewer_data << endl; 
        } 
        dmtviewer_data.close();

        //----- Pointers assigned in GainMon::GainMon().
        //---------- These pointers are always assigned.
        //delete uncalvalues;
        delete calvalues;
        delete amplvalues;
        delete alphavalues;
        delete alphabetavalues;
        delete psd;
        delete mCalibrate;
        delete mOsclist; 

        //----- Pointers assigned in GainMon::ProcessData().
	delete history_alpha;
	delete history_alphabeta; 
	delete history_ugf;
        delete history_ampl; 

        //----- Close files.
	if (mOutput.trend) mTrend.close();
        mOutput.cumlog_file.close(); 
    }
    //----- Send "finished happily" message.
    std::cout << "GainMon MESSAGE: GainMon is finished after "
	      << NStride << " strides.\n";
}


//======================================  Frame processing function.
void
GainMon::ProcessData(void) {

	//---------- Get current GPS time and check that it is an integer.
        Time time = getDacc().getFillTime();
        if (time.getS() != time.totalS()) {
            cerr << "GainMon WARNING: GPS start time not an integer: ";
            cerr << time.totalS() << endl;
        }

	//---------- Check for gap in data.
        if ( (time.totalS() != prev_endtime) && (NStride != 0) ) { 
          char msg[150];
          sprintf(
              msg,"Missing data between GPS times (%0.3f, %0.3f).\n",
              prev_endtime,time.totalS()
          );
	  std::cerr << "GainMon ERROR: " << msg;
          missing_strides = (int) (1+(time.totalS()-prev_endtime)/Run_Par.T);
        } else { 
	  missing_strides = 0;
	}
        prev_endtime = time.totalS();
        prev_endtime += Run_Par.T; 

	//---------- Calculate start time of histories to DMTViewer.
	//           Note:  TSeries start time = "time" - "DMTViewer_Period",
        //           where DMTViewer_Period is a defined quantity copied to 
        //           tserieslength. 
	Time history_start = time;
	history_start -= (tserieslength-1)*(Run_Par.T);
        
	//---------- Set cumlog_file_name.  Choices in order of preference are 
	//           (1) "$DMTHTMLOUT/<IFO>_GainMon_CumLog.txt" if 
        //               -local option NOT selected (default).
	//           (2) User's choice if -local and -logfile options selected.
	//           (3) Use <GPS_start_time>.log if only -local selected. 
        if (!mOutput.local) {  
	    mOutput.cumlog_file_name = string(mOutput.html_dir) + "/" + Run_Par.IFO +
				    "_GainMon_CumLog.txt";
	}
        if ( (mOutput.cumlog_file_name).empty() && !NStride ) {
            char temp_file_name[50]; 
            sprintf(temp_file_name,"%ld",time.getS());
            mOutput.cumlog_file_name = temp_file_name;
	    mOutput.cumlog_file_name += ".log";
        }

	//---------- Things to do once only.
    	if (NStride == 0) {

            //---------- Need some data for initializing certain storage.
            //           Doesn't matter if IFO is locked.

            //---------- Get data to TSeries.
	    ts_asq = getDacc().refData((Run_Par.mChannel).c_str());

	    //---------- Calcuate PSD.
	    psd->generate(PSDed_data,ts_asq);

	    //---------- Output run parameters.  
	    //----- Set starttime
 	    starttime = time.getS();

	    //----- Open logfile
	    mOutput.cumlog_file.open((mOutput.cumlog_file_name).c_str(), std::ios::app);

	    //----- Dump parameters to html log file and cout. 
            DumpParameters(cout);
            DumpParameters(mOutput.cumlog_file);

            DumpReportHeader(cout);
            DumpReportHeader(mOutput.cumlog_file);

            if (!mOutput.local) {  
                //----- Set up two-week summary file
	        std::string summary_name = string(mOutput.html_dir) + "/" + Run_Par.IFO +
		    		           "_summary.txt";
	        mGainSumm.init(summary_name, time);
	    }

            //---------- Write histories to DMTViewer.
            //----- Initialize and subscribe history of
	    //      range with calibration line.
            history_ugf = new TSeries(history_start,Run_Par.T,tserieslength,calvalues);
	    std::string DMTViewer_name = Run_Par.IFO;
            // DMTViewer_name = Run_Par.IFO;
            DMTViewer_name += "GAIN UGF (HZ)";
            serveData(DMTViewer_name.c_str(), history_ugf);
            //----- Initialize and subscribe history of calibration-line amplitude.
            history_ampl = new TSeries(history_start,Run_Par.T,tserieslength,amplvalues);
            DMTViewer_name = Run_Par.IFO;
            DMTViewer_name += "GAIN CAL LINE AMPL (ASQ)";
            serveData(DMTViewer_name.c_str(), history_ampl);
            //----- Initialize and subscribe history of alpha parameter.
            history_alpha = new TSeries(history_start,Run_Par.T,tserieslength,alphavalues);
            DMTViewer_name = Run_Par.IFO;
            DMTViewer_name += "GAIN CAL CAV FAC (ALPHA)";
            serveData(DMTViewer_name.c_str(), history_alpha);
            //----- Initialize and subscribe history of beta parameter.
            history_alphabeta = new TSeries(history_start,Run_Par.T,tserieslength,alphabetavalues);
            DMTViewer_name = Run_Par.IFO;
            DMTViewer_name += "GAIN CAL OLOOP FAC (ALPHAxBETA)";  
            serveData(DMTViewer_name.c_str(), history_alphabeta);

	    //---------- Make fake strain_noise_ampl for DMTViewer initialization. 
	    strain_noise_ampl = PSDed_data;
	    DMTViewer_name = Run_Par.IFO;
	    DMTViewer_name += "GAIN CALIBRATED NOISE SPECTRUM";
	    serveData(DMTViewer_name.c_str(), &strain_noise_ampl);
	    //---------- Generate fake f_7_3 for DMTViewer initialization. 
	    integrand(f_7_3, PSDed_data);
	    DMTViewer_name = Run_Par.IFO;
	    DMTViewer_name += "GAIN RANGE INTEGRAND (ARB UNITS)";
	    serveData(DMTViewer_name.c_str(), &f_7_3);
        }

	//---------- Check that IFO is locked before proceeding.
        if (mOsclist->satisfied(Run_Par.osc_cond.c_str())) {
	
	    //---------- Get input data to TSeries.
	    ts_asq = getDacc().refData((Run_Par.mChannel).c_str());

	    //---------- Calcuate PSD.
	    psd->generate(PSDed_data,ts_asq);

	    //---------- Update and apply calibration.  Extracts subsection 
	    //           of PSD for which we have calibration info.
            if (Run_Par.genCalParam) {
                //----- Use alpha, beta computed on-the-fly.
                mCalibrate->UpdateResponseFunction();

            } else {
                //----- Use alpha, beta stored in reference calibration file.
                mCalibrate->UpdateResponseFunction(time);
		
            }
	    mCalibrate->UpdateUGF();
            PSDed_data = mCalibrate->Apply(PSDed_data);

            //---------- Also retrieve calibration data to report.
            //           MUST BE DONE AFTER "UpdateResponseFunction"!
	    G_Dat.alpha = mCalibrate->GetAlpha();
	    G_Dat.beta = mCalibrate->GetBeta();
	    G_Dat.ampl = mCalibrate->GetLineAmplitudeASQ();


	    //---------- Obtain the unity gain frequency           
	    G_Dat.ugf   = mCalibrate->GetUGF();

	    //---------- Compute strain_noise_ampl from PSDed_data, for DMTViewer.
            //           This code effectively duplicates the Calibrated PSD dump code.
            //           I should remove the latter.

	    //----- Set up array to hold values of calibrated PSD.
            float* strainvalues = new float[PSDed_data.getNStep() + 1];

	    //----- Get values of calibrated PSD.
            PSDed_data.getData(PSDed_data.getNStep() + 1, strainvalues);

            //----- Specify arm length in nm for conversion of PSD units to strain:
            float arm_length = 1.0;  
            if (Run_Par.IFO == "H2") { 
                arm_length = 2.0e12;  
            } else if ((Run_Par.IFO == "L1") || (Run_Par.IFO == "H1")) { 
                arm_length = 4.0e12;  
            }

            //----- Compute sqrt of power spectrum, converted to units of strain/Hz^1/2.  
            for (int j=0; j<(int)(PSDed_data.getNStep()+1); j++) {
                strainvalues[j] = (float) pow((double) strainvalues[j],0.5)/arm_length;
            }

            //----- Copy data to strain_noise_ampl and clean up.
	    strain_noise_ampl = PSDed_data;
	    strain_noise_ampl.setData(PSDed_data.getNStep() + 1, strainvalues);
	    delete[] strainvalues;

	} else { 

	    //---------- IFO not Locked; set range to zero, don't increment
	    //		 4-volume.
	    G_Dat.alpha = 0.0;
	    G_Dat.beta = 0.0;
      	    G_Dat.ugf = 0.0;
	    G_Dat.ampl = 0.0;	

	    //----- Set to 1 PSDs to be send to DMTViewer.
	    int nsteps = PSDed_data.getNStep();
	    float *data = new float[nsteps + 1];
	    for (int i=0; i<(nsteps + 1); i++) {
		data[i] = 1;
	    }
	    strain_noise_ampl.setData(nsteps + 1, data);
	    f_7_3.setData(nsteps + 1, data);
	    delete[] data;
      	}
	
	//---------- Shift DMTViewer TSeries data one time step.
        int nshift = 1+missing_strides;
        for (int ii=0; ii<(tserieslength-nshift); ii++) {
            calvalues[ii] = calvalues[ii+nshift];
            amplvalues[ii] = amplvalues[ii+nshift];
            alphavalues[ii] = alphavalues[ii+nshift];
            alphabetavalues[ii] = alphabetavalues[ii+nshift];
        } 

	//---------- Fill any missing strides with -1.0.
	if (missing_strides) {
            for (int ii=(tserieslength-nshift); ii<(tserieslength-1); ii++) {
                calvalues[ii]   = -1.0;
                amplvalues[ii]  = -1.0;
                alphavalues[ii] = -1.0;
                alphabetavalues[ii]  = -1.0;
            } 
	}

	//---------- Store most recent values.
        calvalues[tserieslength-1] = G_Dat.ugf;
	amplvalues[tserieslength-1] = G_Dat.ampl;
	alphavalues[tserieslength-1] = G_Dat.alpha;
	alphabetavalues[tserieslength-1] = (G_Dat.alpha)*(G_Dat.beta);

	//---------- Export data to DMTViewer. 
	    history_ugf->setData(history_start, Run_Par.T, calvalues, tserieslength);            
 	    history_ampl->setData(history_start, Run_Par.T, amplvalues, tserieslength);
 	    history_alpha->setData(history_start, Run_Par.T, alphavalues, tserieslength);
 	    history_alphabeta->setData(history_start, Run_Par.T, alphabetavalues, tserieslength);

	//---------- Write the range and 4-volume scanned to the log file
	//	     and cout.
	ReportResults(mOutput.cumlog_file, time, Run_Par);
        if (mOutput.screen_output || mOutput.write_log) {
	    ReportResults(cout, time, Run_Par);
        }

	//---------- Update Trends.
	if (mOutput.trend) {
	    std::string trendName;
	        trendName = Run_Par.IFO + ":DMT-GAIN_UGF_HZ";
                mTrend.trendData(trendName.c_str(), time, G_Dat.ugf);
	        trendName = Run_Par.IFO + ":DMT-GAIN_CAL_LINE_AMPL_ASQ";
                mTrend.trendData(trendName.c_str(), time, G_Dat.ampl);
	        trendName = Run_Par.IFO + ":DMT-GAIN_CAL_CAV_FAC";
                mTrend.trendData(trendName.c_str(), time, G_Dat.alpha);
	        trendName = Run_Par.IFO + ":DMT-GAIN_CAL_OLOOP_FAC";
                mTrend.trendData(trendName.c_str(), time, (G_Dat.alpha)*(G_Dat.beta));
	}

        //---------- Do periodic range and calibrated PSD dumps if running in 
        //           background (!local) mode:    
	//           -> two-week summary file  
	//           -> calibrated psd dumped to ascii file
	//           NOTE: Since stride "Run_Par.T" may not exactly divide
        //           the specified update period "CalPSD_Dump_Period",
	//	     not all strides will contribute with equal rate
	//	     to a given average.  Eg: If CalPSD_Dump_Period is 1000 
        //           and Run_Par.T = 300, the first through 
	//	     fourth strides contribute to first 1000-sec average with
	//	     weights 1, 1, 1, 1/3.  Fourth stride also contributes to
	//	     second 1000-sec average with weight 2/3.
	//           Note also that report time (GPS truncated to last multiple
        //           of CalPSD_Dump_Period) is 
	//           guaranteed to lie in the 1000-sec interval reported on.
        //           Times when the IFO is not locked contribute zero to the average.
	if (!mOutput.local) {
	    time_since_update += (int) Run_Par.T;
	    if (Run_Par.CalPSD_Dump_Period && time_since_update >= Run_Par.CalPSD_Dump_Period) {

	        //---------- 2-week summary file update:

	        //----- Add current UGF to running average with
	        //      weight = fraction of stride coming before (CalPSD_Dump_Period)-sec mark.
	        running_average += (Run_Par.T - time_since_update + Run_Par.CalPSD_Dump_Period) /
		   		    Run_Par.T * G_Dat.ugf;
                running_count += (Run_Par.T - time_since_update + Run_Par.CalPSD_Dump_Period)/Run_Par.T;
	        running_average /= running_count;

	        //----- Dump running average to two-week summary file.
                //      Report time to last multiple of CalPSD_Dump_Period 
                //      (eg, to last 000 sec for CalPSD_Dump_Period = 1000).
                //cout <<  (time.getS() % CalPSD_Dump_Period) << endl;
                GainDataSummary tmpstatus = {time.getS() - (time.getS() % Run_Par.CalPSD_Dump_Period), running_average};
                mGainSumm.append(tmpstatus);
                mGainSumm.dumpList();

	        //----- Initialize next running average with whatever is left
	        //      of current stride.
	        time_since_update -= Run_Par.CalPSD_Dump_Period;
		running_average = (time_since_update) / Run_Par.T * G_Dat.ugf;
	        running_count = (time_since_update)/Run_Par.T;

		//----- calibrated psd file dump:  
                //if (mOsclist->satisfied(osc_cond1.c_str())  && 
                //    mOsclist->satisfied(osc_cond2.c_str())  ) {
                if (mOsclist->satisfied(Run_Par.osc_cond.c_str())) {

	            //---------- Set up array to hold values of calibrated PSD.
                    float* CalPSDValues = new float[PSDed_data.getNStep() + 1];
                    char* time_holder = new char[128];

		    //---------- Get values of calibrated PSD.
            	    PSDed_data.getData(PSDed_data.getNStep() + 1, CalPSDValues);

	    	    //---------- Print values to text file
                    //----- Specify arm length in nm for conversion of PSD units to strain:
                    float arm_length = 1.0;  
                    if (Run_Par.IFO == "H2") { 
                      arm_length = 2.0e12;  
                    } else if ((Run_Par.IFO == "L1") || (Run_Par.IFO == "H1")) { 
                      arm_length = 4.0e12;  
                    } else {
                      cerr << "GainMon ERROR: IFO " << Run_Par.IFO << " not recognized.  Calibrated PSD files will \n";
                      cerr << "                     not have units of strain.\n";
                    }
//          	    TimeStr(ts_asq->getStartTime(), time_holder, "%Y%M%d.%H:%N:%S.txt");
            	    TimeStr(ts_asq->getStartTime(), time_holder, "%s.txt");
//          	    TimeStr( operator-(ts_asq->getStartTime(),Interval(time_since_update,0)), time_holder, "%s.txt");
       		    mOutput.calpsd_file_name = time_holder;
            	    mOutput.calpsd_file_name = string(mOutput.html_dir) + "/" + Run_Par.IFO + "_CalPSD_GPS_" + mOutput.calpsd_file_name;
            	    mOutput.calpsd_file.open(mOutput.calpsd_file_name.c_str());
            	    mOutput.calpsd_file << "#Calibrated PSD of Channel, produced by GainMon version 1.0." << endl;
            	    mOutput.calpsd_file << "#Units: (strain)/Hz^{-0.5}" << endl;
                    mOutput.calpsd_file << "#Number of Averages for PSD: " << Run_Par.num_ints << endl;
                    mOutput.calpsd_file << "#Window: " << Run_Par.window_name << endl;
            	    mOutput.calpsd_file << "#GPS start time of this data: " << time.getS() << endl;
                    int npoints = PSDed_data.getNStep() + 1; 
                    mOutput.calpsd_file << "#Number of points: " << npoints << endl;
            	    mOutput.calpsd_file << "#Frequency step (Hz): " << PSDed_data.getFStep() << endl;
            	    mOutput.calpsd_file << "#Min Frequency (Hz): " << PSDed_data.getLowFreq() << endl;
            	    mOutput.calpsd_file << "#Max Frequency (Hz): " << PSDed_data.getLowFreq() + (npoints-1)*(PSDed_data.getFStep())<< endl;
            	    mOutput.calpsd_file << "#UGF (Hz): " << G_Dat.ugf << endl;
            	    mOutput.calpsd_file << "#Calibration file comment: " << mCalibrate->getComment() << endl;
            	    mOutput.calpsd_file << "#alpha parameter: " << mCalibrate->GetAlpha() << endl;
            	    mOutput.calpsd_file << "#beta parameter: " << mCalibrate->GetBeta() << endl;

            	    //----- Dump sqrt of power spectrum, converted to units of strain/Hz^1/2.  
            	    for (int j=0; j<(int)(PSDed_data.getNStep()+1); j++) {
                        mOutput.calpsd_file << pow((double) CalPSDValues[j],0.5)/arm_length << endl;
                    }
            	    mOutput.calpsd_file.close();

                    //---------- Clean up.
                    delete[] CalPSDValues;
                    delete[] time_holder;
	        }
            } else { 

	        //----- Add current range to running average with weight 1.
    	        //running_average += R_Dat.range_uncal;
    	        running_average += G_Dat.ugf;
    	        running_count   += 1.0;
	    }
	}

	//----------  Calculate average of last five time segments.
	G_Dat.T5_array[NStride % 5] = G_Dat.ugf;
	for(int i = 0; i < 5; i++)
	    G_Dat.T5_gain += G_Dat.T5_array[i];
	G_Dat.T5_gain /= 5.0;

	//----------  Calculate average of last fifteen time segments.
	G_Dat.T15_array[NStride % 15] = G_Dat.ugf;
	for(int i = 0; i < 15; i++)
            G_Dat.T15_gain += G_Dat.T15_array[i];
        G_Dat.T15_gain /= 15.0;

	//---------- Print summary webpages.
	RevolverSummary(time, Run_Par, G_Dat);
	Summary(time, Run_Par, G_Dat);

	//---------- Clear old data.
	G_Dat.T15_gain = 0.0;
	G_Dat.T5_gain = 0.0;

	//---------- Check if should finish.
	if (++NStride >= MaxStride) finish();

	//----------  Flush any lingering data.
        fflush(stderr);
	fflush(stdout);
}


//====================================== Write revolver.html page
void
GainMon::RevolverSummary(const Time& t, Parameter Run_Par,
			      Gain_Data G_Dat)
{
    //---------- Html document
    html::document doc (kHtmlTitle);
    char buf[128];

    //---------- Title
    html::block titleblk ("center");
    titleblk.lineBreak();
    html::text title (kHtmlTitle);
    html::font fonttitle ("Helvetica");
    title.setFont (fonttitle);
    title.setColor (html::color (0, 0, 128));
    title.setSize (html::size (+4));
    titleblk.addObject (title);
    titleblk.lineBreak();
    titleblk.addObject (html::text ("&nbsp;"));
    doc.addObject (titleblk);
    doc.addObject (html::hline());

    //---------- Current status section.
    html::block status_blk ("center");
    status_blk.lineBreak();
    html::text status_title ("Current UGF");
    status_title.setFont (fonttitle);
    status_title.setSize (html::size (+2));
    status_blk.addObject (status_title);
    status_blk.lineBreak();

    //---------- Result table.
    html::table results;
    results.addColumn ("GPS Time");
    results.addColumn ("15-Stride Average UGF (Hz)");
    results.addColumn ("Next Update Time (UTC)");
    results.addRow();
    results.insertData(0, 0, html::text (TimeStr(t, buf, "%s")));
    results.insertData(0, 1, html::text (G_Dat.T15_gain));
    results.insertData(0, 2, html::text (t + Run_Par.T));
    for (int j=0; j<3; ++j) {
        results.refCell(0, j).setAlign ("right");
    }
    results.setBorder(true);
    status_blk.addObject(results);
    doc.addObject(status_blk);

    //---------- Last update time.
    doc.addObject (html::hline());
    html::block updateblk ("center");
    html::text lasttime ("This page was last updated: ");
    lasttime << TimeStr (t, buf, "%M %d, %Y at %H:%N:%S");
    updateblk.addObject (lasttime);
    updateblk.lineBreak();
    doc.addObject (updateblk);
    doc.addObject (html::hline());

    //---------- Write web page to file.
    ofstream out (mOutput.revolver_file_name.c_str());
    if (out) {
        html::writer htmlout (out);
        doc.write (htmlout);
    }
}


//====================================== Write summary html page.
void
GainMon::Summary(const Time& t, Parameter Run_Par, Gain_Data G_Dat)
{   
    //---------- Html document
    html::document doc (kHtmlTitle);
    char buf[128];

    //---------- Title
    html::block titleblk ("center");
    titleblk.lineBreak();
    html::text title (kHtmlTitle);
    html::font fonttitle ("Helvetica");
    title.setFont (fonttitle);
    title.setColor (html::color (0, 0, 128));
    title.setSize (html::size (+4));
    titleblk.addObject (title);
    titleblk.lineBreak();
    titleblk.addObject (html::text ("&nbsp;"));
    doc.addObject (titleblk);
    doc.addObject (html::hline());
   
    //---------- Short description
    doc.addObject (html::text (kHtmlDescription));
    doc.addObject (html::hline());

    //---------- Last update time.
    html::block firstupdateblk ("center");
    html::text lasttime1 ("This page was last updated: "); 
    lasttime1 << TimeStr (t, buf, "%M %d, %Y at %H:%N:%S %Z");
    firstupdateblk.addObject (lasttime1);
    firstupdateblk.lineBreak();
    html::text lasttime2 ("Next scheduled update time: ");
    lasttime2 << TimeStr (t+Run_Par.T, buf, "%M %d, %Y at %H:%N:%S %Z");
    firstupdateblk.addObject (lasttime2);
    firstupdateblk.lineBreak();
    doc.addObject (firstupdateblk);
    doc.addObject (html::hline());

    //---------- Run parameters
    html::block run_param_blk ("center");
    run_param_blk.lineBreak();
    html::text param_title ("Run Parameters");
    param_title.setFont (fonttitle);
    param_title.setSize (html::size (+2));
    run_param_blk.addObject (param_title);
    run_param_blk.lineBreak();
    //---------- Parameters table.
    html::table params; 
    //----- WARNING:  Must make columns first, then rows.
    params.addColumn("");
    params.addColumn("");
    for (int j=0; j<11; ++j) {
	params.addRow();
    }
    int i=-1;
    params.insertData(++i, 0, html::text ("Channel: "));
    params.insertData(  i, 1, html::text (Run_Par.mChannel));
    params.insertData(++i, 0, html::text ("Stride: "));
    params.insertData(  i, 1, html::text (Run_Par.T));
    params.insertData(  i, 1, html::text (" sec"));
    params.insertData(++i, 0, html::text ("Number of Strides Already "
					  "Processed: "));
    params.insertData(  i, 1, html::text (NStride+1));
    params.insertData(++i, 0, html::text ("Maximum Number of Strides to "
					  "Process: "));
    params.insertData(  i, 1, html::text (MaxStride));
    params.insertData(++i, 0, html::text ("Number of Averages for PSD: "));
    params.insertData(  i, 1, html::text (Run_Par.num_ints));
    params.insertData(++i, 0, html::text ("Window: "));
    params.insertData(  i, 1, html::text (Run_Par.window_name));
    params.insertData(++i, 0, html::text ("Low Frequency : "));
    params.insertData(  i, 1, html::text (Run_Par.l_freq));
    params.insertData(  i, 1, html::text (" Hz"));
    params.insertData(++i, 0, html::text ("High Frequency : "));
    params.insertData(  i, 1, html::text (Run_Par.h_freq));
    params.insertData(  i, 1, html::text (" Hz"));
    params.insertData(++i, 0, html::text ("Monitor Start Time (GPS): "));
    params.insertData(  i, 1, html::text (TimeStr(Time(starttime), buf,
					  "%s")));
    params.insertData(++i, 0, html::text ("(UTC): "));
    params.insertData(  i, 1, html::text (TimeStr(Time(starttime), buf,
					  "%M %d, %Y at %H:%N:%S")));

    //----- WARNING:  It appears that alignments can only be set AFTER the cell
    //	    is filled.
    //      WARNING:  The (html::table).getNRow() method returns 1+(# of rows).
    //	    Beware!
    for (int j=0; j<params.getNRow()-1; ++j) {
        params.refCell(j, 0).setAlign("right");
        params.refCell(j, 1).setAlign ("left");
    }
    run_param_blk.addObject(params);
    doc.addObject (run_param_blk);
   
    //---------- Current status section.
    html::block status_blk ("center");
    status_blk.lineBreak();
    html::text status_title ("Current UGF");
    status_title.setFont (fonttitle);
    status_title.setSize (html::size (+2));
    status_blk.addObject (status_title);
    status_blk.lineBreak();
   
    //---------- Result table.
    html::table results;
    results.addColumn ("GPS Time");
    results.addColumn ("1-Stride UGF (Hz)");
    results.addColumn ("5-Stride Average UGF (Hz)");
    results.addColumn ("15-Stride Average UGF (Hz)");
    results.addColumn ("Next Update Time (UTC)");
    results.addRow();
    results.insertData(0, 0, html::text (TimeStr(t, buf, "%s")));
    results.insertData(0, 1, html::text (G_Dat.ugf));
    results.insertData(0, 2, html::text (G_Dat.T5_gain));
    results.insertData(0, 3, html::text (G_Dat.T15_gain));
    results.insertData(0, 4, html::text (t + Run_Par.T));
    for (int j=0; j<5; ++j) {
        results.refCell(0, j).setAlign ("right");
    }
    results.setBorder(true);
    status_blk.addObject(results);
    doc.addObject(status_blk);

    if (!mOutput.screen_output) {  
        //---------- Insert error message link
        html::block errorinfo ("center");

        //---------- Link to error-message log.
	html::link errormessage ("Error Messages", mOutput.error_file_link.c_str() );
	errorinfo.addObject (errormessage);
        errorinfo.lineBreak();
        doc.addObject (errorinfo);

        //---------- Insert links to current (this run of monitor) and 
        //           cumulative (all runs) log files.
        html::block output ("center");
	html::link outputlink1 ("Monitor Output: more details", (mOutput.log_file_link).c_str() );
        output.addObject (outputlink1);
        output.lineBreak();
        if (!mOutput.local) {
	    //html::link outputlink2 ("Cumulative Monitor Output from all runs", (cumlog_file_link).c_str() );
	    html::link outputlink2 ("Cumulative Monitor Output from all runs", (mOutput.cumlog_file_link).c_str() );
            output.addObject (outputlink2);
            output.lineBreak();
        }
        output.lineBreak();
        doc.addObject (output);
    }

    //---------- Last update time.
    doc.addObject (html::hline());
    html::block updateblk ("center");
    html::text lasttime ("This page was last updated: ");
    lasttime << TimeStr (t, buf, "%M %d, %Y at %H:%N:%S %Z");
    updateblk.addObject (lasttime);
    updateblk.lineBreak();

    //---------- Author info.
    updateblk.addObject( html::text ("This monitor was written by ") );
    updateblk.addObject( html::link ("Patrick J. Sutton",
				     "mailto:psutton@ligo.caltech.edu, ") );  
    updateblk.addObject( html::link ("Kevin C. Schlaufman",
			             "mailto:kcs149@psu.edu") );  
    updateblk.addObject( html::text (" and ") );
    updateblk.addObject( html::link ("Aaron M. Rogan",
			             "mailto:arogan@wsunix.wsu.edu") );

    doc.addObject (updateblk);
    doc.addObject (html::hline());

    //---------- LIGO logo.
    html::block logoblk ("div");
    logoblk.addAttr ("align", html::align ("right"));
    logoblk.addObject (html::link (kHtmlLIGO, kHtmlLIGOLink));
    html::image logo;
    logo.setSource ("ligo_logo.gif");
    logo.setWidth ("80");
    logoblk.addObject (logo);
    doc.addObject (logoblk);

    //---------- Write web page to file.
    ofstream out (mOutput.summary_file_name.c_str());
    if (out) {
        html::writer htmlout (out);
        doc.write (htmlout);
    }
}


//====================================== Dump run parameters to output stream.
void
GainMon::DumpParameters(std::ostream& out)
{   
    //----- Dump run parameters.
    out << "# Command Line: " << command_line << endl;
    out << "# Launch Directory: " << pwd_dir <<  endl;
    out << "# Log File: " << mOutput.cumlog_file_name << endl;
    out << "# OSC Configuration File: " << Run_Par.osc_file_name << endl; 
    out << "# Reference Calibration File Name: " << Run_Par.xml_file_name << endl; 
    out << "# Reference Calibration File Comment: " << mCalibrate->getComment() << std::endl;
    out << "# Minimum physically allowed alpha*beta: " << mCalibrate->GetMinAlphaBeta() << std::endl;
    out << "# Maximum physically allowed alpha*beta: " << mCalibrate->GetMaxAlphaBeta() << std::endl;
    out << "# Channel: " << Run_Par.mChannel << endl;
    out << "# Stride: " << Run_Par.T << " sec" << endl;
    out << "# Max Number of Strides: " << MaxStride << " sec" << endl;
    out << "# Number of Averages for PSD: " << Run_Par.num_ints << endl;
    out << "# Low Freq.  : " << Run_Par.l_freq << " Hz" << endl;
    out << "# High Freq. : " << Run_Par.h_freq << " Hz" << endl;
    out << "# Window: " << Run_Par.window_name << endl;
    out << "# Monitor Start Time: " << starttime << endl;
    out << "# " <<endl;
}


//====================================== Dump column headers to output.
void
GainMon::DumpReportHeader(std::ostream& out) 
{
    //----- Dump headers for range data that will follow.
    out << "# 1) Data value of 0 means IFO not locked." << endl;
    out << "# 2) Data values of -1, -2 mean monitor not functioning." << endl;
    out << "# " << endl;
    out << "#                                             CAL CAV FAC   CAL OLOOP FAC" << endl;
    out << "#   Start                      High-Freq.     Calibration     Calibration" << endl;
    out << "# Time of             UGF      Line Ampl.       Parameter       Parameter" << endl;
    out << "# Segment            (Hz)    (ASQ Counts)           alpha      alpha*beta" << endl;
//  out << "#       -               -               -               -               -" << endl;
}


//====================================== Dump column headers to output.
void
GainMon::DumpReportHeader(std::ostream& out, double freq) 
{

    //----- Dump headers for range data that will follow.
    out << "# Notes:" << endl;
    out << "# 1) Range of 0 means IFO not locked." << endl;
    out << "# 2) Ranges quoted are adjusted by the" << endl;
    out << "#    calibration-line amplitude only if the -dyncal option is used." << endl;
    out << "# " << endl;
}


//====================================== Read configuration file and pass contents 
//                                       to command-line parser.
void
GainMon::Configure(string& file_name) 
{
    int NEWargc = 1;
    //cout << "Trying to read config file " << file_name << ".\n";
    //---------- Verify that configuration file can be opened.
    ifstream input;
    input.open(file_name.c_str());
    if (input.fail()) {
        std::cerr << "GainMon ERROR: Cannot open configuration file ";
        std::cerr << file_name  << ".  Exiting.\n";
        fatal_error = true;
        finish();
    } else {
        cout << "GainMon MESSAGE: Opened configuration file ";
        cout << file_name << endl;
        //---------- Get number of words in configuration file.
        std::string word;
        while (input >> word) {
            ++NEWargc;
            //cout << word << endl;
        }
        //cout << "NEWargc = " << NEWargc << endl;
    }
    input.close();
    const char* hedr = "Configuration File: ";
    char** NEWargv = new char*[NEWargc];
    NEWargv[0]  = new char[strlen(hedr)+1];
    strcpy(NEWargv[0], hedr);
    //cout << NEWargv[0] << endl;
    //---------- Open file and copy contents to char arrays.
    ifstream inp2;
    inp2.open(file_name.c_str());
    if (inp2.fail()) {
        cout << "Yikes!!!!! Can't open " << file_name << endl;
	return;
    }
    std::string word;
    for (int i=1; i<NEWargc; i++) {
        inp2 >> word;
        NEWargv[i] = new char[80];
        strcpy(NEWargv[i],word.c_str());
        //cout << NEWargv[i] << endl;
    }
    inp2.close();
    Configure(NEWargc,(const char **) NEWargv);
    for (int i=0; i<NEWargc; i++) {
	delete[] NEWargv[i];
    }
    delete[] NEWargv;
}


//====================================== Parse contents of configuration file or command line.
void
GainMon::Configure(int argc, const char *argv[]) 
{
/*
    cout << "GainMon::Configure(int argc, const char *argv[]) called.\n";
    cout << "argc, argv[] are : " << argc << "  ";
    for (int i=0; i<argc; i++) {
        cout << argv[i] << "  ";
    }
    cout << endl;
*/

    //---------- Get command line, for the record.
    for (int i=0; i < argc; i++) {
        std::string arg_i = argv[i];
        command_line += arg_i;        
        command_line += "  ";         
    }
    
    //---------- Get local directory, for the record.
    char *tempdir = getenv("PWD");
    if (tempdir) {
	pwd_dir = string(tempdir) + "/";
    } 
    
    //---------- Get Channel name and IFO.
    Run_Par.IFO = argv[argc - 1];
    Run_Par.mChannel = Run_Par.IFO; 
    Run_Par.mChannel += ":LSC-DARM_ERR";

    //----- Set default parameters.
    Run_Par.osc_cond = Run_Par.IFO + ":SV_IFO_UP";

    //----- Parse command-line arguments, and compare to list of known 
    //      values.  Ignore unrecognized options.
    if ( (argc <= 1) || (argv[argc-1][0] == '-') ) {
        cout << USAGE_INFO << endl;
        fatal_error = true;
        finish();
    } else {
        for (int i=1 ; i<argc-1 ; i++) {
            std::string argi = argv[i];
            if (argi == "-max") {
	        MaxStride = strtol(argv[++i], 0, 0);
 	    } else if (argi == "-stride") {
	        Run_Par.T = strtod(argv[++i], 0);
 	    } else if (argi == "-channel") {
	        Run_Par.mChannel = argv[++i];
	    } else if (argi == "-n") {
	        Run_Par.num_ints = strtol(argv[++i], 0, 0);
	    } else if (argi == "-PSD_DumpPeriod") {
	        Run_Par.CalPSD_Dump_Period = strtol(argv[++i], 0, 0);
	    } else if (argi == "-fmin") {
	        Run_Par.l_freq = strtod(argv[++i], 0);
	    } else if (argi == "-fmax") {
	        Run_Par.h_freq = strtod(argv[++i], 0);
 	    } else if (argi == "-xmlfile") { 
                Run_Par.xml_file_name = argv[++i];
	    } else if (argi == "-logfile") {
	        mOutput.cumlog_file_name = argv[++i];
 	    } else if (argi == "-OSCcond") { 
                Run_Par.osc_cond = Run_Par.IFO + ":" + argv[++i];
 	    } else if (argi == "-OSCfile") { 
                Run_Par.osc_file_name = argv[++i];
	    } else if(argi == "-readalphabeta") {
                Run_Par.genCalParam = false; 
	    } else if(argi == "-screen") {
                mOutput.screen_output = true;
     	        mOutput.write_log = false;
 	        mOutput.error_log = false;
	    } else if(argi == "-trend") {
	        mOutput.trend = true;
  	    } else if(argi == "-window") {
                Run_Par.window_name = argv[++i];
  	        for(int ii = 0; ii < (int)((Run_Par.window_name).length()); ii++) {
                    Run_Par.window_name[ii] = tolower(Run_Par.window_name[ii]);
	        }
	    } else if ( (argi == "-h") || (argi == "--help") ) {
	        cout << USAGE_INFO << endl;
                fatal_error = true;
	        finish();
	    } else if(argi == "-local") {
                mOutput.local = true;
            } else if (isDatEnvArg(argv[i])) {
                //std::cout << "isDatEnvArg == true:  " << argv[i] << std::endl;
	        i++;
            } else {
	        std::cerr << "GainMon WARNING:  Argument: " << argi;
	        std::cerr << " not recognized." << std::endl;
                cout << USAGE_INFO << endl;
                //fatal_error = true;
	        //finish();
	    }
        }
    }
}


//====================================== Run list of checks on parameters.
//:KLUDGE: Since this is a method of the GainMon class, shouldn't need to 
//hand it data members of the same class (Run_Par, fatal_error).
void
GainMon::VerifyParameters(Parameter& Run_Par, bool& fatal_error)
{
    if ( ((int)(Run_Par.T) % Run_Par.num_ints) != 0 ) {
	cout << "GainMon ERROR: The number of averages 'n' must divide the length 'stride'.\n"; 
	cout << "                    Exiting.\n";
	cout << USAGE_INFO << endl;
	fatal_error = true;
	finish();
    } 
    if ( Run_Par.l_freq >= Run_Par.h_freq ) {
        cout << "GainMon ERROR: Low frequency must be less than high ";
	cout << "frequency.  Exiting.\n";
        cout << USAGE_INFO << endl;
        fatal_error = true;
        finish();
    }
/*
    if (((int)(86400.0 / Run_Par.T) != (86400.0 / Run_Par.T)) && mOutput.trend) {
	cout << "GainMon ERROR: 'stride' must divide 86400 (number of seconds in 1 day).\n";
	cout << "                    Exiting.\n";
	cout << USAGE_INFO << endl;
	fatal_error = true;
	finish();
    }
*/
    if (Run_Par.window_name == "square") {
	cout << "GainMon WARNING: Square window (ie, no windowing) selected for\n"; 
	cout << "                      power-spectrum estimation.  This is a VERY BAD\n";
	cout << "                      IDEA; you should select anything else.\n";
    }
    if (!fatal_error) {
        //if ( !mOutput.local && !((Run_Par.cumlog_file_name).empty()) ) {
        if ( !mOutput.local && !((mOutput.cumlog_file_name).empty()) ) {
	    cout << "GainMon WARNING: -local option has not been selected; specification\n";
            cout << "                      of log-file name will be ignored.\n";
        }
//----- :KLUDGE: Put in corresponding check that xml calibration file can be read.
/*
        //---------- Get name of calibration file.
	if (Run_Par.statcal) {
	    if ( (Run_Par.cal_file_name).empty() ) {
  	        cout << "GainMon WARNING: No calibration file specified; will look for default\n";
                cout << "                      calibration file.\n";
                //---------- Look for calibration file in directory specified
                //           in $GAINMON_CALIBRATION instead of in local directory.
                const char* cal_dir = getenv("GAINMON_CALIBRATION");
                if (cal_dir) {
                    Run_Par.cal_file_name = string(cal_dir) + "/";
                } else { 
	            cout << "GainMon WARNING: Environment variable $GainMon_CALIBRATION not set;\n"; 
                    cout << "                      looking in local directory for default calibration file.\n";
                }
                Run_Par.cal_file_name += "GainMon_Calibration_";
                Run_Par.cal_file_name += Run_Par.IFO;
                Run_Par.cal_file_name += ".txt";
            }
            //---------- Verify that calibration file can be opened.
            ifstream input;
            input.open((Run_Par.cal_file_name).c_str());
            if (input.fail()) {
                std::cerr << "GainMon ERROR: Cannot open calibration file ";
                std::cerr << Run_Par.cal_file_name  << ".  Exiting.\n";
                fatal_error = true;
                finish();
            } else {
	        cout << "GainMon MESSAGE: Found calibration file ";
	        cout << Run_Par.cal_file_name << endl;
	    }
            input.close();
	} else {
            //---------- Verify that open-loop-gain and sensing-function files can be opened.
            ifstream input;
            input.open((Run_Par.olg_file_name).c_str());
            if (input.fail()) {
                std::cerr << "GainMon ERROR: Cannot open open-loop-gain file ";
                std::cerr << Run_Par.olg_file_name  << ".  Exiting.\n";
                fatal_error = true;
                finish();
            } else {
	        cout << "GainMon MESSAGE: Found open-loop-gain file ";
	        cout << Run_Par.olg_file_name << endl;
	    }
            input.close();
            input.open((Run_Par.sf_file_name).c_str());
            if (input.fail()) {
                std::cerr << "GainMon ERROR: Cannot open sensing-function file ";
                std::cerr << Run_Par.sf_file_name  << ".  Exiting.\n";
                fatal_error = true;
                finish();
            } else {
	        cout << "GainMon MESSAGE: Found sensing-function file ";
	        cout << Run_Par.olg_file_name << endl;
	    }
            input.close();
	}
*/
        //---------- Get name of OSC configuration files.
        if ( (Run_Par.osc_file_name).empty() ) {
  	    cout << "GainMon WARNING: No OSC configuration file specified; will look for\n";
            cout << "                      default OSC configuration file.\n";
            //---------- Look for OSC config file in directory specified
            //           in $GAINMON_OSCCONF instead of in local directory.
            const char* osc_dir = getenv("GAINMON_OSCCONF");
            if (osc_dir) {
                Run_Par.osc_file_name = string(osc_dir) + "/";
            } else { 
	        cout << "GainMon WARNING: Environment variable $GAINMON_OSCCONF not set;\n"; 
                cout << "                      looking in local directory for default OSC file.\n";
            }
            //Run_Par.osc_file_name += "LockLoss_";
            //Run_Par.osc_file_name += Run_Par.IFO;
            Run_Par.osc_file_name += "GainMon_LockLoss.conf";
        }
//----- :KLUDGE: Put in corresponding check that xml calibration file can be read.
/*
        //---------- Verify that calibration file can be opened.
        ifstream input;  //----- previous declaration inside if () statement therefore gone.
        input.open((Run_Par.osc_file_name).c_str());
        if (input.fail()) {
            std::cerr << "GainMon ERROR: Cannot open OSC configuration file ";
            std::cerr << Run_Par.osc_file_name  << ".  Exiting.\n";
            fatal_error = true;
            finish();
        } else {
	    cout << "GainMon MESSAGE: Found OSC configuration file ";
	    cout << Run_Par.osc_file_name << endl;
	}
        input.close();
*/
    }
}


//====================================== Write current range, etc to specified
//                                       output stream.
void
GainMon::ReportResults(std::ostream& out, const Time& time, Parameter& Run_Par)
{
	//---------- Write the range, alpha, etc. to the specified 
	//	     output stream.
	out << time.getS();
	out << ' ' << setw(15) << G_Dat.ugf;
	out << ' ' << setw(15) << G_Dat.ampl;
	out << ' ' << setw(15) << G_Dat.alpha;
	out << ' ' << setw(15) << (G_Dat.alpha)*(G_Dat.beta);
        out << endl;
}


