/* -*- mode: c++; c-basic-offset: 4; -*- */
//#############################################################################
//#  DuoTone Timing Montoring Program    v0.0
//#  
//#  By:   Szabi Marka, Oct 2005
//#
//#############################################################################

#include "DuoTone.hh"

#ifndef __CINT__

//#include "ProcIdent.hh"
#include <iostream>
#include <fstream>
#include <sstream>
#include <cstdlib>
#include <cstdio>
#include <time.h>
#include <cmath>
#include "TSeries.hh"
#include "FSeries.hh"
#include "Dacc.hh"
#include "DVecType.hh"
#include <cstring>
#include <cstdlib>
#include "FixedLenTS.hh"
#include "lcl_array.hh"
#include <signal.h>
#include "constant.hh"
#if 0
#include "Sine.hh"
#endif

//-------------------------------------- 
typedef double VecType;
typedef DVecType<VecType> data_vect;


EXECDAT(DuoTone)
#endif               // !def(__CINT__)

inline std::string
env_to_str(const char* e, const char* def="") {
    std::string s;
    const char* ev = getenv(e);
    if (ev) {
	s = ev;
    } else {
	s = def;
    }
    return s;
}

using namespace std;

static void
duo_syntax(void) {
    cout << "Usage: DuoTone \n"  
	 << "        [optional -h provides this help" << endl
	 << "        [optional --alarm Alarm threshold (in us)]" << endl
	 << "        [optional -e <Epics Ch name root> enable epics communication"
	 << endl
	 << "        [optional -i <IFO, e.g. H1, H2, L1> specifies ifo \n"
	 << "        [optional -cS<subsys> <dualtone channel name>]" << endl 
	 << "        [optional -cR<subsys> <RAMP channel name>]" << endl
	 << "        [optional --offset channel nominal time offset (in us)]"
	 << endl
	 << "        [optional -t <trend output channel name root>]" << endl
	 << "        [optional -f (read from file) <filename>]" << endl
	 << "        [optional -n N maximum number of frames to read]" << endl
	 << "        [optional -w Web output directory/file]" << endl
	 << "        [optional -l Log output directory/file]" << endl
	 << "        [optional --stride Processing stride (10s default)]"
	 << endl
	 << "Example: DuoTone -e H1:DMT-TM_ -cS H1:LSC-TIME_MON \\" << endl
	 << "                 -cSPEM H1:GDS-TIME_MON -cRPEM H1:DAQ-GPS_RAMP_L1 \\"
	 << endl
	 << "                 -t H1:DMT-DUOT_ -i H1" << endl
	 << endl;
}

//=============================================================================
//
//     Channel processing class
//
//=============================================================================
DuoTone::time_chan::time_chan(const std::string& ch, chan_type t)
    : _chan(ch), _type(t), _f1(960), _f2(961), _offset(0)
{
    size_t ic = ch.find(":");
    size_t id = ch.find("-");
    if (ic != string::npos) ic++;
    if (id > ic) _subsys = ch.substr(ic, id-ic);
}


double
DuoTone::time_chan::calculateTime(const TSeries& ts) {
    params(ts);
    if (_f1 == _f2) return 0.0;
    double phi1  = atan2(_cSin1, _cCos1);
    double phi2  = atan2(_cSin2, _cCos2);
    double phidiff = phi2 - phi1;
    if (phidiff >=  pi) phidiff -= twopi;
    if (phidiff <  -pi) phidiff += twopi;
    return phidiff*1e6/(twopi*(_f2 - _f1)) - _offset; //time in microseconds.
}

void
DuoTone::time_chan::params(const TSeries& ts) {
    if (!ts.getStartTime() || ts.empty()) return;
    Time t0 = ts.getStartTime();
    if (_param_t0 == t0) return;
    
    //----------------------------------  Initialize vectors
    if (_CosF1.empty()) {
	Interval step = ts.getTStep();
	long    nstep = ts.getNSample();
#if 0
	_SinF1 = TSeries(Time(0), step, nstep, Sine(_f1, 1.0, 0.0   ));
	_SinF1.Convert(data_vect::getDataType());
	_CosF1 = TSeries(Time(0), step, nstep, Sine(_f1, 1.0, pi/2.0));
	_CosF1.Convert(data_vect::getDataType());
	_SinF2 = TSeries(Time(0), step, nstep, Sine(_f2, 1.0, 0.0   ));
	_SinF2.Convert(data_vect::getDataType());
	_CosF2 = TSeries(Time(0), step, nstep, Sine(_f2, 1.0, pi/2.0));
	_CosF2.Convert(data_vect::getDataType());
#else
	data_vect dvCosF1(nstep);
	data_vect dvSinF1(nstep);
	data_vect dvCosF2(nstep);
	data_vect dvSinF2(nstep);
	double dPhi1 = _f1 * twopi * step;
	double dPhi2 = _f2 * twopi * step;
	for (long i=0; i<nstep; i++) {
	    double phi = dPhi1 * double(i);
	    dvSinF1[i] = sin(phi);
	    dvCosF1[i] = cos(phi);
	    phi = dPhi2 * double(i);
	    dvSinF2[i] = sin(phi);
	    dvCosF2[i] = cos(phi);
	}
	_SinF1 = TSeries(Time(0), step, dvSinF1);
	_CosF1 = TSeries(Time(0), step, dvCosF1);
	_SinF2 = TSeries(Time(0), step, dvSinF2);
	_CosF2 = TSeries(Time(0), step, dvCosF2);
#endif
	
#if 0
	//------------------------------  Dump the matrix
	cout << "Duotone Hessian matrix: " << endl;
	cout << "      c1    s1    c2    s2    1" << endl;
	cout << " c1 " << 2*(_CosF1*_CosF1)/nstep << endl;
	cout << " s1 " << 2*(_SinF1*_CosF1)/nstep << "  "
	     << 2*(_SinF1*_SinF1)/nstep << endl;
	cout << " c2 " << 2*(_CosF1*_CosF2)/nstep
	     << "  "  << 2*(_SinF1*_CosF2)/nstep
	     << "  "  << 2*(_CosF2*_CosF2)/nstep << endl;
	cout << " s2 " << 2*(_CosF1*_SinF2)/nstep
	     << "  " << 2*(_SinF1*_SinF2)/nstep
	     << "  " << 2*(_CosF2*_SinF2)/nstep
	     << "  " << 2*(_SinF2*_SinF2)/nstep << endl;
	cout << "  1 " << _CosF1.getAverage() << "  " 
	     << _SinF1.getAverage() << "  "
	     << _CosF2.getAverage() << "  " 
	     << _SinF2.getAverage() << "  " << 1.0 << endl;
#endif
    } // end initialization

    //----------------------------------  Parameterize signal
    //
    //   This is basically a 5 parameter least-square fit with 5 parameter
    //   least-square fit that assumes the 5 parameters are normalized and
    //   orthogonal.
    double N = ts.getNSample();
    _cCos1 = 2 * (ts * _CosF1) / N;
    _cSin1 = 2 * (ts * _SinF1) / N;
    _cCos2 = 2 * (ts * _CosF2) / N;
    _cSin2 = 2 * (ts * _SinF2) / N;
    _cOff  = ts.getAverage();
    _param_t0 = ts.getStartTime();
}

//=======================================  Calculate rms
double
DuoTone::time_chan::rms(const TSeries& ts) {
    params(ts);
    size_t N = ts.getNSample();
    if (!N) return 0.0;
    TSeries diffs(ts);
    diffs -= _cOff;
    Time t0 = ts.getStartTime();
    Interval dT = ts.getTStep();
    TSeries temp(t0, dT, *_CosF1.refDVect());
    temp  *= _cCos1;
    diffs -= temp;
    temp   = TSeries(t0, dT, *_SinF1.refDVect());
    temp  *= _cSin1;
    diffs -= temp;
    temp   = TSeries(t0, dT, *_CosF2.refDVect());
    temp  *= _cCos2;
    diffs -= temp;
    temp   = TSeries(t0, dT, *_SinF2.refDVect());
    temp  *= _cSin2;
    diffs -= temp;
    return (diffs*diffs)/double(N);
}

//====================================  Set the frequency
void 
DuoTone::time_chan::setF(double f1, double f2) {
    _f1 = f1;
    _f2 = f2;
}

//====================================  Set the nominal time offset in us
void 
DuoTone::time_chan::setOffset(double off) {
    _offset = off;
}

//====================================  Set the nominal time offset in us
void 
DuoTone::time_chan::set_subsys(const string& ss) {
    _subsys = ss;
}

//=====================================  Set the trend channel name
void 
DuoTone::time_chan::setTrend(const std::string& tchan) {
    _trendChan = tchan;
}

//=============================================================================
//
//     Monitor class
//
//=============================================================================


//======================================  DuoTone  constructor
DuoTone::DuoTone(int argc, const char *argv[])
    : DatEnv(argc, argv),
      MonServer("DuoTone_Junk"),
      MaxFrame(999999999), 
      Epics(true),
      Stride(10.0),
      alarmThreshold(8.0),
      mTrendMinute("DuoToneTrend", Trend::kMinute, 720),
      mAlarm("DUOTONE")
{
    //----------------------------------  initialize defaults from environment
    string site   = env_to_str("LIGOSITE", "L?O");
    WebDir = env_to_str("DMTHTMLOUT");
    LogDir = env_to_str("HOME");
    LogDir += "/log";
    CaPutDir = env_to_str("EPICSBINDIR");

#ifdef DEBUG_DUOTONE
    duo_syntax();
#endif
    time_chan::chan_type type(time_chan::tDuotone);
    double f1(960), f2(961);
    double offset(0.0);
    for (int i=1 ; i<argc ; i++) {
	string argi = argv[i];
        if (argi == "-h" || argi == "--help") {
	    duo_syntax();
	    exit(0);
	}
	else if (argi == "--alarm") {
	    alarmThreshold = strtod(argv[++i], 0);
	}
	else if (argi == "--duotone") {
	    type = time_chan::tDuotone;
	}
        else if (argi == "-e") {
	    EpicsName = argv[++i];
	    Epics = true;
	}
        else if (argi == "-i") {
	    IFO = argv[++i];
	}
        else if (argi.substr(0,3) == "-cR") {
	    time_chan chan(argv[++i], time_chan::tRamp);
	    if (argi.size() > 3) chan.set_subsys(argi.substr(3));
	    mChannels.push_back(chan);
	}
        else if (argi.substr(0,3) == "-cS") {
	    time_chan chan(argv[++i], time_chan::tDuotone);
	    if (argi.size() > 3) chan.set_subsys(argi.substr(3));
	    chan.setF(f1, f2);
	    chan.setOffset(offset);
	    mChannels.push_back(chan);
	}
        else if (argi == "-f1") {
	    f1 = strtod(argv[++i], 0);
	}
        else if (argi == "-f2") {
	    f2 = strtod(argv[++i], 0);
	}
        else if (argi == "-l") {
	    LogDir = argv[++i];
	}
	else if (argi == "-n") {
	    MaxFrame = strtol(argv[++i], 0, 0);
	} 
	else if (argi == "--offset") {
	    offset = strtod(argv[++i], 0);
	} 
	else if (argi == "--stride") {
	    Stride = strtod(argv[++i], 0);
	}
	else if (argi == "--ramp") {
	    type = time_chan::tRamp;
	}
	else if (argi == "-t") {
	    trend = argv[++i];
	}
	else if (argi == "-w") {
	    WebDir = argv[++i];
	}
	else if (isDatEnvArg(argv[i])) {
	    i++;
	}
	else {
	    cerr << "Unrecognized command line argument: " << argi << endl;
	}
    }

    //----------------------------------  Infer channel name
    if (IFO.empty()) {
	if (!mChannels.empty()) {
	    IFO = mChannels[0].chan().substr(0,2);
	} else if (site == "LHO") {
	    IFO = "H1";
	} else if (site == "LLO") {
	    IFO = "L1";
	}
    }

    //----------------------------------  Infer default channel
    std::string def_chan;
    if (IFO == "H1" || IFO == "L1") {
	def_chan = IFO + ":CAL-PCALY_FPGA_DTONE_OUT_DQ";
	if (trend.empty()) trend = IFO + ":DMT-TIME_";
    } else {
	cout << "Unrecognized IFO name: " << IFO << endl;
	finish();
	return;
    }
    
    if (mChannels.empty()) {	
	time_chan chan(def_chan, type);
	chan.setF(f1, f2);
	chan.setOffset(offset);
	mChannels.push_back(chan);	
   }

    //----------------------------------  Set up dmt server and trender
    setServerName((string("DuoTone_DataServer_")+IFO).c_str());
    mTrendMinute.setIFO(IFO.c_str());
    mTrendMinute.setName((string("DuoTone_")+IFO).c_str());
    mTrendMinute.setFrameCount(1);   
    mTrendMinute.setType(Trend::kMinute);
    //mTrendMinute.setAutoUpdate(false);

    //-----------------------------------  Read all about it
    if (Debug()) {
	cout << "Duotone parameter settings: " << endl;
	cout << "    Epics communication is ON! " << EpicsName << endl;
	cout << "    Output trend channel:      " << trend     << endl;
	cout << "    IFO name:                  " << IFO       << endl;
	for (size_t i=0; i<mChannels.size(); i++) {
	    switch (mChannels[i].type()) {
	    case time_chan::tDuotone:
		cout << "    Duotone channel:           ";
		break;
	    case time_chan::tRamp:
		cout << "    Ramp channel:              ";
		break;
	    }
	    cout << mChannels[i].chan() << endl;
	}
	cout << "    Maximum number of frames:  " << MaxFrame  << endl;
	cout << "    Web output is going to:    " << WebDir    << endl;
	cout << "    Log output is going to:    " << LogDir    << endl;
    }

    getDacc().setStride(Stride);             // Set the time stride
    setStrideAlignment(Stride, 0.0);
    string TrendTitle = IFO + "-DuoTone-Minute-Trend";
    for (size_t i=0; i<mChannels.size(); i++) {
	getDacc().addChannel(mChannels[i].chan());
	string trendChan = trend + mChannels[i].subsys()
	                 + "_TimingDifference_us";
	mChannels[i].setTrend(trendChan);
	mTrendMinute.addChannel(trendChan.c_str());
	mTrendMinute.find(trendChan.c_str()).setUnits("microseconds");
	serveData(trendChan.c_str(), 
	    &(mTrendMinute.find(mChannels[i].trend()).refAvgSeries()),
	    TrendTitle.c_str());
    }
    trendChan_ERR = trend + "ErrorFlag";
    mTrendMinute.addChannel(trendChan_ERR.c_str());
    trendChan_QAF = trend + "SignalQualityFlag";
    mTrendMinute.addChannel(trendChan_QAF.c_str());

    serveData(trendChan_ERR.c_str(), 
	      &(mTrendMinute.find(trendChan_ERR.c_str()).refAvgSeries()), 
	      TrendTitle.c_str()); 
    serveData(trendChan_QAF.c_str(), 
	      &(mTrendMinute.find(trendChan_QAF.c_str()).refAvgSeries()), 
	      TrendTitle.c_str());
    //serveData(trend, 
    //          &(mTrendMinute.find(trend.c_str()).refAvgSeries()),
    //          "DUOTONE Minute Trend");
    //mTrendMinute.writeIndex();


    //S--------------------- Defining the alarms
    const char* badTiming="../monitor_reports/DuoTone/";
    AlarmData al1("DUOTONE", IFO+"-LSC_Timing_Is_OFF", Interval(120), 7, 
		  "Timing is bad", "");
    al1.setWebFile(badTiming);
    al1.jamFlags(AlarmData::kReTrigger);
    mAlarm.defineAlarm(al1);

    const char* NoSignal="../monitor_reports/DuoTone/";
    AlarmData al2("DUOTONE", IFO+"-DUOTONE_SIGNAL_IS_NOT_PRESENT",
	          Interval(120), 7, "No DUOTONE signal", "");
    al2.setWebFile(NoSignal);
    al2.jamFlags(AlarmData::kReTrigger);
    mAlarm.defineAlarm(al2);
    //E--------------------- Defining the alarms

    //B--------------------- Setting up EPICS stuff

    // Get environmental variable for CAPUT command...

    //E--------------------- Setting up EPICS stuff

}

//======================================  Duotone destructor
DuoTone::~DuoTone() 
{
    cout << "DuoTone is finished" << endl;
}

//======================================  Attention handler
void DuoTone::Attention(void) {
    MonServer::Attention();
}

//======================================  Process a single stride
void DuoTone::ProcessData(void) {                 // Decodes the time
    size_t nChan = mChannels.size();
    lcl_array<double> Result(nChan+2);
    
    signal(SIGTTOU, SIG_IGN);

    int ErrCode  = 0;
    int QualCode =0;
    Time GT = getDacc().getFillTime() + Stride*0.5;

    for (size_t i=0; i<nChan; ++i) {
	TSeries ts(*getDacc().refData(mChannels[i].chan()));
	ts.Convert(data_vect::getDataType());
	Interval dt = ts.getInterval();
	mChannels[i].params(ts);
	
	TSeries tMean;
	int nsec = int(dt);
	Time t0 = ts.getStartTime();
	for (int isec=0; isec<nsec; isec++) {
	    tMean += ts.extract(t0, Interval(1.0));
	    t0 += Interval(1.0);
	}
	double delta =  mChannels[i].calculateTime(ts);
	Result[i] = double(delta);
	double rms = mChannels[i].rms(ts);
	if (Debug() > 1) cout << mChannels[i].chan() << "  dT=" << delta
			      << " rms=" << rms << endl; 
	mTrendMinute.trendData(mChannels[i].trend(), GT, delta);
    } // End measurement

    //B---------------- Write minute trend
    mTrendMinute.trendData(trendChan_ERR.c_str(), GT, ErrCode);
    mTrendMinute.trendData(trendChan_QAF.c_str(), GT, QualCode);
    //E---------------- Write minute trend

#ifdef EPICS
    //B---------------- Send data to EPICS
    if (Epics) {
	// EPICS communication by the minute...
     	ostringstream Comm0;
	Comm0 << CaPutDir << "/caput " << ALIVE_Ch << " " << abs((int)Alive) 
	      << " >> /dev/null 2>&1 \n";
	sysrc = system(Comm0.str().c_str()); 
	Alive+=128;
	
	time(&TS);
	ostringstream Com00;
	Com00 << CaPutDir << "/caput " << TTAG_Ch
	      << " \" `date -u` \"   >> /dev/null 2>&1" << endl;
	sysrc = system(Com00.str().c_str());

	Alive+=128;
	ostringstream Comm;
	Comm << fixed << precision(3);
	Comm << CaPutDir << "/caput " << EpicsName << "KEEPALIVE "
	     << Alive << " >> /dev/null 2>&1 &";
	Comm << CaPutDir << "/caput " << EpicsName
	     << "TTAG \" `tconvert now` \" >> /dev/null 2>&1 &" << endl;
	Comm << CaPutDir << "/caput " << EpicsName << "LSC "
	     << Result[0] << " >> /dev/null 2>&1 &" << endl;
#if 0
	Comm << CaPutDir << "/caput " >> EpicsName >> "ACC "
	     << Result[4] << " >> /dev/null 2>&1 &" << endl;
	Comm << CaPutDir << "/caput " << EpicsName << "LSC_DUO "
	     << Result[0] << " >> /dev/null 2>&1 &" << endl;
	Comm << CaPutDir << "/caput " << EpicsName << "DAQ_RAMP "
	     << Result[1] << " >> /dev/null 2>&1 &" << endl;
	Comm << CaPutDir << "/caput " << EpicsName << "DAQ_DUO "
	     << Result[2] << " >> /dev/null 2>&1 &" << endl;
#endif
	Comm << CaPutDir << "/caput " << EpicsName << "ERR_BITS "
	     << ErrCode << " >> /dev/null 2>&1 &" << endl;
	Comm << CaPutDir << "/caput " << EpicsName << "QA_BITS "
	     << QualCode << " >> /dev/null 2>&1 &" << endl;
	sysrc = system(Comm.str().c_str());

#ifdef DEBUG_DUOTONE
	cout << "Sent to EPICS: " << Comm.str().c_str() << endl;
#endif

        sysrc = system(EpicsComm);
#ifdef DEBUG_DUOTONE
	cout << "Sent to EPICS: " << EpicsComm << endl;
#endif

	sprintf(EpicsComm, "%s/caput %sLSC %4.2g >> /dev/null 2>&1 \n", 
                CaPutDir, EpicsName, Result[3]);
	sysrc = system(EpicsComm);
#ifdef DEBUG_DUOTONE
	cout << "Sent to EPICS: " << EpicsComm << endl;
#endif


	// H1:DMT-TM_TTAG           (GPS Time of measurement)
	// H1:DMT-TM_KEEPALIVE      (keep alive channel, one per IFO)
	//
	// H1:DMT-TM_LSC            (GPS - LSC time difference, us)
	// H1:DMT-TM_ACC            (Time measurement accuracy (error bar), ns)
	// H1:DMT-TM_DAQ_RAMP       (Timing of ramp in adcu pem)
	// H1:DMT-TM_DAQ_DUO        (Timing of duotone in adcu pem)
	// H1:DMT-TM_LSC_DUO        (Timing of duotone in lsc)
	// H1:DMT-TM_ERR_BITS       (Bit encoded error data)
	// H1:DMT-TM_QA_BITS        (Bit encoded quality data)
	
	// ...end of epics communication...
#ifdef DEBUG_DUOTONE
	cout << "Did EPICS communication..." << endl;
#endif	
    }
	//E---------------- Send data to EPICS
#endif

    //B---------------- Set alarms if necessary
    lmsg::error_t rc = 0;
    AlarmHandle han;

    int lscChan = 0;
    if ( fabs(Result[lscChan]) > alarmThreshold) {
	ostringstream param;
	param << mChannels[lscChan].chan() << " is off by: "
	      << Result[lscChan] << " us " << ends;
	AlarmData al1("DUOTONE", IFO+"-LSC_Timing_Is_OFF", Interval(120), 7, 
		      "Timing is bad", param.str()); 
	rc = mAlarm.setAlarm(al1, han);
	if (rc) cerr << "[DuoTone] Error sending alarm: " << rc << endl;
#ifdef   DEBUG_DUOTONE
	cout << "ALARM: " << mChannels[lscChan].chan() << " is off by: "
	     << Result[lscChan] << " us " << endl;
#endif	
    }
    //E---------------- Set alarms if necessary
}
